#include "Socket.h"

#include "System.h"
#include "Exception.h"
#include "Network.h"

const int Socket::DEFAULT_NUM_PACKETS_PER_RECEIVE = 16;

Socket::Socket( Network* network, int port ){
	this->port = port;
	this->udpSocket = NULL;

	this->network = network;
}

Socket::~Socket(){
}

void Socket::open(){
	if( this->udpSocket != NULL ){
		System::error("(in Socket::open()): Socket is already open.");
		throw Exception("Socket is already open.", "Socket::open()");
	}

	this->udpSocket = SDLNet_UDP_Open(this->port);
	if( this->udpSocket == NULL ){
		System::error("(in Socket::open()): SDLNet_UDP_Open failed: %s. (Maybe you are behind a firewall?)", SDLNet_GetError());
		throw Exception("Couldn't open socket.", "Socket::open()");
	}
}

void Socket::close(){
	if( this->udpSocket == NULL ){
		System::error("(in Socket::close()): Socket is not open.");
		throw Exception("Socket is not open.", "Socket::close()");
	}
	
	SDLNet_UDP_Close( this->udpSocket );
	this->udpSocket = NULL;
}

bool Socket::isOpen(){
	return (this->udpSocket != NULL);
}


void Socket::send( Packet* packet ){
//	int numSent = SDLNet_UDP_Send( this->udpSocket, packet->udpPacket->channel, packet->udpPacket );
	int numSent = SDLNet_UDP_Send( this->udpSocket, -1, packet->udpPacket );
	if( numSent < 1 ){
		// retry once
		System::warn( "(in Socket::send()): SDLNet_UDP_Send to ip %X:%X failed: %s. Trying again."
			, packet->udpPacket->address.host, packet->udpPacket->address.port, SDLNet_GetError() );
		int secondTryNumSent = SDLNet_UDP_Send( this->udpSocket, -1, packet->udpPacket );

		if( secondTryNumSent < 1 ){ // second attempt failed, too => throw Exception
			System::error( "(in Socket::send()): SDLNet_UDP_Send to ip %X:%X failed: %s."
				, packet->udpPacket->address.host, packet->udpPacket->address.port, SDLNet_GetError() );
			throw Exception("Couldn't send packet.", "Socket::send()");
		}
	}

	this->network->notifyPacketSent( packet );
}

void Socket::receive( Packet::packetList_t& packetList, int maxPackets, int maxPacketSize ){
	UDPpacket** pv;
	
	pv = SDLNet_AllocPacketV(maxPackets, maxPacketSize);
	if( pv == NULL ){
		System::error("(in Socket::receive()): Couldnt' alloc packets (SDLNet_AllocPacketV: %s).", SDLNet_GetError());
		throw Exception("Couldn't alloc packets.", "Socket::receive()");
	}

	int numReceived = SDLNet_UDP_RecvV(this->udpSocket, pv);
	if( numReceived == -1 ){
		System::error("(in Socket::receive()): Couldn't receive packets (SDLNet_UDP_RecvV: %s).", SDLNet_GetError());
		throw Exception("Couldn't receive packets.", "Socket::receive()");
	}


	for( int i=0; i<numReceived; i++ ){
		if( pv[i]->len < 1 ){
			System::warn("(in Socket::receive()): Received packet with invalid length (%i). Ignoring.", pv[i]->len);
			continue;
		}
/*
		if( pv[i]->data[0] != Packet::FWP_PACKET_ID ){
			System::warn("(in Socket::receive()): Received packet with invalid id (%i). Ignoring.", pv[i]->data[0]);
			continue;
		}
*/
		Packet* p = Packet::wrapUdpPacket( pv[i] );
		packetList.push_back(p);

		this->network->notifyPacketReceived( p );
	}

	SDLNet_FreePacketV(pv);
}

void Socket::emptyIncomingPacketQueue(){
	int numLoops = 0;
	int numReceived = 0;

	do{
		UDPpacket** pv = SDLNet_AllocPacketV(64, 1);
		if( pv == NULL ){
			System::error("(in Socket::emptyIncomingPacketQueue()): Couldnt' alloc packets (SDLNet_AllocPacketV: %s).", SDLNet_GetError());
			throw Exception("Couldnt' alloc packets", "Socket::emptyIncomingPacketQueue()");
		}

		numReceived = SDLNet_UDP_RecvV(this->udpSocket, pv);
		if( numReceived == -1 ){
			System::error("(in Socket::emptyIncomingPacketQueue()): Couldn't receive packets (SDLNet_UDP_RecvV: %s).", SDLNet_GetError());
			throw Exception("Couldn't receive packets", "Socket::emptyIncomingPacketQueue()");
		}

		SDLNet_FreePacketV(pv);
		numLoops++;
	}while( numReceived > 0 && numLoops < 10 );

	if( numReceived > 0 ){
		System::warn("(in Socket::emptyIncomingPacketQueue()): There may be packets left in queue.");
	}
}
