#include "Sound.h"

#include "System.h"
#include "Exception.h"
#include "Console.h"
#include "Game.h"
#include "Camera.h"

#include <SDL.h>


#define DEG(x) ( (x)*(180.0f/3.1415f) )


Sound::Sound(System* system): SubSystem(system){
	this->enabled = false;

	this->chunkSize = 4096;
	this->numEffectChannels = 32;
	this->samplingRate = 44100;
	this->sampleVolume = 128;
	this->stereo = true;

	this->listener.position = FVector3::ZERO;
	this->listener.direction = FVector3::ZERO;
	this->listener.up = FVector3::ZERO;
	this->listener.velocity = FVector3::ZERO;

	this->samplesByName.clear();
}

Sound::~Sound(){
}

void Sound::initialize(){
	System::log("");
	System::log("*** Initializing Sound ***");
	System::log("");

//	System::log("Initializing FMOD-library...");
//	FSOUND_SetMinHardwareChannels(0);
//	FSOUND_Init(44100, 32, 0);

	this->cVars.sound_enabled->updateReferencedVariable();
	this->cVars.sound_numEffectChannels->updateReferencedVariable();

	if( !this->enabled ){
		System::log("Sound: sound.enabled is set to false. Skipping sound initialization.");
	}else{

		const SDL_version* link_version = Mix_Linked_Version();
		System::log("Sound: Running with SDL_mixer version %d.%d.%d.", link_version->major, link_version->minor, link_version->patch);

		// start SDL with audio support
		System::log("Sound: Starting SDL audio subsystem...");
		if( SDL_InitSubSystem( SDL_INIT_AUDIO ) == -1 ) {
			System::error("(in Sound::inititialize()): SDL_Init failed: %s.", SDL_GetError());
			throw Exception( "Couldn't initialize sound subsystem.", "Sound::initialize()" );
		}

		System::log("Sound: Opening audio device for 16 bit %s sound at %.2fkHz (chunkSize: %i)..."
		, this->stereo ? "stereo" : "mono", this->samplingRate/(float)1000, this->chunkSize );

		if( Mix_OpenAudio( this->samplingRate, MIX_DEFAULT_FORMAT, this->stereo ? 2 : 1, this->chunkSize) == -1 ){
			System::error("(in Sound::inititialize()): Mix_OpenAudio failed: %s.", Mix_GetError());
			throw Exception( "Couldn't initialize sound subsystem.", "Sound::initialize()" );
		}

		System::log("Sound: Allocating %i channels (%i internal and %i for effects)..."
			, Sound::FIRST_EFFECT_CHANNEL + this->numEffectChannels, Sound::FIRST_EFFECT_CHANNEL, this->numEffectChannels);
		Mix_AllocateChannels( this->numEffectChannels );
		Mix_Volume( -1, this->sampleVolume );

		Mix_GroupChannels( Sound::FIRST_EFFECT_CHANNEL, Sound::FIRST_EFFECT_CHANNEL + this->numEffectChannels, this->effectChannelsGroup );
	}

	this->initialized = true;

	System::log("Sound is ready.");
	System::log("");

}

void Sound::shutdown(){
	System::log("");
	System::log("=== Shutting down Sound ===");
	System::log("");

//	System::log("Closing FMOD-library...");
//	FSOUND_Close();

	if( !this->enabled ){
		System::log("Sound: sound.enabled is set to false. Skipping sound shutdown.");
	}else{
		// TODO: free samples!
		System::log("Sound: Closing audio channels...");
		Mix_CloseAudio();	// FIXME: Probleme unter linux, wenn andere programme sound abspielen!!
		System::log("Sound: Closing SDL audio subsystem...");
		SDL_QuitSubSystem(SDL_INIT_AUDIO);
	}

	this->initialized = false;

	System::log("Sound is down.");
	System::log("");

}
void Sound::registerCCmds( Console& console ){
}
void Sound::unregisterCCmds( Console& console ){
}
void Sound::registerCVars( Console& console ){
	this->cVars.sound_enabled = new CVarBool("sound.enabled", &this->enabled, false);
	this->cVars.sound_enabled->setFlags( CVar::FLAG_SYSTEM );
	this->cVars.sound_enabled->setChangeString( "(changes will take effect after you restart fwp)" );
	console.registerCVar(this->cVars.sound_enabled);

	this->cVars.sound_numEffectChannels = new CVarInt("sound.numEffectChannels", &this->numEffectChannels, false);
	this->cVars.sound_numEffectChannels->setValueRange( 1, 32 );
	this->cVars.sound_numEffectChannels->setFlags( CVar::FLAG_SYSTEM );
	this->cVars.sound_enabled->setChangeString( "(changes will take effect after you restart fwp)" );
	console.registerCVar(this->cVars.sound_numEffectChannels);

}
void Sound::unregisterCVars( Console& console ){
	console.unregisterCVar( this->cVars.sound_enabled );
	delete this->cVars.sound_enabled;

	console.unregisterCVar( this->cVars.sound_numEffectChannels );
	delete this->cVars.sound_numEffectChannels;
}



void Sound::update(){
	Camera* cam = this->system->getGame()->getCamera();

	this->setListenerAttributes( cam->position, cam->direction, cam->up, cam->velocity );
}


void Sound::setListenerAttributes( const FVector3& pos, const FVector3& dir, const FVector3& up, const FVector3& vel ){
	this->listener.position = pos;
	this->listener.direction = dir;
	this->listener.up = up;
	this->listener.velocity = vel;
}

Sound::sample_t* Sound::getSample( const string& name ){
	samplesByName_t::const_iterator c_iter = this->samplesByName.find( name );
	if( c_iter == this->samplesByName.end() ){
		//load
		sample_t* s = this->loadSample( this->system->getDataDir() + name );
		this->samplesByName.insert( samplesByName_t::value_type(name, s) );

		return s;
	}

	return c_iter->second;

}

void Sound::playSample( int channel, sample_t* sample, int numLoops, int numMillis ){
	Mix_PlayChannelTimed( channel, sample, numLoops, numMillis );
}

void Sound::playSample( const FVector3& pos, sample_t* sample, int numLoops, int numMillis ){
	int channel = this->acquireEffectChannel();
	if( channel == -1 ){ // no channel available
		return;
	}

	//do pos
	FVector3 toPos = pos - this->listener.position;
	FVector2 a = FVector2( this->listener.direction.x, this->listener.direction.z );
	FVector2 b = FVector2( toPos.x, toPos.z );
	a.normalize();
	b.normalize();

	double rad = acos( a.x*b.x + a.y*b.y );

	int angle;
	FVector3 c = toPos.crossProduct( this->listener.direction );
	if( this->listener.up.dotProduct( c ) > 0.0f ){
		angle = (int)( DEG(rad) );
	}else{
		angle = 360 - (int)( DEG(rad) );
	}

	int distance = (int)( toPos.length() * 0.5f );
//	distance = distance > 255 ? 255 : distance;
	if( distance > 255 ){ // too far away
		return;
	}

	if( !Mix_SetPosition(channel, angle, distance) ){
		//warn("(in sound::playEffect()): couldn't register pos-effect.\n\n");
		return;
	}

//	printf("HALLO! %i\n", distance);
	Mix_PlayChannelTimed( channel, sample, numLoops, numMillis );
}


int Sound::acquireEffectChannel(){
	int channel;
	channel = Mix_GroupAvailable( this->effectChannelsGroup );
	if( channel == -1 ){ // try again
//		channel = Mix_GroupAvailable( this->effectChannelsGroup );
//		if( channel == -1 ){
//			warn("(in aquireEffectChannel()): couldn't aquire ANY effect channel. sound.numEffectChannels set to 0?\n");
//		}
	}

	return channel;
}



Sound::sample_t* Sound::loadSample( const string& path ){
	if( !File::exists( path ) ){
		System::error("(in Sound::loadSample()): File '%s' not found.", path.c_str() );
		throw Exception( "Couldn't load sample.", "Sound::loadSample()" );
	}

	Mix_Chunk* ret = Mix_LoadWAV( path.c_str() );

	if( ret == NULL ){
		System::error( "(in Sound::loadSample()): Mix_LoadWAV() failed for '%s': %s.", path.c_str(), Mix_GetError() );
		throw Exception( "Couldn't load sample.", "Sound::loadSample()" );
	}

//	System::log("Sample '%s' loaded.", path.c_str());

	return ret;
}


/*
Sound::music_t* Sound::loadMusic( const string& path ){
	music_t* ret = FMUSIC_LoadSong( path.c_str() );
	if( ret == NULL ){
		throw Exception("Couldn't load file '" + path + "'.", "Sound::loadMusic()");
	}else{
		return ret;
	}
}

void Sound::freeMusic( music_t* music ){
	FMUSIC_FreeSong( music );
}

void Sound::playMusic( music_t* music ){
	FMUSIC_PlaySong( music );
}
void Sound::stopMusic( music_t* music ){
	FMUSIC_StopSong( music );
}
*/

bool Sound::isEnabled() const {
	return this->enabled;
}

