#include "System.h"

//#include "main.h"
#include "Exception.h"
#include "File.h"
#include "StringTokenizer.h"

#include <SDL.h>

#include <stdlib.h>
#include <stdarg.h>
#include <iostream>

#include "Logging.h"
#include "Console.h"
#include "Input.h"
#include "Graphics.h"
#include "Gui.h"
#include "Network.h"
#include "Sound.h"
#include "Game.h"

#include "MessagesMenu.h"
#include "MessageBoxMenu.h"

System* System::instance = NULL;

System& System::getInstanceRef(){
	return (*System::instance);
}
System* System::getInstance(){
	return System::instance;
}

void System::createInstance(){
	System::instance = new System();
}

void System::deleteInstance(){
	delete System::instance;
	System::instance = NULL;
}




System::System(){
	this->initialized = false;

#if FWP_PLATFORM == FWP_PLATFORM_WIN32
	this->platform = PLATFORM_WIN32;
#elif FWP_PLATFORM == FWP_PLATFORM_LINUX
	this->platform = PLATFORM_LINUX;
#else
	#error Unknown platform!!
#endif

	this->setPlatformString( FWP_PLATFORM_STRING );
	this->setVersionString( FWP_VERSION_STRING );
	this->setDataDir( FWP_DEFAULT_DATA_DIR );
	this->setConfigDir( FWP_DEFAULT_CONFIG_DIR );
	this->setLogDir( FWP_DEFAULT_LOG_DIR );

	this->shouldQuit = false;
	this->shouldStartGame = false;

	this->timer = NULL;
	this->random = new Random();

	this->logging = new Logging(this);
	this->console = new Console(this);
	this->graphics = new Graphics(this);
	this->input = new Input(this);
	this->sound = new Sound(this);
	this->gui = new Gui(this);
	this->network = new Network(this);
	this->game = new Game(this);

	this->commandLineConsoleStrings.clear();
}

System::~System(){
	if( logging != NULL ){
		delete logging;
	}
	if( console != NULL ){
		delete console;
	}
	if( graphics != NULL ){
		delete graphics;
	}
	if( input != NULL ){
		delete input;
	}
	if( sound != NULL ){
		delete sound;
	}
	if( gui != NULL ){
		delete gui;
	}
	if( network != NULL ){
		delete network;
	}
	if( game != NULL ){
		delete game;
	}

	delete random;
}

void System::initialize(){
	System::log("");
	System::log("***************************");
	System::log("*** Initializing System ***");
	System::log("***************************");
	System::log("");
	System::log("Platform: %s", FWP_PLATFORM_STRING);
	System::log("Version: %s", FWP_VERSION_STRING);
	System::log("ConfigDir: %s", this->configDir.c_str() );
	System::log("DataDir: %s", this->dataDir.c_str() );
	System::log("LogDir: %s", this->logDir.c_str() );
	System::log("Time: %s", Timer::getSystemTimeString().c_str() );

	System::log("Checking directories...");
	this->checkDirectories();

	// needed for SDL_net
	System::log("Initializing SDL...");
	SDL_Init(SDL_INIT_NOPARACHUTE);


	logging->initialize();
	console->initialize();
	this->registerCCmds();
	this->registerCVars();
	input->initialize();
	this->readSystemConfig();
	graphics->initialize();
	this->timer = new Timer();
	sound->initialize();
	gui->initialize();
	network->initialize();
	game->initialize();

	this->readAutoExecConfig();

	System::log("");
	System::log("System: Passing command line arguments to console...");
	for( vector<string>::const_iterator c_iter = this->commandLineConsoleStrings.begin(); c_iter != this->commandLineConsoleStrings.end(); c_iter++ ){
//		System::log("Passing [%s]...", c_iter->c_str());
		this->console->parse( *c_iter );
	}

	this->initialized = true;
	System::log("System is ready.");
	System::log("");
}

void System::shutdown(){
	System::log("");
	System::log("============================");
	System::log("=== Shutting down System ===");
	System::log("============================");
	System::log("");
	System::log("Time: %s", Timer::getSystemTimeString().c_str() );

	game->shutdown();
	network->shutdown();
	gui->shutdown();
	sound->shutdown();
	delete this->timer;
	graphics->shutdown();
	this->writeSystemConfig();
	input->shutdown();
	this->unregisterCVars();
	this->unregisterCCmds();
	console->shutdown();
	logging->shutdown();

	System::log("");
	System::log("System: Shutting down SDL...");
	SDL_Quit();

	this->initialized = false;
	System::log("System is down.");
	System::log("");
}

bool System::isInitialized() const {
	return this->initialized;
}




void System::registerCCmds(){
	System::log("");
	System::log("*** Registering CCmds ***");
	System::log("");

	System::log("System: Registering ccmds for 'system'...");
	cCmds.system_quit = new CCmdSystemQuit();
	console->registerCCmd( cCmds.system_quit );
	cCmds.system_debugccmd1 = new CCmdSystemDebugCCmd1();
	console->registerCCmd( cCmds.system_debugccmd1 );

	System::log("System: Registering ccmds for 'logging'...");
	logging->registerCCmds( *console );
	System::log("System: Registering ccmds for 'console'...");
	console->registerCCmds( *console );
	System::log("System: Registering ccmds for 'input'...");
	input->registerCCmds( *console );
	System::log("System: Registering ccmds for 'graphics'...");
	graphics->registerCCmds( *console );
	System::log("System: Registering ccmds for 'sound'...");
	sound->registerCCmds( *console );
	System::log("System: Registering ccmds for 'gui'...");
	gui->registerCCmds( *console );
	System::log("System: Registering ccmds for 'network'...");
	network->registerCCmds( *console );
	System::log("System: Registering ccmds for 'game'...");
	game->registerCCmds( *console );
}

void System::unregisterCCmds(){
	System::log("");
	System::log("=== Unregistering CCmds ===");
	System::log("");

	System::log("System: unregistering ccmds for 'system'...");
	console->unregisterCCmd( cCmds.system_quit );
	delete cCmds.system_quit;
	console->unregisterCCmd( cCmds.system_debugccmd1 );
	delete cCmds.system_debugccmd1;

	System::log("System: Unregistering ccmds for 'logging'...");
	logging->unregisterCCmds( *console );
	System::log("System: Unregistering ccmds for 'console'...");
	console->unregisterCCmds( *console );
	System::log("System: Unregistering ccmds for 'input'...");
	input->unregisterCCmds( *console );
	System::log("System: Unregistering ccmds for 'graphics'...");
	graphics->unregisterCCmds( *console );
	System::log("System: Unregistering ccmds for 'sound'...");
	sound->unregisterCCmds( *console );
	System::log("System: Unregistering ccmds for 'gui'...");
	gui->unregisterCCmds( *console );
	System::log("System: Unregistering ccmds for 'network'...");
	network->unregisterCCmds( *console );
	System::log("System: Unregistering ccmds for 'game'...");
	game->unregisterCCmds( *console );
}


void System::registerCVars(){
	System::log("");
	System::log("*** Registering CVars ***");
	System::log("");

	System::log("System: Registering cvars for 'logging'...");
	logging->registerCVars( *console );
	System::log("System: Registering cvars for 'console'...");
	console->registerCVars( *console );
	System::log("System: Registering cvars for 'graphics'...");
	graphics->registerCVars( *console );
	System::log("System: Registering cvars for 'input'...");
	input->registerCVars( *console );
	System::log("System: Registering cvars for 'sound'...");
	sound->registerCVars( *console );
	System::log("System: Registering cvars for 'gui'...");
	gui->registerCVars( *console );
	System::log("System: Registering cvars for 'network'...");
	network->registerCVars( *console );
	System::log("System: Registering cvars for 'game'...");
	game->registerCVars( *console );
}
void System::unregisterCVars(){
	System::log("");
	System::log("=== Unregistering CVars ===");
	System::log("");

	System::log("System: Unregistering cvars for 'logging'...");
	logging->unregisterCVars( *console );
	System::log("System: Unregistering cvars for 'console'...");
	console->unregisterCVars( *console );
	System::log("System: Unregistering cvars for 'graphics'...");
	graphics->unregisterCVars( *console );
	System::log("System: Unregistering cvars for 'input'...");
	input->unregisterCVars( *console );
	System::log("System: Unregistering cvars for 'sound'...");
	sound->unregisterCVars( *console );
	System::log("System: Unregistering cvars for 'gui'...");
	gui->unregisterCVars( *console );
	System::log("System: Unregistering cvars for 'network'...");
	network->unregisterCVars( *console );
	System::log("System: Unregistering cvars for 'game'...");
	game->unregisterCVars( *console );
}

void System::checkDirectories(){
	// check config dir
	if( File::directoryExists( this->configDir ) ){
		System::log("System: ConfigDir is ok.");
	}else{
		System::warn("(in System::checkDirectories()): ConfigDir '%s' not found. Trying to create it...", this->configDir.c_str());
		if( !File::makeDirectory( this->configDir ) ){
			System::error("(in System::checkDirectories()): Couldn't create configDir '%s'.", this->configDir.c_str());
			throw Exception("Couldn't create configDir.", "System::checkDirectories()");
		}else{
			System::log("System: ConfigDir created.");
		}
	}


	// check data dir
	if( File::directoryExists( this->dataDir ) ){
		System::log("System: DataDir is ok.");
	}else{
		System::error("(in System::checkDirectories()): DataDir '%s' not found.", this->dataDir.c_str());
		throw Exception("Couldn't find dataDir.", "System::checkDirectories()");
	}

	// check log dir
	if( File::directoryExists( this->logDir ) ){
		System::log("System: LogDir is ok.");
	}else{
		System::warn("(in System::checkDirectories()): LogDir '%s' not found. Trying to create it...", this->logDir.c_str() );
		if( !File::makeDirectory( this->logDir ) ){
			System::error("(in System::checkDirectories()): Couldn't create logDir '%s'.", this->logDir.c_str());
			throw Exception("Couldn't create logDir.", "System::checkDirectories()");
		}else{
			System::log("System: LogDir created.");
		}
	}
}

void System::readSystemConfig(){

	System::log("");
	System::log("*** Reading fwp.cfg ***");
	System::log("");

	File systemConfigFile(configDir + "fwp.cfg", ios::in);
	if( !systemConfigFile.isOpen() ){
		System::warn("(in System::readSystemConfig()): Couldn't open '%s'. Using default values.", systemConfigFile.getPath().c_str() );
		return;
	}

	string line;
	while( systemConfigFile.readLine(line, "#") ){
		console->parse( line );
	}
}

void System::writeSystemConfig(){
//	int i;

	System::log("");
	System::log("=== Writing fwp.cfg ===");
	System::log("");

	File systemConfigFile(configDir + "fwp.cfg", ios::out);
	if( !systemConfigFile.isOpen() ){
		System::warn("(in System::writeSystemConfig()): Couldn't open '%s'. System configuration not written!", systemConfigFile.getPath().c_str() );
		return;
	}

	systemConfigFile.writeLine("# FWP system configuration - generated by FWP");
	systemConfigFile.writeLine("# Note: FWP will overwrite this file! Place your personal console-scripts, keybindings, etc in 'autoexec.config'.");
	systemConfigFile.writeLine("");
	systemConfigFile.writeLine("# CVARS");
	this->console->writeCVarSettingsToFile( systemConfigFile );

	systemConfigFile.writeLine("");
	systemConfigFile.writeLine("# BINDINGS");
	systemConfigFile.writeLine("");
	this->input->writeBindingsToFile( systemConfigFile );

	systemConfigFile.close();
	System::log("Wrote fwp.cfg.");
}

void System::readAutoExecConfig(){

	File autoExecConfigFile(configDir + "autoexec.cfg", ios::in);

	System::log( "System: Checking for '%s'...", autoExecConfigFile.getPath().c_str() );
	if( !autoExecConfigFile.isOpen() ){
		System::log( "System: '%s' not found.", autoExecConfigFile.getPath().c_str() );
		return;
	}

	System::log( "System: Executing '%s'...", autoExecConfigFile.getPath().c_str() );

	string line;
	while( autoExecConfigFile.readLine(line, "#") ){
		console->parse( line );
	}
}




void System::log(const char* formatString, ...){
	char toStringBuffer[1024];

	va_list ap;
	va_start (ap, formatString);
	vsprintf(toStringBuffer, formatString, ap);
	va_end (ap);

	string str( toStringBuffer );

	cout << str << endl;

	if( instance != NULL ){
		if( instance->logging != NULL && instance->logging->isInitialized() ){
			instance->logging->log( str );
		}

		if( instance->console != NULL && instance->console->isInitialized() ){
			instance->console->queuePrintString( str );
		}

		if( instance->gui != NULL && instance->gui->isInitialized()
			&& instance->gui->getMessagesMenu() != NULL && instance->gui->getMessagesMenu()->isVisible() ){

			instance->gui->getMessagesMenu()->addMessage( str );
		}
	}

}
void System::log(const FColor& color, const char* formatString, ...){
	char toStringBuffer[1024];

	va_list ap;
	va_start (ap, formatString);
	vsprintf(toStringBuffer, formatString, ap);
	va_end (ap);

	string str( toStringBuffer );

	cout << str << endl;

	if( instance != NULL ){
		if( instance->logging != NULL && instance->logging->isInitialized() ){
			instance->logging->log( str );
		}

		if( instance->console != NULL && instance->console->isInitialized() ){
			instance->console->queuePrintString( str, color );
		}

		if( instance->gui != NULL && instance->gui->isInitialized()
			&& instance->gui->getMessagesMenu() != NULL && instance->gui->getMessagesMenu()->isVisible() ){

			instance->gui->getMessagesMenu()->addMessage( str, color );
		}
	}

}

void System::warn(const char* formatString, ...){
	char toStringBuffer[1024];

	va_list ap;
	va_start (ap, formatString);
	vsprintf(toStringBuffer, formatString, ap);
	va_end (ap);

	string str = string("[WARNING]: ") + string( toStringBuffer );

	cerr << str << endl;

	if( instance != NULL ){
		if( instance->logging != NULL && instance->logging->isInitialized() ){
			instance->logging->log( str );
		}

		if( instance->console != NULL && instance->console->isInitialized() ){
			instance->console->queuePrintString( str, FColor::YELLOW );
		}

		if( instance->gui != NULL && instance->gui->isInitialized()
			&& instance->gui->getMessagesMenu() != NULL && instance->gui->getMessagesMenu()->isVisible() ){

			instance->gui->getMessagesMenu()->addMessage( str, FColor::YELLOW );
		}
	}

}

void System::error(const char* formatString, ...){
	char toStringBuffer[1024];

	va_list ap;
	va_start (ap, formatString);
	vsprintf(toStringBuffer, formatString, ap);
	va_end (ap);

	string str = string("[ERROR]: ") + string( toStringBuffer );

	cerr << str << endl;

	if( instance != NULL ){
		if( instance->logging != NULL && instance->logging->isInitialized() ){
			instance->logging->log( str );
		}

		if( instance->console != NULL && instance->console->isInitialized() ){
			instance->console->queuePrintString( str, FColor::RED );
		}

		if( instance->gui != NULL && instance->gui->isInitialized()
			&& instance->gui->getMessagesMenu() != NULL && instance->gui->getMessagesMenu()->isVisible() ){

			instance->gui->getMessagesMenu()->addMessage( str, FColor::RED );
		}
	}
}


string System::debugStrings[10] = {"dragonfly", "desertfortress", "", "", "", "", "", "", ""};
void System::parseCommandLineArguments( const string& arguments ){
	System::log("Parsing command-line arguments...");

//	System::log("arguments: %s", arguments.c_str());

	StringTokenizer t1(" \t", "\"");
	const vector<string>& argumentTokens = t1.tokenize( arguments );

	vector<string>::const_iterator c_iter;
	for(c_iter = argumentTokens.begin(); c_iter != argumentTokens.end(); c_iter++){
		if( c_iter->length() < 1 ){
			continue;
		}

//		System::log( "Processing argument [%s]...", c_iter->c_str() );

		if( c_iter->length() > 1 && (*c_iter)[0] == '-' ){	// command-line param
			StringTokenizer t2("=", "\"");
			const vector<string>& tokens = t2.tokenize( *(c_iter) );

			if( tokens.size() != 2 ){
				System::warn("(in System::parseCommandLineParameters()): Unknown parameter format: '%s'. Must be: '-param=value'!", c_iter->c_str() );
				continue;
			}

			if( tokens[0] == "-dataDir" ){
				this->setDataDir( tokens[1] );
			}else if( tokens[0] == "-logDir" ){
				this->setLogDir( tokens[1] );
			}else if( tokens[0] == "-configDir" ){
				this->setConfigDir( tokens[1] );

			}else{
				System::warn("(in System::parseCommandLineParameters()): Unknown parameter: '%s'. Ignoring.", tokens[0].c_str() );
			}
		}else if( c_iter->length() > 1 && (*c_iter)[0] == '+' ){	// console param
			string str = c_iter->substr(1, c_iter->length()-1);
			this->commandLineConsoleStrings.push_back(str);
		}else{
			System::warn("(in System::parseCommandLineParameters()): Unknown parameter type: '%s'. Must be: '-param=value' or '+consolecmd'!", c_iter->c_str() );
		}
	}
}





void System::mainLoop(){
	do{
		if( this->shouldStartGame ){ // start a game
			this->setShouldStartGame( false );

			try{
			try{
			try{
			try{
				this->createGame();
				this->game->mainLoop();
				this->destroyGame();
			}catch( std::exception& e ){ throw Exception(e); }
			}catch( Ogre::Exception& e ){ throw Exception(e); }
			}catch( CEGUI::Exception& e ){ throw Exception(e); }
			}catch( Exception& e ){
				System::warn("(in System::mainLoop()): An exception has been caught. Description: %s", e.getDescription().c_str() );

				System::log("System: Trying to recover system...");
				this->game->cleanup();
				this->network->onGameDestruction();
				this->gui->onGameDestruction();
				this->gui->hideLoadingScreen(); // to be sure it is hidden
				System::log("System: Recovery done.");

				this->gui->getMessageBoxMenu()->show("An Exception has occurred",
					"An Exception has been caught in System::mainLoop(). Details:\n"
					+ e.getFullInfo()
					+ "\n(The console may provide more information.)", "Ok");
			}

		}else{	// use dummy scene as background for menus...
			this->gui->setupBackgroundScene();
//			System::log("Calling Gui::backgroundSceneMainLoop()...");
			this->gui->backgroundSceneMainLoop();
			this->gui->cleanupBackgroundScene();
		}
	}while( !shouldQuit );
}


void System::startGame(){
	this->setShouldStartGame(true);

	if( this->game->isRunning() ){
		this->game->setShouldBreakMainLoop(true);
	}else{
		this->gui->setShouldBreakBackgroundSceneMainLoop(true);
	}
}
void System::endGame(){
	this->setShouldStartGame(false);

	if( this->game->isRunning() ){
		this->game->setShouldBreakMainLoop(true);
	}else{
		this->gui->setShouldBreakBackgroundSceneMainLoop(true);
	}

}

void System::quit(){
	this->shouldQuit = true;

	if( this->game->isRunning() ){
		this->game->setShouldBreakMainLoop(true);
	}else{
		this->gui->setShouldBreakBackgroundSceneMainLoop(true);
	}
}









void System::createGame(){
	System::log("");
	System::log("*** Creating game ***");
	System::log("");

	// update gui
	this->gui->onGameCreation();
//	this->gui->hide(); // FIXME: gui is hidden here because otherwise it would be rendered in front of loading screen overlay.
	this->gui->showLoadingScreen();

	// init network
	this->network->onGameCreation();

	this->game->setup();

	this->gui->hideLoadingScreen();

	// game->mainLoop is called right after return!
	System::log("System::createGame finished. Calling Game::mainLoop()...");
}

void System::destroyGame(){
	System::log("");
	System::log("=== Destroying game ===");
	System::log("");

	this->game->cleanup();

	this->network->onGameDestruction();

	// update gui
	this->gui->onGameDestruction();
}









Timer* System::getTimer() const{
	return this->timer;
}
Random* System::getRandom() const{
	return this->random;
}

Logging* System::getLogging() const{
	return this->logging;
}
Console* System::getConsole() const{
	return this->console;
}
Console& System::getConsoleRef(){
	return (*this->console);
}
Graphics* System::getGraphics() const{
	return this->graphics;
}
Gui* System::getGui() const{
	return this->gui;
}
Input* System::getInput() const{
	return this->input;
}
Game* System::getGame() const{
	return this->game;
}
Sound* System::getSound() const{
	return this->sound;
}
Network* System::getNetwork() const{
	return this->network;
}


System::platforms_e System::getPlatform() const {
	return this->platform;
}
void System::setPlatformString(const string& newPlatformString){
	this->platformString = newPlatformString;
}
const string& System::getPlatformString() const {
	return this->platformString;
}

void System::setVersionString(const string& newVersionString){
	this->versionString = newVersionString;
}
const string& System::getVersionString() const {
	return this->versionString;
}

void System::setConfigDir(const string& newConfigDir){
	this->configDir = File::appendFileSeparator( newConfigDir );
}
const string& System::getConfigDir() const {
	return this->configDir;
}

void System::setDataDir(const string& newDataDir){
	this->dataDir = File::appendFileSeparator( newDataDir );
}
const string& System::getDataDir() const {
	return this->dataDir;
}

void System::setLogDir(const string& newLogDir){
	this->logDir = File::appendFileSeparator( newLogDir );
}
const string& System::getLogDir() const {
	return this->logDir;
}


void System::setShouldStartGame(bool newShouldStartGame){
	this->shouldStartGame = newShouldStartGame;
}
bool System::getShouldStartGame() const {
	return this->shouldStartGame;
}

void System::setShouldQuit(bool newShouldQuit){
	this->shouldQuit = newShouldQuit;
}
bool System::getShouldQuit() const {
	return this->shouldQuit;
}
