#include "TeleportationArea.h"

#include "System.h"
#include "Exception.h"
#include "XmlUtils.h"
#include "Game.h"
#include "Vehicle.h"
#include "Graphics.h"
#include "Effect.h"

const unsigned long TeleportationArea::TELEPORTATION_INTERVAL = 500;

TeleportationArea::TeleportationArea(): Area( Area::AREA_TYPE_TELEPORTATION ) {
	this->teamFlags = 0;
	this->targetLocation = FVector3::ZERO;
	this->lastTeleportationMillis = 0;

	this->sceneNode = NULL;
}

TeleportationArea::~TeleportationArea(){
}


Area* TeleportationArea::clone() const {
	TeleportationArea* ret = new TeleportationArea();

	*ret = *this;

	return ret;
}


void TeleportationArea::attach( Game* game ){

	if( this->game != NULL ){
		throw Exception("TeleportationArea is already attached to a game.", "TeleportationArea::attach()");
	}

	Ogre::SceneManager* sm = game->getSystem()->getGraphics()->getSceneManager();
	Ogre::SceneNode* rootNode = sm->getRootSceneNode();

	this->sceneNode = rootNode->createChildSceneNode( this->name );
	this->sceneNode->setPosition( this->position.toOgreVector3() );
	this->sceneNode->setOrientation( this->orientation.toOgreQuaternion() );
//	this->ogreSceneNode->setScale( this->edgeLengths.toOgreVector3() );

	// attach core billboard
	Ogre::BillboardSet* bs = sm->createBillboardSet( this->name + "/CoreBillboard" );
	bs->setBillboardType(Ogre::BBT_POINT);
	bs->setBillboardOrigin(Ogre::BBO_CENTER);
	bs->setDefaultWidth( this->edgeLengths.length() );
	bs->setDefaultHeight( this->edgeLengths.length() );
	bs->setMaterialName("are/teleportation_area_core");
	bs->createBillboard( Ogre::Vector3::ZERO );
	this->sceneNode->attachObject( bs );

	this->lastTeleportationMillis = game->getSystem()->getTimer()->getMilliseconds();

	this->game = game;
}

void TeleportationArea::detach(){
	if( this->game == NULL ){
		throw Exception("TeleportationArea is not attached to a game.", "TeleportationArea::detach()");
	}

	Ogre::SceneManager* sm = this->game->getSystem()->getGraphics()->getSceneManager();
	this->sceneNode->getParentSceneNode()->removeAndDestroyChild( this->sceneNode->getName() );
	this->sceneNode = NULL;

	this->game = NULL;
}


void TeleportationArea::fromXmlElement( TiXmlElement* xmlElement ){
	// name
	const char* tmp = xmlElement->Attribute("name");
	if( tmp != NULL ){
		this->name = string( tmp );
	}else{
		throw Exception( "TeleportationArea element has no name attribute.", "TeleportationArea::fromXmlElement()" );
	}
	if( this->name.length() > 32 ){
		System::error( "(in TeleportationArea::fromXmlElement()): TeleportationArea name '%s' is longer than 32 characters.", this->name.c_str() );
		throw Exception( "XmlElement is not a valid teleportation area description.", "TeleportationArea::fromXmlElement()" );
	}

	// teamFlags
	tmp = xmlElement->Attribute("teamFlags");
	if( tmp != NULL ){
		string flags = string( tmp );
		if( flags.find("s") != string::npos ){
			this->teamFlags |= Game::TEAM_FLAG_SPECTATORS;
		}
		if( flags.find("p") != string::npos ){
			this->teamFlags |= Game::TEAM_FLAG_PLAYERS;
		}
		if( flags.find("r") != string::npos ){
			this->teamFlags |= Game::TEAM_FLAG_RED;
		}
		if( flags.find("b") != string::npos ){
			this->teamFlags |= Game::TEAM_FLAG_BLUE;
		}
	}else{
		throw Exception( "TeleportationArea element has no teamFlags attribute.", "TeleportationArea::fromXmlElement()" );
	}



	for( TiXmlElement* elt = xmlElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );
		
		if(value == "edgeLengths"){
			this->edgeLengths = XmlUtils::readFVector3( elt );
		}else if(value == "targetLocation"){
			this->parseTargetLocation(elt);

		}else if(value == "position"){
			this->position = XmlUtils::readFVector3( elt );
		}else if(value == "orientation"){
			this->orientation = XmlUtils::readFQuaternion( elt );

		}else{
			System::warn("(in TeleportationArea::fromXmlElement()): Unknown child element: '%s' (row: %i). Ignoring.", value.c_str(), elt->Row() );
		}
	}
}

TiXmlElement TeleportationArea::toXmlElement() const {
	return TiXmlElement("dummy");
}
void TeleportationArea::calculateChecksum(){
	this->checksum = 0;
}




bool TeleportationArea::canAffectVehicle( Vehicle* vehicle ) const {
	unsigned long currentMillis = this->game->getSystem()->getTimer()->getMilliseconds();

	if( this->lastTeleportationMillis + TeleportationArea::TELEPORTATION_INTERVAL > currentMillis ){
		return false;
	}

	FVector3 doubleDiff = (vehicle->position - this->position) * 2.0f;
	return ( fabs(doubleDiff.x) < this->edgeLengths.x 
			&& fabs(doubleDiff.y) < this->edgeLengths.y 
			&& fabs(doubleDiff.z) < this->edgeLengths.z );
}

void TeleportationArea::affectVehicle( Vehicle* vehicle ){
	unsigned long currentMillis = this->game->getSystem()->getTimer()->getMilliseconds();

	Effect* e = this->game->spawnEffect( "vehicle_teleportation", NULL );	// vanish
	e->position = vehicle->position;
	e->orientation = vehicle->orientation;

	vehicle->position = this->targetLocation;
	this->game->spawnEffect( "vehicle_teleportation", vehicle );	// show up

	this->lastTeleportationMillis = currentMillis;
}




void TeleportationArea::parseTargetLocation(TiXmlElement* targetLocationElement){
	this->targetLocation = XmlUtils::readFVector3( targetLocationElement );
	FVector3 doubleDiff = (this->targetLocation - this->position) * 2.0f;
	if( fabs(doubleDiff.x) < this->edgeLengths.x 
		&& fabs(doubleDiff.y) < this->edgeLengths.y 
		&& fabs(doubleDiff.z) < this->edgeLengths.z ){
		throw Exception("TargetLocation must not lie inside teleportation area.", "TeleportationArea::parseTargetArea()");
	}
}
