#include "Vehicle.h"

#include "Exception.h"
#include "System.h"
#include "XmlUtils.h"
#include "Effect.h"
#include "Client.h"
#include "Game.h"
#include "Graphics.h"
#include "EffectFactory.h"
#include "Weapon.h"
#include "Camera.h"
#include "Network.h"
#include "Geometry.h"

Vehicle::Vehicle( Vehicle::vehicleTypes_e vehicleType ): GameObject(GameObject::GAME_OBJECT_TYPE_VEHICLE) {
	this->vehicleType = vehicleType;

	this->info.description = "<no description available>";

	this->graphicalProperties.mainUnitMesh.clear();
	this->graphicalProperties.propulsionUnitMesh.clear();
	this->graphicalProperties.propulsionEffect.clear();
	this->graphicalProperties.boosterEffect.clear();
	this->graphicalProperties.explosionEffect.clear();

	this->graphicalProperties.firstPersonViewPoint = FVector3::ZERO;
	this->graphicalProperties.chaseViewPoint = FVector3::ZERO;

//	this->physicalProperties.surfaceType = SURFACE_TYPE_METAL;
	this->physicalProperties.affectedByGravity = true;
	this->physicalProperties.bouncyness = 0.0f;
	this->physicalProperties.friction = 0.0f;
	this->physicalProperties.propulsiveForce = 0.0f;
	this->physicalProperties.propulsiveTorque = 0.0f;
	this->physicalProperties.maxLinearVelocity = 0.0f;
	this->physicalProperties.maxAngularVelocity = 0.0f;
	this->physicalProperties.boosterForce = 0.0f;
//	this->physicalProperties.hoverForce = 0.0f;
	this->physicalProperties.hoverHeight = 0.0f;
	this->physicalProperties.airControlModifier = 0.0f;
	this->physicalProperties.minMainUnitYaw = 0.0f;
	this->physicalProperties.maxMainUnitYaw = 0.0f;
	this->physicalProperties.minMainUnitPitch = 0.0f;
	this->physicalProperties.maxMainUnitPitch = 0.0f;
	this->physicalProperties.healthCapacity = 0.0f;
	this->physicalProperties.energyCapacity = 0.0f;

	this->physicalProperties.hoverPoints.clear();
	this->physicalProperties.shape.clear();
	dMassSetZero( &this->physicalProperties.mass );
//	this->physicalProperties.body = 0;

	this->physicalProperties.weaponSlots.clear();
	this->physicalProperties.debris.clear();

	this->health = 0.0f;
	this->energy = 0.0f;
	this->boostersActive = false;
	this->body = 0;

	this->mainUnitSceneNode = NULL;
	this->mainUnitEntity = NULL;
	this->propulsionUnitSceneNode = NULL;
	this->propulsionUnitEntity = NULL;
	this->propulsionEffect = NULL;
	this->boosterEffect = NULL;

	this->propulsiveForce = FVector3::ZERO;
	this->propulsiveTorque = FVector3::ZERO;
	this->boostersActive = false;
	this->recoveryActive = false;

	this->client = NULL;
	for( int i=0; i<MAX_WEAPONS; i++ ){
		this->weapons[i] = NULL;
	}

	this->attackerClientId = -1;
}

Vehicle::~Vehicle(){
}








void Vehicle::attach( Game* game, Client* client ){
	if( this->game != NULL ){
		throw Exception("Vehicle is already attached to a game.", "Vehicle::attach()");
	}

	this->position = client->state.position;
	this->orientation = client->state.orientation;

	Ogre::ParticleSystemManager& psm = Ogre::ParticleSystemManager::getSingleton();
	Ogre::SceneManager* sm = game->getSystem()->getGraphics()->getSceneManager();
	Ogre::SceneNode* rootNode = sm->getRootSceneNode();

	string fullName = client->properties.name + ":" + this->name;
	this->mainUnitSceneNode = rootNode->createChildSceneNode( fullName + "/MainUnit" );
	this->mainUnitSceneNode->setPosition( this->position.toOgreVector3() );
	this->mainUnitSceneNode->setOrientation( this->orientation.toOgreQuaternion() );
	if( !this->graphicalProperties.mainUnitMesh.empty() ){
		this->mainUnitEntity = sm->createEntity( fullName + "/MainUnit", this->graphicalProperties.mainUnitMesh );
		this->mainUnitEntity->setCastShadows( true );
		this->mainUnitSceneNode->attachObject( this->mainUnitEntity );
	}

	this->propulsionUnitSceneNode = rootNode->createChildSceneNode( fullName + "/PropulsionUnit" );
	this->propulsionUnitSceneNode->setPosition( this->position.toOgreVector3() );
	this->propulsionUnitSceneNode->setOrientation( this->orientation.toOgreQuaternion() );
	if( !this->graphicalProperties.propulsionUnitMesh.empty() ){
		this->propulsionUnitEntity = sm->createEntity( fullName + "/PropulsionUnit", this->graphicalProperties.propulsionUnitMesh);
		this->propulsionUnitEntity->setCastShadows( true );
		this->propulsionUnitSceneNode->attachObject( this->propulsionUnitEntity );
	}
	if( game->isRunningTeamMode() ){
		this->setTeamMaterial( client->info.team );
	}


	if( !this->graphicalProperties.propulsionEffect.empty() ){
		this->propulsionEffect = game->getEffectFactory()->createEffect( this->graphicalProperties.propulsionEffect );
		this->propulsionEffect->attach( game, this->propulsionUnitSceneNode );
	}
	if( !this->graphicalProperties.boosterEffect.empty() ){
		this->boosterEffect = game->getEffectFactory()->createEffect( this->graphicalProperties.boosterEffect );
	}



	Physics* physics = game->getPhysics();
	this->physicalProperties.shape.createGeoms( this );

	this->body = dBodyCreate( physics->getWorld() );
	dBodySetMass( this->body, &this->physicalProperties.mass );
	dBodySetPosition( this->body, this->position.x, this->position.y, this->position.z );
	dVector4 tmp;
	this->orientation.toOdeVector4(tmp);
	dBodySetQuaternion( this->body, tmp );
	dBodySetGravityMode( this->body, this->physicalProperties.affectedByGravity );
	dBodySetAutoDisableFlag( this->body, false );
	dBodySetData( this->body, this );
	this->physicalProperties.shape.attachGeomsToBody( this->body );
	this->physicalProperties.shape.addGeomsToSpace( physics->getDynamicGeometrySpace() );

	this->client = client;
	this->game = game;
}

void Vehicle::detach(){
	if( this->game == NULL ){
		throw Exception("Vehicle is not attached to a game.", "Vehicle::detach()");
	}

	Physics* physics = this->game->getPhysics();
	this->physicalProperties.shape.detachGeomsFromBody( );
	dBodyDestroy( this->body );
	this->physicalProperties.shape.removeGeomsFromSpace( physics->getDynamicGeometrySpace() );
	this->physicalProperties.shape.destroyGeoms();

	if( this->propulsionEffect != NULL ){
		this->propulsionEffect->detach();
		this->game->getEffectFactory()->destroyEffect( this->propulsionEffect );
		this->propulsionEffect = NULL;
	}
	if( this->boosterEffect != NULL ){
		if( this->boosterEffect->isAttached() ){
			this->boosterEffect->detach();
		}
		this->game->getEffectFactory()->destroyEffect( this->boosterEffect );
		this->boosterEffect = NULL;
	}

	Ogre::SceneManager* sm = this->game->getSystem()->getGraphics()->getSceneManager();

	if( this->mainUnitEntity ){
		sm->destroyEntity( this->mainUnitEntity );
	}
	this->mainUnitSceneNode->getParentSceneNode()->removeAndDestroyChild( this->mainUnitSceneNode->getName() );
	this->mainUnitSceneNode = NULL;

	if( this->propulsionUnitEntity ){
		sm->destroyEntity( this->propulsionUnitEntity );
	}
	this->propulsionUnitSceneNode->getParentSceneNode()->removeAndDestroyChild( this->propulsionUnitSceneNode->getName() );
	this->propulsionUnitSceneNode = NULL;


	this->game = NULL;
}

void Vehicle::update( float deltaT ){
	for( int i=0; i<MAX_WEAPONS; i++ ){
		if( this->weapons[i] != NULL ){
			this->weapons[i]->update( deltaT );
		}
	}
}



void Vehicle::setupCamera(Camera* camera){
	camera->position = this->mainUnitSceneNode->_getDerivedPosition();
	const Ogre::Quaternion& ori = this->mainUnitSceneNode->_getDerivedOrientation();
	camera->orientation = ori;
	camera->direction = -ori.zAxis();
	camera->up = ori.yAxis();
	camera->right = ori.xAxis();

	camera->angles[0] = ori.getYaw().valueRadians();
	camera->angles[1] = ori.getPitch().valueRadians();
	camera->angles[2] = ori.getRoll().valueRadians();

	if( camera->getMode() == Camera::MODE_FIRST_PERSON ){
		Ogre::Matrix3 m;
		ori.ToRotationMatrix( m );
		camera->position += FVector3( m * this->graphicalProperties.firstPersonViewPoint.toOgreVector3() );
	}else if( camera->getMode() == Camera::MODE_CHASE ){
		Ogre::Matrix3 m;
		ori.ToRotationMatrix( m );
		camera->position += FVector3( m * this->graphicalProperties.chaseViewPoint.toOgreVector3() );
	}

	if( camera->getMode() == Camera::MODE_FIRST_PERSON ){
		this->mainUnitSceneNode->setVisible(false);
//		this->mainUnitEntity->setVisible(false);
	}else{
		this->mainUnitSceneNode->setVisible(true);
	}
}

void Vehicle::processInput(Input::actionMap_t& actionMap, Input::mouseState_t& mouseState){

}





void Vehicle::fromXmlElement( TiXmlElement* xmlElement ){
	// name
	const char* tmp = xmlElement->Attribute("name");
	if( tmp != NULL ){
		this->name = string( tmp );
	}else{
		throw Exception( "vehicle element has no name attribute.", "Vehicle::fromXmlElement()" );
	}
	if( this->name.length() > 32 ){
		System::error( "(in Vehicle::fromXmlElement()): Vehicle name '%s' is longer than 32 characters.", this->name.c_str() );
		throw Exception( "XmlElement is not a valid vehicle description.", "Vehicle::fromXmlElement()" );
	}

	for( TiXmlElement* elt = xmlElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );

		if(value == "info"){
			this->parseInfo(elt);
		}else if(value == "graphicalProperties"){
			this->parseGraphicalProperties(elt);
		}else if(value == "physicalProperties"){
			this->parsePhysicalProperties(elt);
		}else if(value == "acousticalProperties"){
			this->parseAcousticalProperties(elt);

		}else{
			System::warn("(in Vehicle::fromXmlElement()): Unknown child element: '%s' (row: %i). Ignoring.", value.c_str(), elt->Row() );
		}
	}

}

TiXmlElement Vehicle::toXmlElement() const {
	return TiXmlElement("dummy");
}


void Vehicle::parseInfo(TiXmlElement* infoElement){

	TiXmlElement* elt = infoElement->FirstChildElement("description");
	if( elt != NULL ){
		this->parseDescription(elt);
	}

}

void Vehicle::parseDescription(TiXmlElement* descriptionElement){

	TiXmlNode* n = descriptionElement->FirstChild();
	if( n != NULL && n->ToText() != NULL ){
		this->info.description = string( n->ToText()->Value() );
	}
}


void Vehicle::parseGraphicalProperties( TiXmlElement* graphicalPropertiesElement ){
	XmlUtils::readStringProperty(graphicalPropertiesElement, "mainUnitMesh", this->graphicalProperties.mainUnitMesh);
	XmlUtils::readStringProperty(graphicalPropertiesElement, "propulsionUnitMesh", this->graphicalProperties.propulsionUnitMesh);
	XmlUtils::readStringProperty(graphicalPropertiesElement, "propulsionEffect", this->graphicalProperties.propulsionEffect);
	XmlUtils::readStringProperty(graphicalPropertiesElement, "boosterEffect", this->graphicalProperties.boosterEffect);
	XmlUtils::readStringProperty(graphicalPropertiesElement, "explosionEffect", this->graphicalProperties.explosionEffect);

	XmlUtils::readFVector3Property(graphicalPropertiesElement, "firstPersonViewPoint", this->graphicalProperties.firstPersonViewPoint);
	XmlUtils::readFVector3Property(graphicalPropertiesElement, "chaseViewPoint", this->graphicalProperties.chaseViewPoint);
/*
	for( TiXmlElement* elt = graphicalPropertiesElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );

		if( value == string("firstPersonViewPoint") ){
			this->graphicalProperties.firstPersonViewPoint = XmlUtils::readFVector3(elt);
		}else if( value == string("chaseViewPoint") ){
			this->graphicalProperties.chaseViewPoint = XmlUtils::readFVector3(elt);
		}else if( value == string("property") ){
			// already read

		}else{
			System::warn("(in Vehicle::parseGraphicalProperties()): Unknown child element: '%s' (row: %i). Ignoring.", value.c_str(), elt->Row() );
		}
	}
*/
}


void Vehicle::parsePhysicalProperties( TiXmlElement* physicalPropertiesElement ){

	XmlUtils::readBoolProperty(physicalPropertiesElement, "affectedByGravity", this->physicalProperties.affectedByGravity);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "bouncyness", this->physicalProperties.bouncyness);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "friction", this->physicalProperties.friction);

	XmlUtils::readFloatProperty(physicalPropertiesElement, "energyCapacity", this->physicalProperties.energyCapacity);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "healthCapacity", this->physicalProperties.healthCapacity);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "boosterForce", this->physicalProperties.boosterForce);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "propulsiveForce", this->physicalProperties.propulsiveForce);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "propulsiveTorque", this->physicalProperties.propulsiveTorque);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "maxLinearVelocity", this->physicalProperties.maxLinearVelocity);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "maxAngularVelocity", this->physicalProperties.maxAngularVelocity);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "minMainUnitYaw", this->physicalProperties.minMainUnitYaw);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "maxMainUnitYaw", this->physicalProperties.maxMainUnitYaw);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "minMainUnitPitch", this->physicalProperties.minMainUnitPitch);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "maxMainUnitPitch", this->physicalProperties.maxMainUnitPitch);

//	XmlUtils::readFloatProperty(physicalPropertiesElement, "hoverForce", this->physicalProperties.hoverForce);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "hoverHeight", this->physicalProperties.hoverHeight);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "airControlModifier", this->physicalProperties.airControlModifier);

	for( TiXmlElement* elt = physicalPropertiesElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );

		if( value == string("hoverPoint") ){
			this->physicalProperties.hoverPoints.push_back( XmlUtils::readFVector3(elt) );
		}else if( value == string("weaponSlot") ){
			this->physicalProperties.weaponSlots.push_back( XmlUtils::readFVector3(elt) );
		}else if( value == string("shape") ){
			this->physicalProperties.shape.fromXmlElement( elt );
		}else if( value == string("mass") ){
			this->parseMass( elt );
		}else if( value == string("debris") ){
			this->parseDebris( elt );
		}else if( value == string("property") ){
			// already read
		}else{
			System::warn("(in Vehicle::parsePhysicalProperties()): Unknown child element: '%s' (row: %i). Ignoring.", value.c_str(), elt->Row() );
		}
	}

}

void Vehicle::parseMass( TiXmlElement* massElement ){
	for( TiXmlElement* elt = massElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );
		if(value == "sphere"){
			float m = elt->Attribute("mass") ? atof(elt->Attribute("mass")) : 0.0f;
			float r = elt->Attribute("radius") ? atof(elt->Attribute("radius")) : 0.0f;

			dMass tempMass;
			dMassSetSphereTotal( &tempMass, m, r );
			dMassAdd( &this->physicalProperties.mass, &tempMass );

		}else if(value == "box"){
			float m = elt->Attribute("mass") ? atof(elt->Attribute("mass")) : 0.0f;
			float w = elt->Attribute("width") ? atof(elt->Attribute("width")) : 0.0f;
			float h = elt->Attribute("height") ? atof(elt->Attribute("height")) : 0.0f;
			float d = elt->Attribute("depth") ? atof(elt->Attribute("depth")) : 0.0f;

			dMass tempMass;
			dMassSetBoxTotal( &tempMass, m, w, h, d );
			dMassAdd( &this->physicalProperties.mass, &tempMass );

		}else if(value == "capsule"){
			float m = elt->Attribute("mass") ? atof(elt->Attribute("mass")) : 0.0f;
			float r = elt->Attribute("radius") ? atof(elt->Attribute("radius")) : 0.0f;
			float l = elt->Attribute("length") ? atof(elt->Attribute("length")) : 0.0f;

			dMass tempMass;
			dMassSetCappedCylinderTotal( &tempMass, m, 3, r, l );
			dMassAdd( &this->physicalProperties.mass, &tempMass );

		}else{
			System::warn("(in Vehicle::parseMass()): Unknown child element: '%s' (row: %i). Ignoring.", value.c_str(), elt->Row() );
		}
	}
}

void Vehicle::parseDebris( TiXmlElement* debrisElement ){
	physicalProperties_s::debris_s debris;

	debris.initialPosition = FVector3::ZERO;
	debris.initialAngularImpulse = FVector3::ZERO;
	debris.initialLinearImpulse = FVector3::ZERO;

	if( debrisElement->Attribute("geometry") != NULL ){
		debris.geometry = string( debrisElement->Attribute("geometry") );
	}
	if( debrisElement->Attribute("effect") != NULL ){
		debris.effect = string( debrisElement->Attribute("effect") );
	}
	if( debrisElement->Attribute("initialPosition") != NULL ){
		const char* val = debrisElement->Attribute("initialPosition");
		int n = sscanf( val, "x:%f y:%f z:%f", &debris.initialPosition.x, &debris.initialPosition.y, &debris.initialPosition.z );
	}
	if( debrisElement->Attribute("initialAngularImpulse") != NULL ){
		const char* val = debrisElement->Attribute("initialAngularImpulse");
		int n = sscanf( val, "x:%f y:%f z:%f", &debris.initialAngularImpulse.x, &debris.initialAngularImpulse.y, &debris.initialAngularImpulse.z );
	}
	if( debrisElement->Attribute("initialLinearImpulse") != NULL ){
		const char* val = debrisElement->Attribute("initialLinearImpulse");
		int n = sscanf( val, "x:%f y:%f z:%f", &debris.initialLinearImpulse.x, &debris.initialLinearImpulse.y, &debris.initialLinearImpulse.z );
	}

	this->physicalProperties.debris.push_back( debris );
}


void Vehicle::parseAcousticalProperties( TiXmlElement* acousticalPropertiesElement ){
}






bool Vehicle::collidedWithGameObject(const FVector3& contactPoint, GameObject* other, dContact& contact){
	if( other == NULL ){ // arena boundary
		return true;

	}else if( other->getGameObjectType() == GameObject::GAME_OBJECT_TYPE_SHOT ){
//		Shot* shot = (Shot*)other;
//		return this->collidedWithShot( contactPoint, shot, contact );
		return false;

//	}else if( other->getGameObjectType() == GameObject::GAME_OBJECT_TYPE_VEHICLE ){
//		Vehicle* vehicle = (Vehicle*)other;
//		return this->collidedWithVehicle( contactPoint, vehicle, contact);

	}else{

		contact.surface.mu += Physics::frictionToMu(this->physicalProperties.friction) * 0.5f;
		if( this->physicalProperties.bouncyness > 0.0f ){
			contact.surface.mode |= dContactBounce;
			contact.surface.bounce += this->physicalProperties.bouncyness*0.5f;
		}
		return true;
	}
}

bool Vehicle::collidedWithTerrain(const FVector3& contactPoint, dContact& contact){

	contact.surface.mu += Physics::frictionToMu(this->physicalProperties.friction) * 0.5f;
	if( this->physicalProperties.bouncyness > 0.0f ){
		contact.surface.mode |= dContactBounce;
		contact.surface.bounce += this->physicalProperties.bouncyness*0.5f;
	}
	return true;
}
/*
bool Vehicle::collidedWithShot( const FVector3& contactPoint, Shot* shot, dContact& contact ){
	if( shot->getClientId() == this->client->properties.clientId ){ // shot fired by this vehicle => ignore
		return false;
	}


	return false;
}
*/
void Vehicle::addLinearImpulse( const FVector3& impulse, const FVector3& position ){
	// convert impulse into force and apply it
	float deltaT = this->game->getSystem()->getTimer()->getDeltaT();
	dVector3 force;
	dWorldImpulseToForce( this->game->getPhysics()->getWorld(), deltaT,
		impulse.x, impulse.y, impulse.z, force);

	dBodyAddForceAtPos( this->body,
		force[0], force[1], force[2],
		position.x, position.y, position.z );

	dBodyEnable( this->body );

//			printf("FORCE: %f, %f, %f\n", force[0], force[1], force[2]);

}

void Vehicle::addAngularImpulse( const FVector3& impulse ){
	// convert impulse into force and apply it
	float deltaT = this->game->getSystem()->getTimer()->getDeltaT();
	dVector3 torque;
	dWorldImpulseToForce( this->game->getPhysics()->getWorld(), deltaT,
		impulse.x, impulse.y, impulse.z, torque);

	dBodyAddTorque( this->body,
		torque[0], torque[1], torque[2] );

	dBodyEnable( this->body );

}

void Vehicle::damage( float amount ){
	this->health -= amount;

	if( this->health < 0.0f ){
		// THINKABOUTME: explode von NW aufrufen
//		this->explode();
		if( this->client == this->game->getSystem()->getNetwork()->getClient() ){
			// send kill packet
			this->client->sendKillPacket( this->client->properties.clientId, this->attackerClientId );
			// this->destroyed = true;
		}
		this->health = 0.0f;
	}

	// in case the damage amount was negative
//	if( amount < 0.0f ){
//		this->heal( -amount );
//	}

}

void Vehicle::heal( float amount ){
	this->health += amount;

	if( this->health > this->physicalProperties.healthCapacity ){
		this->health = this->physicalProperties.healthCapacity;
	}

	this->game->spawnEffect( "vehicle_regeneration", this );

}

void Vehicle::explode(){
	// effect
	if( !this->graphicalProperties.explosionEffect.empty() ){
		Effect* e = this->game->spawnEffect( this->graphicalProperties.explosionEffect, this->position, this->orientation );
	}

	// debris
	for( vector<physicalProperties_s::debris_s>::const_iterator c_iter = this->physicalProperties.debris.begin();
		c_iter != this->physicalProperties.debris.end(); ++c_iter )
		{

		Ogre::Matrix3 m;
		m.FromAxes( this->right.toOgreVector3(), this->up.toOgreVector3(), -this->direction.toOgreVector3() );
		FVector3 pos = this->position + FVector3( m * c_iter->initialPosition.toOgreVector3() );

		if( !c_iter->geometry.empty() ){
			Geometry* g = this->game->spawnGeometry( c_iter->geometry, pos, this->orientation );
			if( !c_iter->effect.empty() ){
				this->game->spawnEffect( c_iter->effect, g );
			}
			g->velocity = this->velocity;
			g->angularVelocity = this->angularVelocity;

			g->addLinearImpulse( c_iter->initialLinearImpulse, pos );
			g->addAngularImpulse( c_iter->initialAngularImpulse );

		}else{ // only spawn effect if there is one
			if( !c_iter->effect.empty() ){
				this->game->spawnEffect( c_iter->effect, pos, this->orientation );
			}
		}
	}
}


void Vehicle::restockAmmoRel( float relAmount ){
	for( int i=0; i<MAX_WEAPONS; i++ ){
		if( this->weapons[i] != NULL ){
			this->weapons[i]->restockAmmoRel( relAmount );
		}
	}
}

void Vehicle::thermalShock(){
	this->energy = 0.0f;

	for( int i=0; i<MAX_WEAPONS; i++ ){
		if( this->weapons[i] != NULL ){
			this->weapons[i]->overheat();
		}
	}
}


void Vehicle::setTeamMaterial( int team ){
	string materialName;

	if( team == Game::TEAM_RED ){
		materialName = "veh/camo_red";
	}else if( team == Game::TEAM_BLUE ){
		materialName = "veh/camo_blue";
	}else{
		materialName = "veh/camo_white";
	}

	// main unit
	int i;
	int numSubEntities = this->mainUnitEntity->getNumSubEntities();
	for( i=0; i<numSubEntities; i++ ){
		Ogre::SubEntity* se = this->mainUnitEntity->getSubEntity( i );
		if( se->getMaterialName() == "veh/camo_white" ){
			se->setMaterialName( materialName );
		}
	}

	// prop unit
	numSubEntities = this->propulsionUnitEntity->getNumSubEntities();
	for( i=0; i<numSubEntities; i++ ){
		Ogre::SubEntity* se = this->propulsionUnitEntity->getSubEntity( i );
		if( se->getMaterialName() == "veh/camo_white" ){
			se->setMaterialName( materialName );
		}
	}
}


void Vehicle::calculateChecksum(){
	this->checksum = 0;
}


void Vehicle::setName( const string& newName ){
	this->name = newName;
}

const string& Vehicle::getName() const {
	return this->name;
}

const string& Vehicle::getDescription() const {
	return this->info.description;
}

Vehicle::vehicleTypes_e Vehicle::getVehicleType() const {
	return this->vehicleType;
}

unsigned int Vehicle::getChecksum() const {
	return this->checksum;
}

Client* Vehicle::getClient() const {
	return this->client;
}

const Vehicle::physicalProperties_s& Vehicle::getPhysicalProperties() const {
	return this->physicalProperties;
}

Weapon* Vehicle::getWeapon( int slot ) const {
	return this->weapons[slot];
}

float Vehicle::getHealth() const {
	return this->health;
}
float Vehicle::getEnergy() const {
	return this->energy;
}
void Vehicle::setAttackerClientId( int id ){
	this->attackerClientId = id;
}
