#include "WeaponFactory.h"

#include <tinyxml.h>

#include "System.h"
#include "Exception.h"
#include "Weapon.h"

WeaponFactory::WeaponFactory(Game* game){
	this->game = game;

	this->weaponsByName.clear();
}

WeaponFactory::~WeaponFactory(){
	// delete weapons
	for( weaponsByName_t::iterator iter = weaponsByName.begin(); iter != weaponsByName.end(); iter++ ){
		delete iter->second;
	}

}

void WeaponFactory::addWeapon(const string& weaponFilePath){
//	System::log( "WeaponFactory: Parsing weapon '%s'...", weaponFilePath.c_str() );

	try{
		TiXmlDocument doc( weaponFilePath.c_str() );
		if( !doc.LoadFile() ){
			throw Exception("Couldn't load file '" + weaponFilePath + "'. (Maybe not a correct xml file?)", "WeaponFactory::addWeapon()");
		}
//		doc.Print();

		TiXmlNode* rootElement = doc.RootElement();
		TiXmlElement* element = rootElement->ToElement();

		string rootElementName = element->Value();
		if( rootElementName != "weapon" ){
			throw Exception("Weapon file '" + weaponFilePath + "' does not have a <weapon> root node.", "WeaponFactory::addWeapon()");
		}

		// check version
		const char* tmp = element->Attribute( "formatVersion" );
		if( tmp == NULL ){
			throw Exception("Weapon file '" + weaponFilePath + "' has no version information.", "WeaponFactory::addWeapon()");
		}else{
			string version = tmp;

			if( version != System::getInstance()->getVersionString() ){
				System::warn("(in WeaponFactory::addWeapon()): file '%s' has version %s. Trying to parse anyway.", weaponFilePath.c_str(), version.c_str() );
			}
		}


		Weapon* w = new Weapon();
/*
		// check type and create a weapon of the proper subclass
		tmp = element->Attribute( "type" );
		if( tmp == NULL ){
			throw Exception("Weapon file '" + weaponFilePath + "' has no type information.", "WeaponFactory::addWeapon()");
		}else{
			string type = tmp;

			if( type == string("projectile") ){
				w = new ProjectileWeapon();
			}else if( type == string("ballistic") ){
//				w = new BeamWeapon();
			}else if( type == string("ray") ){
//				w = new BeamWeapon();
			}else if( type == string("beam") ){
//				w = new BeamWeapon();

			}else{
				System::warn("(in WeaponFactory::addWeapon()): Weapon file '%s' references unknown weapon type ('%s'). Ignoring weapon.", weaponFilePath.c_str(), type.c_str() );
			}
		}
*/

		if( w != NULL ){
			w->fromXmlElement( element );
			w->calculateChecksum();

			weaponsByName_t::const_iterator c_iter = weaponsByName.find( w->getName() );
			if( c_iter != weaponsByName.end() ){ // another weapon with that name is already registered -> throw exception
				throw Exception("A weapon with name '" + w->getName() + "' is already registered.", "WeaponFactory::addWeapon()");
			}
			weaponsByName.insert( weaponsByName_t::value_type(w->getName(), w) );
			System::log("WeaponFactory: Added weapon '%s' to list.", w->getName().c_str());
		}

	}catch( Exception& e ){
		System::warn("(WeaponFactory::addWeapon()): An exception occured while parsing weapon file '%s'. Ignoring weapon.", weaponFilePath.c_str());
		System::log("WeaponFactory: The exception was: %s", e.getDescription().c_str());
	}

}


void WeaponFactory::removeWeapon(const string& name){
	weaponsByName_t::iterator iter = weaponsByName.find( name );
	if( iter == weaponsByName.end() ){
		throw Exception("A weapon with name '" + name +"' is not registered.", "WeaponFactory::removeWeapon()");
	}
	this->weaponsByName.erase(iter);
}





Weapon* WeaponFactory::createWeapon(const string& name) const {

	weaponsByName_t::const_iterator c_iter = weaponsByName.find( name );
	if( c_iter == weaponsByName.end() ){
		throw Exception("A weapon with name '" + name +"' is not registered.", "WeaponFactory::createWeapon()");
	}

	Weapon* ret = c_iter->second->clone();

	ret->reset();

	return ret;
}

void WeaponFactory::destroyWeapon(Weapon* weapon) const {
	delete weapon;
}


bool WeaponFactory::isWeaponAvailable( const string& name ) const {
	weaponsByName_t::const_iterator c_iter = this->weaponsByName.find( name );

	return (c_iter != this->weaponsByName.end());

}

vector<string> WeaponFactory::getWeaponNames() const {
	vector<string> ret;

	for( weaponsByName_t::const_iterator c_iter = this->weaponsByName.begin(); c_iter != this->weaponsByName.end(); c_iter++ ){
		ret.push_back( c_iter->first );
	}

	return ret;
}

const string& WeaponFactory::getWeaponDescription( const string& name ) const {
	weaponsByName_t::const_iterator c_iter = weaponsByName.find( name );
	if( c_iter == weaponsByName.end() ){
		throw Exception("A weapon with name '" + name +"' is not registered.", "WeaponFactory::getWeaponDescription()");
	}else{
		return c_iter->second->getDescription();
	}

}
