#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "ganglia.h" /* for the libgmond messaging */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <apr.h>
#include <apr_strings.h>
#include <apr_hash.h>
#include <apr_time.h>
#include <apr_pools.h>
#include <apr_poll.h>
#include <apr_network_io.h>
#include <apr_signal.h>       
#include <apr_thread_proc.h>  /* for apr_proc_detach(). no threads used. */
#include <apr_tables.h>

#include "cmdline.h"   /* generated by cmdline.sh which runs gengetopt */
#include "confuse.h"   /* header for libconfuse in ./srclib/confuse */
#include "become_a_nobody.h"
#include "libmetrics.h"/* libmetrics header in ./srclib/libmetrics */
#include "apr_net.h"   /* our private network functions based on apr */
#include "debug_msg.h" 
#include "protocol.h"  /* generated header from ./lib/protocol.x xdr definition file */
#include "dtd.h"       /* the DTD definition for our XML */
#include "g25_config.h" /* for converting old file formats to new */
#include "daemon_init.h"

/* When this gmond was started */
apr_time_t started;
/* My name */
char myname[APRMAXHOSTLEN+1];
/* The commandline options */
struct gengetopt_args_info args_info;
/* The configuration file */
cfg_t *config_file;
/* The debug level (in debug_msg.c) */
extern int debug_level;
/* The global context */
Ganglia_pool global_context = NULL;
/* Deaf mode boolean */
int deaf;
/* Mute mode boolean */
int mute;
/* Cluster tag boolean */
int cluster_tag = 0;
/* This host's location */
char *host_location = NULL;
/* Boolean. Will this host received gexec requests? */
int gexec_on = 0;
/* This is tweakable by globals{max_udp_msg_len=...} */
int max_udp_message_len = 1472;
/* The default configuration for gmond.  Found in conf.c. */
extern char *default_gmond_configuration;
/* The number of seconds to hold "dead" hosts in the hosts hash */
int host_dmax = 0;
/* The amount of time between cleanups */
int cleanup_threshold = 300;

/* The array for outgoing UDP message channels */
Ganglia_udp_send_channels udp_send_channels = NULL;

/* TODO: The array for outgoing TCP message channels (later) */
apr_array_header_t *tcp_send_array = NULL;

enum Ganglia_action_types {
  GANGLIA_ACCESS_DENY = 0,
  GANGLIA_ACCESS_ALLOW = 1
};
typedef enum Ganglia_action_types Ganglia_action_types;

/* This is the structure used for the access control lists */
struct Ganglia_access {
  apr_ipsubnet_t *ipsub;
  Ganglia_action_types action;
};
typedef struct Ganglia_access Ganglia_access;

struct Ganglia_acl {
  apr_array_header_t *access_array;
  Ganglia_action_types default_action;
};
typedef struct Ganglia_acl Ganglia_acl;

/* This is the channel definitions */
enum Ganglia_channel_types {
  TCP_ACCEPT_CHANNEL,
  UDP_RECV_CHANNEL
};
typedef enum Ganglia_channel_types Ganglia_channel_types;

struct Ganglia_channel {
  Ganglia_channel_types type;
  Ganglia_acl *acl;
  int timeout;
};
typedef struct Ganglia_channel Ganglia_channel;

/* This pollset holds the tcp_accept and udp_recv channels */
apr_pollset_t *listen_channels = NULL;

/* The hash to hold the hosts (key = host IP) */
apr_hash_t *hosts = NULL;

/* The "hosts" hash contains values of type "hostdata" */
struct Ganglia_host {
  /* Name of the host */
  char *hostname;
  /* The IP of this host */
  char *ip;
  /* The location of this host */
  char *location;
  /* Timestamp of when the remote host gmond started */
  unsigned int gmond_started;
  /* The pool used to malloc memory for this host */
  apr_pool_t *pool;
  /* A hash containing the registered data from the host */
  apr_hash_t *metrics;
  /* A hash containing the arbitrary data from gmetric */
  apr_hash_t *gmetrics;
  /* First heard from */
  apr_time_t first_heard_from;
  /* Last heard from */
  apr_time_t last_heard_from;
};
typedef struct Ganglia_host Ganglia_host;

/* This is the structure of the data save to each host->metric hash */
struct Ganglia_metric {
  /* The pool used for allocating memory */
  apr_pool_t *pool;
  /* The name of the metric */
  char *name;
  /* The ganglia message */
  Ganglia_message message;
  /* Last heard from */
  apr_time_t last_heard_from;
};
typedef struct Ganglia_metric Ganglia_metric;

/* The hash to hold the metrics available on this platform */
apr_hash_t *metric_callbacks = NULL;

/* The "metrics" hash contains values of type "Ganglia_metric_callback" */
/* This is where libmetrics meets gmond */
struct Ganglia_metric_callback {
   char *name;          /* metric name */
   float value_threshold;/* the value threshold */
   Ganglia_25metric *info;/* the information about this metric */
   g_val_t (*cb)(void); /* callback function */
   g_val_t now;         /* the current value */
   g_val_t last;        /* the last value */
   Ganglia_message msg; /* the message to send */
};
typedef struct Ganglia_metric_callback Ganglia_metric_callback;

/* This is the structure of a collection group */
struct Ganglia_collection_group {
  apr_time_t next_collect;  /* When to collect next */
  apr_time_t next_send;     /* When to send next (tmax) */
  int once;
  int collect_every;
  int time_threshold;
  apr_array_header_t *metric_array;
};
typedef struct Ganglia_collection_group Ganglia_collection_group;

/* This is the array of collection groups that we are processing... */
apr_array_header_t *collection_groups = NULL;

/* this is just a temporary function */
void
process_configuration_file(void)
{
  cfg_t *tmp;

  /* this is a global for now */
  config_file = Ganglia_gmond_config_create( args_info.conf_arg, !args_info.conf_given );

  /* Initialize a few variables */
  tmp = cfg_getsec( config_file, "globals");
  /* Get the maximum UDP message size */
  max_udp_message_len = cfg_getint( tmp, "max_udp_msg_len");
  /* Get the gexec status requested */
  gexec_on            = cfg_getbool(tmp, "gexec");
  /* Get the host dmax ... */
  host_dmax           = cfg_getint( tmp, "host_dmax");
  /* Get the cleanup threshold */
  cleanup_threshold   = cfg_getint( tmp, "cleanup_threshold");
}

static void
daemonize_if_necessary( char *argv[] )
{
  int should_daemonize;
  cfg_t *tmp;
  tmp = cfg_getsec( config_file, "globals");
  should_daemonize = cfg_getbool( tmp, "daemonize");

  /* Commandline for debug_level trumps configuration file behaviour ... */
  if (args_info.debug_given) 
    {
      debug_level = args_info.debug_arg;
    }
  else
    {
      debug_level = cfg_getint ( tmp, "debug_level");
    }

  /* Daemonize if needed */
  if(!args_info.foreground_flag && should_daemonize && !debug_level)
    {
      apr_proc_detach(1);
    }
}

static void
setuid_if_necessary( void )
{
  cfg_t *tmp;
  int setuid;
  char *user;

  tmp    = cfg_getsec( config_file, "globals");
  setuid = cfg_getbool( tmp, "setuid" );
  if(setuid)
    {
#ifdef CYGWIN
      fprintf(stderr,"Windows does not support setuid.\n");
#else
      user = cfg_getstr(tmp, "user" );
      become_a_nobody(user);
#endif
    }
}

static void
process_deaf_mute_mode( void )
{
  cfg_t *tmp = cfg_getsec( config_file, "globals");
  deaf =       cfg_getbool( tmp, "deaf");
  mute =       cfg_getbool( tmp, "mute");
  if(deaf && mute)
    {
      fprintf(stderr,"Configured to run both deaf and mute. Nothing to do. Exiting.\n");
      exit(1);
    }
}

static Ganglia_acl *
Ganglia_acl_create ( cfg_t *channel, apr_pool_t *pool )
{
  apr_status_t status;
  Ganglia_acl *acl = NULL;
  cfg_t *acl_config;
  char *default_action;
  int num_access = 0;
  int i;

  if(!channel || !pool)
    {
      return acl;
    }


  acl_config = cfg_getsec(channel, "acl");
  if(!acl_config)
    {
      return acl;
    }

  /* Find out the number of access entries */
  num_access = cfg_size( acl_config, "access" );
  if(!num_access)
    {
      return acl;
    }

  /* Create a new ACL from the pool */
  acl = apr_pcalloc( pool, sizeof(Ganglia_acl));
  if(!acl)
    {
      fprintf(stderr,"Unable to allocate memory for ACL. Exiting.\n");
      exit(1);
    }

  default_action = cfg_getstr( acl_config, "default");
  if(!apr_strnatcasecmp( default_action, "deny"))
    {
      acl->default_action = GANGLIA_ACCESS_DENY;
    }
  else if(!apr_strnatcasecmp( default_action, "allow"))
    {
      acl->default_action = GANGLIA_ACCESS_ALLOW;
    }
  else
    {
      fprintf(stderr,"Invalid default ACL '%s'. Exiting.\n", default_action);
      exit(1);
    }

  /* Create an array to hold each of the access instructions */
  acl->access_array  = apr_array_make( pool, num_access, sizeof(Ganglia_acl *));
  if(!acl->access_array)
    {
      fprintf(stderr,"Unable to malloc access array. Exiting.\n");
      exit(1);
    }
  for(i=0; i< num_access; i++)
    {
      cfg_t *access_config   = cfg_getnsec( acl_config, "access", i);
      Ganglia_access *access = apr_pcalloc( pool, sizeof(Ganglia_access));
      char *ip, *mask, *action;

      if(!access_config)
	{
	  /* This shouldn't happen unless maybe acl is empty and
	   * the safest thing to do it exit */
	  fprintf(stderr,"Unable to process ACLs. Exiting.\n");
	  exit(1);
	}

      ip     = cfg_getstr( access_config, "ip");
      mask   = cfg_getstr( access_config, "mask");
      action = cfg_getstr( access_config, "action");
      if(!ip && !mask && !action)
	{
	  fprintf(stderr,"An access record requires an ip, mask and action. Exiting.\n");
	  exit(1);
	}

      /* Process the action first */
      if(!apr_strnatcasecmp( action, "deny" ))
	{
          access->action = GANGLIA_ACCESS_DENY;
	}
      else if(!apr_strnatcasecmp( action, "allow"))
	{
          access->action = GANGLIA_ACCESS_ALLOW;
	}
      else
	{
          fprintf(stderr,"ACL access entry has action '%s'. Must be deny|allow. Exiting.\n", action);
          exit(1);
	}	  

      /* Create the subnet */
      access->ipsub = NULL;
      status = apr_ipsubnet_create( &(access->ipsub), ip, mask, pool);
      if(status != APR_SUCCESS)
	{
	  fprintf(stderr,"ACL access entry has invalid ip('%s')/mask('%s'). Exiting.\n", ip, mask);
	  exit(1);
	}

      /* Save this access entry to the acl */
      *(Ganglia_access **)apr_array_push( acl->access_array ) = access;
    }
  return acl;
}


static int
Ganglia_acl_action( Ganglia_acl *acl, apr_sockaddr_t *addr )
{
  int i;

  if(!acl)
    {
      /* If no ACL is specified, we assume there is no access control */
      return GANGLIA_ACCESS_ALLOW; 
    }

  for(i=0; i< acl->access_array->nelts; i++)
    {
      Ganglia_access *access = ((Ganglia_access **)(acl->access_array->elts))[i];
      if(!apr_ipsubnet_test( access->ipsub, addr ))
	{
	  /* no action will occur because addr is not in this subnet */
	  continue;
	}
      else
	{
          return access->action;
	}
    }

  /* No matches in the access list so we return the default */
  return acl->default_action;
}

static int32_t
get_sock_family( char *family )
{
  if( strchr( family, '4' ))
    {
      return APR_INET;
    }
  else if( strchr( family, '6'))
    {
#if APR_INET6
      return APR_INET6;
#else
      fprintf(stderr,"IPv6 is not supported on this host. Exiting.\n");
      exit(1);
#endif
    }

  fprintf(stderr,"Unknown family '%s'. Try inet4|inet6. Exiting.\n", family);
  exit(1);
  /* shouldn't get here */
  return APR_UNSPEC;
}

static void
setup_listen_channels_pollset( void )
{
  apr_status_t status;
  int i;
  int num_udp_recv_channels   = cfg_size( config_file, "udp_recv_channel");
  int num_tcp_accept_channels = cfg_size( config_file, "tcp_accept_channel");
  int total_listen_channels   = num_udp_recv_channels + num_tcp_accept_channels;
  Ganglia_channel *channel;

  /* Create my incoming pollset */
  apr_pollset_create(&listen_channels, total_listen_channels, global_context, 0);

  /* Process all the udp_recv_channels */
  for(i = 0; i< num_udp_recv_channels; i++)
    {
      cfg_t *udp_recv_channel;
      char *mcast_join, *mcast_if, *bindaddr, *family;
      int port;
      apr_socket_t *socket = NULL;
      apr_pollfd_t socket_pollfd;
      apr_pool_t *pool = NULL;
      int32_t sock_family = APR_INET;

      udp_recv_channel = cfg_getnsec( config_file, "udp_recv_channel", i);
      mcast_join     = cfg_getstr( udp_recv_channel, "mcast_join" );
      mcast_if       = cfg_getstr( udp_recv_channel, "mcast_if" );
      port           = cfg_getint( udp_recv_channel, "port");
      bindaddr       = cfg_getstr( udp_recv_channel, "bind");
      family         = cfg_getstr( udp_recv_channel, "family");

      debug_msg("udp_recv_channel mcast_join=%s mcast_if=%s port=%d bind=%s",
		  mcast_join? mcast_join:"NULL", 
		  mcast_if? mcast_if:"NULL", port,
		  bindaddr? bindaddr: "NULL");

      /* Create a sub-pool for this channel */
      apr_pool_create(&pool, global_context);

      sock_family = get_sock_family(family);

      if( mcast_join )
	{
	  /* Listen on the specified multicast channel */
	  socket = create_mcast_server(pool, sock_family, mcast_join, port, bindaddr, mcast_if );
	  if(!socket)
	    {
	      fprintf(stderr,"Error creating multicast server mcast_join=%s port=%d mcast_if=%s family='%s'. Exiting.\n",
		      mcast_join? mcast_join: "NULL", port, mcast_if? mcast_if:"NULL",family);
	      exit(1);
	    }
	}
      else
	{
	  /* Create a UDP server */
          socket = create_udp_server( pool, sock_family, port, bindaddr );
          if(!socket)
            {
              fprintf(stderr,"Error creating UDP server on port %d bind=%s. Exiting.\n",
		      port, bindaddr? bindaddr: "unspecified");
	      exit(1);
	    }
	}

      /* Build the socket poll file descriptor structure */
      socket_pollfd.desc_type   = APR_POLL_SOCKET;
      socket_pollfd.reqevents   = APR_POLLIN;
      socket_pollfd.desc.s      = socket;

      channel = apr_pcalloc( pool, sizeof(Ganglia_channel));
      if(!channel)
	{
	  fprintf(stderr,"Unable to malloc memory for channel.  Exiting. \n");
	  exit(1);
	}

      /* Mark this channel as a udp_recv_channel */
      channel->type = UDP_RECV_CHANNEL;

      /* Make sure this socket never blocks */
      channel->timeout = 0;
      apr_socket_timeout_set( socket, channel->timeout);

      /* Save the ACL information */
      channel->acl = Ganglia_acl_create ( udp_recv_channel, pool );

      /* Save the pointer to this socket specific data */
      socket_pollfd.client_data = channel;

      /* Add the socket to the pollset */
      status = apr_pollset_add(listen_channels, &socket_pollfd);
      if(status != APR_SUCCESS)
	{
	  fprintf(stderr,"Failed to add socket to pollset. Exiting.\n");
	  exit(1);
	}
    }

  /* Process all the tcp_accept_channels */ 
  for(i=0; i< num_tcp_accept_channels; i++)
    {
      cfg_t *tcp_accept_channel = cfg_getnsec( config_file, "tcp_accept_channel", i);
      char *bindaddr, *interface, *family;
      int port, timeout;
      apr_socket_t *socket = NULL;
      apr_pollfd_t socket_pollfd;
      apr_pool_t *pool = NULL;
      int32_t sock_family;

      port           = cfg_getint( tcp_accept_channel, "port");
      bindaddr       = cfg_getstr( tcp_accept_channel, "bind");
      interface      = cfg_getstr( tcp_accept_channel, "interface"); 
      timeout        = cfg_getint( tcp_accept_channel, "timeout");
      family         = cfg_getstr( tcp_accept_channel, "family");

      debug_msg("tcp_accept_channel bind=%s port=%d",
		  bindaddr? bindaddr: "NULL", port);

      /* Create a subpool context */
      apr_pool_create(&pool, global_context);

      sock_family = get_sock_family(family);

      /* Create the socket for the channel */
      socket = create_tcp_server(pool, sock_family, port, bindaddr, interface,
		      1);// blocking w/timeout 
      if(!socket)
	{
	  fprintf(stderr,"Unable to create tcp_accept_channel. Exiting.\n");
	  exit(1);
	}

      /* Build the socket poll file descriptor structure */
      socket_pollfd.desc_type   = APR_POLL_SOCKET;
      socket_pollfd.reqevents   = APR_POLLIN;
      socket_pollfd.desc.s      = socket;

      channel = apr_pcalloc( pool, sizeof(Ganglia_channel));
      if(!channel)
	{
	  fprintf(stderr,"Unable to malloc data for channel. Exiting.\n");
	  exit(1);
	}
      
      channel->type = TCP_ACCEPT_CHANNEL;

      /* Save the timeout for this socket */
      channel->timeout = timeout;

      /* Save the ACL information */
      channel->acl = Ganglia_acl_create( tcp_accept_channel, pool ); 

      /* Save the pointer to this channel data */
      socket_pollfd.client_data = channel;

      /* Add the socket to the pollset */
      status = apr_pollset_add(listen_channels, &socket_pollfd);
      if(status != APR_SUCCESS)
         {
            fprintf(stderr,"Failed to add socket to pollset. Exiting.\n");
            exit(1);
         }
    }
}

static Ganglia_host *
Ganglia_host_get( char *remoteip, apr_sockaddr_t *sa, Ganglia_message *fullmsg)
{
  apr_status_t status;
  apr_pool_t *pool;
  Ganglia_host *hostdata;
  char *hostname = NULL;

  if(!remoteip || !sa || !fullmsg)
    {
      return NULL;
    }

  hostdata =  (Ganglia_host *)apr_hash_get( hosts, remoteip, APR_HASH_KEY_STRING );
  if(!hostdata)
    {
      /* Lookup the hostname or use the proxy information if available */
      if( !hostname )
	{
	  /* We'll use the resolver to find the hostname */
          status = apr_getnameinfo(&hostname, sa, 0);
          if(status != APR_SUCCESS)
	    {
	      /* If hostname lookup fails.. set it to the ip */
	      hostname = remoteip;
	    }
	}

      /* This is the first time we've heard from this host.. create a new pool */
      status = apr_pool_create( &pool, global_context );
      if(status != APR_SUCCESS)
	{
	  return NULL;
	}

      /* Malloc the hostdata_t from the new pool */
      hostdata = apr_pcalloc( pool, sizeof( Ganglia_host ));
      if(!hostdata)
	{
	  return NULL;
	}

      /* Save the pool address for later.. freeing this pool free everthing
       * for this particular host */
      hostdata->pool = pool;

      /* Save the hostname */
      hostdata->hostname = apr_pstrdup( pool, hostname );

      /* Dup the remoteip (it will be freed later) */
      hostdata->ip =  apr_pstrdup( pool, remoteip);

      /* We don't know the location yet */
      hostdata->location = NULL;

      /* Set the timestamps */
      hostdata->first_heard_from = hostdata->last_heard_from = apr_time_now();

      /* Create a hash for the metric data */
      hostdata->metrics = apr_hash_make( pool );
      if(!hostdata->metrics)
	{
	  apr_pool_destroy(pool);
	  return NULL;
	}

      /* Create a hash for the gmetric data */
      hostdata->gmetrics = apr_hash_make( pool );
      if(!hostdata->gmetrics)
	{
	  apr_pool_destroy(pool);
	  return NULL;
	}

      /* Save this host data to the "hosts" hash */
      apr_hash_set( hosts, hostdata->ip, APR_HASH_KEY_STRING, hostdata); 
    }
  else
    {
      /* We already have this host in our "hosts" hash update timestamp */
      hostdata->last_heard_from = apr_time_now();
    }

  debug_msg("Processing a Ganglia_message from %s", hostdata->hostname);

  if(fullmsg->id == metric_location)
    {
      /* We have to manage this memory here because.. returning NULL
       * will not cause Ganglia_message_save to be run.  Maybe this
       * could be done better later i.e should these metrics be
       * in the host->metrics list instead of the host structure? */
      if(hostdata->location)
	{
	  /* Free old location */
	  free(hostdata->location);
	}
      /* Save new location */
      hostdata->location = strdup(fullmsg->Ganglia_message_u.str);
      debug_msg("Got a location message %s\n", hostdata->location);
      /* Processing is finished */
      return NULL;
    }
  if(fullmsg->id == metric_heartbeat)
    {
      /* nothing more needs to be done. we handled the timestamps above. */
      hostdata->gmond_started = fullmsg->Ganglia_message_u.u_int;
      debug_msg("Got a heartbeat message %d\n", hostdata->gmond_started);
      /* Processing is finished */
      return NULL;
    }

  return hostdata;
}

static Ganglia_metric *
Ganglia_metric_create( Ganglia_host *host )
{
  apr_status_t status;
  Ganglia_metric *metric = NULL;
  apr_pool_t *pool = NULL;

  if(!host)
    return NULL;

  /* Create the context for this metric */
  status = apr_pool_create( &pool, host->pool);
  if(status != APR_SUCCESS)
    return NULL;

  /* Allocate a new metric from this context */
  metric = apr_pcalloc( pool, sizeof(Ganglia_metric));
  if(!metric)
    {
      /* out of mem... kill context.. return null */
      apr_pool_destroy(pool);
      return NULL;
    }

  metric->pool = pool;
  return metric;
}

#if 0
static void
Ganglia_metric_free( Ganglia_metric *metric )
{
  if(!metric)
    return;
  apr_pool_destroy( metric->pool );
}
#endif

static Ganglia_metric *
Ganglia_message_find_gmetric( Ganglia_host *host, Ganglia_message *message)
{
  /* Keyed on the name element of the gmetric sent */
  return (Ganglia_metric *)apr_hash_get( host->gmetrics,
				   message->Ganglia_message_u.gmetric.name,
				   APR_HASH_KEY_STRING);
}

static Ganglia_metric *
Ganglia_message_find_metric( Ganglia_host *host, Ganglia_message *message)
{
  /* Keyed on the message id number */
  return (Ganglia_metric *)apr_hash_get( host->metrics, 
				  &(message->id), 
				  sizeof(message->id) );
}

static void
Ganglia_message_save( Ganglia_host *host, Ganglia_message *message )
{
  Ganglia_metric *metric = NULL;

  if(!host || !message)
    return;

  /* Search for the Ganglia_metric in the Ganglia_host */
  if(message->id == metric_user_defined)
    {
      metric = Ganglia_message_find_gmetric( host, message);
    }
  else
    {
      metric = Ganglia_message_find_metric( host, message);
    }

  if(!metric)
    {
      /* This is a new metric sent from this host... allocate space for this data */
      metric = Ganglia_metric_create( host );
      if(!metric)
	{
	  /* no memory */
	  return;
	}
      
      /* NOTE: In order for gmetric messages to be properly saved to the hash table
       * based on the name of the gmetric sent...we need to strdup() the name
       * since the xdr_free below will blast the value later (along with the other
       * allocated structure elements).  This is only performed once at gmetric creation */
      if(message->id == metric_user_defined)
	{
	  metric->name = apr_pstrdup( metric->pool, message->Ganglia_message_u.gmetric.name );
	} 
    }
  else
    {
      /* This is a metric update.  Flush the old data. */
      xdr_free((xdrproc_t)xdr_Ganglia_message, (char *)&(metric->message));
    }

  /* Copy in the data */
  memcpy(&(metric->message), message, sizeof(Ganglia_message));

  if(message->id == metric_user_defined)
    {
      /* Save the gmetric */
      apr_hash_set( host->gmetrics, metric->name, APR_HASH_KEY_STRING, metric);
    }
  else
    {
      /* Save the metric */
      apr_hash_set( host->metrics, &(message->id), sizeof(message->id), metric );
    }

  /* Timestamp */
  metric->last_heard_from = apr_time_now();
}

static void
process_udp_recv_channel(const apr_pollfd_t *desc, apr_time_t now)
{
  apr_status_t status;
  apr_socket_t *socket;
  apr_sockaddr_t *remotesa = NULL;
  char  remoteip[256];
  char buf[max_udp_message_len];
  apr_size_t len = max_udp_message_len;
  Ganglia_channel *channel;
  XDR x;
  Ganglia_message msg;
  Ganglia_host *hostdata = NULL;

  socket         = desc->desc.s;
  /* We could also use the apr_socket_data_get/set() functions
   * to have per socket user data .. see APR docs */
  channel       = desc->client_data;

  apr_socket_addr_get(&remotesa, APR_REMOTE, socket);

  /* Grab the data */
  status = apr_socket_recvfrom(remotesa, socket, 0, buf, &len);
  if(status != APR_SUCCESS)
    {
      return;
    }	  

  /* This function is in ./lib/apr_net.c and not APR. The
   * APR counterpart is apr_sockaddr_ip_get() but we don't 
   * want to malloc memory evertime we call this */
  apr_sockaddr_ip_buffer_get(remoteip, 256, remotesa);

  /* Check the ACL */
  if(Ganglia_acl_action( channel->acl, remotesa) != GANGLIA_ACCESS_ALLOW)
    return;

  /* Create the XDR receive stream */
  xdrmem_create(&x, buf, max_udp_message_len, XDR_DECODE);

  /* Flush the data... */
  memset( &msg, 0, sizeof(Ganglia_message));

  /* Read the gangliaMessage from the stream */
  if(!xdr_Ganglia_message(&x, &msg))
    {	
      return;
    }

  hostdata = Ganglia_host_get( remoteip, remotesa, &msg);
  if(!hostdata)
    {
      /* Processing of this message is finished ... */
      xdr_free((xdrproc_t)xdr_Ganglia_message, (char *)&msg);
      return;
    }

  /* Save the message from this particular host */
  Ganglia_message_save( hostdata, &msg );
}

static apr_status_t
print_xml_header( apr_socket_t *client )
{
  apr_status_t status;
  apr_size_t len = strlen(DTD);
  char gangliaxml[128];
  char clusterxml[1024];
  static int clusterinit = 0;
  static char *name = NULL;
  static char *owner = NULL;
  static char *latlong = NULL;
  static char *url = NULL;
  apr_time_t now = apr_time_now();

  status = apr_socket_send( client, DTD, &len );
  if(status != APR_SUCCESS)
    return status;

  len = apr_snprintf( gangliaxml, 128, "<GANGLIA_XML VERSION=\"%s\" SOURCE=\"gmond\">\n",
		      VERSION);
  status = apr_socket_send( client, gangliaxml, &len);
  if(status != APR_SUCCESS)
    return status;

  if(!clusterinit)
    {
      /* We only run this on the first connection we process */
      cfg_t *cluster = cfg_getsec(config_file, "cluster");
      if(cluster)
	{
          name    = cfg_getstr( cluster, "name" );
          owner   = cfg_getstr( cluster, "owner" );
          latlong = cfg_getstr( cluster, "latlong" );
          url     = cfg_getstr( cluster, "url" );
	  if(name || owner || latlong || url)
	    {
	      cluster_tag =1;
	    }
	}
      clusterinit = 1;
    }

  if(cluster_tag)
    {
      len = apr_snprintf( clusterxml, 1024, 
	"<CLUSTER NAME=\"%s\" LOCALTIME=\"%d\" OWNER=\"%s\" LATLONG=\"%s\" URL=\"%s\">\n", 
		      name?name:"unspecified", 
		      (int)(now / APR_USEC_PER_SEC),
		      owner?owner:"unspecified", 
		      latlong?latlong:"unspecified",
		      url?url:"unspecified");

      return apr_socket_send( client, clusterxml, &len);
    }

  return APR_SUCCESS;
}

static apr_status_t
print_xml_footer( apr_socket_t *client )
{
  apr_status_t status;
  apr_size_t len; 
  if(cluster_tag)
    {
      len = 11;
      status = apr_socket_send(client, "</CLUSTER>\n", &len);
      if(status != APR_SUCCESS)
	{
	  return status;
	}
    }
  len = 15;
  return apr_socket_send( client, "</GANGLIA_XML>\n", &len);
}

static apr_status_t
print_host_start( apr_socket_t *client, Ganglia_host *hostinfo)
{
  apr_size_t len;
  char hostxml[1024]; /* for <HOST></HOST> */
  apr_time_t now = apr_time_now();
  int tn = (now - hostinfo->last_heard_from) / APR_USEC_PER_SEC;

  len = apr_snprintf(hostxml, 1024, 
           "<HOST NAME=\"%s\" IP=\"%s\" REPORTED=\"%d\" TN=\"%d\" TMAX=\"%d\" DMAX=\"%d\" LOCATION=\"%s\" GMOND_STARTED=\"%d\">\n",
                     hostinfo->hostname, 
		     hostinfo->ip, 
		     (int)(hostinfo->last_heard_from / APR_USEC_PER_SEC),
		     tn,
		     20, /*tmax for now is always 20 */
		     host_dmax,
		     hostinfo->location? hostinfo->location: "unspecified", 
		     hostinfo->gmond_started);

  return apr_socket_send(client, hostxml, &len);
}

/* NOT THREAD SAFE */
static char *
host_metric_type( Ganglia_value_types type)
{
  switch(type)
    {
    case GANGLIA_VALUE_UNKNOWN:
      return "unknown";
    case GANGLIA_VALUE_STRING:
      return "string";
    case GANGLIA_VALUE_UNSIGNED_SHORT:
      return "uint16";
    case GANGLIA_VALUE_SHORT:
      return "int16";
    case GANGLIA_VALUE_UNSIGNED_INT:
      return "uint32";
    case GANGLIA_VALUE_INT:
      return "int32";
    case GANGLIA_VALUE_FLOAT:
      return "float";
    case GANGLIA_VALUE_DOUBLE:
      return "double";
    }
  return "undef";
}

/* NOT THREAD SAFE */
static char *
host_metric_value( Ganglia_25metric *metric, Ganglia_message *message )
{
  static char value[1024];
  if(!metric||!message)
    {
      return "unknown";
    }

  switch(metric->type)
    {
    case GANGLIA_VALUE_UNKNOWN:
      return "unknown";
    case GANGLIA_VALUE_STRING:
      return message->Ganglia_message_u.str;
    case GANGLIA_VALUE_UNSIGNED_SHORT:
      apr_snprintf(value, 1024, metric->fmt, message->Ganglia_message_u.u_short);
      return value;
    case GANGLIA_VALUE_SHORT:
      /* For right now.. there are no metrics which are signed shorts... use u_short */
      apr_snprintf(value, 1024, metric->fmt, message->Ganglia_message_u.u_short);
      return value;
    case GANGLIA_VALUE_UNSIGNED_INT:
      apr_snprintf(value, 1024, metric->fmt, message->Ganglia_message_u.u_int);
      return value;
    case GANGLIA_VALUE_INT:
      /* For right now.. there are no metric which are signed ints... use u_int */
      apr_snprintf(value, 1024, metric->fmt, message->Ganglia_message_u.u_int);
      return value;
    case GANGLIA_VALUE_FLOAT:
      apr_snprintf(value, 1024, metric->fmt, message->Ganglia_message_u.f);
      return value;
    case GANGLIA_VALUE_DOUBLE:
      apr_snprintf(value, 1024, metric->fmt, message->Ganglia_message_u.d);
      return value;
    }

  return "unknown";
}

/*
 * struct Ganglia_gmetric_message {
 *   char *type;
 *   char *name;
 *   char *value;
 *   char *units;
 *   u_int slope;
 *   u_int tmax;
 *   u_int dmax;
 * };
 */
static apr_status_t
print_host_gmetric( apr_socket_t *client, Ganglia_metric *metric, apr_time_t now )
{
  apr_size_t len;
#define GMETRIC_BUFFER_SIZE (GANGLIA_MAX_MESSAGE_LEN + 1024)
  char metricxml[GMETRIC_BUFFER_SIZE];
  Ganglia_gmetric_message *msg = &(metric->message.Ganglia_message_u.gmetric);

  if(!msg)
    return APR_SUCCESS;

  len = apr_snprintf(metricxml, GMETRIC_BUFFER_SIZE,
          "<METRIC NAME=\"%s\" VAL=\"%s\" TYPE=\"%s\" UNITS=\"%s\" TN=\"%d\" TMAX=\"%d\" DMAX=\"%d\" SLOPE=\"%s\" SOURCE=\"gmetric\"/>\n",
           msg->name,
	   msg->value,
	   msg->type,
	   msg->units,
           (int)((now - metric->last_heard_from) / APR_USEC_PER_SEC),
	   msg->tmax,
	   msg->dmax,
	   msg->slope? "both": "zero");

  return apr_socket_send(client, metricxml, &len );
}

static apr_status_t
print_host_metric( apr_socket_t *client, Ganglia_metric *data, apr_time_t now )
{
  Ganglia_25metric *metric;
  char metricxml[1024];
  apr_size_t len;

  metric = Ganglia_25metric_bykey( data? data->message.id: -1 );
  if(!metric)
    return APR_SUCCESS;

  len = apr_snprintf(metricxml, 1024,
          "<METRIC NAME=\"%s\" VAL=\"%s\" TYPE=\"%s\" UNITS=\"%s\" TN=\"%d\" TMAX=\"%d\" DMAX=\"0\" SLOPE=\"%s\" SOURCE=\"gmond\"/>\n",
	  metric->name,
	  host_metric_value( metric, &(data->message) ),
	  host_metric_type( metric->type ),
	  metric->units? metric->units: "",
	  (int)((now - data->last_heard_from) / APR_USEC_PER_SEC),
	  metric->tmax,
	  metric->slope );

  return apr_socket_send(client, metricxml, &len);
}

static apr_status_t
print_host_end( apr_socket_t *client)
{
  apr_size_t len = 8;
  return apr_socket_send(client, "</HOST>\n", &len); 
}

static void
process_tcp_accept_channel(const apr_pollfd_t *desc, apr_time_t now)
{
  apr_status_t status;
  apr_hash_index_t *hi, *metric_hi;
  void *val;
  apr_socket_t *client, *server;
  apr_sockaddr_t *remotesa = NULL;
  char  remoteip[256];
  apr_pool_t *client_context = NULL;
  Ganglia_channel *channel;

  server         = desc->desc.s;
  /* We could also use the apr_socket_data_get/set() functions
   * to have per socket user data .. see APR docs */
  channel        = desc->client_data;

  /* Create a context for the client connection */
  apr_pool_create(&client_context, global_context);

  /* Accept the connection */
  status = apr_accept(&client, server, client_context);
  if(status != APR_SUCCESS)
    {
      goto close_accept_socket;
    }

  /* Set the timeout for writing to the client */
  apr_socket_timeout_set( client, channel->timeout);

  apr_socket_addr_get(&remotesa, APR_REMOTE, client);
  /* This function is in ./lib/apr_net.c and not APR. The
   * APR counterpart is apr_sockaddr_ip_get() but we don't 
   * want to malloc memory evertime we call this */
  apr_sockaddr_ip_buffer_get(remoteip, 256, remotesa);

  /* Check the ACL */
  if(Ganglia_acl_action( channel->acl, remotesa ) != GANGLIA_ACCESS_ALLOW)
    goto close_accept_socket;

  /* Print the DTD, GANGLIA_XML and CLUSTER tags */
  status = print_xml_header(client);
  if(status != APR_SUCCESS)
    goto close_accept_socket;

  /* Walk the host hash */
  for(hi = apr_hash_first(client_context, hosts);
      hi;
      hi = apr_hash_next(hi))
    {
      apr_hash_this(hi, NULL, NULL, &val);
      status = print_host_start(client, (Ganglia_host *)val);
      if(status != APR_SUCCESS)
        {
          goto close_accept_socket;
        }

      /* Send the metric info for this particular host */
      for(metric_hi = apr_hash_first(client_context, ((Ganglia_host *)val)->metrics);
          metric_hi;
	  metric_hi = apr_hash_next(metric_hi))
        {
	  void *metric;
	  apr_hash_this(metric_hi, NULL, NULL, &metric);

	  /* Print each of the metrics for a host ... */
	  if(print_host_metric(client, metric, now) != APR_SUCCESS)
	    {
	      goto close_accept_socket;
	    }
	}

      /* Send the gmetric info for this particular host */
      for(metric_hi = apr_hash_first(client_context, ((Ganglia_host *)val)->gmetrics);
	  metric_hi;
	  metric_hi = apr_hash_next(metric_hi))
	{
	  void *metric;
	  apr_hash_this(metric_hi, NULL, NULL, &metric);

	  /* Print each of the metrics from gmetric for this host... */
	  if(print_host_gmetric(client, metric, now) != APR_SUCCESS)
	    {
	      goto close_accept_socket;
	    }
	}

      /* Close the host tag */
      status = print_host_end(client);
      if(status != APR_SUCCESS)
        {
	  goto close_accept_socket;
	}
    }

  /* Close the CLUSTER and GANGLIA_XML tags */
  print_xml_footer(client);

  /* Close down the accepted socket */
close_accept_socket:
  apr_shutdown(client, APR_SHUTDOWN_READ);
  apr_socket_close(client);
  apr_pool_destroy(client_context);
}


static void
poll_listen_channels( apr_interval_time_t timeout, apr_time_t now)
{
  apr_status_t status;
  const apr_pollfd_t *descs = NULL;
  apr_int32_t num = 0;
  apr_int32_t i;

  /* Poll for incoming data */
  status = apr_pollset_poll(listen_channels, timeout, &num, &descs);
  if(status != APR_SUCCESS)
    return;

  for(i = 0; i< num ; i++)
    {
      Ganglia_channel *channel = descs[i].client_data;
      switch( channel->type )
	{
	case UDP_RECV_CHANNEL:
          process_udp_recv_channel(descs+i, now); 
	  break;
	case TCP_ACCEPT_CHANNEL:
	  process_tcp_accept_channel(descs+i, now);
	  break;
	default:
	  continue;
	}
    }
}

static int
tcp_send_message( char *buf, int len )
{
  /* Mirror of UDP send message for TCP channels */
  return 0;
}

static int
send_message( char *buf, int len )
{
  return Ganglia_udp_send_message(udp_send_channels, buf, len ) + tcp_send_message( buf, len );
}

static Ganglia_metric_callback *
Ganglia_metric_cb_define(  char *name, g_val_t (*cb)(void))
{
  Ganglia_metric_callback *metric = apr_pcalloc( global_context, sizeof(Ganglia_metric_callback));
  if(!metric)
    return NULL;

  metric->name = apr_pstrdup( global_context, name );
  if(!metric->name)
    return NULL;

  metric->cb = cb;

  apr_hash_set( metric_callbacks, metric->name, APR_HASH_KEY_STRING, metric);
  return metric;
}

g_val_t
gexec_func ( void )
{
   g_val_t val;
   if( gexec_on )
      snprintf(val.str, 32, "%s", "ON");
   else
      snprintf(val.str, 32, "%s", "OFF");
   return val;
}

g_val_t
heartbeat_func( void )
{
   g_val_t val;
   val.uint32 = started / APR_USEC_PER_SEC;
   return val;
}

g_val_t
location_func(void)
{
   g_val_t val;
   if(!host_location)
     {
       cfg_t *host = cfg_getsec(config_file, "host");
       host_location = cfg_getstr( host, "location");
     }
   strncpy(val.str, host_location, 32);
   return val;
}


/* This function imports the metrics from libmetrics right now but in the future
 * we could easily do this via DSO. */
static void
setup_metric_callbacks( void )
{
  /* Initialize the libmetrics library in ./srclib/libmetrics */
  libmetrics_init();

  /* Create the metric_callbacks hash */
  metric_callbacks = apr_hash_make( global_context );

  /* All platforms support these metrics */
  Ganglia_metric_cb_define("cpu_num",        cpu_num_func);
  Ganglia_metric_cb_define("cpu_speed",      cpu_speed_func);
  Ganglia_metric_cb_define("mem_total",      mem_total_func);
  Ganglia_metric_cb_define("swap_total",     swap_total_func);
  Ganglia_metric_cb_define("boottime",       boottime_func);
  Ganglia_metric_cb_define("sys_clock",      sys_clock_func);
  Ganglia_metric_cb_define("machine_type",   machine_type_func);
  Ganglia_metric_cb_define("os_name",        os_name_func);
  Ganglia_metric_cb_define("os_release",     os_release_func);
  Ganglia_metric_cb_define("mtu",            mtu_func);
  Ganglia_metric_cb_define("cpu_user",       cpu_user_func);
  Ganglia_metric_cb_define("cpu_nice",       cpu_nice_func);
  Ganglia_metric_cb_define("cpu_system",     cpu_system_func);
  Ganglia_metric_cb_define("cpu_wio",        cpu_wio_func);
  Ganglia_metric_cb_define("cpu_intr",       cpu_intr_func);
  Ganglia_metric_cb_define("cpu_sintr",       cpu_sintr_func);
  Ganglia_metric_cb_define("cpu_idle",       cpu_idle_func);
  Ganglia_metric_cb_define("cpu_aidle",      cpu_aidle_func);
  Ganglia_metric_cb_define("load_one",       load_one_func);
  Ganglia_metric_cb_define("load_five",      load_five_func);
  Ganglia_metric_cb_define("load_fifteen",   load_fifteen_func);
  Ganglia_metric_cb_define("proc_run",       proc_run_func);
  Ganglia_metric_cb_define("proc_total",     proc_total_func);
  Ganglia_metric_cb_define("mem_free",       mem_free_func);
  Ganglia_metric_cb_define("mem_shared",     mem_shared_func);
  Ganglia_metric_cb_define("mem_buffers",    mem_buffers_func);
  Ganglia_metric_cb_define("mem_cached",     mem_cached_func);
  Ganglia_metric_cb_define("swap_free",      swap_free_func);
  Ganglia_metric_cb_define("bytes_in",       bytes_in_func);
  Ganglia_metric_cb_define("bytes_out",      bytes_out_func);
  Ganglia_metric_cb_define("pkts_in",        pkts_in_func);
  Ganglia_metric_cb_define("pkts_out",       pkts_out_func);
  Ganglia_metric_cb_define("disk_total",     disk_total_func);
  Ganglia_metric_cb_define("disk_free",      disk_free_func);
  Ganglia_metric_cb_define("part_max_used",  part_max_used_func);

  /* These are "internal" metrics for host heartbeat,location,gexec */
  Ganglia_metric_cb_define("heartbeat",      heartbeat_func);
  Ganglia_metric_cb_define("location",       location_func);
  Ganglia_metric_cb_define("gexec",          gexec_func);

  /* Add platform specific metrics here... */
#if SOLARIS
  Ganglia_metric_cb_define("bread_sec",      bread_sec_func);
  Ganglia_metric_cb_define("bwrite_sec",     bwrite_sec_func);
  Ganglia_metric_cb_define("lread_sec",      lread_sec_func);
  Ganglia_metric_cb_define("lwrite_sec",     lwrite_sec_func);
  Ganglia_metric_cb_define("phread_sec",     phread_sec_func);
  Ganglia_metric_cb_define("phwrite_sec",    phwrite_sec_func);
  Ganglia_metric_cb_define("rcache",         rcache_func);
  Ganglia_metric_cb_define("wcache",         wcache_func);
#endif

#if HPUX
  Ganglia_metric_cb_define("mem_arm",        mem_arm_func);
  Ganglia_metric_cb_define("mem_rm",         mem_rm_func);
  Ganglia_metric_cb_define("mem_avm",        mem_avm_func);
  Ganglia_metric_cb_define("mem_vm",         mem_vm_func);
#endif

}

double
setup_collection_groups( void )
{
  int i, num_collection_groups = cfg_size( config_file, "collection_group" );
  double bytes_per_sec = 0;
  
  /* Create the collection group array */
  collection_groups = apr_array_make( global_context, num_collection_groups,
				      sizeof(Ganglia_collection_group *));

  for(i = 0; i < num_collection_groups; i++)
    {
      int j, num_metrics;
      cfg_t *group_conf;
      Ganglia_collection_group *group = apr_pcalloc( global_context, 
						     sizeof(Ganglia_collection_group));
      if(!group)
	{
	  fprintf(stderr,"Unable to malloc memory for collection group. Exiting.\n");
	  exit(1);
	}

      group_conf  = cfg_getnsec( config_file, "collection_group", i);
      group->once = cfg_getbool( group_conf, "collect_once");
      group->collect_every = cfg_getint( group_conf, "collect_every");
      group->time_threshold = cfg_getint( group_conf, "time_threshold");

      if(group->once)
        {
          /* TODO: this isn't pretty but simplifies the code( next collect in a year)
             since we will collect the value in this function */
	  group->next_collect = apr_time_now() + (31536000 * APR_USEC_PER_SEC);
	}
      else
        {
          group->next_collect = 0;
        }

      group->next_send    = 0;

      num_metrics = cfg_size( group_conf, "metric" );
      group->metric_array = apr_array_make(global_context, num_metrics,
					  sizeof(Ganglia_metric_callback *)); 
      for(j=0; j< num_metrics; j++)
	{
	  cfg_t *metric         = cfg_getnsec( group_conf, "metric", j );
	  char *name            = cfg_getstr  ( metric, "name");
	  float value_threshold = cfg_getfloat( metric, "value_threshold");

	  Ganglia_metric_callback *metric_cb =  (Ganglia_metric_callback *)
	                apr_hash_get( metric_callbacks, name, APR_HASH_KEY_STRING );
	  Ganglia_25metric *metric_info = Ganglia_25metric_byname(name);

	  if(!metric_cb)
	    {
	      fprintf(stderr,"Unable to collect metric '%s' on this platform. Exiting.\n", name);
	      exit(1);
	    }

	  if(!metric_info)
	    {
	      fprintf(stderr,"Unable to send metric '%s' (not in protocol.x). Exiting.\n", name);
	      exit(1);
	    }

	  /* This sets the key for this particular metric.
	   * The value is set by the callback function later */
	  metric_cb->msg.id = metric_info->key;

	  /* Save the location of information about this particular metric */
	  metric_cb->info   = metric_info;

	  /* Set the value threshold for this particular metric */
	  metric_cb->value_threshold = value_threshold;

	  if(group->once)
	    {
	      /* If this metric will only be collected once, run it now at setup... */
	      metric_cb->now = metric_cb->cb();
	    }
	  else
	    {
	      /* ... otherwise set it to zero */
	      memset( &(metric_cb->now), 0, sizeof(g_val_t));
	    }
	  memset( &(metric_cb->last), 0, sizeof(g_val_t));

	  /* Calculate the bandwidth this metric will use */
	  bytes_per_sec += ( (double)metric_info->msg_size / (double)group->time_threshold );

	  /* Push this metric onto the metric_array for this group */
	  *(Ganglia_metric_callback **)apr_array_push(group->metric_array) = metric_cb;
	}

      /* Save the collection group the collection group array */
      *(Ganglia_collection_group **)apr_array_push(collection_groups) = group;
    }

  return bytes_per_sec;
}

void
Ganglia_collection_group_collect( Ganglia_collection_group *group, apr_time_t now)
{
  int i;

  /* Collect data for all the metrics in the groups metric array */
  for(i=0; i< group->metric_array->nelts; i++)
    {
      Ganglia_metric_callback *cb = ((Ganglia_metric_callback **)(group->metric_array->elts))[i];

      debug_msg("\tmetric '%s' being collected now", cb->name);
      cb->last = cb->now;
      cb->now  = cb->cb();

      /* Check the value threshold.  If passed.. set this group to send immediately. */
      if( cb->value_threshold >= 0.0 )
        {
	  debug_msg("\tmetric '%s' has value_threshold %f", cb->name, cb->value_threshold);
	  switch(cb->info->type)
	    {
	    case GANGLIA_VALUE_UNKNOWN:
	    case GANGLIA_VALUE_STRING:
	      /* do nothing for non-numeric data */
	      break;
	    case GANGLIA_VALUE_UNSIGNED_SHORT:
	      if( abs( cb->last.uint16 - cb->now.uint16 ) >= cb->value_threshold )
		group->next_send = 0; /* send immediately */
	      break;
	    case GANGLIA_VALUE_SHORT:
              if( abs( cb->last.int16 - cb->now.int16 ) >= cb->value_threshold )
		group->next_send = 0; /* send immediately */
	      break;
	    case GANGLIA_VALUE_UNSIGNED_INT:
              if( abs( cb->last.uint32 - cb->now.uint32 ) >= cb->value_threshold )
		group->next_send = 0; /* send immediately */
	      break;
	    case GANGLIA_VALUE_INT:
              if( abs( cb->last.int32 - cb->now.int32 ) >= cb->value_threshold )
		group->next_send = 0; /* send immediately */
	      break;
	    case GANGLIA_VALUE_FLOAT:
              if( abs( cb->last.f - cb->now.f ) >= cb->value_threshold )
		group->next_send = 0; /* send immediately */
	      break;
	    case GANGLIA_VALUE_DOUBLE:
              if( abs( cb->last.d - cb->now.d ) >= cb->value_threshold )
		group->next_send = 0; /* send immediately */
	      break;
	    default:
	      break;
	    }

	}

    }

  /* Set the next time this group should be collected */
  group->next_collect = now + (group->collect_every * APR_USEC_PER_SEC);
}

void
Ganglia_collection_group_send( Ganglia_collection_group *group, apr_time_t now)
{
  int i;

  /* This group needs to be sent */
  for(i=0; i< group->metric_array->nelts; i++)
    {
      XDR x;
      int len, errors;
      char metricmsg[max_udp_message_len];
      Ganglia_metric_callback *cb = ((Ganglia_metric_callback **)(group->metric_array->elts))[i];

      /* Build the message */
      switch(cb->info->type)
        {
	  case GANGLIA_VALUE_UNKNOWN:
	    /* The 2.5.x protocol doesn't allow for unknown values. :(  Do nothing. */
	    continue;
	  case GANGLIA_VALUE_STRING:
	    cb->msg.Ganglia_message_u.str = cb->now.str; 
	    break;
	  case GANGLIA_VALUE_UNSIGNED_SHORT:
	    cb->msg.Ganglia_message_u.u_short = cb->now.uint16;
	    break;
	  case GANGLIA_VALUE_SHORT:
	    /* TODO: currently there are no signed short values in protocol.x
	     * As soon as we add one, we need to implement it here... */
	    break;
	  case GANGLIA_VALUE_UNSIGNED_INT:
	    cb->msg.Ganglia_message_u.u_int = cb->now.uint32;
	    break;
	  case GANGLIA_VALUE_INT:
	    /* TODO: currently there are no signed int value in protocol.x
	     * As soon as we add one, we need to implement it here... */
	    break;
	  case GANGLIA_VALUE_FLOAT:
	    cb->msg.Ganglia_message_u.f = cb->now.f;
	    break;
	  case GANGLIA_VALUE_DOUBLE:
	    cb->msg.Ganglia_message_u.d = cb->now.d;
	    break;
	  default:
	    continue;
	}

      /* Send the message */
      xdrmem_create(&x, metricmsg, max_udp_message_len, XDR_ENCODE);
      xdr_Ganglia_message(&x, &(cb->msg));
      len = xdr_getpos(&x); 
      errors = send_message( metricmsg, len );
      debug_msg("\tsent message '%s' of length %d with %d errors", cb->name, len, errors);
      if(!errors)
	{
	  /* If the message send ok. Schedule the next time threshold. */
	  group->next_send = now + (group->time_threshold * APR_USEC_PER_SEC);
	}
    }
}
 
/* TODO: It might be necessary in the future to use a heap for the collection groups.
 * Running through an array should suffice for now */
apr_time_t
process_collection_groups( apr_time_t now )
{
  int i;
  apr_time_t next = 0;

  /* Run through each collection group and collect any data that needs collecting... */
  for(i=0; i< collection_groups->nelts; i++)
    {
      Ganglia_collection_group *group = ((Ganglia_collection_group **)(collection_groups->elts))[i];
      if(group->next_collect <= now)
        {
	  Ganglia_collection_group_collect(group, now);
	}
    }

  /* Run through each collection group and send any data that needs sending... */
  for(i=0; i< collection_groups->nelts; i++)
    {
      Ganglia_collection_group *group = ((Ganglia_collection_group **)(collection_groups->elts))[i];
      if( group->next_send <= now )
	{
	  Ganglia_collection_group_send(group, now);
	}
    }

  /* Run through each collection group and find when our next event (collect|send) occurs */
  for(i=0; i< collection_groups->nelts; i++)
    {
      apr_time_t min;
      Ganglia_collection_group *group = ((Ganglia_collection_group **)(collection_groups->elts))[i];
      min = group->next_send < group->next_collect? group->next_send : group->next_collect;
      if(!next)
	{
	  next = min;
	}
      else
	{
	  if(min < next)
	    {
	      next = min;
	    }
	}
    }

  /* make sure we don't schedule for the past */
  return next < now ? now + 1 * APR_USEC_PER_SEC: next;
}

static void
print_metric_list( void )
{
  apr_hash_index_t *hi;
  void *val;

  for(hi = apr_hash_first(global_context, metric_callbacks);
      hi;
      hi = apr_hash_next(hi))
    {
      Ganglia_metric_callback *cb;
      apr_hash_this(hi, NULL, NULL, &val);
      cb = val;
      fprintf(stdout, "%s\n", cb->name);
    }
}

static void
cleanup_data( Ganglia_pool pool, apr_time_t now)
{
  apr_hash_index_t *hi, *gmetric_hi;

  /* Walk the host hash */
  for(hi = apr_hash_first(pool, hosts);
      hi;
      hi = apr_hash_next(hi))
    {
      void *val;
      Ganglia_host *host;
      apr_hash_this(hi, NULL, NULL, &val);
      host = val;

      if( host_dmax && (now - host->last_heard_from) > (host_dmax * APR_USEC_PER_SEC) )
	{
	  /* this host is older than dmax... delete it */
	  debug_msg("deleting old host '%s' from host hash'", host->hostname);
	  /* remove it from the hash */
	  apr_hash_set( hosts, host->ip, APR_HASH_KEY_STRING, NULL);
	  /* free all its memory */
	  apr_pool_destroy( host->pool);
	} 
      else
	{
	  /* this host isn't being deleted but it might have some stale gmetric data */
	  for( gmetric_hi = apr_hash_first( pool, host->gmetrics );
	       gmetric_hi;
	       gmetric_hi = apr_hash_next( gmetric_hi ))
	    {
	      void *val;
	      Ganglia_metric *gmetric;
	      int dmax;

	      apr_hash_this( gmetric_hi, NULL, NULL, &val );
	      gmetric = val;

	      if(!gmetric)
		continue;  /* this shouldn't happen */

	      dmax = gmetric->message.Ganglia_message_u.gmetric.dmax;
	      if( dmax && (now - gmetric->last_heard_from) > (dmax * APR_USEC_PER_SEC) )
	        {
		  /* this is a stale gmetric */
		  debug_msg("deleting old gmetric '%s' from host '%s'", gmetric->name, host->hostname);
		  /* remove the metric from the gmetric hash */
		  apr_hash_set( host->gmetrics, gmetric->name, APR_HASH_KEY_STRING, NULL);
		  /* destroy any memory that was allocated for this gmetric */
		  apr_pool_destroy( gmetric->pool );
                }
	    }
	}
    }

  apr_pool_clear( pool );
}

int
main ( int argc, char *argv[] )
{
  apr_time_t now, next_collection, last_cleanup;
  Ganglia_pool cleanup_context;

  if (cmdline_parser (argc, argv, &args_info) != 0)
        exit(1) ;

  if(args_info.convert_given)
    {
      exit (print_ganglia_25_config( args_info.convert_arg ));
    }

  /* Create the global context */
  global_context = Ganglia_pool_create(NULL);

  /* Create the cleanup context from the global context */
  cleanup_context = Ganglia_pool_create(global_context);

  /* Mark the time this gmond started */
  started = apr_time_now();

  /* Builds a default configuration based on platform */
  build_default_gmond_configuration(global_context);

  if(args_info.default_config_flag)
    {
      fprintf(stdout, default_gmond_configuration);
      fflush( stdout );
      exit(0);
    }

  if(args_info.metrics_flag)
    {
      setup_metric_callbacks();
      print_metric_list();
      fflush( stdout );
      exit(0);
    }

  process_configuration_file();

  if(args_info.bandwidth_flag)
    {
      double bytes_per_sec;
      setup_metric_callbacks();
      bytes_per_sec = setup_collection_groups();
      fprintf(stdout, "%f bytes/sec\n", bytes_per_sec);
      exit(0);
    }

  if(args_info.location_given)
    {
      host_location = args_info.location_arg;
    }

  daemonize_if_necessary( argv );

  if (args_info.pid_file_given)
    {
      update_pidfile (args_info.pid_file_arg);
    }
  
  /* Collect my hostname */
  apr_gethostname( myname, APRMAXHOSTLEN+1, global_context);

  apr_signal( SIGPIPE, SIG_IGN );

  /* This must occur before we setuid_if_necessary() particularly on freebsd
   * where we need to be root to access /dev/mem to initialize metric collection */
  setup_metric_callbacks();

  setuid_if_necessary(); 

  process_deaf_mute_mode();

  if(!deaf)
    {
      setup_listen_channels_pollset();
    }

  if(!mute)
    {
      setup_collection_groups();
      udp_send_channels = Ganglia_udp_send_channels_create( global_context, config_file );
    }

  if(!udp_send_channels)
    {
      /* if there are no send channels defined, we are equivalent to mute */
      mute = 1;
    }
  if(!listen_channels)
    {
      /* if there are no listen channels defined, we are equivalent to deaf */
      deaf = 1;
    }

  /* Create the host hash table */
  hosts = apr_hash_make( global_context );

  /* Initialize time variables */
  last_cleanup = next_collection = now = apr_time_now();

  /* Loop */
  for(;;)
    {
      /* Make sure we never wait for negative seconds (shouldn't happen) */
      apr_interval_time_t wait = next_collection >= now ? next_collection - now : 1;
      if(!deaf)
	{
	  /* Pull in incoming data */
	  poll_listen_channels(wait, now);
	}
      else
	{
	  /* Sleep until next collection */
          apr_sleep( wait );
	}

      /* only continue if it's time to process our collection groups */
      now = apr_time_now();
      if(now < next_collection)
	continue;

      if(!deaf)
	{
	  /* cleanup the data if the cleanup threshold has been met */
	  if( (now - last_cleanup) > cleanup_threshold )
	    {
	      cleanup_data( cleanup_context, now );
	      last_cleanup = now;
	    }
	}

      if(!mute)
	{
	  /* collect data from collection_groups */
	  next_collection = process_collection_groups( now );
	}
      else
	{
	  /* we're mute. nothing to collect and send. */
	  next_collection = now + 60 * APR_USEC_PER_SEC;
	}
    }

  return 0;
}
