/*******************************************************************************
*---|
*---|     Garnaxbot by Freakster http://garnax.mircx.com/
*---|
*---|     Garnaxbot is distributed in the hope that it will be useful,
*---|     but WITHOUT ANY WARRANTY; without even the implied warranty of
*---|     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*---|
********************************************************************************
*---|
*---|     File:            fserve.c
*---|     Created:         Fri Dec 13 23:09:29 CST 2002
*---|     Last Edited:
*---|     Description:     This file contains the declarations of the
*---|                      basic structures of the fserve.
*---|
*******************************************************************************/

#include "fserve.h"

/*******************************************************************************
*---|     Function:        FSERVE_AD
*---|     Created:         Wed Aug  6 21:47:19 CDT 2003
*---|     Last Edited:
*---|     Description:     Puts an fserve ad into a buffer.
*******************************************************************************/

char *fserve_ad(channelstruct *chan)
{
    int i = 0;
    char *ad = (char *)malloc(sizeof(char) * BUFFSIZE), *size = NULL, *br = NULL;

    if (ad == NULL)
    {
        return NULL;
    }

    pthread_mutex_lock(&mutex);
#ifdef debug

    logprintf( "MUTEX LOCK: FSERVE_AD\n");
#endif

    /* Do not remove this */
    ad[i++] = 3;
    ad[i++] = bot->color1[0];
    ad[i++] = bot->color1[1];
    ad[i++] = '\0';

    /* [Fserve] */
    br = bracket("%s", "Fserve");
    strcat(ad, br);
    free(br);

    /* Trigger */
    strcat(ad, " Trigger:");
    if (bot->settings & CTCPTRIG)
    {
        br = bracket("/ctcp %s %s", bot->nick, chan->trigger);
    }
    else
    {
        br = bracket("%s", chan->trigger);
    }
    strcat(ad, br);
    free(br);

    /* Fserve Activity */
    strcat(ad, " Online:");
    br = bracket("%d/%d", bot->fserves->count, bot->fserves->total);
    strcat(ad, br);
    free(br);

    /* Send Activity */
    strcat(ad, " Sends:");
    br = bracket("%d/%d", bot->sends->count, bot->sends->total);
    strcat(ad, br);
    free(br);

    /* Queue Activity */
    strcat(ad, " Queues:");
    br = bracket("%d/%d", bot->queues->count, bot->queues->total);
    strcat(ad, br);
    free(br);

    /* Data Sent */
    strcat(ad, " Sent:");
    size = filesize(bot->bytecount, bot->overflowcount);
    br = bracket("%s in %d transfers", size, bot->filecount + bot->failcount);
    free(size);
    strcat(ad, br);
    free(br);

    /* Record Speed */
    strcat(ad, " Record:");
    size = filesize(bot->maxspeed, 0);
    br = bracket("%sPS", size);
    free(size);
    strcat(ad, br);
    free(br);

    /* Average Speed */
    strcat(ad, " Average:");
    size = filesize(bot->avespeed, 0);
    br = bracket("%sPS", size);
    free(size);
    strcat(ad, br);
    free(br);

    /* Description */
    strcat(ad, " Desc:");
    br = bracket("%s", chan->desc);
    strcat(ad, br);
    free(br);

    /* Bot Version (Please don't remove) */
    strcat(ad, " ");
    br = bracket("%s %s", MODEL, VERSION);
    strcat(ad, br);
    free(br);

    /* Ends the color code */
    if (bot->color1)
    {
        i = strlen(ad);
        ad[i++] = 3;
        ad[i++] = '\0';
    }

    br = NULL;
    size = NULL;

    pthread_mutex_unlock(&mutex);
#ifdef debug

    logprintf( "MUTEX RELEASE: FSERVE_AD\n");
#endif

    return ad;
}

/*******************************************************************************
*---|     Function:        FILESIZE
*---|     Created:         Wed Aug  6 00:26:26 CDT 2003
*---|     Last Edited:
*---|     Description:     Given a size in bytes and an overflow count, it
*---|                      returns how many bytes it equals to in readable form.
*******************************************************************************/

char *filesize(unsigned int size, unsigned int overflow)
{
    int state = 0;
    char buff[BUFFSIZE], type = ' ';
    float newsize = (float)size;

    /* an overflow constitutes that an unsigned int has overrun it's 4 gig maximimum
    ** so magically jump to that state and add what is left
    */
    if (overflow)
    {
        newsize = (overflow * 4.0) + (newsize / 1024.0 / 1024.0 / 1024.0);
        state = 3;
    }

    while ((state < 4) && ((newsize / 1024.0) > 1))
    {
        newsize = newsize / 1024.0;
        state++;
    }

    switch (state)
    {
    case 1:
        type = 'K';
        break;

    case 2:
        type = 'M';
        break;

    case 3:
        type = 'G';
        break;

    case 4:
        type = 'T';
        break;

    default:
        break;
    }

    sprintf(buff, "%.1f %cB", newsize, type);

    return mymalloc(buff);
}

/*******************************************************************************
*---|     Function:        BRACKET
*---|     Created:         Wed Aug  6 21:47:19 CDT 2003
*---|     Last Edited:
*---|     Description:     Place text within colored brackets.
*******************************************************************************/

char *bracket(char *msg, ...)
{

    va_list ap;
    char buff[BUFFSIZE];
    char *bracketed = NULL;

    va_start(ap, msg);

    /* vsprintf(buff, msg, ap); */
    vsnprintf(buff, BUFFSIZE, msg, ap);

    va_end(ap);

    if (bot->color1)
    {
        bracketed = (char*)malloc(sizeof(char) * (10 + strlen(bot->ob) + strlen(bot->cb) + strlen(buff)));

        if (bracketed)
        {
            sprintf(bracketed, "%s%c%s%s%c%s%s", bot->ob, 3, bot->color2, buff, 3, bot->color1, bot->cb);
        }
    }
    else
    {
        bracketed = (char*)malloc(sizeof(char) * (1 + strlen(bot->ob) + strlen(bot->cb) + strlen(buff)));
        sprintf(bracketed, "%s%s%s", bot->ob, buff, bot->cb);
    }

    return bracketed;
}

/*******************************************************************************
*---|     Function:        RESUME_FILE
*---|     Created:         Wed Aug  6 23:31:23 CDT 2003
*---|     Last Edited:
*---|     Description:     Takes in a DCC resume and sets the offset
*---|                      for the corresponding transfer.
*******************************************************************************/

int resume_file(char *host, int dccport, int offset)
{

    int status = SUCCESS;
    unsigned short port = (unsigned short)dccport;
    dccstruct *dcc;

    pthread_mutex_lock(&mutex);
#ifdef debug

    logprintf( "MUTEX LOCK: RESUME_FILE\n");
#endif

    for (dcc = bot->sends->start; dcc != NULL; dcc = dcc->next)
    {
        if ((dcc->port == port) && (lowercasecmp(host, dcc->host) == 0))
        {
            dcc->offset = offset;
            status = quote(bot->sock, "PRIVMSG %s :\001DCC ACCEPT file.ext %i %d\001\r\n", dcc->nick, port, offset);
            break;
        }
    }

    pthread_mutex_unlock(&mutex);
	#ifdef debug
    logprintf( "MUTEX RELEASE: RESUME_FILE\n");
	#endif

    return status;
}

/*******************************************************************************
*---|   Function:      FSERVE
*---|   Created:       Wed Aug  6 23:49:16 CDT 2003
*---|   Last Edited:   Sun Nov 30 15:57:21 CST 2003
*---|   Description:   Starts an fserve with someone.
*******************************************************************************/

int fserve(channelstruct *chan, char *nick, char *host)
{
    dccstruct *fserve = NULL;
    int status = SUCCESS;

    /* Invalid request, but keep on going */
    if ((chan == NULL) || (nick == NULL) || (host == NULL))
    {
		#ifdef debug
        logprintf("ERROR: Bad chan, nick, or host for fserve.\n");
		#endif

        return SUCCESS;
    }

    pthread_mutex_lock(&mutex);
	#ifdef debug
    logprintf("MUTEX LOCK: FSERVE\n");
    logprintf("fservecount = %d, fservetotal = %d, fserveach = %d.\n", bot->fserves->count, bot->fserves->total, bot->fserves->each);
    logprintf("Channel:[%s] Nick:[%s] Host [%s] is trying to use the fserve.\n", chan->name, nick, host);
	#endif

    if (bot->fserves->count >= bot->fserves->total)
    {
        status = quote(bot->sock, "NOTICE %s :Sorry, All fserve slots [%d] are full. Try again later.\r\n", nick, bot->fserves->total);
    }
    else if (in_use(nick, host, bot->fserves, "fserves") >= bot->fserves->each)
    {
        status = quote(bot->sock, "NOTICE %s :Sorry, all of your max fserve slots [%d] are full.  Try again in a minute.\r\n", nick, bot->fserves->each);
    }
    else
    {
        fserve = create_dcc();
        if (fserve == NULL)
        {
            status = FAILURE;
        }
        else
        {
            fserve->nick = mymalloc(nick);
            fserve->host = mymalloc(host);
            fserve->chan = mymalloc(chan->name);

            /* any mallocs failed? */
            if (!(fserve->nick) || !(fserve->host) || !(fserve->chan))
            {

                if (fserve->nick)
				free(fserve->nick);
                fserve->nick = NULL;

                if (fserve->host)
				free(fserve->host);
                fserve->host = NULL;

                if (fserve->chan)
				free(fserve->chan);
                fserve->chan = NULL;

                free(fserve);
                status = FAILURE;
            }
            else
            {

                fserve->lastdata = time(NULL);
                fserve->state = DCCACCEPT;
                fserve->type = DCCCHAT;

                status = bind_listen(fserve);
                if (status == FAILURE)
                {
                    fserve = destroy_dcc(fserve);
                }
                else
                {
                    /* add it to the list of fserves for the fserve_thread to handle */
                    add_dcc(fserve, bot->fserves);
                }

            }
        }
    }

    pthread_mutex_unlock(&mutex);
#ifdef debug

    logprintf("MUTEX RELEASE: FSERVE\n");
#endif

    return status;
}

/*******************************************************************************
*---|     Function:        IN_USE
*---|     Created:         Thu Aug  7 00:00:21 CDT 2003
*---|     Last Edited:
*---|     Description:     Checks to see if a user/host is already using a dcc.
*******************************************************************************/

int in_use(char *nick, char *host, dccptr *dcclist, char *type)
{
    int ctr = 0;
    dccstruct *ptr = NULL;

#ifdef debug

    logprintf( "Searching %s for matches to %s and %s\n", type, nick, host);
#endif

    for (ptr = dcclist->start; ptr != NULL; ptr = ptr->next)
    {
#ifdef debug
        logprintf( "Comparing [%s] to [%s] and [%s] to [%s]\n", nick, ptr->nick, host, ptr->host);
#endif

        if ((lowercasecmp(nick, ptr->nick) == 0) || (lowercasecmp(host, ptr->host) == 0))
        {
            ctr++;
        }
    }

#ifdef debug
    logprintf( "There are %d %s in use by %s or %s.\n", ctr, type, nick, host);
#endif

    ptr = NULL;
    return ctr;
}

/*******************************************************************************
*---|   Function:      CREATE_DCC
*---|   Created:       Wed Aug  6 23:49:16 CDT 2003
*---|   Last Edited:
*---|   Description:   Claims memory for a dcc struct.
*******************************************************************************/

dccstruct *create_dcc()
{

    dccstruct *newdcc = (dccstruct*)malloc(sizeof(dccstruct));

    if (newdcc)
    {
        newdcc->chan = NULL;
        newdcc->nick = NULL;
        newdcc->host = NULL;
        newdcc->path = NULL;
        newdcc->filename = NULL;
        newdcc->sock = 0;
        newdcc->size = 0;
        newdcc->starttime = 0;
        newdcc->lastdata = 0;
        newdcc->warned = 0;
        newdcc->next = NULL;
        newdcc->prev = NULL;
        newdcc->head = NULL;
        newdcc->tail = NULL;
        newdcc->offset = 0;
        newdcc->ctr = 0;
        newdcc->port = 0;
        newdcc->die = 0;
        newdcc->ackval = 0;
        newdcc->speed = 0;
        newdcc->state = 0;
        newdcc->type = 0;
        newdcc->file = NULL;
        newdcc->dir_name = NULL;
        newdcc->dir_path = NULL;
    }

    return newdcc;
}

/*******************************************************************************
*---|   Function:      BIND_AND_LISTEN
*---|   Created:       Fri Aug 15 10:46:35 CDT 2003
*---|   Last Edited:   Sun Nov 30 16:04:08 CST 2003
*---|   Description:   Create a socket to listen for an 
*---|                  incoming dcc connection.
*******************************************************************************/

int bind_listen(dccstruct *dcc)
{
    struct sockaddr_in my_addr;
    int status = 0, oldport;
    static int port = -1;

    /*pthread_mutex_lock(&mutex);
    */

    if (port <= bot->minport) port = bot->minport + 1;
    else if (port >= bot->maxport) port = bot->minport + 1;

	#ifdef debug
    logprintf( "Trying To find a port starting with %d\n", port);
	#endif

    /* oldport is equal to the current value of the static port num */
    oldport = port;

    if ((dcc->sock = socket(AF_INET, SOCK_STREAM, 0)) == -1)
    {
#ifdef debug
        logprintf("DCC Socket Error - [%s]\n", strerror(errno) );
#endif

        return FAILURE;
    }

    memset(&my_addr, 0, sizeof(struct sockaddr_in));
    my_addr.sin_family = AF_INET;
    my_addr.sin_addr.s_addr = htonl(INADDR_ANY);

    /* Find an available port */
    do
    {
        my_addr.sin_port = htons(port);

        if (bind(dcc->sock, (struct sockaddr *)&my_addr, sizeof(my_addr)) == -1)
        {
            if (errno != EADDRINUSE)
            {
#ifdef debug
                logprintf( "DCC Bind Error: Can't bind to port [%d] - %s\n", port, strerror(errno) );
#endif

                status = 1;
            }
        }
        else
        {
            /* success! */
            status = 2;
        }

        port++;
        if (port >= bot->maxport)
        {
            port = bot->minport + 1;
        }

        /* we've looped around */
        if (port == oldport)
        {
#ifdef debug
            logprintf( "Could not find an open port between %d and %d\n", bot->minport, bot->maxport);
#endif

            status = 1;
        }
    }
    while (status == 0);

    /*pthread_mutex_unlock(&mutex);
    */

    /* No port found or error. */
    if (status == 1)
    {
        close(dcc->sock);
        return FAILURE;
    }

    if (listen(dcc->sock, 1) == -1)
    {
#ifdef debug
        logprintf( "Listen Error [%s]\n", strerror(errno) );
#endif

        close(dcc->sock);
        return FAILURE;
    }

    if (dcc->type == DCCCHAT)
    { /* fserve */

        status = quote(bot->sock, "PRIVMSG %s :%cDCC CHAT CHAT %lu %i%c\r\n", dcc->nick, 1, htonl(bot->ip), ntohs(my_addr.sin_port), 1);

    }
    else if (dcc->type == DCCSEND)
    { /* send */

        dcc->port = ntohs(my_addr.sin_port);
        status = quote(bot->sock, "PRIVMSG %s :%cDCC SEND \"%s\" %lu %i %d%c\r\n", dcc->nick, 1, dcc->filename, htonl(bot->ip), dcc->port, dcc->size, 1);

    }

    if (status == FAILURE)
    {
        close(dcc->sock);
        return FAILURE;
    }

    return status;
}

/*******************************************************************************
*---|   Function:      DCC_ACCEPT
*---|   Created:       Sat Dec  6 17:50:29 CST 2003
*---|   Last Edited:   
*---|   Description:   Establishes a dcc chat.
*******************************************************************************/

int dcc_accept(dccstruct *dcc)
{

    unsigned int sin_size = sizeof(struct sockaddr_in);
    struct sockaddr_in their_addr;
    int status = 0, sock = accept(dcc->sock, (struct sockaddr *) & their_addr, &sin_size);

    /* close the old socket */
    close(dcc->sock);

    /* replace dcc sock with the new socket */
    if (sock >= 0)
    {
#ifdef debug
        logprintf( "DCC Established with %s\n", dcc->nick);
#endif

        dcc->sock = sock;
        if (dcc->type == DCCCHAT)
        {
            dcc->state = DCCCHAT;
        }
        else if (dcc->type == DCCSEND)
        {
            dcc->state = DCCSEND;
        }
        status = SUCCESS;
    }
    else
    {
#ifdef debug
        logprintf( "DCC Accept Error with %s - %s\n", dcc->nick, strerror(errno));
#endif

        dcc->sock = 0;
        status = FAILURE;

    }

    return status;

}

/*******************************************************************************
*---|     Function:        PARSE_DCCCHAT
*---|     Created:         Thu Aug  7 00:53:49 CDT 2003
*---|     Last Edited:
*---|     Description:     Handles the dcc chat.
*******************************************************************************/

int fserve_entry(dccstruct *fserve)
{
    int status = FAILURE;

    bot->accesscount++;
    fserve->starttime = time(NULL);

    if ((quote(fserve->sock, "%c%s%s %s Windows/Unix/Linux IRC Fileserver\r\n", 3, bot->color1, MODEL, VERSION)) &&
		(quote(fserve->sock, "%c%sThe filesystem is case sensitive under unix/linux.\r\n", 3, bot->color1)) &&
		(quote(fserve->sock, "%c%sCommands: %c%scd pwd dir ls get who sends queues del_queue clr_queues help bye quit exit\r\n", 3, bot->color1, 3, bot->color2)) &&
		(readmessage(fserve)))
    {
        status = SUCCESS;
        fserve->state = DCCCHAT;
    }

    return status;
}

/*******************************************************************************
*---|     Function:        READMESSAGE
*---|     Created:         Thu Aug  7 01:00:21 CDT 2003
*---|     Last Edited:
*---|     Description:     Reads the welcome message and displayes 
*---|                      it to the fserve.
*******************************************************************************/

int readmessage(dccstruct *fserve)
{

    char buff[129] = { '\0' };
    int status = SUCCESS, i = 0;
    FILE *welcome = NULL;
    channelstruct *chan = is_channel(fserve->chan);

    if ((chan == NULL) || (chan->welcomemsg == NULL))
    {
        return SUCCESS;
    }

    welcome = fopen(chan->welcomemsg, "r");

    if (welcome == NULL)
    {
		#ifdef debug
        logprintf( "Couldn't open welcome file %s\n", chan->welcomemsg);
		#endif

        return SUCCESS;
    }

    while ((status) && (fgets(buff, 128, welcome) != NULL))
    {
        i = strlen(buff);

        if (buff[i - 1] == '\n')
        {
            buff[i - 1] = ' ';
            status = quote(fserve->sock, "%s\r\n", buff);
        }
        else
        {
            status = quote(fserve->sock, "%s", buff);
        }
        if (status && feof(welcome))
        {
            status = quote(fserve->sock, "\r\n");
        }
    }

    fclose(welcome);

    return status;

}

/*******************************************************************************
*---|   Function:      CHAT_READ
*---|   Created:       Thu Aug  7 00:53:49 CDT 2003
*---|   Last Edited:   Sat Dec  6 18:45:32 CST 2003
*---|   Description:   Reads in input from a dcc chat.
*******************************************************************************/

int chat_read(dccstruct *fserve)
{
    int i = 0;
    char buff[BUFFSIZE] = { '\0' };

    do
    {
        if (recv(fserve->sock, &buff[i++], 1, 0) <= 0)
        {
            return FAILURE;
        }
    }
    while ((buff[i - 1] != '\n') && (i < BUFFSIZE));
    buff[i - 1] = '\0';

    if ((i > 1) && (buff[i - 2] == '\r')) {
        buff[i - 2] = '\0';
	}

    return chat_parse(fserve, buff);

}

/*******************************************************************************
*---|     Function:        CHAT_PARSE
*---|     Created:         Thu Aug  7 00:53:49 CDT 2003
*---|     Last Edited:
*---|     Description:     Determines what to do with the text given as input.
*******************************************************************************/

int chat_parse(dccstruct *fserve, char *buff)
{
    int fileptr, status = SUCCESS;
    char *cmd = NULL, *args = NULL, *tmp = NULL, *cwd = NULL;
	#ifdef WIN32
    WIN32_FIND_DATA findData;
    HANDLE hFind;
	#else
    struct stat buf;
	#endif

	#ifdef debug
    logprintf( "Recieved from %s: [%s]\n", fserve->nick, buff);
	#endif

    /* Remove Leading spaces */
    for (cmd = &buff[0]; cmd[0] == ' '; cmd = &cmd[1]);
    if ((cmd == NULL) || (cmd[0] == '\0'))
    {
        return quote(fserve->sock, "%c%sInvalid Command.\r\n", 3, bot->color1);
    }

    /* find the next space or end */
    for (tmp = cmd; ((tmp[0] != ' ') && (tmp[0] != '\0')); tmp = &tmp[1]);

    if (tmp[0] != '\0')
    {
        args = &tmp[1];
        tmp[0] = '\0';
        if (args[0] == '\0') args = NULL;
    }
    else
    {
        args = NULL;
    }

#ifdef debug
    logprintf( "cmd = [%s]\n", cmd);
    if (args) logprintf( "args = [%s]\n", args);
#endif


    if (lowercasecmp(cmd, "bye") == 0 || lowercasecmp(cmd, "quit") == 0 || lowercasecmp(cmd, "exit") == 0)
    {
        return FAILURE;
    }
    else if ((lowercasecmp(cmd, "ls") == 0) || (lowercasecmp(cmd, "dir") == 0) || (lowercasecmp(cmd, "ll") == 0))
    {
        return senddirlist(fserve);
    }
    else if (lowercasecmp(cmd, "pwd") == 0)
    {
        return pwd(fserve);
    }
    else if (lowercasecmp(cmd, "help") == 0)
    {
        if (quote(fserve->sock, "%c%sNOTE: %c%sThe filesystem is case sensitive\r\n", 3, bot->color1, 3, bot->color2) &&
			quote(fserve->sock, "%c%s cd - %c%sUse cd to traverse directories. (cd newfolder)\r\n", 3, bot->color1, 3, bot->color2) &&
			quote(fserve->sock, "%c%s pwd - %c%sPrints out your current path\r\n", 3, bot->color1, 3, bot->color2) &&
			quote(fserve->sock, "%c%s dir - %c%sLists the directories contents.\r\n", 3, bot->color1, 3, bot->color2) &&
			quote(fserve->sock, "%c%s ls - %c%sLists the directories contents.\r\n", 3, bot->color1, 3, bot->color2) &&
			quote(fserve->sock, "%c%s get - %c%sRetrieves a file. (get filename)\r\n", 3, bot->color1, 3, bot->color2) &&
			quote(fserve->sock, "%c%s who - %c%sLists all of the current users on the fserve.\r\n", 3, bot->color1, 3, bot->color2) &&
			quote(fserve->sock, "%c%s sends - %c%sLists all of the current sends in progress.\r\n", 3, bot->color1, 3, bot->color2) &&
			quote(fserve->sock, "%c%s queues - %c%sLists all of the queues waiting to be sent.\r\n", 3, bot->color1, 3, bot->color2) &&
			quote(fserve->sock, "%c%s del_queue - %c%sRemoves a specific queue if your nick and host match. (del_queue #)\r\n", 3, bot->color1, 3, bot->color2) &&
			quote(fserve->sock, "%c%s clr_queues - %c%sRemoves all of your queues.\r\n", 3, bot->color1, 3, bot->color2) &&
			quote(fserve->sock, "%c%s help - %c%sDisplays this help guide.\r\n", 3, bot->color1, 3, bot->color2) &&
			quote(fserve->sock, "%c%s bye - %c%sEnds an fserve session.\r\n", 3, bot->color1, 3, bot->color2) &&
			quote(fserve->sock, "%c%s quit - %c%sEnds an fserve session.\r\n", 3, bot->color1, 3, bot->color2) &&
			quote(fserve->sock, "%c%s exit - %c%sEnds an fserve session.\r\n", 3, bot->color1, 3, bot->color2))
        {
            return SUCCESS;
        }
        else
        {
            return FAILURE;
        }
    }
    else if (lowercasecmp(cmd, "who") == 0)
    {
        return show_dccs(fserve->sock, NULL, NULL, 0, "Fserve");
    }
    else if (lowercasecmp(cmd, "sends") == 0)
    {
        return show_dccs(fserve->sock, NULL, NULL, 0, "Send");
    }
    else if (lowercasecmp(cmd, "queues") == 0)
    {
        return show_dccs(fserve->sock, NULL, NULL, 0, "Queue");
    }
    else if (lowercasecmp(cmd, "del_queue") == 0)
    {
        if (args) return del_queue(fserve->sock, atoi(args), fserve->nick, fserve->host);
        else return quote(fserve->sock, "%c%sSyntax: %c%sdel_queue #\r\n", 3, bot->color1, 3, bot->color2);
    }
    else if (lowercasecmp(cmd, "clr_queues") == 0)
    {
        return clear_queues(fserve->nick, fserve->sock, NULL, NULL);
    }
    else if (lowercasecmp(cmd, "get") == 0)
    {
        if (args == NULL)
        {
            return quote(fserve->sock, "%c%sSyntax: %c%sget filename\r\n", 3, bot->color1, 3, bot->color2);
        }
	
		for (fileptr = strlen(args) - 1; fileptr >= 0; fileptr--)
		{
			#ifdef WIN32
			if ((args[fileptr] == '/') || (args[fileptr] == '\\'))
			#else
			if (args[fileptr] == '/')
			#endif
			{
				return quote(fserve->sock, "%c%sYou are only allowed to get files from the current directory.\r\n", 3, bot->color1);
			}
		}
		if (fserve->head == NULL)
		{
			return quote(fserve->sock, "%c%sInvalid File [%c%s%s%c%s]\r\n%c%sVerify case and space sensitity.\r\n",
						 3, bot->color1, 3, bot->color2, args, 3, bot->color1, 3, bot->color1);
		}

		cwd = getpath(fserve->head);
		if (cwd == NULL)
		{
			return FAILURE;
		}

		tmp = (char*)malloc(sizeof(char) * (strlen(cwd) + strlen(args) + 1) );

		if (tmp == NULL)
		{
			free(cwd);
			cwd = NULL;
			return FAILURE;
		}

		fileptr = strlen(cwd);
		sprintf(tmp, "%s%s", cwd, args);

		#ifdef debug
		logprintf( "%s is trying to get file %s\n", fserve->nick, tmp);
		#endif

#ifdef WIN32

		pthread_mutex_lock(&mutex);
		if (SetCurrentDirectory(cwd))
		{
			hFind = FindFirstFile(args, &findData);
			if (hFind)
			{
				if (FILE_ATTRIBUTE_DIRECTORY & GetFileAttributes(findData.cFileName))
				{
					status = quote(fserve->sock, "%c%sYou can not get a directory [%c%s%s%c%s] Use cd to change to that directory.\r\n",
								   3, bot->color1, 3, bot->color2, args, 3, bot->color1);
				}
				else
				{
					status = send_handler(fserve, tmp, fileptr, findData.nFileSizeLow);
				}
			}
			else
			{
				status = quote(fserve->sock, "%c%sInvalid File [%c%s%s%c%s]\r\n%c%sVerify case and space sensitity.\r\n",
							   3, bot->color1, 3, bot->color2, args, 3, bot->color1, 3, bot->color1);
			}
		}
		SetCurrentDirectory(botdir);
		pthread_mutex_unlock(&mutex);

#else

		if (stat(tmp, &buf) != 0)
		{
			status = quote(fserve->sock, "%c%sInvalid File [%c%s%s%c%s]\r\n%c%sVerify case and space sensitity.\r\n",
						   3, bot->color1, 3, bot->color2, args, 3, bot->color1, 3, bot->color1);
		}
		else if ( buf.st_mode & S_IFDIR )
		{
			status = quote(fserve->sock, "%c%sYou can not get a directory [%c%s%s%c%s]\r\n%c%sUse cd to change to that directory.\r\n",
						   3, bot->color1, 3, bot->color2, args, 3, bot->color1, 3, bot->color1);
		}
		else if ( !(buf.st_mode & S_IFREG))
		{
			status = quote(fserve->sock, "%c%sInvalid File [%c%s%s%c%s]\r\n%c%sVerify case and space sensitity.\r\n",
						   3, bot->color1, 3, bot->color2, args, 3, bot->color1, 3, bot->color1);
		}
		else
		{
			status = send_handler(fserve, tmp, fileptr, buf.st_size);
		}

#endif

		free(tmp);
		tmp = NULL;

		free(cwd);
		cwd = NULL;

    }
    else if (lowercasecmp(cmd, "cd..") == 0)
    {
        removepath(fserve);
        status = pwd(fserve);
    }
    else if (lowercasecmp(cmd, "cd") == 0)
    {
        if (args)
        {
            if (strcmp(args, "/") == 0)
            {
                while (fserve->head)
                {
                    removepath(fserve);
                }
                return pwd(fserve);
            }
            for (fileptr = strlen(args) - 1; fileptr >= 0 ; fileptr--)
            {
				#ifdef WIN32
                if ((args[fileptr] == '/') || (args[fileptr] == '\\'))
				#else
                if (args[fileptr] == '/')
				#endif
                {
                    return quote(fserve->sock, "%c%sYou are only allowed to traverse one directory at a time.\r\n", 3, bot->color1);
                }
            }
            if (strcmp(args, ".") == 0)
            {
                status = pwd(fserve);
            }
            else if (strcmp(args, "..") == 0)
            {
                removepath(fserve);
                status = pwd(fserve);
            }
			#ifdef WIN32
            else if (strcmp(args, "...") == 0)
            {
                status = quote(fserve->sock, "%c%sInvalid Folder [%c%s%s%c%s]\r\n%c%sVerify case and space sensitivity\r\n",
                               3, bot->color1, 3, bot->color2, args, 3, bot->color1, 3, bot->color2);
            }
			#endif
            else
            {
                status = change_directory(fserve, args);
            }
        }
        else
        {
            status = quote(fserve->sock, "%c%sSyntax: %c%scd new directory\r\n", 3, bot->color1, 3, bot->color2);
        }
    }
    else
    {
        status = quote(fserve->sock, "%c%sInvalid Command.\r\n", 3, bot->color1);
    }
    return status;
}

/*******************************************************************************
*---|     Function:        CHANGE_DIRECTORY
*---|     Created:         Thu Aug  7 20:26:10 CDT 2003
*---|     Last Edited:     Thu Dec 11 15:56:55 CST 2003
*---|     Description:     Changes to a new directory.
*******************************************************************************/

int change_directory(dccstruct *fserve, char *newpath)
{

    channelstruct *chan = is_channel(fserve->chan);
    dirstruct *dir = NULL;
    char *cwd = NULL;
    int status = SUCCESS;
#ifdef WIN32
    int good = 1;
#else
    struct stat buf;
#endif

    if (chan == NULL)
    {
        return FAILURE;
    }

    if (fserve->head == NULL)
    {
        for (dir = chan->directories; dir; dir = dir->next)
        {
            if (strcmp(newpath, dir->name) == 0)
            {
                break;
            }
        }
        if (dir == NULL)
        {
            return quote(fserve->sock, "%c%sInvalid Folder [%c%s%s%c%s]\r\n%c%sVerify case and space sensitivity\r\n",
                         3, bot->color1, 3, bot->color2, newpath, 3, bot->color1, 3, bot->color2);
        }
        fserve->dir_name = mymalloc(dir->name);
		if (!fserve->dir_name) {
			return FAILURE;
		}

        fserve->dir_path = mymalloc(dir->dir);
		if (!fserve->dir_path) {
			free(fserve->dir_name);
			return FAILURE;
		}

        fserve->head = (pathstruct*)malloc(sizeof(pathstruct));
		if (!fserve->head) {
			free(fserve->dir_name);
			free(fserve->dir_path);
			return FAILURE;
		}

        fserve->head->name = mymalloc(dir->dir);
		if (!fserve->head->name) {
			free(fserve->dir_name);
			free(fserve->dir_path);
			free(fserve->head);
			return FAILURE;
		}
        fserve->tail = fserve->head;
        fserve->tail->next = NULL;
        fserve->tail->prev = NULL;

    }
    else
    {
        fserve->tail->next = (pathstruct*)malloc(sizeof(pathstruct));
		if (!fserve->tail->next) {
			return FAILURE;
		}

        fserve->tail->next->name = (char*)malloc(sizeof(char) * (strlen(newpath) + 2));
		if (!fserve->tail->next->name) {
			free(fserve->tail->next);
			fserve->tail->next = NULL;
			return FAILURE;
		}

        fserve->tail->next->prev = fserve->tail;
        fserve->tail->next->next = NULL;
        fserve->tail = fserve->tail->next;

		sprintf(fserve->tail->name, "%s/", newpath);
    }
    cwd = getpath(fserve->head);

    if (cwd == NULL)
    {
        return FAILURE;
    }

#ifdef WIN32

    pthread_mutex_lock(&mutex);
    if (SetCurrentDirectory(cwd))
    {
        status = pwd(fserve);
    }
    else
    {
        good = 0;
        removepath(fserve);
        if (fserve->head == NULL)
            status = quote(fserve->sock, "%c%sImproperly Configured Fserve. Inform owner of invalid path:[%s]\r\n", 3, bot->color1, newpath);
        else
            status = quote(fserve->sock, "%c%s[%s] - Invalid Folder or Permission Denied\r\n", 3, bot->color1, newpath);
    }
    SetCurrentDirectory(botdir);
    pthread_mutex_unlock(&mutex);


#else

    if ((stat(cwd, &buf) != 0) || !(buf.st_mode & S_IFDIR))
    {
        removepath(fserve);
        if (fserve->head == NULL)
            status = quote(fserve->sock, "%c%sImproperly Configured Fserve. Inform owner of invalid path:[%s]\r\n", 3, bot->color1, newpath);
        else
            status = quote(fserve->sock, "%c%s[%s] - Invalid Folder or Permission Denied\r\n", 3, bot->color1, newpath);
    }
    else
    {
        status = pwd(fserve);
    }

#endif

    free(cwd);

    return status;
}

/*******************************************************************************
*---|   Function:      SENDDIRLIST
*---|   Created:       Thu Aug  7 10:14:07 CDT 2003
*---|   Last Edited:   Thu Dec 11 15:13:48 CST 2003
*---|   Description:   Sends the directory list to an fserve.
*******************************************************************************/

int senddirlist(dccstruct *fserve)
{
    int status = SUCCESS;
    char *cwd = NULL, *file = NULL;
    filestruct *head = NULL, *files = NULL, *tmp = NULL;
    channelstruct *chan = is_channel(fserve->chan);
    dirstruct *dirs = NULL;

#ifdef WIN32
    WIN32_FIND_DATA findData;
    HANDLE hFind;
#else
    DIR *dirp;
    struct dirent *dp;
    struct stat buf;
#endif

    if (chan == NULL)
    {
        return FAILURE;
    }
    if (fserve->head == NULL)
    {
        for (dirs = chan->directories; dirs; dirs = dirs->next)
        {
            head = addfile(dirs->name, 0, head);
        }
    }
    else
    {

        cwd = getpath(fserve->head);
        if (cwd == NULL)
        {
            return FAILURE;
        }

#ifdef WIN32

        pthread_mutex_lock(&mutex);
        if (SetCurrentDirectory(cwd))
        {
            hFind = FindFirstFile("*", &findData);
            if (hFind)
            {
                while (1)
                {
                    if (strcmp(findData.cFileName, "."))
                    {
                        if (FILE_ATTRIBUTE_DIRECTORY & GetFileAttributes(findData.cFileName))
                        {
                            head = addfile(findData.cFileName, 0, head);
                        }
                        else
                        {
                            files = addfile(findData.cFileName, findData.nFileSizeLow, files);
                        }
                    }
                    if (FindNextFile(hFind, &findData) == 0)
                        break;
                }

                FindClose(hFind);
            }
        }
        SetCurrentDirectory(botdir);
        pthread_mutex_unlock(&mutex);

#else

        dirp = opendir(cwd);
        if (dirp == NULL)
        {
            free(cwd);
            if (quote(fserve->sock, "Access denied: %s\r\n", strerror(errno)) == SUCCESS)
            {
                removepath(fserve);
                return pwd(fserve);
            }
            else
            {
                return FAILURE;
            }
        }

        while ((dp = readdir(dirp)))
        {
            file = (char*)malloc(sizeof(char) * (strlen(cwd) + strlen(dp->d_name) + 1));
			sprintf(file, "%s%s", cwd, dp->d_name);

            if (stat(file, &buf) == 0)
            {
                if (buf.st_mode & S_IFREG)
                {
                    files = addfile(dp->d_name, buf.st_size, files);
                }
                else if (buf.st_mode & S_IFDIR)
                {
                    if (strcmp(dp->d_name, ".") != 0)
                    {
                        head = addfile(dp->d_name, 0, head);
                    }
                }
				#ifdef debug
                else
                {
                    logprintf( "Stat Error on [%s]: %s\n", file, strerror(errno) );
                }
				#endif

            }
            free(file);
        }
        closedir(dirp);
#endif

        free(cwd);
    }

    for (tmp = head; (status && (tmp != NULL)); tmp = tmp->next)
    {
        status = quote(fserve->sock, "%s\r\n", tmp->name);
    }

    for (tmp = files; (status && (tmp != NULL)); tmp = tmp->next)
    {
        status = quote(fserve->sock, "%s  [%d bytes]\r\n", tmp->name, tmp->size);
    }

    while (head != NULL)
    {
        tmp = head;
        head = head->next;
        free(tmp->name);
        tmp->name = NULL;
        tmp->size = 0;
        tmp->next = NULL;
        tmp->prev = NULL;
        free(tmp);
        tmp = NULL;
    }

    while (files != NULL)
    {
        tmp = files;
        files = files->next;
        free(tmp->name);
        tmp->name = NULL;
        tmp->size = 0;
        tmp->next = NULL;
        tmp->prev = NULL;
        free(tmp);
        tmp = NULL;
    }

    file = NULL;
    head = NULL;
    files = NULL;
    tmp = NULL;
    cwd = NULL;

    if (status)
    {
        status = quote(fserve->sock, "End of list.\r\n");
    }
    return status;
}

/*******************************************************************************
*---|     Function:        ADDFILE
*---|     Created:         Thu Aug  7 10:28:16 CDT 2003
*---|     Last Edited:
*---|     Description:     Adds a file in alphabetical order
*---|                      to be displayed in an fserve.
*******************************************************************************/

filestruct *addfile(char *filename, int filesize, filestruct *start)
{

    filestruct *tmp = NULL, *ptr = malloc(sizeof(filestruct));

	/* error */
	if (ptr == NULL) {
		return start;
	}

    ptr->name = mymalloc(filename);
    ptr->size = filesize;
    ptr->next = NULL;
    ptr->prev = NULL;

	/* error */
	if (ptr->name == NULL) {
		return start;
	}

	/* empty */
    if (start == NULL) {
        return ptr;
	}

    tmp = start;
    while (tmp->next != NULL)
    {
        if (filecompare(tmp->name, ptr->name) == 0)
        {
            ptr->next = tmp;
            ptr->prev = tmp->prev;
            if (tmp->prev != NULL)
            {
                tmp->prev->next = ptr;
            }
            else
            {
                start = ptr;
            }
            tmp->prev = ptr;
            return start;
        }
        tmp = tmp->next;
    }

    if (filecompare(tmp->name, ptr->name) == 0)
    {
        ptr->next = tmp;
        ptr->prev = tmp->prev;
        if (tmp->prev != NULL)
        {
            tmp->prev->next = ptr;
        }
        else
        {
            start = ptr;
        }
        tmp->prev = ptr;
    }
    else
    {
        tmp->next = ptr;
        ptr->prev = tmp;
    }

    return start;
}


/*******************************************************************************
*---|     Function:        FILECOMPARE
*---|     Created:         Thu Aug  7 10:39:29 CDT 2003
*---|     Last Edited:
*---|     Description:     Compares alphabetic file names.
*******************************************************************************/

int filecompare(char *f1, char *f2)
{

    int i = 0;
    for (i = 0; i < (int)strlen(f1); i++)
    {
        if (f1[i] == '\0')
        {
            break;
        }
        else if (LCASE(f1[i]) < LCASE(f2[i]))
        {
            break;
        }
        else if (LCASE(f1[i]) > LCASE(f2[i]))
        {
            return 0;
        }
    }
    return 1;
}

/*******************************************************************************
*---|     Function:        GETPATH
*---|     Created:         Thu Aug  7 10:39:29 CDT 2003
*---|     Last Edited:
*---|     Description:     Puts a total path string together.
*******************************************************************************/

char *getpath(pathstruct *path)
{
    pathstruct *ptr;
    char *cwd = NULL;
    int len = 0;

    for (ptr = path; ptr != NULL; ptr = ptr->next)
    {
        len = len + strlen(ptr->name);
    }

    if (len)
    {
        cwd = (char *)malloc(sizeof(char) * (len + 1));

        if (cwd == NULL)
        {
            return NULL;
        }

        strcpy(cwd, path->name);

        for (ptr = path->next; ptr != NULL; ptr = ptr->next)
        {
            strcat(cwd, ptr->name);
        }
        cwd[len] = '\0';
    }

    return cwd;
}

/*******************************************************************************
*---|     Function:        PWD
*---|     Created:         Thu Aug  7 10:43:35 CDT 2003
*---|     Last Edited:
*---|     Description:     Returns the current path.
*******************************************************************************/

int pwd (dccstruct *fserve)
{
    char *pwd = NULL;
    int status = FAILURE;

    if (fserve->head)
    {

        if (fserve->head == fserve->tail)
        {
            status = quote(fserve->sock, "[/%s/]\r\n", fserve->dir_name);
        }
        else
        {
            pwd = getpath(fserve->head->next);
            status = quote(fserve->sock, "[/%s/%s]\r\n", fserve->dir_name, pwd);
            free(pwd);
            pwd = NULL;
        }
    }
    else
    {
        status = quote(fserve->sock, "[/]\r\n");
    }

    return status;
}

/*******************************************************************************
*---|   Function:      MAKE_QUEUE
*---|   Created:       Thu Aug  7 22:43:32 CDT 2003
*---|   Last Edited:   Thu Dec 11 16:49:07 CST 2003
*---|   Description:   Enables the parts of a queue.
*******************************************************************************/

dccstruct *make_queue(dccstruct *fserve, char *path, int fileptr, int size)
{

    dccstruct *newqueue = create_dcc();

	if (newqueue == NULL) {
		return NULL;
	}

    newqueue->path = mymalloc(path);
    newqueue->nick = mymalloc(fserve->nick);
    newqueue->host = mymalloc(fserve->host);

	if ((!newqueue->path) || (!newqueue->nick) || (!newqueue->host)) {

		newqueue = destroy_dcc(newqueue);
		return NULL;
	}
    newqueue->filename = &newqueue->path[fileptr];
    newqueue->size = size;
    newqueue->type = DCCSEND;

#ifdef debug
    logprintf( "DCC File: %s\n", newqueue->filename);
    logprintf( "DCC Path: %s\n", newqueue->path);
    logprintf( "DCC Nick: %s\n", newqueue->nick);
    logprintf( "DCC Host: %s\n", newqueue->host);
#endif

    return newqueue;
}

/*******************************************************************************
*---|     Function:        SEND_HANDLER
*---|     Created:         Thu Aug  7 22:30:46 CDT 2003
*---|     Last Edited:
*---|     Description:     Determines if a send can be issued.
*******************************************************************************/

int send_handler(dccstruct *fserve, char *path, int fileptr, int size)
{

    char *fsize = NULL, *br = NULL, buff[BUFFSIZE] = { '\0' };
    dccstruct *dcc = NULL;
    int status = FAILURE, sndcnt = 0, i = 0;

    buff[i++] = 3;
    buff[i++] = bot->color1[0];
    buff[i++] = bot->color1[1];
    buff[i++] = '\0';

    pthread_mutex_lock(&mutex);
	#ifdef debug
    logprintf( "MUTEX LOCK: SEND_HANDLER %s\n", fserve->nick);
	#endif

    if (size < bot->forcesize)
    {
        dcc = make_queue(fserve, path, fileptr, size);
		if (dcc != NULL) {
			strcat(buff, "Force Sending File");
			br = bracket("%s", &path[fileptr]);
			strcat(buff, br);
			free(br);

			strcat(buff, " Size");
			fsize = filesize(size, 0);
			br = bracket("%s", fsize);
			strcat(buff, br);
			free(br);
			free(fsize);

			strcat(buff, " because it's smaller than ");
			br = bracket("%d", bot->forcesize);
			strcat(buff, br);
			free(br);

			strcat(buff, " bytes.");

			status = quote(fserve->sock, "%s\r\n", buff);

			dcc_send(dcc);
		}
		else {
			status = FAILURE;
		}
    }
	/* there is a send available */
    else if (bot->sends->count < bot->sends->total)
    {
        sndcnt = in_use(fserve->nick, fserve->host, bot->sends, "sends");
		/* you can have another send */
        if (sndcnt < bot->sends->each)
        {
            dcc = make_queue(fserve, path, fileptr, size);
			if (dcc != NULL) {
				strcat(buff, "Attempting to send File");
				br = bracket("%s", &path[fileptr]);
				strcat(buff, br);
				free(br);

				strcat(buff, " Size");
				fsize = filesize(size, 0);
				br = bracket("%s", fsize);
				strcat(buff, br);
				free(br);
				free(fsize);

				strcat(buff, " to Nick");
				br = bracket("%s", fserve->nick);
				strcat(buff, br);
				free(br);

				status = quote(fserve->sock, "%s\r\n", buff);

				dcc_send(dcc);
			}
			else {
				status = FAILURE;
			}

        }
        else
        {
            /* queue stuff */
            status = queue_handler(fserve, path, fileptr, size, sndcnt);
        }
    }
    else
    {
        /* queue stuff */
        status = queue_handler(fserve, path, fileptr, size, sndcnt);
    }

    pthread_mutex_unlock(&mutex);
#ifdef debug

    logprintf( "MUTEX RELEASE: SEND_HANDLER %s\n", fserve->nick);
#endif

    return status;
}

/*******************************************************************************
*---|     Function:        QUEUE_HANDLER
*---|     Created:         Thu Aug  7 22:49:29 CDT 2003
*---|     Last Edited:
*---|     Description:     Determines if a user can have a queue.
*******************************************************************************/

int queue_handler(dccstruct *fserve, char *path, int fileptr, int size, int sndcnt)
{
    char *fsize = NULL, *br = NULL, buff[BUFFSIZE] = { '\0' };
    dccstruct *dcc = NULL;
    int status = FAILURE, queuecnt = 0, i = 0;

    buff[i++] = 3;
    buff[i++] = bot->color1[0];
    buff[i++] = bot->color1[1];
    buff[i++] = '\0';

    if (bot->queues->count < bot->queues->total)
    {
#ifdef debug
        logprintf( "There is an open queue slot.\n");
        logprintf( "-----------------------------------\n");
#endif

        queuecnt = in_use(fserve->nick, fserve->host, bot->queues, "queues");
		/* you can have another queue */
        if ((queuecnt + sndcnt) < bot->queues->each)
        {
            dcc = make_queue(fserve, path, fileptr, size);
			if (dcc) {
				add_dcc(dcc, bot->queues);

				strcat(buff, "Adding Queue");
				br = bracket("%d", bot->queues->count);
				strcat(buff, br);
				free(br);

				strcat(buff, " Nick");
				br = bracket("%s", fserve->nick);
				strcat(buff, br);
				free(br);

				strcat(buff, " File");
				br = bracket("%s", &path[fileptr]);
				strcat(buff, br);
				free(br);

				strcat(buff, " Size");
				fsize = filesize(size, 0);
				br = bracket("%s", fsize);
				strcat(buff, br);
				free(br);
				free(fsize);

				status = quote(fserve->sock, "%s\r\n", buff);

				#ifdef debug
				print_queues(bot->queues, "Queue");
				#endif

			}
			else {
				status = FAILURE;
			}
        }
        else
        {
            status = quote(fserve->sock, "%c%sSorry, you have used up all of your slots. Sends:%s%d%s Queues:%s%d%s\r\n",
                           3, bot->color1, bot->ob, sndcnt, bot->cb, bot->ob, queuecnt, bot->cb);
        }
    }
    else
    {
        status = quote(fserve->sock, "%c%sSorry, all of the queues %s%d%s are full.\r\n",
                       3, bot->color1, bot->ob, bot->queues->total, bot->cb);
    }


    return status;
}

/*******************************************************************************
*---|     Function:        DCC_SEND
*---|     Created:         Thu Aug  7 22:55:47 CDT 2003
*---|     Last Edited:
*---|     Description:     Starts a dcc send.
*******************************************************************************/

int dcc_send(dccstruct *dcc)
{
    int status = bind_listen(dcc);

    if (status == FAILURE)
    {
        destroy_dcc(dcc);
		return FAILURE;
    }
    else
    {
		#ifdef debug
        print_queues(bot->sends, "Send");
		#endif

        dcc->file = fopen(dcc->path, "rb");

        if (dcc->file == NULL)
        {
			#ifdef debug
            logprintf( "Could not open file:[%s]\n", dcc->path);
			#endif
            destroy_dcc(dcc);
            return FAILURE;
        }

        dcc->state = DCCACCEPT;
        dcc->lastdata = time(NULL);
        add_dcc(dcc, bot->sends);
    }

    return status;
}

/*******************************************************************************
*---|     Function:        PRINT_QUEUES
*---|     Created:         Thu Aug  7 23:02:36 CDT 2003
*---|     Last Edited:
*---|     Description:     Prints out the elements of a dccptr list.
*******************************************************************************/

#ifdef debug
void print_queues(dccptr *list, char *type)
{

    dccstruct *ptr = NULL;
    int i = 0;

    for (ptr = list->start; ptr != NULL; ptr = ptr->next)
    {
        logprintf( "%s #%d: %s, %s\n", type, ++i, ptr->nick, ptr->filename);
    }

    return ;
}
#endif

/*******************************************************************************
*---|   Function:      SEND_NEXT
*---|   Created:       Sun Aug 10 17:28:09 CDT 2003
*---|   Last Edited:   Sun Dec  7 11:10:04 CST 2003
*---|   Description:   Deletes a send and sends the next appropriate queue.
*******************************************************************************/

int send_next()
{
    dccstruct *nextdcc = NULL;
    int status = FAILURE;

    if ((alive) && (bot->sends->count < bot->sends->total))
    {
        for (nextdcc = bot->queues->start; nextdcc != NULL; nextdcc = nextdcc->next)
        {
            if (in_use(nextdcc->nick, nextdcc->host, bot->sends, "sends") < bot->sends->each)
            {
                remove_dcc(nextdcc, bot->queues);
                status = dcc_send(nextdcc);
                if (status == SUCCESS) {
					#ifdef debug
					logprintf( "The next send is File:[%s] to Nick:[%s].\n", nextdcc->filename, nextdcc->nick);
					#endif
                    break;
				}
            }
        }

    }
    return status;
}

/*******************************************************************************
*---|     Function:        DISPLAY_FSERVE
*---|     Created:         Sun Aug 10 17:38:41 CDT 2003
*---|     Last Edited:
*---|     Description:     Display the fserve into a channel.
*******************************************************************************/

int display_fserve(channelstruct *chan)
{

    int status = SUCCESS;
    char *ad = NULL;

    if (bot->settings & FSERVEON)
    {
        ad = fserve_ad(chan);
        status = quote(bot->sock, "PRIVMSG %s :%s\r\n", chan->name, ad);
        free(ad);
        ad = NULL;
    }

    return status;
}

/*******************************************************************************
*---|   Function:      TOTAL_SEND_SPEED
*---|   Created:       Sun Aug 10 17:42:59 CDT 2003
*---|   Last Edited:
*---|   Description:   Calculates the total speed across all sends.
*******************************************************************************/

unsigned int total_send_speed()
{
    unsigned int curspeed = 0, i = 0;
    dccstruct *dcc = NULL;

    if (bot->sends->start != NULL)
    {
        for (dcc = bot->sends->start; dcc != NULL; dcc = dcc->next)
        {
            i++;
			/*
			#ifdef debug
			logprintf("Send #%d: Speed = %d\n", i, dcc->speed);
			#endif
			*/

            curspeed += dcc->speed;
        }
    }

	/*
	#ifdef debug
	logprintf( "Current Total Speed = %u\n", curspeed);
	#endif
	*/

    return curspeed;
}

/*******************************************************************************
*---|   Function:      ADJUST_SPEEDCAP
*---|   Created:       Sun Dec  7 01:54:53 CST 2003
*---|   Last Edited:
*---|   Description:   
*******************************************************************************/

void adjust_speedcap()
{
    unsigned int speed = 0;

    if ((bot->totalcap) && (bot->sends->count))
    {
        speed = total_send_speed();

        if (speed > bot->totalcap)
        {
            bot->tmpcap -= ((speed - bot->totalcap) / bot->sends->count);
        }
        else if (speed < bot->totalcap)
        {
            bot->tmpcap += ((bot->totalcap - speed) / bot->sends->count);
            if ((bot->sendcap) && (bot->tmpcap > bot->sendcap))
            {
                bot->tmpcap = bot->sendcap;
            }
            if (bot->tmpcap > bot->totalcap)
            {
                bot->tmpcap = bot->totalcap;
            }
        }
        if (bot->tmpcap <= 0)
        {
            bot->tmpcap = bot->totalcap;
        }

    }

    return ;
}

void update_stats(dccstruct *dcc)
{

    unsigned int sent_size = (bot->bytecount + dcc->ctr);
    FILE *log = NULL;

    if (sent_size < bot->bytecount)
    {
        bot->overflowcount++;
    }
    bot->bytecount = sent_size;

#ifdef debug

    logprintf( "Sent %lu bytes of %s [%d] to %s at %u Bps\n", dcc->ackval - dcc->offset,
               dcc->filename, dcc->size, dcc->nick, dcc->speed);
#endif

    if ((bot->transferlog) && ((log = fopen(bot->transferlog, "a+"))))
    {
        if (dcc->ackval >= dcc->size)
        {
            fprintf(log, "COMPLETED - ");
        }
        else
        {
            fprintf(log, "FAILED    - ");
        }

        fprintf(log, "%s recieved %lu Bytes of %s [%d Bytes] at %u Bps\n", dcc->nick,
                dcc->ackval - dcc->offset, dcc->filename, dcc->size, dcc->speed);

        fclose(log);
    }

    if (dcc->speed > bot->maxspeed)
    {
        bot->maxspeed = dcc->speed;
    }

    if (dcc->speed > 0)
    {
        bot->avespeed = (bot->avespeed * (bot->filecount + bot->failcount) + dcc->speed) / (bot->filecount + bot->failcount + 1);
    }

    if (dcc->ackval >= dcc->size)
    {
        bot->filecount++;
    }
    else
    {
        bot->failcount++;
    }

    fclose(dcc->file);

    return ;
}

/*******************************************************************************
*---|     Function:        SHOW_DCCS
*---|     Created:         Mon Aug 11 23:00:19 CDT 2003
*---|     Last Edited:
*---|     Description:     Displays sends to a channel, private 
*---|                      message, or an fserve.
*******************************************************************************/

int show_dccs(int sock, char *action, char *target, int num, char *type)
{
    dccstruct *dcc = NULL;
    int i = 0, status = SUCCESS;
    unsigned int curspeed = 0, avgspeed = 0;
    char *size = NULL, *speed = NULL, *timeleft = NULL, *br = NULL, buff[BUFFSIZE] = { '\0' };

    pthread_mutex_lock(&mutex);
#ifdef debug

    logprintf( "MUTEX LOCK: SHOW_DCCS\n");
#endif

    if (lowercasecmp("FSERVE", type) == 0)
    {
        dcc = bot->fserves->start;
    }
    else if (lowercasecmp("QUEUE", type) == 0)
    {
        dcc = bot->queues->start;
    }
    else if (lowercasecmp("SEND", type) == 0)
    {
        dcc = bot->sends->start;
    }

    buff[0] = 3;
    buff[1] = bot->color1[0];
    buff[2] = bot->color1[1];
    buff[3] = '\0';

    if (dcc == NULL)
    {
        if (target)
        {
            status = quote(sock, "%s %s :%c%sThere are no %ss in progress.\r\n", action, target, 3, bot->color1, type);
        }
        else
        {
            status = quote(sock, "%c%sThere are no %ss in progress.\r\n", 3, bot->color1, type);
        }
    }
    else
    {

        for (; ((status) && (dcc != NULL)); dcc = dcc->next)
        {
            buff[3] = '\0';
            i++;
            avgspeed = dcc->speed;
            curspeed += avgspeed;
            size = filesize(dcc->size, 0);
            speed = filesize(avgspeed, 0);
            if (avgspeed == 0)
            {
                timeleft = mymalloc("Unknown");
            }
            else
            {
                timeleft = gettime((long)((dcc->size - dcc->ctr) / avgspeed));
            }

            strcat(buff, type);
            strcat(buff, ":");
            br = bracket("#%d", i);
            strcat(buff, br);
            free(br);

            strcat(buff, " Nick:");
            br = bracket("%s", dcc->nick);
            strcat(buff, br);
            free(br);

            if (lowercasecmp("FSERVE", type) == 0)
            {
                strcat(buff, " Host:");
                br = bracket("%s", dcc->host);
                strcat(buff, br);
                free(br);
            }
            else
            { /* queue or send */

                strcat(buff, " File:");
                br = bracket("%s", dcc->filename);
                strcat(buff, br);
                free(br);

                strcat(buff, " Size:");
                br = bracket("%s", size);
                strcat(buff, br);
                free(br);

                if (lowercasecmp("Send", type) == 0)
                {
                    strcat(buff, " Complete:");
                    if (dcc->size > 0)
                    {
                        br = bracket("%.0f%%", (100.0 * (dcc->ctr)) / (dcc->size));
                    }
                    else
                    {
                        br = bracket("%d%%", 0);
                    }
                    strcat(buff, br);
                    free(br);

                    strcat(buff, " Speed:");
                    br = bracket("%sPS", speed);
                    strcat(buff, br);
                    free(br);

                    strcat(buff, " Remaining:");
                    br = bracket("%s", timeleft);
                    strcat(buff, br);
                    free(br);
                }
            }

            /* Only send 5 to a channel or pm so as to not flood off */
            if (target)
            {
                if ((i >= num) && (i < num + 5))
                {
                    status = quote(sock, "%s %s :%s\r\n", action, target, buff);
                }
            }

            /* When sending to an fserve, you can send
            ** all you want to the socket without flooding
            ** off, so display all of them
            */
            else
            {
                status = quote(sock, "%s\r\n", buff);
            }

            free(size);
            free(speed);
            free(timeleft);

        }

        if ((status) && (lowercasecmp("Send", type) == 0))
        {
            speed = filesize(curspeed, 0);
            buff[3] = '\0';

            strcat(buff, "Total Bandwidth:");
            br = bracket("%sPS", speed);
            strcat(buff, br);
            free(br);

            strcat(buff, " for ");
            br = bracket("%d", i );
            strcat(buff, br);
            free(br);

            strcat(buff, " sends.");
            free(speed);

            if (target)
            {
                status = quote(sock, "%s %s :%s\r\n", action, target, buff);
            }
            else
            {
                status = quote(sock, "%s\r\n", buff);
            }
        }
    }
    pthread_mutex_unlock(&mutex);
#ifdef debug

    logprintf( "MUTEX RELEASE: SHOW_DCCS\n");
#endif

    return status;
}

/*******************************************************************************
*---|     Function:        CLEAR_QUEUES
*---|     Created:         Mon Aug 11 23:00:19 CDT 2003
*---|     Last Edited:
*---|     Description:     Clears all of the queues for a given nick.
*******************************************************************************/

int clear_queues(char *nick, int sock, char *action, char *target)
{

    int i = 0, status = SUCCESS;
    dccstruct *tmp, *dcc = NULL;
    char *br = NULL, buff[BUFFSIZE] = { '\0' };

    buff[0] = 3;
    buff[1] = bot->color1[0];
    buff[2] = bot->color1[1];
    buff[3] = '\0';

    pthread_mutex_lock(&mutex);
#ifdef debug

    logprintf( "MUTEX LOCK: CLEAR_QUEUES %s\n", nick);
#endif

    dcc = bot->queues->start;

    if (dcc == NULL)
    {
        if (target)
            status = quote(sock, "%s %s :%c%sThere are no queues.\r\n", action, target, 3, bot->color1);
        else
            status = quote(sock, "%c%sThere are no queues.\r\n", 3, bot->color1);
    }
    else
    {
        do
        {
            tmp = dcc;
            dcc = dcc->next;
            if (lowercasecmp(nick, tmp->nick) == 0)
            {
                i++;
                remove_dcc(tmp, bot->queues);
                destroy_dcc(tmp);
            }
        }
        while (dcc != NULL);

        strcat(buff, "Removed:");
        br = bracket("%d", i);
        strcat(buff, br);
        free(br);

        strcat(buff, " Queues for Nick:");
        br = bracket("%s", nick);
        strcat(buff, br);
        free(br);

        if (target)
        {
            status = quote(sock, "%s %s :%s\r\n", action, target, buff);
        }
        else
        {
            status = quote(sock, "%s\r\n", buff);
        }
    }

    pthread_mutex_unlock(&mutex);
#ifdef debug

    logprintf( "MUTEX RELEASE: CLEAR_QUEUES %s\n", nick);
#endif

    return status;
}

/*******************************************************************************
*---|     Function:        DEL_QUEUE
*---|     Created:         Tue Aug 12 17:21:17 CDT 2003
*---|     Last Edited:
*---|     Description:     Deletes a queue if nick & host match
*******************************************************************************/

int del_queue(int sock, int num, char *nick, char *host)
{

    int i = 0, status = SUCCESS;
    dccstruct *queue = NULL;
    char *br = NULL, buff[BUFFSIZE] = { '\0' };

    buff[0] = 3;
    buff[1] = bot->color1[0];
    buff[2] = bot->color1[1];
    buff[3] = '\0';

    pthread_mutex_lock(&mutex);
#ifdef debug

    logprintf( "MUTEX LOCK: DEL_QUEUE\n");
#endif

    for (queue = bot->queues->start; queue; queue = queue->next)
    {
        if (++i == num)
            break;
    }

    if (queue == NULL)
    {

        status = quote(sock, "%c%sThere is no queue %d.\r\n", 3, bot->color1, num);

    }
    else if ((lowercasecmp(nick, queue->nick) == 0) && (lowercasecmp(host, queue->host) == 0))
    {

        strcat(buff, "Removing Queue:");
        br = bracket("#%d", num);
        strcat(buff, br);
        free(br);

        strcat(buff, " File:");
        br = bracket("%s", queue->filename);
        strcat(buff, br);
        free(br);

        strcat(buff, " Nick:");
        br = bracket("%s", queue->nick);
        strcat(buff, br);
        free(br);

        status = quote(sock, "%s\n", buff);

        remove_dcc(queue, bot->queues);
        destroy_dcc(queue);

    }
    else
    {
        strcat(buff, "Permission Denied: You do not match Nick:");
        br = bracket("%s", queue->nick);
        strcat(buff, br);
        free(br);

        strcat(buff, " Host:");
        br = bracket("%s", queue->host);
        strcat(buff, br);
        free(br);

        status = quote(sock, "%s\n", buff);
    }

    pthread_mutex_unlock(&mutex);
#ifdef debug

    logprintf( "MUTEX RELEASE: DEL_QUEUE\n");
#endif

    return status;

}

/*******************************************************************************
*---|     Function:        SEND_QUEUE
*---|     Created:         Mon Aug 11 23:00:19 CDT 2003
*---|     Last Edited:
*---|     Description:     Makes a particular queue auto send.
*******************************************************************************/

int send_queue(char *action, char *target, int ctr)
{

    int x = 0;
    dccstruct *queue = NULL;
    char *br = NULL, buff[BUFFSIZE] = { '\0' };

    buff[0] = 3;
    buff[1] = bot->color1[0];
    buff[2] = bot->color1[1];
    buff[3] = '\0';

    pthread_mutex_lock(&mutex);

    queue = bot->queues->start;

    if ((ctr > bot->queues->count) || (queue == NULL))
    {
        x = quote(bot->sock, "%s %s :%c%sThere is no queue %d.\n", action, target, 3, bot->color1, ctr);
    }
    else
    {
        for (x = 1; x < ctr; x++)
        {
            queue = queue->next;
        }
        remove_dcc(queue, bot->queues);
        dcc_send(queue);

        strcat(buff, "Sending Queue:");
        br = bracket("#%d", ctr);
        strcat(buff, br);
        free(br);

        strcat(buff, " File:");
        br = bracket("%s", queue->filename);
        strcat(buff, br);
        free(br);

        strcat(buff, " to Nick:");
        br = bracket("%s", queue->nick);
        strcat(buff, br);
        free(br);

        x = quote(bot->sock, "%s %s :%s\n", buff);
    }

    pthread_mutex_unlock(&mutex);

    return x;

}

/*******************************************************************************
*---|     Function:        PURGE_DCC
*---|     Created:         Mon Aug 11 23:00:19 CDT 2003
*---|     Last Edited:
*---|     Description:     Gets rid of a dcc struct.
*******************************************************************************/

int purge_dcc(char *action, char *target, int ctr, char *type)
{

    int x = 0;
    dccstruct *dcc = NULL;
    dccptr *list = NULL;
    char *br = NULL, buff[BUFFSIZE] = { '\0' };

    buff[0] = 3;
    buff[1] = bot->color1[0];
    buff[2] = bot->color1[1];
    buff[3] = '\0';

    pthread_mutex_lock(&mutex);
#ifdef debug

    logprintf( "MUTEX LOCK: PURGE_DCC\n");
#endif

    if (lowercasecmp("FSERVE", type) == 0)
    {
        list = bot->fserves;
    }
    else if (lowercasecmp("QUEUE", type) == 0)
    {
        list = bot->queues;
    }
    else if (lowercasecmp("SEND", type) == 0)
    {
        list = bot->sends;
    }

    dcc = list->start;

    if ((ctr > list->count) || (dcc == NULL))
    {
        x = quote(bot->sock, "%s %s :%c%sThere is no %s %d.\n", action, target, 3, bot->color1, type, ctr);
    }
    else
    {
        for (x = 1; x < ctr; x++)
        {
            dcc = dcc->next;
        }

        strcat(buff, "Removing ");
        strcat(buff, type);
        strcat(buff, ":");
        br = bracket("#%d", ctr);
        strcat(buff, br);
        free(br);

        if (lowercasecmp("FSERVE", type) == 0)
        {
            strcat(buff, " Host:");
            br = bracket("%s", dcc->host);
            strcat(buff, br);
            free(br);

        }
        else
        {
            strcat(buff, " File:");
            br = bracket("%s", dcc->filename);
            strcat(buff, br);
            free(br);

        }

        if (lowercasecmp("QUEUE", type) == 0)
        {
            remove_dcc(dcc, list);
            destroy_dcc(dcc);
        }
        else
        {
            dcc->die = 1;
        }

        strcat(buff, " Nick:");
        br = bracket("%s", dcc->nick);
        strcat(buff, br);
        free(br);

        x = quote(bot->sock, "%s %s :%s\n", action, target, buff);

    }
    pthread_mutex_unlock(&mutex);
#ifdef debug

    logprintf( "MUTEX RELEASE: PURGE_DCC\n");
#endif

    return x;
}

/*******************************************************************************
*---|     Function:        CHANGE_QUEUE_NICK
*---|     Created:         Tue Aug 12 16:46:37 CDT 2003
*---|     Last Edited:
*---|     Description:     Traverses the queues and changes
*---|                      any matching queues to the new nick.
*******************************************************************************/

void change_queue_nick(char *nick, char *newnick, char *host)
{

    dccstruct *queue = NULL;

    pthread_mutex_lock(&mutex);
#ifdef debug

    logprintf( "MUTEX LOCK: CHANGE_QUEUE_NICK\n");
#endif

    for (queue = bot->queues->start; queue; queue = queue->next)
    {

        if ((lowercasecmp(nick, queue->nick) == 0) && (lowercasecmp(host, queue->host) == 0))
        {

            if (queue->nick)
                free(queue->nick);
            queue->nick = mymalloc(newnick);

        }
    }

    for (queue = bot->sends->start; queue; queue = queue->next)
    {

        if ((lowercasecmp(nick, queue->nick) == 0) && (lowercasecmp(host, queue->host) == 0))
        {

            if (queue->nick)
                free(queue->nick);
            queue->nick = mymalloc(newnick);

        }
    }

    for (queue = bot->fserves->start; queue; queue = queue->next)
    {

        if ((lowercasecmp(nick, queue->nick) == 0) && (lowercasecmp(host, queue->host) == 0))
        {

            if (queue->nick)
                free(queue->nick);
            queue->nick = mymalloc(newnick);

        }
    }

    pthread_mutex_unlock(&mutex);
#ifdef debug

    logprintf( "MUTEX RELEASE: CHANGE_QUEUE_NICK\n");
#endif

    return ;
}

/*******************************************************************************
*---|   Function:      FSERVE_THREAD
*---|   Created:       Sat Dec  6 16:31:22 CST 2003
*---|   Last Edited:   Tue Dec  9 15:41:24 CST 2003
*---|   Description:   Thread to handle all of the fserves.  
*---|                  Accepts clients.  Reads in fserve commands.
*---|                  Cleans up dead fserves.
*******************************************************************************/

THREADFUNC fserve_thread(THREADARG arg)
{

    int largest = 0, status = 0;
    dccstruct *dcc = NULL, *tmp = NULL;
    fd_set socklist;
    struct timeval tv;

	#ifdef debug
    logprintf("Fserves thread running.\n");
	#endif

    while (alive)
    {

		/* At some point the bot may be told to restart. If it can't
		create the bot structure again then we don't want
		the threads trying to access dccs that don't even
		have a pointer */
        pthread_mutex_lock(&mutex);
		if (alive == 0)
        {
            pthread_mutex_unlock(&mutex);
            break;
        }
        /*
        #ifdef debug
        logprintf( "MUTEX LOCK #1: FSERVE_THREAD\n");
        #endif
        */

        /* Reset the list */
        FD_ZERO(&socklist);
        largest = 0;

        /* Add all of the sockets to the set and find the largest */
        for (dcc = bot->fserves->start; dcc; dcc = dcc->next)
        {

            FD_SET(dcc->sock, &socklist);

            if (dcc->sock > largest)
            {
                largest = dcc->sock;
            }

        }

        pthread_mutex_unlock(&mutex);
        /*
        #ifdef debug
        logprintf( "MUTEX RELEASE #1: FSERVE_THREAD\n");
        #endif
        */

        /* Wait up to 5 seconds to see if there is any activity */
        tv.tv_sec = 5;
        tv.tv_usec = 0;
        if (largest == 0)
        {
            sleep(tv.tv_sec);
			continue;
        }
        else
        {
            status = select(largest + 1, &socklist, (fd_set*)0, (fd_set*)0, &tv);
        }

        pthread_mutex_lock(&mutex);
        if (alive == 0)
        {
            pthread_mutex_unlock(&mutex);
            break;
        }
        /*
        #ifdef debug
        logprintf( "MUTEX LOCK #2: FSERVE_THREAD\n");
        #endif
        */

        for (dcc = bot->fserves->start; dcc; dcc = dcc->next)
        {

            status = SUCCESS;

            pthread_mutex_unlock(&mutex);
            /*
            #ifdef debug
            logprintf( "MUTEX RELEASE #3: FSERVE_THREAD\n");
            #endif
            */

            /* this socket is in the list */
            if (FD_ISSET(dcc->sock, &socklist) != 0)
            {

                dcc->lastdata = time(NULL);
                dcc->warned = 0;

                switch (dcc->state)
                {

                case DCCACCEPT:
					if (dcc_accept(dcc) == FAILURE)
					{
						status = FAILURE;
					}
					else if (fserve_entry(dcc) == FAILURE)
					{
						status = FAILURE;
					}
                    break;

                case DCCCHAT:
                    status = chat_read(dcc);
                    break;

                default:
                    status = FAILURE;
                    break;
                }

            }
            /* haven't received any data in 2 minutes */
            else if ((dcc->warned == 0) && (time(NULL) - dcc->lastdata > 120))
            {
                status = quote(dcc->sock, "%c%sIdle for 2 minutes. Fserve will close in 30 seconds.\r\n", 3, bot->color1);
                dcc->warned = 1;
            }
            /* haven't received any data in 30 seconds since the warning */
            else if (time(NULL) - dcc->lastdata > 150)
            {
                status = quote(dcc->sock, "%c%sClosing Fserve.\r\n", 3, bot->color1);
                status = FAILURE;
            }

            /* Been on the fserve too long */
            if ((dcc->starttime) && (time(NULL) - dcc->starttime > 600))
            {
                quote(dcc->sock, "%c%sYou've been on the fserve for too long.  10 minute limit.\n", 3, bot->color1);
                status = FAILURE;
            }

            if ((status == FAILURE) || (dcc->die == 1))
            {
                dcc->state = DCCDONE;
            }

            pthread_mutex_lock(&mutex);
            if (alive == 0)
            {
                pthread_mutex_unlock(&mutex);
                break;
            }
            /*
            #ifdef debug
            logprintf( "MUTEX LOCK #3: FSERVE_THREAD\n");
            #endif
            */

        }

        /* Remove dead dccs */
        dcc = bot->fserves->start;

        while (dcc)
        {

            tmp = dcc->next;

            if (dcc->state == DCCDONE)
            {

                remove_dcc(dcc, bot->fserves);
                dcc = destroy_dcc(dcc);

            }

            dcc = tmp;
        }

        pthread_mutex_unlock(&mutex);
        /*
        #ifdef debug
        logprintf( "MUTEX RELEASE #2: FSERVE_THREAD\n");
        #endif
        */

    }

    pthread_exit(0);

}

/*******************************************************************************
*---|   Function:      SENDS_THREAD
*---|   Created:       Sat Dec  6 21:01:33 CST 2003
*---|   Last Edited:   Tue Dec  9 15:41:24 CST 2003
*---|   Description:   Thread to handle all of the sends.  It checks to
*---|                  see if they have any acks to be read from.  If it
*---|                  needs to send some data. Cleans up the dead sends.
*---|                  Adjusts for the speedcaps.
*******************************************************************************/

THREADFUNC sends_thread(THREADARG arg)
{

    dccstruct *dcc = NULL, *tmp = NULL;

    struct timeval stv, etv;
    double etime = 0;

    int largest = 0, status = 0;
    fd_set rd, wd;
    struct timeval tv;

	#ifdef debug
    logprintf("Sends thread running.\n");
	#endif

	adjust_speedcap();

    while (alive)
    {

        pthread_mutex_lock(&mutex);
        if (alive == 0)
        {
            pthread_mutex_unlock(&mutex);
            break;
        }
        /*
        #ifdef debug
        logprintf( "MUTEX LOCK #1: SENDS_THREAD\n");
        #endif
        */

        /* Reset the list */
        FD_ZERO(&rd);
        FD_ZERO(&wd);
        largest = 0;

        /* Add all of the sockets to the set and find the largest */
        for (dcc = bot->sends->start; dcc; dcc = dcc->next)
        {

            FD_SET(dcc->sock, &rd);
            FD_SET(dcc->sock, &wd);

            if (dcc->sock > largest)
            {
                largest = dcc->sock;
            }

        }

        pthread_mutex_unlock(&mutex);
        /*
        #ifdef debug
        logprintf( "MUTEX RELEASE #1: SENDS_THREAD\n");
        #endif
        */

        /* Wait up to 5 seconds to see if there is any activity */
        tv.tv_sec = 5;
        tv.tv_usec = 0;
        if (largest == 0)
        {
            sleep(tv.tv_sec);
			continue;
        }
        else
        {
            status = select(largest + 1, &rd, &wd, (fd_set*)0, &tv);
        }

        pthread_mutex_lock(&mutex);
        if (alive == 0)
        {
            pthread_mutex_unlock(&mutex);
            break;
        }
        /*
        #ifdef debug
        logprintf( "MUTEX LOCK #2: SENDS_THREAD\n");
        #endif
        */

        gettimeofday(&stv, NULL);

        for (dcc = bot->sends->start; dcc; dcc = dcc->next)
        {

            status = SUCCESS;

            pthread_mutex_unlock(&mutex);
            /*
            #ifdef debug
            logprintf( "MUTEX RELEASE #3: SENDS_THREAD\n");
            #endif
            */

            /* this socket has something for me to read */
            if (FD_ISSET(dcc->sock, &rd) != 0)
            {

                switch (dcc->state)
                {

                case DCCACCEPT:
					if (dcc_accept(dcc) == FAILURE)
					{
						status = FAILURE;
					}
					else if (fseek(dcc->file, (long)dcc->offset, 0) != 0)
                    {
                        status = FAILURE;
                    }
					#ifdef debug
					else {
						logprintf( "Starting transfer at offest %lu\n", (long)dcc->offset);
					}
					#endif

                    dcc->ctr = dcc->offset;
                    break;

                case DCCSEND:
                    status = retr_ack(dcc);
                    break;

                default:
                    status = FAILURE;
                    break;
                }
                dcc->lastdata = time(NULL);
            }

            /* this socket has something for me to write to */
            if ((status) && (dcc->state == DCCSEND) && (FD_ISSET(dcc->sock, &wd) != 0))
            {
                status = send_bytes(dcc);
            }

            /* Haven't sent/received any data in 30 seconds */
            if (time(NULL) - dcc->lastdata > 30)
            {
				#ifdef debug
                logprintf("DCC Send with %s timed out after 30 seconds of inactivity.\n", dcc->nick);
				#endif

                status = FAILURE;
            }

            if ((status == FAILURE) || (dcc->die == 1))
            {
                dcc->state = DCCDONE;
            }

            pthread_mutex_lock(&mutex);
			if (alive == 0)
            {
                pthread_mutex_unlock(&mutex);
                break;
            }
            /*
            #ifdef debug
            logprintf( "MUTEX LOCK #3: SENDS_THREAD\n");
            #endif
            */

        }

        /* Remove dead dccs */
        dcc = bot->sends->start;

        while (dcc)
        {

            tmp = dcc->next;

            if (dcc->state == DCCDONE)
            {

                update_stats(dcc);
                remove_dcc(dcc, bot->sends);
                dcc = destroy_dcc(dcc);

                send_next();
            }

            dcc = tmp;
        }

        pthread_mutex_unlock(&mutex);

        /*
        #ifdef debug
        logprintf( "MUTEX RELEASE #2: SENDS_THREAD\n");
        #endif
        */

		/* If we've sent our fill of the data in under a second, then wait the remainder
		** of the time before we try to send any data so as to not violate the cap */
        if (bot->tmpcap)
        {
            gettimeofday(&etv, NULL);
            etime = (double)(etv.tv_sec - stv.tv_sec) + (double)(etv.tv_usec - stv.tv_usec) / 1000000;
            if (etime < 1)
            {
                usleep((1 - etime)*1000000);
            }
        }

        adjust_speedcap();

    }

    pthread_exit(0);

}

/*******************************************************************************
*---|   Function:      SEND_BYTES
*---|   Created:       Sun Dec  7 10:51:46 CST 2003
*---|   Last Edited:
*---|   Description:   Transfers so many bytes of a file.  It allows the
*---|                  dcc to send data up to the cap amount or as much
*---|                  as it can in one second.
*******************************************************************************/

int send_bytes(dccstruct *dcc)
{

    int tmp = 0, ret = 0, status = SUCCESS, ctr = 0, amt = 0;
    unsigned char buff[BLOCKSIZE];
    struct timeval stv, etv;
    double etime = 0;

    if (dcc->ctr >= dcc->size)
    {
        return status;
    }

    gettimeofday(&stv, NULL);

    while (tmp == ret)
    {

		/* If there is more than a block's worth of data
		** send a block, otherwise send what's left */
		if (bot->tmpcap != 0) {
			if (bot->tmpcap - ctr >= BLOCKSIZE) {
				amt = BLOCKSIZE;
			}
			else {
				amt = bot->tmpcap - ctr;
			}
		}
		else {
			amt = BLOCKSIZE;
		}

        ret = fread(buff, 1, amt, dcc->file);
        if (ret < 0)
        {
			#ifdef debug
            logprintf("fread %s returned %d\n", dcc->path, ret);
			#endif

            return FAILURE;
        }
        /* nothing to read and sent everything */
        else if ((ret == 0) && (dcc->ctr >= dcc->size))
        {
            break;
        }

        tmp = send(dcc->sock, buff, ret, 0);
        if (tmp <= 0)
        {
			#ifdef debug
            logprintf("send %s returned %d\n", dcc->path, tmp);
			#endif

            return FAILURE;
        }
        dcc->lastdata = time(NULL);
        dcc->ctr += tmp;
        ctr += tmp;

        /* I've matched the amount I can send */
        if ((bot->tmpcap != 0) && (ctr >= bot->tmpcap))
            break;

        /* Any send can only send data up to a second */
        gettimeofday(&etv, NULL);
        etime = (double)(etv.tv_sec - stv.tv_sec) + (double)(etv.tv_usec - stv.tv_usec) / 1000000;
        if (etime > 1)
            break;
    }

	if (dcc->ctr < dcc->size) {
		dcc->speed = ctr;
	}

    if ((dcc->ctr < dcc->size) && (fseek(dcc->file, 0 - ((long)ret - tmp), SEEK_CUR) != 0))
    {
        errorlog( "fseek failed %s\n", strerror(errno));
        status = FAILURE;
    }

    return status;
}

/*******************************************************************************
*---|   Function:      RETR_ACK  
*---|   Created:       Sun Dec  7 11:41:24 CST 2003
*---|   Last Edited:   Tue Dec  9 15:40:49 CST 2003
*---|   Description:   Reads in the acks sent from the client.
*******************************************************************************/

int retr_ack(dccstruct *dcc)
{
    int status = SUCCESS;
    unsigned char buff[BLOCKSIZE] = { 0 };

    status = recv(dcc->sock, buff, BLOCKSIZE, 0);
    if (status <= 0)
    {
        return FAILURE;
    }

    /* only interested in the last 4 byte ack sent */
    dcc->ackval = (unsigned long)((buff[status - 4] << 24) + (buff[status - 3] << 16) + (buff[status - 2] << 8) + buff[status - 1]);
    if (dcc->ackval >= dcc->size)
    {
		#ifdef debug
		logprintf( "File Transfer finished.\n");
		#endif

        dcc->state = DCCDONE;
    }

    return status;
}
