/*******************************************************************************
*---|
*---|   Garnaxbot by Freakster http://garnax.mircx.com/
*---|
*---|   Garnaxbot is distributed in the hope that it will be useful,
*---|   but WITHOUT ANY WARRANTY; without even the implied warranty of
*---|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*---|
********************************************************************************
*----|
*----|   File:          readcfg.c
*----|   Created:       Sun Dec 15 22:47:27 CST 2002
*----|   Last Edited:   Wed Nov 26 03:06:50 CST 2003
*----|   Description:   Reads a config file and sets the values
*----|                  into the bot.
*----|
*******************************************************************************/

#include "readcfg.h"

#ifdef UPTIMECONTEST
#include "uptime.h"
long uptimectr = 0;
#endif

/*******************************************************************************
*---|   Function:      READCFG
*---|   Created:       Sun Dec 15 22:47:27 CST 2002
*---|   Last Edited:   Thu Jul 24 22:23:37 CDT 2003
*---|   Description:   Reads in the config file values and parses
*---|                  out data from comments.
*******************************************************************************/

int readcfg()
{

    FILE *config;
    char *line = NULL;
    int i = 0, comment = 0, linecount = 0, err = 0;

	#ifdef debug
    logprintf("Attempting to open config file. [%s]\n", bot->filename);
	#endif

    if ((config = fopen(bot->filename, "r")) == NULL)
    {
        errorlog("FATAL: Error opening config file:[%s] - %s\n", bot->filename, strerror(errno));
        return FAILURE;
    }

    while ((err == 0) && (line = readin_a_string(config)) != NULL)
    {
        linecount++;

        /* Empty line - Skip it */
        if (line[0] == '\n')
        {
            free(line);
            line = NULL;
            continue;
        }

        i = strlen(line);

        /* Any line that starts with a '#' is a comment */
        if (line[0] == '#')
        {}

        /* Any line that is only one character is an error.
        ** This is a check so that the following checks don't
        ** go out of bounds
        */
        else if (i == 1)
        {
            err = 1;
        }

        /* Any Line with "//" at the begining is a comment */
        else if ((line[0] == '/') && (line[1] == '/'))
        {}

        /* Any Line that starts with a slash star and
        ** ends with a star slash is a comment.  If the line
        ** only has the slash star, then it starts a comment
        ** block.
        */
        else if ((line[0] == '/') && (line[1] == '*'))
        {
            if (!((line[i - 1] == '/') && (line[i - 2] == '*')))
            {
                comment = 1;
            }
        }

        /* If we are in a comment block then skip this line. */
        else if (comment)
        {

            /* As long as we are in a comment block then any
            ** line that ends with star slash ends the comment block
            */
            if ((line[i - 1] == '/') && (line[i - 2] == '*'))
            {
                comment = 0;
            }

        }

        /* This is where text should mean something. */
        else
        {

            /* fprintf(stdout, "Line:[#%d][Len: %d][%s]\n", linecount, i, line); */

            if (parse_cfg_lines(line) == FAILURE)
            {
                err = 1;
            }
        }

        free(line);
        line = NULL;
    }

    fclose(config);

	#ifdef debug
    fprintf(stdout, "*******************************************************************************\n");
	#endif

    if (err)
    {
        errorlog("FATAL: Error parsing the config file. Line:[%d]\n", linecount);
        return FAILURE;
    }

    return verify_bot();
}

/*******************************************************************************
*---|   Function:      READIN_A_STRING
*---|   Created:       Sun Dec 15 22:47:27 CST 2002
*---|   Last Edited:   Wed Nov 26 02:59:10 CST 2003
*---|   Description:   Reads data into a buffer and parses it
*---|                  into readable format and checks for errors.
*******************************************************************************/

char * readin_a_string(FILE *fstream)
{

    char *rastmp = NULL;
    char temp[BUFFSIZE], *tmp = NULL;
    int i = 0, len = 0;

    /* This is a list of all the different kinds of input to handle.
    **
    ** num) strlen "buffer" - type
    **
    **  #1) 0 "\0"   - Empty
    **  #2) 1 "x\0"   - No Newline
    **  #3) 1 "\n\0"  - Newline Only
    **  #4) 2 "x\n\0"  - text and no spaces
    **  #5) 2 " \n\0"  - only spaces
    **  #6) 3 " x\n\0"  - Leading space only
    **  #7) 3 "x \n\0"  - Trailing space only
    **  #8) 4 " x \n\0"  - Leading and trailing spaces
    */

    /* Handles #1 - Completely Empty Lines, Errors, or End of files. */
    if (fgets(temp, BUFFSIZE, fstream) == NULL)
    {
        return NULL;
    }

    len = strlen(temp);

    /* fprintf(stdout, "Before:[%d][%s]\n", len, temp);
    */

    /* Handles #2 - Lines read in that are too long to contain a newline */
    if ((temp[len - 1] != '\n') && (len == BUFFSIZE))
    {
        errorlog( "FATAL: Read in a string larger than %d characters.\n", BUFFSIZE);
        errorlog( "No line in the config file should be that long.\n");
        errorlog( "String:[%s]\n", temp);
        return NULL;
    }

    /* Remove any leading spaces. */
    for (i = 0; (temp[i] == '\t') || (temp[i] == ' ') ; i++);

    /*
    **  After removing leading spaces the cases condense into other cases.
    **  #5 becomes #3
    **  #6 becomes #4
    **  #8 becomes #7
    **
    **  We are left with:
    **  #3) 1 "\n\0"  - Newline Only
    **  #4) 2 "x\n\0"  - text and no spaces
    **  #7) 3 "x \n\0"  - Trailing space only
    **
    */
    tmp = &temp[i];
    len = strlen(tmp);

    /* Remove any trailing spaces, tabs, carriage returns, or newlines. */
    for (len--; (len > 0) && ((tmp[len - 1] == ' ') || (tmp[len - 1] == '\t') || (tmp[len - 1] == '\r') || (tmp[len - 1] == '\n')); len--);

    /* Handle #3
    ** Send back a newline to indicate an empty line
    ** rather than sending a NULL which indicates
    ** an error */
    if (len == 0)
    {
        tmp[0] = '\n';
        tmp[1] = '\0';
    }
    /* We have data! */
    else
    {
        tmp[len] = '\0';
    }

    /* By this point we either have an empty line or a line of text.
    **  #3) 1 "\n\0"    - Newline Only
    **  #9) 2 "x\0"     - text and no spaces
    */

    /* rastmp will be NULL if mymalloc fails indicating an error
    ** to the higher layer so I don't have to check it */
    rastmp = mymalloc(tmp);

    /* fprintf(stdout, "After:[%d][%s]\n", strlen(rastmp), rastmp);
    */

    return rastmp;
}

/*******************************************************************************
*---|     Function:        PARSE_CFG_LINES
*---|     Created:         Sun Dec 15 22:47:27 CST 2002
*---|     Last Edited:     Thu Jul 24 22:54:50 CDT 2003
*---|     Description:     once the data is properly formatted it comes here
*---|                      to be read into the appropriate data structures
*******************************************************************************/

int parse_cfg_lines(char *text)
{

    char *tmp = text;
    int i = 0;
    channelstruct *chan = bot->channels;

    /* all text at this point is in the format "option <space or tabs> settings\0"
    ** This will make text point to "option\0" and tmp point to "settings\0" */

    /* Find the first space or tab */
    while ((tmp[i] != ' ') && (tmp[i] != '\t') && (tmp[i] != '\0'))
        i++;
    if (tmp[i] == '\0')
    {
        errorlog( "Error: Option with no parameters.\n");
        return FAILURE;
    }
    tmp[i++] = '\0';

    /* Find the first non space/tab */
    while ((tmp[i] == ' ') || (tmp[i] == '\t'))
        i++;
    if (tmp[i] == '\0')
    {
        errorlog("Error: Option with no parameters.\n");

        return FAILURE;
    }
    tmp = &tmp[i];

    /*
    fprintf(stdout, "***\n");
    fprintf(stdout, "Option:[%s] setting:[%s]\n", text, tmp);
    */

    if (lowercasecmp(text, "STATFILE") == 0)
    {
        if (bot->statfile)
        {
            errorlog( "Error: Assigned multiple statfile:[%s][%s]\n", bot->statfile, tmp);
            return FAILURE;
        }
        bot->statfile = mymalloc(tmp);
        if (!bot->statfile) return FAILURE;
		#ifdef debug
        logprintf( "Stat Filename:[%s]\n", bot->statfile);
		#endif

        load_stats();
    }
    else if (lowercasecmp(text, "PIDFILE") == 0)
    {
        if (bot->pidfile)
        {
            errorlog( "Error: Assigned multiple pidfile:[%s][%s]\n", bot->pidfile, tmp);
            return FAILURE;
        }
        bot->pidfile = mymalloc(tmp);
        if (!bot->pidfile) return FAILURE;
		#ifdef debug
		logprintf( "PID Filename:[%s]\n", bot->pidfile);
		#endif

        write_pid();
    }
    else if (lowercasecmp(text, "TRANSFERLOG") == 0)
    {
        if (bot->transferlog)
        {
            errorlog( "Error: Assigned multiple transferlog:[%s][%s]\n", bot->transferlog, tmp);
            return FAILURE;
        }
        bot->transferlog = mymalloc(tmp);
        if (!bot->transferlog) return FAILURE;
		#ifdef debug
		logprintf( "Transfer Log:[%s]\n", bot->transferlog);
		#endif

    }
    else if (lowercasecmp(text, "BOTNICK") == 0)
    {
        if (bot->nickkeep)
        {
            errorlog( "Error: Assigned multiple nickkeep:[%s][%s]\n", bot->nickkeep, tmp);
            return FAILURE;
        }
        bot->nickkeep = mymalloc(tmp);
        if (!bot->nickkeep) return FAILURE;
		#ifdef debug
        logprintf( "Nickname:[%s]\n", bot->nickkeep);
		#endif

    }
    else if (lowercasecmp(text, "NICKPASS") == 0)
    {
        if (bot->nickpass)
        {
            errorlog( "Error: Assigned multiple nickpass:[%s][%s]\n", bot->nickpass, tmp);
            return FAILURE;
        }
        bot->nickpass = mymalloc(tmp);
        if (!bot->nickpass) return FAILURE;
		#ifdef debug
        logprintf( "NickServ Password:[%s]\n", bot->nickpass);
		#endif

    }
    else if (lowercasecmp(text, "SERVER") == 0)
    {
        if (add_server(tmp) == FAILURE)
        {
            return FAILURE;
        }
    }
    else if (lowercasecmp(text, "CONNECTDELAY") == 0)
    {
        if (bot->connectdelay != 60)
        {
            errorlog( "Error: Assigned multiple connectdelay:[%d][%s]\n", bot->connectdelay, tmp);
            return FAILURE;
        }
        bot->connectdelay = atoi(tmp);
		#ifdef debug
        logprintf( "Connection Delay:[%d]\n", bot->connectdelay);
		#endif

    }
    else if (lowercasecmp(text, "SERVEREXPAND") == 0)
    {
        if ((lowercasecmp(tmp, "1") == 0) || (lowercasecmp(tmp, "YES") == 0) || (lowercasecmp(tmp, "ON") == 0))
        {
            BITSET(bot->settings, EXPAND);
        }
        else
        {
            BITCLR(bot->settings, EXPAND);
        }
		#ifdef debug
        logprintf( "Server Expansion:[%d].\n", (int)(bot->settings & EXPAND));
		#endif

    }
    else if (lowercasecmp(text, "CTCPTRIGGERS") == 0)
    {
        if ((lowercasecmp(tmp, "1") == 0) || (lowercasecmp(tmp, "YES") == 0) || (lowercasecmp(tmp, "ON") == 0))
        {
            BITSET(bot->settings, CTCPTRIG);
        }
        else
        {
            BITCLR(bot->settings, CTCPTRIG);
        }
		#ifdef debug
        logprintf( "CTCP Triggers:[%d].\n", (int)(bot->settings & CTCPTRIG));
		#endif

    }
    else if (lowercasecmp(text, "IPFROMSERVER") == 0)
    {
        if ((lowercasecmp(tmp, "1") == 0) || (lowercasecmp(tmp, "YES") == 0) || (lowercasecmp(tmp, "ON") == 0))
        {
            BITSET(bot->settings, DCCIP);
        }
        else
        {
            BITCLR(bot->settings, DCCIP);
        }
		#ifdef debug
        logprintf( "IP From Server:[%d].\n", (int)(bot->settings & DCCIP));
		#endif

    }
    else if (lowercasecmp(text, "VHOST") == 0)
    {
        if (bot->vhost)
        {
            errorlog( "Error: Assigned multiple vhost:[%s][%s]\n", bot->vhost, tmp);
            return FAILURE;
        }
        bot->vhost = mymalloc(tmp);
        if (!bot->vhost) return FAILURE;
		#ifdef debug
        logprintf( "Vhost:[%s]\n", bot->vhost);
		#endif

    }
    else if (lowercasecmp(text, "MINPORT") == 0)
    {
        if (bot->minport != 1024)
        {
            errorlog( "Error: Assigned multiple minport:[%d][%s]\n", bot->minport, tmp);
            return FAILURE;
        }
        bot->minport = atoi(tmp);
        if ((bot->minport <= 0) || (bot->minport > 65535))
        {
            errorlog("Invalid MinPort:[%d]\n", bot->minport);
            return FAILURE;
        }
		#ifdef debug
        logprintf( "Minimum Port:[%d]\n", bot->minport);
		#endif

    }
    else if (lowercasecmp(text, "MAXPORT") == 0)
    {
        if (bot->maxport != 65535)
        {
            errorlog( "Error: Assigned multiple maxport:[%d][%s]\n", bot->maxport, tmp);
            return FAILURE;
        }
        bot->maxport = atoi(tmp);
        if ((bot->maxport <= 0) || (bot->maxport > 65535))
        {
            errorlog("Invalid MaxPort:[%d]\n", bot->maxport);
            return FAILURE;
        }
		#ifdef debug
        logprintf( "Maximum Port:[%d]\n", bot->maxport);
		#endif

    }
    else if (lowercasecmp(text, "TOTALCAP") == 0)
    {
        if (bot->totalcap != 0)
        {
            errorlog( "Error: Assigned multiple totalcap:[%d][%s]\n", bot->totalcap, tmp);
            return FAILURE;
        }
        bot->totalcap = atoi(tmp) * 1024;
		#ifdef debug
        logprintf( "Total sends speed:[%dkBps]\n", bot->totalcap);
		#endif

    }
    else if (lowercasecmp(text, "SENDCAP") == 0)
    {
        if (bot->sendcap != 0)
        {
            errorlog( "Error: Assigned multiple sendcap:[%d][%s]\n", bot->sendcap, tmp);
            return FAILURE;
        }
        bot->sendcap = atoi(tmp) * 1024;
		#ifdef debug
        logprintf( "Individual send speed:[%dkBps]\n", bot->sendcap);
		#endif

    }
    else if (lowercasecmp(text, "MINSPEED") == 0)
    {
        if (bot->minspeed != 0)
        {
            errorlog( "Error: Assigned multiple minspeed:[%d][%s]\n", bot->minspeed, tmp);
            return FAILURE;
        }
        bot->minspeed = atoi(tmp);
		#ifdef debug
        logprintf( "Minimum send speed:[%dkBps]\n", bot->minspeed);
		#endif

    }
    else if (lowercasecmp(text, "FSERVETOTAL") == 0)
    {
        if (bot->fserves->total != 0)
        {
            errorlog( "Error: Assigned multiple fservetotal:[%d][%s]\n", bot->fserves->total, tmp);
            return FAILURE;
        }
        bot->fserves->total = atoi(tmp);
		#ifdef debug
        logprintf( "Fserve Total:[%d]\n", bot->fserves->total);
		#endif

    }
    else if (lowercasecmp(text, "FSERVEEACH") == 0)
    {
        if (bot->fserves->each != 0)
        {
            errorlog( "Error: Assigned multiple fserveeach:[%d][%s]\n", bot->fserves->each, tmp);
            return FAILURE;
        }
        bot->fserves->each = atoi(tmp);
		#ifdef debug
        logprintf( "Fserve Each:[%d]\n", bot->fserves->each);
		#endif

    }
    else if (lowercasecmp(text, "SENDTOTAL") == 0)
    {
        if (bot->sends->total != 0)
        {
            errorlog( "Error: Assigned multiple sendtotal:[%d][%s]\n", bot->sends->total, tmp);
            return FAILURE;
        }
        bot->sends->total = atoi(tmp);
		#ifdef debug
        logprintf( "Sends Total:[%d]\n", bot->sends->total);
		#endif

    }
    else if (lowercasecmp(text, "SENDEACH") == 0)
    {
        if (bot->sends->each != 0)
        {
            errorlog( "Error: Assigned multiple sendeach:[%d][%s]\n", bot->sends->each, tmp);
            return FAILURE;
        }
        bot->sends->each = atoi(tmp);
		#ifdef debug
        logprintf( "Sends Each:[%d]\n", bot->sends->each);
		#endif

    }
    else if (lowercasecmp(text, "QUEUETOTAL") == 0)
    {
        if (bot->queues->total != 0)
        {
            errorlog( "Error: Assigned multiple queuetotal:[%d][%s]\n", bot->queues->total, tmp);
            return FAILURE;
        }
        bot->queues->total = atoi(tmp);
		#ifdef debug
        logprintf( "Queues Total:[%d]\n", bot->queues->total);
		#endif

    }
    else if (lowercasecmp(text, "QUEUEEACH") == 0)
    {
        if (bot->queues->each != 0)
        {
            errorlog( "Error: Assigned multiple queueeach:[%d][%s]\n", bot->queues->each, tmp);
            return FAILURE;
        }
        bot->queues->each = atoi(tmp);
		#ifdef debug
        logprintf( "Queues Each:[%d]\n", bot->queues->each);
		#endif

    }
    else if (lowercasecmp(text, "FORCESIZE") == 0)
    {
        if (bot->forcesize != 0)
        {
            errorlog( "Error: Assigned multiple forcesize:[%d][%s]\n", bot->forcesize, tmp);
            return FAILURE;
        }
        bot->forcesize = atoi(tmp);
		#ifdef debug
        logprintf( "Force Size:[%d]\n", bot->forcesize);
		#endif

    }
    else if (lowercasecmp(text, "COLOR1") == 0)
    {
        if (strcmp(bot->color1, "00") != 0)
        {
            errorlog( "Error: Assigned multiple color1:[%s][%s]\n", bot->color1, tmp);
            return FAILURE;
        }
        i = atoi(tmp);
        if (i > 15)
        {
            errorlog( "ERROR: invalid color. 0-15 [%d]\n", i);
            return FAILURE;
        }
        else if (i < 10) sprintf(bot->color1, "0%d", i);
        else sprintf(bot->color1, "%d", i);
		#ifdef debug
        logprintf( "Color 1:[%s]\n", bot->color1);
		#endif

    }
    else if (lowercasecmp(text, "COLOR2") == 0)
    {
        if (strcmp(bot->color2, "00") != 0)
        {
            errorlog( "Error: Assigned multiple color2:[%s][%s]\n", bot->color2, tmp);
            return FAILURE;
        }

        i = atoi(tmp);
        if (i > 15)
        {
            errorlog( "ERROR: invalid color. 0-15 [%d]\n", i);
            return FAILURE;
        }
        else if (i < 10) sprintf(bot->color2, "0%d", i);
        else sprintf(bot->color2, "%d", i);
		#ifdef debug
        logprintf( "Color 2:[%s]\n", bot->color2);
		#endif

    }
    else if (lowercasecmp(text, "OB") == 0)
    {
        if (bot->ob)
        {
            errorlog( "Error: Assigned multiple open brackets:[%s][%s]\n", bot->ob, tmp);
            return FAILURE;
        }
        bot->ob = mymalloc(tmp);
        if (!bot->ob) return FAILURE;
		#ifdef debug
        logprintf( "Open bracket Style: %s\n", bot->ob);
		#endif

    }
    else if (lowercasecmp(text, "CB") == 0)
    {
        if (bot->cb)
        {
            errorlog( "Error: Assigned multiple closed brackets:[%s][%s]\n", bot->cb, tmp);
            return FAILURE;
        }
        bot->cb = mymalloc(tmp);
        if (!bot->cb) return FAILURE;
		#ifdef debug
        logprintf( "Closed bracket Style: %s\n", bot->cb);
		#endif

    }
    else if (lowercasecmp(text, "CHANNEL") == 0)
    {

		#ifdef debug
        fprintf(stdout, "*******************************************************************************\n");
		#endif

        if ((chan = is_channel(tmp)))
        {

            /*
            ** Since the channel already exists, we will remove it and put it at
            ** the begining so the following channel options can be added to it.
            ** Which either means duplicate channel or a rehash.
            */

            if ((chan->settings & REHASHED) == 0)
            {
                errorlog( "ERROR: Duplicate Channel Declaration. [%s]\n", chan->name);
                return FAILURE;
            }
            chan = remove_channel(chan);
            BITCLR(chan->settings, REHASHED);
        }
        else
        {
            chan = create_channel();
            if (chan == NULL)
            {
                return FAILURE;
            }
            chan->name = mymalloc(tmp);
            if (!chan->name)
                return FAILURE;
        }
        add_channel(chan);
    }

    /* All channel options are added to the first channel in
    ** the channel list.  As this is the last channel added. */

    else if (chan)
    {
        if (lowercasecmp(text, "KEY") == 0)
        {
            if (chan->key)
            {
                errorlog( "Error: Assigned multiple channel keys:[%s][%s] to %s\n", chan->key, tmp, chan->name);
                return FAILURE;
            }
            chan->key = mymalloc(tmp);
            if (!chan->key) return FAILURE;
			#ifdef debug
            logprintf( "Channel Key:[%s]\n", chan->key);
			#endif

        }
        else if (lowercasecmp(text, "INVITE") == 0)
        {
            if ((lowercasecmp(tmp, "1") == 0) || (lowercasecmp(tmp, "YES") == 0) || (lowercasecmp(tmp, "ON") == 0))
            {
                BITSET(chan->settings, INVITEIN);
            }
            else
            {
                BITCLR(chan->settings, INVITEIN);
            }
			#ifdef debug
            logprintf( "Message chanserv to invite in:[%d].\n", (int)(chan->settings & INVITEIN));
			#endif

        }
        else if (lowercasecmp(text, "JOINADVERTISE") == 0)
        {
            if ((lowercasecmp(tmp, "1") == 0) || (lowercasecmp(tmp, "YES") == 0) || (lowercasecmp(tmp, "ON") == 0))
            {
                BITSET(chan->settings, DISPLAYAD);
            }
            else
            {
                BITCLR(chan->settings, DISPLAYAD);
            }
			#ifdef debug
            logprintf( "Advertise on Join:[%d].\n", (int)(chan->settings & DISPLAYAD));
			#endif

        }
        else if (lowercasecmp(text, "DCCWATCH") == 0)
        {
            if ((lowercasecmp(tmp, "1") == 0) || (lowercasecmp(tmp, "YES") == 0) || (lowercasecmp(tmp, "ON") == 0))
            {
                BITSET(chan->settings, DCCWATCH);
            }
            else
            {
                BITCLR(chan->settings, DCCWATCH);
            }
			#ifdef debug
            logprintf( "DCC Watch:[%d].\n", (int)(chan->settings & DCCWATCH));
			#endif

        }
        else if (lowercasecmp(text, "BANNED") == 0)
        {
            chan->banned = add_member(chan->banned, tmp);
        }
        else if (lowercasecmp(text, "MASTER") == 0)
        {
            chan->masters = add_member(chan->masters, tmp);
        }
        else if (lowercasecmp(text, "DESCRIPTION") == 0)
        {
            if (chan->desc)
            {
                errorlog( "Error: Assigned multiple fserve descriptions:[%s][%s] to [%s]\n", chan->desc, tmp, chan->name);
                return FAILURE;
            }
            chan->desc = mymalloc(tmp);
            if (!chan->desc) return FAILURE;
			#ifdef debug
            logprintf( "Fserve Description:[%s]\n", chan->desc);
			#endif

        }
        else if (lowercasecmp(text, "WELCOME") == 0)
        {
            if (chan->welcomemsg)
            {
                errorlog( "Error: Assigned multiple welcome messages:[%s][%s] to [%s]\n", chan->welcomemsg, tmp, chan->name);
                return FAILURE;
            }
            chan->welcomemsg = mymalloc(tmp);
            if (!chan->welcomemsg) return FAILURE;
			#ifdef debug
            logprintf( "Welcome Message:[%s]\n", chan->welcomemsg);
			#endif

        }
        else if (lowercasecmp(text, "TRIGGER") == 0)
        {
            if (chan->trigger)
            {
                errorlog( "Error: Assigned multiple triggers:[%s][%s] to [%s]\n", chan->trigger, tmp, chan->name);
                return FAILURE;
            }
            chan->trigger = mymalloc(tmp);
            if (!chan->trigger) return FAILURE;
			#ifdef debug
            logprintf( "Trigger:[%s]\n", chan->trigger);
			#endif

        }
        else if (lowercasecmp(text, "DELAY") == 0)
        {
            if (chan->delay != 0)
            {
                errorlog( "Error: Assigned multiple chan delay:[%d][%s] to [%s]\n", chan->delay, tmp, chan->name);
                return FAILURE;
            }
            chan->delay = atoi(tmp);
			#ifdef debug
            logprintf( "Advertise Delay:[%d]\n", chan->delay);
			#endif

        }
        else if (lowercasecmp(text, "PATH") == 0)
        {
            if (add_dir(tmp) == FAILURE)
            {
                return FAILURE;
            }
        }
        else
        {
            errorlog( "Error in the config file. With three possibilities.\n");
            errorlog( "1. Syntax Error.\n2. Unsupported option.\n3. A channel option without a channel decleration.\n");
            return FAILURE;
        }
    }
    else
    {
        errorlog( "Error in the config file. With three possibilities.\n");
        errorlog( "1. Syntax Error.\n2. Unsupported option.\n3. A channel option without a channel decleration.\n");
        return FAILURE;
    }

    return SUCCESS;
}

/*******************************************************************************
*---|     Function:        VERIFY_BOT
*---|     Created:         Thu Jul 24 23:18:05 CDT 2003
*---|     Last Edited:     Tue Aug 12 01:46:36 CDT 2003
*---|     Description:     Verifies that all necessary bot structs
*---|                      are initialized.
*******************************************************************************/

int verify_bot()
{
    channelstruct *vchan = NULL;

    if (bot->nickkeep == NULL)
    {
        errorlog( "Bot needs a nickname.\n");
        return FAILURE;
    }

    if (bot->servers == NULL)
    {
        errorlog( "Bot needs at least one server.\n");
        return FAILURE;
    }

    if (bot->channels == NULL)
    {
        errorlog( "Bot needs at least one channel.\n");
        return FAILURE;
    }

    if ((bot->fserves->each <= 0) || (bot->fserves->each <= 0) || (bot->sends->total <= 0) || (bot->sends->each <= 0))
    {
        errorlog( "Set Fserve and Send total/each greater than 0.\n");
        return FAILURE;
    }

    if ((bot->ob == NULL) || (bot->cb == NULL))
    {
        errorlog( "Need to specify opening and closing brackets.\n");
        return FAILURE;
    }

    for (vchan = bot->channels; vchan != NULL; vchan = vchan->next)
    {

        if (vchan->settings & REHASHED)
            continue;
        if ((vchan->desc == NULL) || (vchan->trigger == NULL) || (vchan->directories == NULL))
        {
            errorlog( "Fserve isn't configured correctly for %s\n", vchan->name);
            errorlog( "Either desc, trigger, or paths aren't setup.\n");
            return FAILURE;
        }

        if (((6 + strlen(bot->ob) + strlen(bot->cb)) * 10 + 140 + strlen(bot->nickkeep) + strlen(vchan->trigger) + strlen(vchan->desc) + strlen(VERSION) + strlen(MODEL)) >= BUFFSIZE)
        {
            errorlog( "Fserve ad is too long. Shorten your trigger, desc, or brackets for channel %s.\n", vchan->name);
            return FAILURE;
        }

    }

    return SUCCESS;
}

/*******************************************************************************
*---|     Function:        REHASH
*---|     Created:         Sun Dec 15 22:47:27 CST 2002
*---|     Last Edited:     Sat Jul 19 20:04:03 CST 2003
*---|     Description:     Prepares the bot and channels for
*---|                      a rehash of the config file
*******************************************************************************/

int rehash()
{

    channelstruct *chan = NULL;
    int status = SUCCESS;
    unsigned char tmpsettings;

    rehash_bot();

    for (chan = bot->channels; chan != NULL; chan = chan->next)
    {
        /* rehash_channel will wipe out the settings for the channel
        ** and we don't want that when we are still online
        */
        purge_fserves(chan->name);
        tmpsettings = chan->settings;
        rehash_channel(chan);
        chan->settings = tmpsettings;
        BITCLR(chan->settings, DISPLAYAD);
        BITCLR(chan->settings, DCCWATCH);
        BITCLR(chan->settings, INVITEIN);
        BITSET(chan->settings, REHASHED);
    }

    status = readcfg();

    for (chan = bot->channels;(status && (chan != NULL)); chan = chan->next)
    {
        if (chan->settings & REHASHED)
        {
            status = quote(bot->sock, "PART %s\r\n", chan->name);
        }
    }

    if ((status) && (bot->settings & EXPAND))
    {
        status = quote(bot->sock, "LINKS\r\n");
    }
    if ((status) && (bot->settings & DCCIP))
    {
        status = quote(bot->sock, "USERHOST %s\r\n", bot->nick);
    }

    bot->chktime = time(NULL) - 301;
    if (status)
        status = checkup();

    return status;
}


/*******************************************************************************
*---|     Function:        WRITE_PID
*---|     Created:         Fri Dec 13 22:14:39 CST 2002
*---|     Last Edited:     Sat Jul 19 16:02:51 CDT 2003
*---|     Description:     Writes the pid of the program into a file.
*******************************************************************************/
void write_pid()
{
    FILE *f;

	#ifdef debug
    logprintf( "Attempting to create a pid file. [%s]\n", bot->pidfile);
	#endif

    if ((f = fopen(bot->pidfile, "w")))
    {
        fprintf(f, "%d\n", getpid());
        fclose(f);
    }
	#ifdef debug
    else
    {
        logprintf( "Could not open pid file:[%s] - %s\n", bot->pidfile, strerror(errno));
    }
	#endif

    return ;
}

/*******************************************************************************
*---|     Function:        CHECKUP
*---|     Last Edited: Tue Aug  5 14:10:02 CDT 2003
*---|     Created:         Mon Dec 16 01:08:38 CST 2002
*---|     Description:     Checkup on the bot. i.e. nick, channels
*******************************************************************************/

int checkup()
{

    int status = SUCCESS;
    channelstruct *chan = bot->channels;

	#ifdef UPTIMECONTEST
    /* Every 6 hours send in an update */
    if ((time(NULL) - uptimectr) > 21600)
    {
        uptimectr = time(NULL);
        send_uptime();
    }
	#endif

    /* Every 5 minutes
    ** see if we have our nick
    ** see if we are in all of our channels
    ** save the fserve stats
    */
    if (time(NULL) - bot->chktime > 300)
    {
        bot->chktime = time(NULL);
        save_stats();
        if (lowercasecmp(bot->nick, bot->nickkeep) != 0)
        {
            status = quote(bot->sock, "NICK %s\r\n", bot->nickkeep);
        }
        if (status)
        {
            status = join_chans();
        }
    }

    /* Check ad delays and advertise if the time has elapsed */
    for (chan = bot->channels; ((status) && (chan != NULL)); chan = chan->next)
    {
        if (status && (chan->delay > 0) && (time(NULL) - chan->delaytime > (chan->delay * 60)))
        {
            chan->delaytime = time(NULL);
            status = display_fserve(chan);
        }
    }

    return status;
}

