/*******************************************************************************
*---|
*---|   Garnaxbot by Freakster http://garnax.mircx.com/
*---|
*---|   Garnaxbot is distributed in the hope that it will be useful,
*---|   but WITHOUT ANY WARRANTY; without even the implied warranty of
*---|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*---|
********************************************************************************
*---|
*---|   File:          routine.c
*---|   Created:       Sun Dec 15 22:47:27 CST 2002
*---|   Last Edited:   Wed Nov 26 02:31:49 CST 2003
*---|   Description:   State machine that maintains
*---|                  the flow of the bot.
*---|
*******************************************************************************/

#include "routine.h"

/*******************************************************************************
*---|   Function:      BOT_ROUTINE
*---|   Created:       Sun Dec 15 22:47:27 CST 2002
*---|   Last Edited:   Thu Dec 11 01:36:17 CST 2003
*---|   Description:   Finite state machine for the bot
*******************************************************************************/

void bot_routine(char *filename)
{
	pthread_t fthread, sthread;
	#ifdef WIN32
	int fid = 0, sid = 0;
	#endif

	/* I hate the way this all looks */

	#ifdef debug
    char *t = NULL;
	#else
    /* If on unix and not in debug mode, fork into the background */
	#ifndef WIN32
    int pid = fork();
    if (pid != 0) return;
	#endif
	#endif

	#ifdef UPTIMECONTEST
	init_uptime();
	#endif

    /* create the bot structure */
    if (create_bot(filename) == FAILURE)
    {
        errorlog("Error Creating the bot structure.\n");
        return;
    }

#ifdef WIN32
	mutex = CreateMutex(NULL, 0, NULL);
	if (mutex == NULL)
	{
        errorlog("Error creating the DCC mutex.\n");
		return;
	}
	fthread = CreateThread(NULL, 0, fserve_thread, NULL, 0, NULL);
	if (fthread == NULL)
	{
		errorlog("Error creating fserve thread. %s\n", strerror(errno) );
		return;
	}
	sthread = CreateThread(NULL, 0, sends_thread, NULL, 0, NULL);
	if (sthread == NULL)
	{
		errorlog("Error creating sends thread. %s\n", strerror(errno) );
		return;
	}
#else
    if (pthread_mutex_init(&mutex, NULL) != 0)
    {
        errorlog("Error creating the mutex.\n");
		return;
    }
    if (pthread_mutex_init(&dccmutex, NULL) != 0)
    {
        errorlog("Error creating the DCC mutex.\n");
		return;
    }
    if (pthread_create(&fthread, NULL, fserve_thread, NULL) != 0)
	{
        errorlog("Error creating fserve thread. %s\n", strerror(errno) );
		return;
	}
	if (pthread_create(&sthread, NULL, sends_thread, NULL) != 0)
    {
        errorlog("Error creating sends thread. %s\n", strerror(errno) );
		return;
    }
#endif

    /* State machine */
    while (alive)
    {
        if (bot->restart == 1)
        {
            bot->restart = 0;
            bot->state = RESTART;
        }
        else if (bot->restart == 2)
        {
            bot->restart = 0;
            bot->state = REHASH;
        }

        switch (bot->state)
        {

        case REHASH:
            if (rehash() == SUCCESS)
            {
                bot->state = READDATA;
            }
            else
            {
                alive = 0;
            }
            break;
        case READCFG:
            if (readcfg() == SUCCESS)
            {
                bot->state = CONNECT;
            }
            else
            {
                alive = 0;
            }
            break;

        case CONNECT:
            if (bot_connect() == SUCCESS)
            {
                bot->state = READDATA;
            }
            else
            {
                if (alive) sleep(bot->connectdelay);
            }
            break;

        case READDATA:
            if ((bot_read() == FAILURE) || (checkup() == FAILURE))
            {
                bot->state = RECONNECT;
                if (alive) sleep(bot->connectdelay);
            }
            break;

        case RECONNECT:
            reset_bot();
            bot->state = CONNECT;
            break;

        case RESTART:
            pthread_mutex_lock(&mutex);
            destroy_bot();
            if (create_bot(filename) == FAILURE)
            {
                alive = 0;
            }
            pthread_mutex_unlock(&mutex);
            break;

        default:
            /* Error */
            errorlog("FATAL: Entered an unknown bot state.\n");
            alive = 0;
            break;

        }
    }

	/* Cleanup threads */
	#ifdef WIN32
    GetExitCodeThread(fthread, &fid);
    GetExitCodeThread(sthread, &sid);
	CloseHandle(mutex);
	#else
    pthread_join(fthread, NULL);
    pthread_join(sthread, NULL);
    pthread_mutex_destroy(&mutex);
	#endif

	/* Clean up bot */
    destroy_bot();

	#ifdef debug
    t = gettime(time(NULL) - runtime);
    fprintf(stdout, "Runtime: %s\n", t);
    free(t);
    t = NULL;
	#endif

    return;
}

/*******************************************************************************
*---|   Function:      BOT_CONNECT
*---|   Created:       Fri Dec 13 22:14:39 CST 2002
*---|   Last Edited:   Wed Nov 26 12:00:57 CST 2003
*---|   Description:   Connects to a server. Setup vhost if specified.
*******************************************************************************/

int bot_connect()
{

    struct hostent *he, *virt;
    struct sockaddr_in addr, my_addr, server_addr;
    unsigned int sin_size = 0;

    /* Find the next server */
    if ((bot->server == NULL) || (bot->server->next == NULL))
    {
        bot->server = bot->servers;
    }
    else
    {
        bot->server = bot->server->next;
    }

	#ifdef debug
    logprintf("Attempting connection to [%s]\n", bot->server->name);
	#endif

    if ((bot->sock = socket(AF_INET, SOCK_STREAM, 0)) == -1)
    {
		#ifdef debug
        errorlog("Socket() in bot_connect() failed. %s\n", strerror(errno));
		#endif

        return FAILURE;
    }

    memset(&my_addr, 0, sizeof(struct sockaddr_in));
    my_addr.sin_family = AF_INET;
    my_addr.sin_port = htons(0);
    my_addr.sin_addr.s_addr = htonl(INADDR_ANY);

    /* Bind to a particular IP/host */
    if (bot->vhost)
    {
        if ((virt = gethostbyname(bot->vhost)) != NULL)
        {
            my_addr.sin_addr = *((struct in_addr *)virt->h_addr);
#ifdef debug

            logprintf("Bound to address:[%s]\n", virt->h_name);
#endif

        }
        else
        {
            errorlog( "Can not resolve virtualhost:[%s] - %s\n", bot->vhost, strerror(errno) );
            errorlog( "This host must be active on the current system.\n");
        }
    }

    if (bind(bot->sock, (struct sockaddr *)&my_addr, sizeof(my_addr)) != 0)
    {
        errorlog("bind() in bot_connect() failed. %s\n", strerror(errno));
        if (bot->vhost)
        {
            errorlog( "Can not bind to address:[%s] - %s\n", bot->vhost, strerror(errno) );
            errorlog( "This host must be active on the current system.\n");
        }
        close(bot->sock);
        bot->sock = 0;
        return FAILURE;
    }

    /* Resolve the server name */
    if ((he = gethostbyname(bot->server->name)) == NULL)
    {
        errorlog( "Can not resolve:[%s] - %s\n", bot->server->name, strerror(errno));

        close(bot->sock);
        bot->sock = 0;
        return FAILURE;
    }

    memset(&addr, 0, sizeof(struct sockaddr_in));
    addr.sin_family = AF_INET;
    addr.sin_port = htons(bot->server->port);
    addr.sin_addr = *((struct in_addr *)he->h_addr);

    /* Make the connection */
    if (connect(bot->sock, (struct sockaddr *)&addr, sizeof(struct sockaddr)) < 0)
    {
        errorlog( "Connection to [%s] failed. - %s\n", bot->server->name, strerror(errno));

        if (bot->sock)
            close(bot->sock);
        bot->sock = 0;
        return FAILURE;
    }

    /*
       FD_ZERO(&rfd);
       FD_SET(bot->sock, &rfd);
       wfd = rfd;

       tv.tv_sec = 30;
       tv.tv_usec = 0;

       if (select(bot->sock + 1, &rfd, &wfd, NULL, &tv) <= 0)
       {
           errorlog( "Connection to [%s] timed out. - %s\n", bot->server->name, strerror(errno));

           close(bot->sock);
     bot->sock = 0;
           return FAILURE;
       }

       if ((FD_ISSET(bot->sock, &rfd) > 0) || (FD_ISSET(bot->sock, &wfd) > 0))
       {
    */

    bot->contime = time(NULL);
    bot->chktime = time(NULL);

    if ((bot->settings & DCCIP) == 0)
    {
        /* figure out which IP is used to connect to the irc server */
        sin_size = sizeof(struct sockaddr_in);
        if (getsockname(bot->sock, (struct sockaddr *)&server_addr, &sin_size) < 0)
        {
            errorlog( "bot_connect getsockname Error [%s]\n", strerror(errno) );
            close(bot->sock);
            bot->sock = 0;
            return FAILURE;
        }
        bot->ip = (unsigned long)server_addr.sin_addr.s_addr;
    }
    /*
      }
      else
      {
          close(bot->sock);
    bot->sock = 0;
          return FAILURE;
      }
    */

    return quote(bot->sock, "USER %s 8 8 :%s\r\nNICK %s\r\n", bot->nickkeep, bot->nickkeep, bot->nickkeep);
}

/*******************************************************************************
*---|   Function:      BOT_READ
*---|   Created:       Fri Dec 13 22:14:39 CST 2002
*---|   Last Edited:   Thu Jul 31 15:57:16 CDT 2003
*---|   Description:   Reads in the data on the socket.
*******************************************************************************/

int bot_read()
{
    int ret, readingdata = 1, i = 0, status = SUCCESS;
    char buff[BUFFSIZE + 1];
    struct timeval tv;
    fd_set fd;

    do
    {

        FD_ZERO(&fd);
        FD_SET(bot->sock, &fd);
        tv.tv_sec = 120;
        tv.tv_usec = 0;

        ret = select(bot->sock + 1, &fd, (fd_set *)0, (fd_set *)0, &tv);

        /* error */
        if (ret < 0)
        {
#ifdef debug
            fprintf(stdout, "Select failed for bot_read. %s\n", strerror(errno));
#endif

            return FAILURE;
        }

        /* timed out without receiving any data try to ping the
        ** server to verify the connection. If we are still connected 
        ** the next select call will not time out.
		*/
        else if (ret == 0)
        {
            if (readingdata == 2)
            {
                return FAILURE;
            }
            else
            {
                readingdata = 2;
                status = quote(bot->sock, "PING :%s\r\n", bot->servername);
            }
        }

        /* We've got data! */
        else if (ret > 0)
        {
            if (FD_ISSET(bot->sock, &fd) <= 0)
            {

                errorlog( "FATAL: Major system error.\n");
                errorlog( "Select() was valid but our socket wasn't ready.\n");
                errorlog( "This should never happen.\n");
                return FAILURE;
            }

            /* Read in one byte at a time until you find the newline.
            ** If there is no newline then we got sent an invalid data
            ** chunk.  Chop off the end and make it fit.
			** I should probably make this better.
			*/
            do
            {
                if (recv(bot->sock, &buff[i++], 1, 0) <= 0)
                {
                    return FAILURE;
                }
            }
            while ((buff[i - 1] != '\n') && (i <= BUFFSIZE));
            if (i > BUFFSIZE)
            {
                buff[i - 1] = '\n';
            }
            buff[i] = '\0';
            readingdata = 0;
        }
    }
    while ((status) && (readingdata > 0));

    if (status)
    {
        status = data_parse(buff);
    }

    return status;
}

/*******************************************************************************
*---|   Function:      DATA_PARSE
*---|   Created:       Mon Dec 16 00:39:52 CST 2002
*---|   Last Edited:   Wed Nov 26 12:12:21 CST 2003
*---|   Description:   Split the data into argc/argv
*******************************************************************************/

int data_parse(char *buff)
{
    int i = 0, argc = 0;
    char *argv[BUFFSIZE] = {'\0'};

#ifdef debug

    fprintf(stdout, "*******************************************************************************\n");
    logprintf( "Received: [%d] %s", strlen(buff), buff);
#endif

    while (buff[i] == ' ' || buff[i] == ':')
    {
        i++;
    }
    while (buff[i] != '\0')
    {
        argv[argc++] = &buff[i];
        while (buff[i] != '\0' && buff[i] != ' ' && buff[i] != '\r' && buff[i] != '\n')
        {
            i++;
        }
        if (buff[i] == '\0')
        {
            break;
        }
        buff[i++] = '\0';
        while (buff[i] == ' ' || buff[i] == '\r' || buff[i] == '\n')
        {
            i++;
        }
    }

#ifdef debug
    for (i = 0; i < argc; i++)
    {
        printf("%d:[%s] ", i, argv[i]);
    }
    printf("\n");
#endif

    if (lowercasecmp("PING", argv[0]) == 0)
    {
        return quote(bot->sock, "PONG %s\r\n", argv[1]);
    }
    else if (lowercasecmp("ERROR", argv[0]) == 0)
    {
        return FAILURE;
    }

    return process_data(argc, argv);
}

