/*******************************************************************************
*---|
*---|   Garnaxbot by Freakster http://garnax.mircx.com/
*---|
*---|   Garnaxbot is distributed in the hope that it will be useful,
*---|   but WITHOUT ANY WARRANTY; without even the implied warranty of
*---|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*---|
********************************************************************************
*---|
*---|   File:          STRUCTS.C
*---|   Created:       Fri Dec 13 22:14:39 CST 2002
*---|   Last Edited:   Wed Nov 26 02:35:27 CST 2003
*---|   Description:   This file contains the functions to create,
*---|                  modify, parse, and destroy the basic
*---|                  structures of the program.
*---|
*******************************************************************************/

#include "structs.h"

/*******************************************************************************
*---|   Function:      CREATE_BOT
*---|   Created:       Sat Dec 14 00:28:36 CST 2002
*---|   Last Edited:   Thu Dec 11 01:11:30 CST 2003
*---|   Description:   Creates and initializes a network struct.
*******************************************************************************/

int create_bot(char *filename)
{
    bot = (networkstruct *)malloc(sizeof(networkstruct));

	if (bot == NULL) {
		errorlog("Malloc failed in create_bot\n");
		return FAILURE;
	}

	bot->filename = mymalloc(filename);
	if (bot->filename == NULL) {
		free(bot);
		return FAILURE;
	}

	if (create_dccptrs() == FAILURE)
	{
		free(bot);
		free(bot->filename);
		return FAILURE;
	}

	bot->pidfile = NULL;
	bot->statfile = NULL;
	bot->transferlog = NULL;

	bot->nickkeep = NULL;
	bot->nickpass = NULL;
	bot->nick = NULL;

	bot->ob = NULL;
	bot->cb = NULL;

	bot->color1[0] = '0';
	bot->color1[1] = '0';
	bot->color1[2] = '\0';

	bot->color2[0] = '0';
	bot->color2[1] = '0';
	bot->color2[2] = '\0';

	bot->servername = NULL;
	bot->vhost = NULL;
	bot->settings = FSERVEON;

	bot->sock = 0;
	bot->state = READCFG;
	bot->restart = 0;

	bot->contime = 0;
	bot->chktime = 0;
	bot->connectdelay = 60;

	bot->server = NULL;
	bot->servers = NULL;
	bot->channels = NULL;

	bot->minport = 1024;
	bot->maxport = 65535;

	bot->totalcap = 0;
	bot->sendcap = 0;
	bot->tmpcap = 0;
	bot->minspeed = 0;

	bot->forcesize = 0;

	bot->failcount = 0;
	bot->filecount = 0;
	bot->bytecount = 0;
	bot->overflowcount = 0;
	bot->accesscount = 0;
	bot->maxspeed = 0;
	bot->avespeed = 0;

	bot->ip = 0;

	bot->fserves->count = 0;
	bot->fserves->total = 0;
	bot->fserves->each = 0;
	bot->fserves->start = NULL;
	bot->fserves->end = NULL;

	bot->sends->count = 0;
	bot->sends->total = 0;
	bot->sends->each = 0;
	bot->sends->start = NULL;
	bot->sends->end = NULL;

	bot->queues->count = 0;
	bot->queues->total = 0;
	bot->queues->each = 0;
	bot->queues->start = NULL;
	bot->queues->end = NULL;

    return SUCCESS;
}

/*******************************************************************************
*---|   Function:      REHASH_BOT
*---|   Created:       Sun Dec 15 22:47:27 CST 2002
*---|   Last Edited:   Sat Jul 19 20:04:03 CST 2003
*---|   Description:   Prepares the bot and channels for
*---|                  a rehash of the config file
*******************************************************************************/

void rehash_bot()
{

    if (bot)
    {

        if (bot->nickkeep)
            free(bot->nickkeep);
        bot->nickkeep = NULL;

        if (bot->nickpass)
            free(bot->nickpass);
        bot->nickpass = NULL;

        remove_pid();
        if (bot->pidfile)
            free(bot->pidfile);
        bot->pidfile = NULL;

        save_stats();
        if (bot->statfile)
            free(bot->statfile);
        bot->statfile = NULL;

        if (bot->transferlog)
            free(bot->transferlog);
        bot->transferlog = NULL;

        if (bot->ob)
            free(bot->ob);
        bot->ob = NULL;

        if (bot->cb)
            free(bot->cb);
        bot->cb = NULL;

        bot->color1[0] = '0';
        bot->color1[1] = '0';
        bot->color1[2] = '\0';

        bot->color2[0] = '0';
        bot->color2[1] = '0';
        bot->color2[2] = '\0';

        bot->server = NULL;
        destroy_servers();

        bot->connectdelay = 60;

        bot->settings = FSERVEON;

        if (bot->vhost)
            free(bot->vhost);
        bot->vhost = NULL;

        bot->minport = 1024;
        bot->maxport = 65535;

        bot->totalcap = 0;
        bot->tmpcap = 0;
        bot->sendcap = 0;
        bot->minspeed = 0;
        bot->forcesize = 0;

        bot->failcount = 0;
        bot->filecount = 0;
        bot->bytecount = 0;
        bot->overflowcount = 0;
        bot->accesscount = 0;
        bot->maxspeed = 0;
        bot->avespeed = 0;

        bot->fserves->total = 0;
        bot->fserves->each = 0;
        bot->sends->total = 0;
        bot->sends->each = 0;
        bot->queues->total = 0;
        bot->queues->each = 0;
    }
    return ;
}

/*******************************************************************************
*---|     Function:        DESTROY_BOT
*---|     Created:         Sat Dec 14 00:28:36 CST 2002
*---|     Last Edited:     Fri Aug  1 16:00:23 CDT 2003
*---|     Description:     Cleans up and reclaims a network struct.
*******************************************************************************/

void destroy_bot()
{

    if (bot)
    {

#ifdef debug
        fprintf(stdout, "*******************************************************************************\n");
        logprintf("Cleaning up memory structures.\n");
#endif

        rehash_bot();

        if (bot->filename)
		free(bot->filename);
        bot->filename = NULL;

        if (bot->nick)
		free(bot->nick);
        bot->nick = NULL;

        if (bot->servername)
		free(bot->servername);
        bot->servername = NULL;

        bot->settings = 0;

        if (bot->sock)
		close(bot->sock);
        bot->sock = 0;
        bot->state = 0;
        bot->restart = 0;

        bot->chktime = 0;
        bot->contime = 0;
        bot->connectdelay = 0;

        destroy_channels();

        bot->minport = 0;
        bot->maxport = 0;

        killall_dccs(bot->fserves, 1);
        killall_dccs(bot->queues, 0);
        killall_dccs(bot->sends, 2);

        destroy_dccptrs();

        free(bot);
        bot = NULL;

    }
    return;
}

/*******************************************************************************
*---|   Function:      CREATE_DCCPTRS
*---|   Created:       Tue Aug  5 16:48:50 CDT 2003
*---|   Last Edited:   Thu Dec 11 00:50:40 CST 2003
*---|   Description:   Creates and initializes dccptrs.
*******************************************************************************/

int create_dccptrs()
{
	int status = SUCCESS;

    bot->fserves = (dccptr*)malloc(sizeof(dccptr));
    bot->sends = (dccptr*)malloc(sizeof(dccptr));
    bot->queues = (dccptr*)malloc(sizeof(dccptr));

    if ((!bot->fserves) || (!bot->sends) || (!bot->queues))
    {
		errorlog("Could not malloc dcc ptrs\n");
		destroy_dccptrs();
        status = FAILURE;
    }

    return status;
}

/*******************************************************************************
*---|     Function:        DESTROY_DCCPTRS
*---|     Created:         Tue Aug  5 16:56:07 CDT 2003
*---|     Last Edited:     
*---|     Description:     Reclaims the dcc ptrs.
*******************************************************************************/
void destroy_dccptrs()
{

	if (bot->queues)
	free(bot->queues);
	bot->queues = NULL;

	if (bot->sends)
	free(bot->sends);
	bot->sends = NULL;

	if (bot->fserves)
	free(bot->fserves);
	bot->fserves = NULL;

    return ;
}
/*******************************************************************************
*---|     Function:        DESTROY_SERVERS
*---|     Created:         Sat Dec 14 00:19:08 CST 2002
*---|     Last Edited:     
*---|     Description:     Wipes out the server list.
*******************************************************************************/

void destroy_servers()
{
    serverstruct *ptr = bot->servers;

    while (ptr)
    {
        bot->servers = bot->servers->next;
        delete_server(ptr);
        ptr = bot->servers;
    }

    return ;
}

/*******************************************************************************
*---|     Function:        DELETE_SERVER
*---|     Created:         Sat Dec 14 00:19:25 CST 2002
*---|     Last Edited:     Fri Jul 25 14:46:39 CDT 2003
*---|     Description:     Deletes a server struct and reclaims the memory.
*******************************************************************************/

void delete_server(serverstruct *ptr)
{

    if (ptr)
    {
        if (ptr->name)
            free(ptr->name);
        ptr->name = NULL;
        ptr->next = NULL;
        ptr->port = 0;
        free(ptr);
        ptr = NULL;
    }

    return ;
}

/*******************************************************************************
*---|   Function:      CREATE_SERVER
*---|   Created:       Fri Dec 13 22:14:39 CST 2002
*---|   Last Edited:   Fri Jul 25 14:52:39 CDT 2003
*---|   Description:   Takes in a servername:port text string and
*---|                  creates a server struct.
*******************************************************************************/

serverstruct *create_server(char *text)
{

    serverstruct *ptr = (serverstruct *)malloc(sizeof(serverstruct));

    if (ptr)
    {
        text = strtok(text, ":");

        if ((ptr->name = mymalloc(text)) == NULL)
        {
            return NULL;
        }

        text = strtok(NULL, ":");
        if (text == NULL)
        {
            ptr->port = 6667;
        }
        else
        {
            ptr->port = atoi(text);
        }
        ptr->next = NULL;
    }
    else
    {
        errorlog( "ERROR: Could not malloc memory for server: [%s]\n", text);
    }

    return ptr;

}

/*******************************************************************************
*---|   Function:      ADD_SERVER
*---|   Created:       Fri Dec 13 23:22:07 CST 2002
*---|   Last Edited:   Fri Jul 25 14:53:40 CDT 2003
*---|   Description:   Adds a server struct to the master list.
*---|                  If it doesn't exist already.
*******************************************************************************/

int add_server(char *text)
{
    serverstruct *temp = create_server(text), *ptr = bot->servers;

    if (!temp)
        return FAILURE;

    /* Try to find a match for the name AND the port, if the
    ** port doesn't match, then keep looking */
    while ((ptr) && ((lowercasecmp(ptr->name, temp->name) != 0) || (ptr->port != temp->port)))
    {
        ptr = ptr->next;
    }

    if (ptr)
    {
#ifdef debug
        logprintf("Server Already Exists: [%s:%d]\n", temp->name, temp->port);
#endif

        delete_server(temp);
    }
    else
    {
#ifdef debug
        logprintf("Added Server: [%s:%d]\n", temp->name, temp->port);
#endif

        temp->next = bot->servers;
        bot->servers = temp;
    }

    temp = NULL;
    ptr = NULL;

    return SUCCESS;

}

/*******************************************************************************
*---|     Function:        CREATE_CHANNEL
*---|     Created:         Sat Dec 14 00:28:36 CST 2002
*---|     Last Edited:     
*---|     Description:     Creates and initializes a channel struct.
*******************************************************************************/

channelstruct *create_channel()
{
    channelstruct *ptr = (channelstruct *)malloc(sizeof(channelstruct));
    if (ptr)
    {
        ptr->next = NULL;
        ptr->settings = 0;

        ptr->name = NULL;
        ptr->key = NULL;

        ptr->desc = NULL;
        ptr->welcomemsg = NULL;
        ptr->trigger = NULL;

        ptr->delay = 0;
        ptr->delaytime = time(NULL);

        ptr->directories = NULL;

        ptr->masters = NULL;
        ptr->banned = NULL;
        ptr->users = NULL;
    }
    else
    {
        errorlog("ERROR: Not enough memory to create another channel.\n");
    }
    return ptr;

}

/*******************************************************************************
*---|     Function:        ADD_CHANNEL
*---|     Created:         Sat Dec 14 00:28:36 CST 2002
*---|     Last Edited:     Fri Jul 25 15:57:03 CDT 2003
*---|     Description:     Adds a channel to the channel list
*---|                      as long as it doesn't already exist.
*******************************************************************************/

void add_channel(channelstruct *addchan)
{
    channelstruct *tmp = bot->channels;

    /* If the list is empty or it is not in the list tmp will end up null */
    while ((tmp) && (lowercasecmp(tmp->name, addchan->name) != 0))
    {
        tmp = tmp->next;
    }

    if (tmp == NULL)
    {
        addchan->next = bot->channels;
        bot->channels = addchan;
#ifndef debug

    }
#else
        logprintf("Added Channel: [%s]\n", addchan->name);
    }
    else
    {
        logprintf("Channel [%s] already exists.\n", addchan->name);
    }
#endif

    tmp = NULL;
    addchan = NULL;

    return ;

}



/*******************************************************************************
*---|     Function:        REMOVE_CHANNEL
*---|     Created:         Sat Dec 14 00:28:36 CST 2002
*---|     Last Edited:     Fri Jul 25 15:18:22 CDT 2003
*---|     Description:     Removes a channel from the channel list.
*---|     
*---|     There are 3 possible options for this function:
*---|       1. The list is empty.
*---|       2. The channel isn't in the list.
*---|       3. The channel is in the list and will be removed.
*---|     
*---|      No matter the option, the channel will not be in the
*---|      channel list and can be used for whatever it was removed
*---|      for (Probably deletion)
*---|      
*******************************************************************************/

channelstruct *remove_channel(channelstruct *chan)
{

    channelstruct *ptr = bot->channels;

    if (ptr && chan)
    {
        /* First channel is the one that needs removed */
        if (ptr == chan)
        {
            bot->channels = ptr->next;
            chan->next = NULL;
        }
        else
        {
            /* Traverse the list looking for a matching channel */
            while ((ptr) && (ptr->next != chan))
            {
                ptr = ptr->next;
            }
            if (ptr)
            {
                ptr->next = chan->next;
                chan->next = NULL;
            }
            else
            {
                errorlog( "ERROR: Trying to remove a channel that's not in the list.\n");
            }
        }
    }
    else
    {
        errorlog( "ERROR: Trying to remove a channel from an empty channel list.\n");
    }


    return chan;
}



/*******************************************************************************
*---|     Function:        DELETE_CHANNEL
*---|     Created:         Sat Dec 14 00:28:36 CST 2002
*---|     Last Edited:     
*---|     Description:     Deletes a channel and reclaims its memory.
*******************************************************************************/

channelstruct *delete_channel(channelstruct *chan)
{
    if (chan)
    {
#ifdef debug
        logprintf( "Deleting channel [%s]\n", chan->name);
#endif

        destroy_users(chan);
        rehash_channel(chan);

        chan->next = NULL;
        chan->settings = 0;

        if (chan->name)
            free(chan->name);
        chan->name = NULL;

        free(chan);
        chan = NULL;
    }

    return chan;
}


/*******************************************************************************
*---|     Function:        DESTROY_CHANNELS
*---|     Created:         Sat Dec 14 00:28:36 CST 2002
*---|     Last Edited:     
*---|     Description:     Wipes out the channel list.
*******************************************************************************/

void destroy_channels()
{
    channelstruct *ptr = bot->channels;

    while (ptr)
    {
        ptr = remove_channel(ptr);
        ptr = delete_channel(ptr);
        ptr = bot->channels;
    }

    return ;
}

/*******************************************************************************
*---|     Function:        IS_CHANNEL
*---|     Created:         Sat Dec 14 00:28:36 CST 2002
*---|     Last Edited:
*---|     Description:     Searches a networks channel listing
*---|                      for a specific channel.
*******************************************************************************/

channelstruct *is_channel(char *text)
{
    channelstruct *ptr = NULL;
    for (ptr = bot->channels; ((ptr) && (lowercasecmp(ptr->name, text) != 0)); ptr = ptr->next)
        ;
    return ptr;
}


/*******************************************************************************
*---|     Function:        ADD_MEMBER
*---|     Created:         Sat Dec 14 00:28:36 CST 2002
*---|     Last Edited:     
*---|     Description:     Add a member if it doesn't already exist to a list.
*******************************************************************************/

userstruct *add_member(userstruct *head, char *host)
{

    userstruct *ptr = head;
    while ((ptr) && (lowercasecmp(ptr->host, host) != 0))
    {
        ptr = ptr->next;
    }

    if (ptr == NULL)
    {
        lowercase(host);
        if ((ptr = create_user("x", host, 0)))
        {
            ptr->next = head;
            head = ptr;
        }
    }

    ptr = NULL;

    return head;
}

/*******************************************************************************
*---|     Function:        REMOVE_MEMBER
*---|     Created:         Sat Dec 14 00:28:36 CST 2002
*---|     Last Edited:     
*---|     Description:     Remove and delete a member from a list.
*******************************************************************************/

userstruct *remove_member(userstruct *head, char *host)
{

    userstruct *ptr = head, *tmp = NULL;

    if (ptr)
    {
        if (lowercasecmp(ptr->host, host) == 0)
        {
            head = head->next;
            tmp = ptr;
        }
        else
        {
            while ((ptr->next) && (lowercasecmp(ptr->next->host, host) == 0))
            {
                ptr = ptr->next;
            }
            if (ptr->next)
            {
                tmp = ptr->next;
                ptr->next = tmp->next;
            }
        }

        if (tmp)
            delete_user(tmp);
        tmp = NULL;

    }

    return head;
}

/*******************************************************************************
*---|     Function:        DESTROY_MEMBERS
*---|     Created:         Sat Dec 14 00:28:36 CST 2002
*---|     Last Edited:     Fri Jul 25 16:11:26 CDT 2003
*---|     Description:     Wipes out the member list.
*******************************************************************************/

void destroy_members(channelstruct *chan)
{
    userstruct *uptr = chan->masters;

#ifdef debug

    logprintf( "Deleting Masters Masks.\n");
#endif

    while (uptr)
    {
        chan->masters = uptr->next;
        delete_user(uptr);
        uptr = chan->masters;
    }

#ifdef debug
    logprintf( "Deleting Banned Masks.\n");
#endif

    uptr = chan->banned;
    while (uptr)
    {
        chan->banned = uptr->next;
        delete_user(uptr);
        uptr = chan->banned;
    }

    uptr = NULL;

    return ;

}

/*******************************************************************************
*---|     Function:        CREATE_USER
*---|     Created:         Sat Dec 14 00:28:36 CST 2002
*---|     Last Edited:     
*---|     Description:     Creates a user element.
*******************************************************************************/

userstruct *create_user(char *nick, char *host, unsigned char modes)
{
    userstruct *cuser = (userstruct*)malloc(sizeof(userstruct));

    if (cuser)
    {
        if (host)
        {
            cuser->host = mymalloc(host);
            if (cuser->host == NULL)
            {
                errorlog( "ERROR: Could not malloc host [%s] for user struct.\n", host);
                cuser = delete_user(cuser);
                return NULL;
            }
        }
        else
        {
            cuser->host = NULL;
        }
        cuser->nick = mymalloc(nick);
        if (cuser->nick == NULL)
        {
            errorlog( "ERROR: Could not malloc nick [%s] for user struct.\n", nick);
            cuser = delete_user(cuser);
            return NULL;
        }
        cuser->modes = modes;
        cuser->next = NULL;

#ifdef debug

        if (host)
            logprintf( "Created User Struct: Nick[%s] Host[%s] Modes[%x]\n", nick, host, modes);
        else
            logprintf( "Created User Struct: Nick[%s] Modes[%x]\n", nick, modes);
#endif

    }
    else
    {
        if (host)
            errorlog( "Error Creating User Struct: Nick[%s] Host[%s] Modes[%x]\n", nick, host, modes);
        else
            errorlog( "Error Creating User Struct: Nick[%s] Modes[%x]\n", nick, modes);
    }

    return cuser;
}


/*******************************************************************************
*---|     Function:        IS_USER
*---|     Created:         Sat Dec 14 00:28:36 CST 2002
*---|     Last Edited:     
*---|     Description:     Detect if a user is a part of a channel.
*******************************************************************************/
userstruct *is_user(channelstruct *chan, char *nick)
{
    userstruct *ptr = chan->users;
    if (nick == NULL)
    {
        return NULL;
    }
    while ((ptr) && (lowercasecmp(ptr->nick, nick) != 0))
    {
        ptr = ptr->next;
    }
#ifdef debug
    if (ptr)
        logprintf("User [%s] is a user of [%s].\n", nick, chan->name);
    else
        logprintf("User [%s] is NOT a user of [%s].\n", nick, chan->name);
#endif

    return ptr;
}

/*******************************************************************************
*---|   Function:      ADD_USER
*---|   Created:       Sat Dec 14 00:28:36 CST 2002
*---|   Last Edited:   Wed Nov 26 15:52:03 CST 2003
*---|   Description:   Adds a user to a channel.
*******************************************************************************/

int add_user(channelstruct *chan, char *nick, char *host, unsigned char modes)
{

    userstruct *auser = is_user(chan, nick);

    if (auser)
    {
        if ((host) && (!auser->host))
        {
#ifdef debug
            logprintf( "Updating host [%s] of User [%s] on Channel [%s]\n", host, nick, chan->name);
#endif

            auser->host = mymalloc(host);
            if (!auser->host)
                return FAILURE;
        }
    }
    else
    {
        auser = create_user(nick, host, modes);
        if (auser)
        {
            auser->next = chan->users;
            chan->users = auser;
        }
        else
        {
            return FAILURE;
        }
    }
    auser = NULL;
    return SUCCESS;
}


/*******************************************************************************
*---|     Function:        REMOVE_USER
*---|     Created:         Sat Dec 14 00:28:36 CST 2002
*---|     Last Edited:     Fri Jul 25 15:41:59 CDT 2003
*---|     Description:     Removes a user from a channel.
*******************************************************************************/
userstruct *remove_user(channelstruct *chan, char *nick)
{

    userstruct *remuser = chan->users, *temp = NULL;

    if (remuser)
    {
        /* User is the first user in the list */
        if (lowercasecmp(remuser->nick, nick) == 0)
        {
            chan->users = chan->users->next;
            temp = remuser;
            temp->next = NULL;
        }
        else
        {
            /* Traverse the list until you find a matching nickname */
            while ((remuser->next) && (lowercasecmp(remuser->next->nick, nick) != 0))
            {
                remuser = remuser->next;
            }
            if (remuser->next)
            {
                temp = remuser->next;
                remuser->next = temp->next;
                temp->next = NULL;
            }
        }
    }
    else
    {
        errorlog( "ERROR: Trying to remove a user from an empty user list.\n");
    }
    if (!temp)
    {
        errorlog("User [%s] does not exist in [%s].\n", nick, chan->name);
    }
#ifdef debug
    else
    {
        logprintf( "Removed user Nick:[%s] ", temp->nick);
        if (temp->host)
            fprintf(stdout, "Host:[%s] ", temp->host);
        fprintf(stdout, "Modes:[%d] from Channel:[%s] \n", temp->modes, chan->name);
    }
#endif
    return temp;

}

/*******************************************************************************
*---|     Function:        DELETE_USER
*---|     Created:         Sat Dec 14 00:28:36 CST 2002
*---|     Last Edited:     Fri Jul 25 15:39:53 CDT 2003
*---|     Description:     Deletes a user structures.
*******************************************************************************/

userstruct *delete_user(userstruct *deluser)
{
    if (deluser)
    {
#ifdef debug
        if (deluser->host)
            logprintf( "Deleting user Nick:[%s] Host:[%s] Modes:[%d]\n", deluser->nick, deluser->host, deluser->modes);
        else
            logprintf( "Deleting user Nick:[%s] Modes:[%d]\n", deluser->nick, deluser->modes);
#endif

        if (deluser->nick)
            free(deluser->nick);
        deluser->nick = NULL;

        if (deluser->host)
            free(deluser->host);
        deluser->host = NULL;

        deluser->modes = 0;
        deluser->next = NULL;

        free(deluser);
        deluser = NULL;
    }
    else
    {
        errorlog( "ERROR: Trying to delete a NULL user.\n");
    }

    return deluser;
}


/*******************************************************************************
*---|     Function:        DESTROY_USERS
*---|     Created:         Sat Dec 14 00:28:36 CST 2002
*---|     Last Edited:     
*---|     Description:     Wipes out the user list from a channel.
*******************************************************************************/

void destroy_users(channelstruct *chan)
{
    userstruct *ptr = NULL;

    while (chan->users)
    {
        ptr = delete_user(remove_user(chan, chan->users->nick));
    }

    return ;
}

/*******************************************************************************
*---|   Function:      KILLALL_DCCS
*---|   Created:       Thu Aug  7 02:07:18 CDT 2003
*---|   Last Edited:   Sun Dec  7 11:32:48 CST 2003
*---|   Description:   Makes all dccs stop.
*******************************************************************************/

void killall_dccs(dccptr *list, int type)
{

    dccstruct *rem = NULL, *tmp = NULL;

#ifdef debug

    logprintf( "Cleaning up ");

    switch (type)
    {
    case 1:
        fprintf(stdout, "Fserves.\n");
        break;
    case 2:
        fprintf(stdout, "Sends.\n");
        break;
    default:
        fprintf(stdout, "Queues.\n");
        break;
    }
#endif

    pthread_mutex_lock(&mutex);
#ifdef debug

    logprintf( "MUTEX LOCK: KILLALL DCCS\n");
#endif

    rem = list->start;

    if (rem)
    {
        do
        {
            if (type)
            {
                rem->die = 1;
            }
            tmp = rem->next;
            remove_dcc(rem, list);
            destroy_dcc(rem);

            rem = tmp;

        }
        while (rem != NULL);
    }

    pthread_mutex_unlock(&mutex);
#ifdef debug

    logprintf( "MUTEX RELEASE: KILLALL DCCS\n");
#endif

    return ;
}

/*******************************************************************************
*---|   Function:      ADD_DCC
*---|   Created:       Thu Aug  7 00:51:13 CDT 2003
*---|   Last Edited:
*---|   Description:   Adds a dcc to a dcc list.
*******************************************************************************/

void add_dcc(dccstruct *adddcc, dccptr *list)
{

    if (list->end == NULL)
    {
        list->start = adddcc;
        list->end = adddcc;
    }
    else
    {
        adddcc->prev = list->end;
        list->end->next = adddcc;
        list->end = adddcc;
    }

    list->count++;

    return ;
}


/*******************************************************************************
*---|     Function:        DESTROY_DCC
*---|     Created:         
*---|     Last Edited:     
*---|     Description:     
*******************************************************************************/
dccstruct *destroy_dcc(dccstruct *deldcc)
{

	#ifdef debug
	logprintf( "Deleting DCC User:[%s] Host:[%s]\n", deldcc->nick, deldcc->host);
	#endif

	if (deldcc->chan)
	free(deldcc->chan);
	deldcc->chan = NULL;

	if (deldcc->nick)
	free(deldcc->nick);
	deldcc->nick = NULL;

	if (deldcc->host)
	free(deldcc->host);
	deldcc->host = NULL;

	deldcc->next = NULL;
	deldcc->prev = NULL;

	if (deldcc->path)
	free(deldcc->path);
	deldcc->path = NULL;

	deldcc->filename = NULL;

	while (deldcc->head)
	{
		removepath(deldcc);
	}

	if (deldcc->sock)
	close(deldcc->sock);
	deldcc->sock = 0;

	deldcc->port = 0;
	deldcc->starttime = 0;
	deldcc->lastdata = 0;
	deldcc->warned = 0;
	deldcc->die = 0;

	deldcc->size = 0;
	deldcc->state = 0;
	deldcc->type = 0;
	deldcc->offset = 0;
	deldcc->ctr = 0;
	deldcc->ackval = 0;
	deldcc->file = NULL;

	if (deldcc->dir_name)
	free(deldcc->dir_name);
	deldcc->dir_name = NULL;

	if (deldcc->dir_path)
	free(deldcc->dir_path);
	deldcc->dir_path = NULL;

	free(deldcc);
	deldcc = NULL;

	return deldcc;
}

/*******************************************************************************
*---|     Function:        rem_elements
*---|     Created:         Thu Aug  7 00:42:21 CDT 2003
*---|     Last Edited:     
*---|     Description:     
*******************************************************************************/
void remove_dcc(dccstruct *remdcc, dccptr *list)
{

    dccstruct *elemptr = NULL;

    for (elemptr = list->start; elemptr != NULL; elemptr = elemptr->next)
    {
        if (elemptr == remdcc)
        {
            if (elemptr->prev)
            {
                elemptr->prev->next = elemptr->next;
            }
            else
            {
                list->start = elemptr->next;
            }
            if (elemptr->next)
            {
                elemptr->next->prev = elemptr->prev;
            }
            else
            {
                list->end = elemptr->prev;
            }
            elemptr->prev = NULL;
            elemptr->next = NULL;

            list->count--;

            return ;
        }
    }

    return ;
}
/*******************************************************************************
*----|     Function:        REMOVEPATH
*----|     Created:         Thu Aug  7 10:55:16 CDT 2003
*----|     Last Edited:     
*----|     Description:     Removes the last element off of the path list.
*******************************************************************************/
void removepath(dccstruct *fserve)
{
    pathstruct *tmp = NULL;

    if (fserve->tail)
    {
        tmp = fserve->tail;

        fserve->tail = fserve->tail->prev;

        if (fserve->tail == NULL)
        {
            fserve->head = NULL;

            if (fserve->dir_name)
                free(fserve->dir_name);
            fserve->dir_name = NULL;

            if (fserve->dir_path)
                free(fserve->dir_path);
            fserve->dir_path = NULL;
        }
        else
        {
            fserve->tail->next = NULL;
        }

        if (tmp->name)
            free(tmp->name);
        tmp->name = NULL;

        tmp->prev = NULL;
        tmp->next = NULL;

        free(tmp);
        tmp = NULL;
    }

    return ;
}

/*******************************************************************************
*----|     Function:        free_ptrs
*----|     Last Edited:
*----|     Created:         
*----|     Description:    
*******************************************************************************/ 
/*
ptrstruct *free_ptrs(ptrstruct *ptr)
{
 
    ptr->start = NULL;
    ptr->end = NULL;
    ptr->total = 0;
    ptr->each = 0;
    ptr->count = 0;
    free(ptr);
    ptr = NULL;
 
    return ptr;
 
}
*/

/*******************************************************************************
*---|     Function:        REHASH_CHANNEL
*---|     Created:         Sat Jul 19 21:10:03 CDT 2003
*---|     Last Edited:     Sat Jul 26 13:30:04 CDT 2003
*---|     Description:     
*******************************************************************************/

void rehash_channel(channelstruct *chan)
{
    if (chan)
    {
#ifdef debug
        logprintf( "Resetting channel [%s]\n", chan->name);
#endif

        if (chan->key)
            free(chan->key);
        chan->key = NULL;

        if (chan->desc)
            free(chan->desc);
        chan->desc = NULL;

        if (chan->welcomemsg)
            free(chan->welcomemsg);
        chan->welcomemsg = NULL;

        if (chan->trigger)
            free(chan->trigger);
        chan->trigger = NULL;

        chan->delay = 0;
        chan->delaytime = 0;

        destroy_dirs(chan);
        destroy_members(chan);

        /* I'm not cleaning up users because a rehash might be done while in a channel */

        reset_channel(chan);
    }

    return ;
}

/*******************************************************************************
*---|     Function:        RESET_CHANNEL
*---|     Created:         Sat Dec 14 00:28:36 CST 2002
*---|     Last Edited:     Sat Jul 26 13:30:04 CDT 2003
*---|     Description:     Resets the bot after a disconnection.
*---|     
*---|     This function does not clear out the users as it is used in a rehash.
*---|     On disconnection the users must be deleted sperately.
*---|     
*---|     On disconnection, the chan settings need cleared, on
*---|     rehash, the settings are saved before calling this and then restored
*---|     afterwards so it won't wipe out the right settings.
*******************************************************************************/
void reset_channel(channelstruct *chan)
{

    /* There isn't much to do here compared to the protection bot. */

    BITCLR(chan->settings, BOTONCHAN);
    BITCLR(chan->settings, BOTISOP);

    return ;
}

/*******************************************************************************
*---|     Function:        RESET_BOT
*---|     Created:         Sat Dec 14 00:28:36 CST 2002
*---|     Last Edited:     
*---|     Description:     Resets the bot after a disconnection.
*******************************************************************************/

void reset_bot()
{
    channelstruct *chan = NULL;

    if (bot)
    {
#ifdef debug
        logprintf( "Resetting the bot for reconnection.\n");
#endif

        if (bot->nick)
            free(bot->nick);
        bot->nick = NULL;

        if (bot->servername)
            free(bot->servername);
        bot->servername = NULL;

        for (chan = bot->channels; chan != NULL; chan = chan->next)
        {
            destroy_users(chan);
            reset_channel(chan);
        }

    }

    return ;
}

/*******************************************************************************
*---|     Function:        SAVE_STATS
*---|     Created:         Tue Aug  5 18:25:55 CDT 2003
*---|     Last Edited:     
*---|     Description:     
*******************************************************************************/

void save_stats()
{
    FILE *fptr;

    if (bot->statfile == NULL)
    {
        return ;
    }

    fptr = fopen(bot->statfile, "wb");

    if (fptr == NULL)
    {
        return ;
    }

#ifdef debug
    logprintf( "Writing stats to [%s]\n", bot->statfile);
#endif

    fprintf(fptr, "%u %u %u %u %u %u %u", bot->accesscount,
            bot->filecount, bot->bytecount, bot->overflowcount,
            bot->failcount, bot->maxspeed, bot->avespeed);

    fclose(fptr);

    return ;
}


/*******************************************************************************
*---|     Function:        LOAD_STATS
*---|     Created:         Tue Aug  5 18:25:55 CDT 2003
*---|     Last Edited:     
*---|     Description:     
*******************************************************************************/

void load_stats()
{
    FILE *fptr;

    if (bot->statfile == NULL)
    {
        return ;
    }

    fptr = fopen(bot->statfile, "rb");

    if (fptr == NULL)
    {
        return ;
    }

    fscanf(fptr, "%u %u %u %u %u %u %u", &bot->accesscount, &bot->filecount, &bot->bytecount, &bot->overflowcount, &bot->failcount, &bot->maxspeed, &bot->avespeed);


    fclose(fptr);

#ifdef debug

    logprintf( "Reading stats from: [%s]\n", bot->statfile);
    logprintf( "Accessed:[%u]\n", bot->accesscount);
    logprintf( "Files Sent:[%u]\n", bot->filecount);
    logprintf( "Bytes Sent:[%u]\n", bot->bytecount);
    logprintf( "Bytes Sent:[%u]\n", bot->bytecount);
#endif

    return ;
}

/*******************************************************************************
*---|     Function:        CREATE_DIR
*---|     Created:         Wed Aug  6 20:17:30 CDT 2003
*---|     Last Edited:     
*---|     Description:     Creates a dir struct for use with 
*---|                      multiple paths per trigger.
*******************************************************************************/

dirstruct *create_dir(char *dname, char *ddir)
{

    dirstruct *cdir = (dirstruct*)malloc(sizeof(dirstruct));
    char *tmp = (char*)malloc(sizeof(char) * strlen(ddir) + 2);

    if ((cdir == NULL) || (tmp == NULL))
    {
        if (cdir)
		free(cdir);
        if (tmp)
		free(tmp);
        return NULL;
    }

    cdir->name = mymalloc(dname);
    if (cdir->name == NULL)
    {
		free(tmp);
        free(cdir);
        return NULL;
    }

#ifdef WIN32
    if (ddir[strlen(ddir) - 1] != '\\')
    {
        strcpy(tmp, ddir);
        strcat(tmp, "\\");
        cdir->dir = mymalloc(tmp);
    }
#else
    if (ddir[strlen(ddir) - 1] != '/')
    {
        strcpy(tmp, ddir);
        strcat(tmp, "/");
        cdir->dir = mymalloc(tmp);
    }
#endif
    else
    {
        cdir->dir = mymalloc(ddir);
    }
    free(tmp);
    if (cdir->dir == NULL)
    {
        free(cdir->name);
        cdir->name = NULL;
        free(cdir);
        cdir = NULL;
        return NULL;
    }

    cdir->next = NULL;

    return cdir;
}

/*******************************************************************************
*---|     Function:        ADD_DIR
*---|     Created:         Wed Aug  6 20:24:06 CDT 2003
*---|     Last Edited:     
*---|     Description:     Adds a dir struct to the first channel on the list.
*******************************************************************************/

int add_dir(char *text)
{

    dirstruct *newdir = NULL;
    char *tmp = NULL;
    int i = 0;

    if ((text == NULL) || (bot->channels == NULL))
    {
        return FAILURE;
    }

    while ((text[i] != '\0') && (text[i] != ':'))
        i++;

    if (text[i] == '\0')
    {
        return FAILURE;
    }
    text[i++] = '\0';

    if (text[i] == '\0')
    {
        return FAILURE;
    }
    tmp = &text[i];

    newdir = create_dir(text, tmp);

    if (newdir == NULL)
    {
        return FAILURE;
    }

    newdir->next = bot->channels->directories;
    bot->channels->directories = newdir;

#ifdef debug

    logprintf( "Added Dir:[%s][%s] to Channel:[%s]\n", newdir->name, newdir->dir, bot->channels->name);
#endif

    newdir = NULL;

    return SUCCESS;
}

/*******************************************************************************
*---|     Function:        DESTROY_DIRS
*---|     Created:         Wed Aug  6 20:24:06 CDT 2003
*---|     Last Edited:     
*---|     Description:     Reclaims memory from channel dirs.
*******************************************************************************/

void destroy_dirs(channelstruct *chan)
{

    dirstruct *deldir = chan->directories, *tmp = NULL;

    while (deldir)
    {
        tmp = deldir->next;

        if (deldir->name)
            ;
        free(deldir->name);
        deldir->name = NULL;

        if (deldir->dir)
            ;
        free(deldir->dir);
        deldir->dir = NULL;

        deldir->next = NULL;
        free(deldir);
        deldir = tmp;
    }

    chan->directories = NULL;

    return ;
}

/*******************************************************************************
*---|     Function:        PURGE_FSERVES
*---|     Created:         Tue Aug 12 17:46:24 CDT 2003
*---|     Last Edited:    Mon Sep  1 16:18:24 CDT 2003
*---|     Description:     Delete all fserves which match the channel name.
*---|     
*---|     When the bot leaves a channel or is rehashed, it deletes all of the
*---|     structures relative to that channel.  If there are fserves open,
*---|     then dirs correlated to that channel would be NULL.
*******************************************************************************/

void purge_fserves(char *name)
{

    dccstruct *fserve = NULL;

    pthread_mutex_lock(&mutex);
#ifdef debug

    logprintf( "MUTEX LOCK: PURGE_FSERVES\n");
#endif

    for (fserve = bot->fserves->start; fserve; fserve = fserve->next)
    {
        if (lowercasecmp(name, fserve->chan) == 0)
        {
            quote(fserve->sock, "%c%sBot rehashed by owner. Closing all fserves. Please reconnect.\n", 3, bot->color1);
            fserve->die = 1;
        }
    }

    pthread_mutex_unlock(&mutex);
#ifdef debug

    logprintf( "MUTEX RELEASE: PURGE_FSERVES\n");
#endif

    return ;
}

/*******************************************************************************
*---|     Function:        REMOVE_PID
*---|     Created:         Fri Dec 13 22:14:39 CST 2002
*---|     Description:     Removes the pid file when exiting.
*******************************************************************************/

void remove_pid()
{
    char *cmd = NULL;

    if (bot->pidfile)
    {
        cmd = (char*)malloc(sizeof(char) * (strlen(bot->pidfile) + 10));
        if (cmd)
        {
#ifdef WIN32
            sprintf(cmd, "del %s", bot->pidfile);
#else

            sprintf(cmd, "/bin/rm %s", bot->pidfile);
#endif

#ifdef debug

            logprintf( "Removing pid file. [%s]\n", cmd);
#endif

            system(cmd);
            free(cmd);
            cmd = NULL;
        }
    }

    return ;
}

