/* Copyright (C) 2002 CANON INC.

   Ghostscript printer driver
   for Canon CPCA printer devices (LIPS IV)

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any
   later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied
   warranty of MERCHANTABILITY or FITNESS FOR A
   PARTICULAR PURPOSE.  See the GNU General Public
   License for more details.

   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA  02111-1307, USA.

*/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include "cpca_mk.h"

/*$Id: mk_cpca_cmd.c $ */

/***********************************************************
  Define
***********************************************************/
#define null            0x00
#define nil             0x00

#define SWAP_WORD(n)    (((n & 0x00FF) << 8) | \
                        ((n & 0xFF00) >> 8 ))
#define SWAP_DWORD(n)   (((n & 0x000000FF) << 24) | \
                        (( n & 0x0000FF00) <<  8) | \
                        (( n & 0x00FF0000) >>  8) | \
                        (( n & 0xFF000000) >> 24))

/***********************************************************
  CPCA Packet Define
***********************************************************/
#define cpcaPKT_HDR_ID                                      0xCACD
#define cpcaVERSION_V10                                     0x10
#define cpcaPKTFLG_SEND_V10                                 0x00
#define cpcaPKTFLG_SEND_CONT                                0x02
#define cpcaPKTFLG_SEND_END                                 0x00

/***********************************************************
  CPCA-ID Define
***********************************************************/
#define  id_oc_print_job                                    0x0000C900
#define  id_oc_print_document                               0x00009101
#define  id_oc_print_binder                                 0x00002D01

#define  id_LIPS                                            0x27

#define  id_val_operation_job_start                         0x1100
#define  id_val_operation_set_job                           0x1200
#define  id_val_operation_job_end                           0x1300
#define  id_val_operation_binder_start                      0x1400
#define  id_val_operation_set_binder                        0x1500
#define  id_val_operation_binder_end                        0x1600
#define  id_val_operation_document_start                    0x1700
#define  id_val_operation_set_document                      0x1800
#define  id_val_operation_execute_method                    0x1D00
#define  id_val_operation_document_end                      0x1900
#define  id_val_operation_send                              0x1A00
#define  id_val_code_set_Shift_JIS                          0x1100
#define  id_val_resolution_type_dpi                         0x08
#define  id_val_gradation_depth_1                           0x01
#define  id_val_color_mode_color                            0x01
#define  id_val_color_mode_mono                             0x02
#define  id_val_plex_simplex                                0x01
#define  id_val_plex_duplex                                 0x02
#define  id_val_binding_top_edge                            0x03
#define  id_val_binding_left_edge                           0x04
#define  id_val_send_mode_raw                               0x01
#define  id_val_input_tray_named                            0x02
#define  id_val_output_bin_group                            0x11
#define  id_val_output_bin_numbered                         0x12
#define  id_val_output_bin_stack                            0x15
#define  id_val_output_bin_face_up                          0x01
#define  id_val_output_bin_face_down                        0x02
#define  id_val_finishing_type_staple_stitching             0x0C
#define  id_val_finishing_location_left                     0xD300
#define  id_val_finishing_location_top_left                 0xCA00
#define  id_val_finishing_location_bottom_left              0xCB00
#define  id_val_output_partition_offset                     0x01
#define  id_val_output_partition_none                       0x04

#define  id_att_job_name                                    0x0400
#define  id_att_job_owner                                   0x0600
#define  id_att_job_submission_time_on_client               0x0900
#define  id_att_impressions                                 0xD407
#define  id_att_resolution                                  0x3A00
#define  id_att_document_format                             0x2E00
#define  id_att_color_mode_select                           0x3808
#define  id_att_copies                                      0xD707
#define  id_att_plex                                        0xDF07
#define  id_att_binding_edge                                0xE007
#define  id_att_binding_edge_image_shift                    0xE107
#define  id_att_input_tray                                  0xEE07
#define  id_att_output_bin                                  0xD907
#define  id_att_finishing                                   0xDA07

#define  JOB_NAME                      id_att_job_name
#define  JOB_OWNER                     id_att_job_owner
#define  COLOR                         id_val_color_mode_color
#define  MONO                          id_val_color_mode_mono
#define  SIMPLEX                       id_val_plex_simplex
#define  DUPLEX                        id_val_plex_duplex


/***************************************
  glue alloc memory size
***************************************/
#define  kGluePacketMemSize            512


/*******************************************************
  TypeDefine
*******************************************************/
typedef char *          Ptr;
typedef char *          StringPtr;
typedef struct tm       timeStr;

typedef char            cpcaBYTE;               /* signed 1 byte */
typedef unsigned char   cpcaUBYTE;              /* unsigned 1 byte */
typedef short           cpcaSHORT;              /* signed 2 byte */
typedef unsigned short  cpcaUSHORT;             /* unsigned 2 byte  */
typedef long            cpcaLONG;               /* signed 4 byte */
typedef unsigned long   cpcaULONG;              /* unsigned 4 byte  */
typedef void*           cpcaANY;                /* Any Data */
typedef cpcaUSHORT      cpcaATTRIB;             /* Attribute ID */
typedef cpcaULONG       cpcaDATE;               /* Date */
typedef cpcaULONG       cpcaHANDLE;             /* Object Handle */
typedef cpcaULONG       cpcaTIME;               /* Time */
typedef cpcaULONG       cpcaUSER;               /* User ID */
typedef cpcaULONG       cpcaUSER_PASSWORD;      /* User Password */


/*******************************************************
  CPCA headder structure
*******************************************************/
typedef struct _SendPktHdr {
    cpcaUSHORT          HdrID;
    cpcaUBYTE           Ver;
    cpcaUBYTE           Flgs;
    cpcaUSHORT          Oper;
    cpcaUSHORT          BlkNo;
    unsigned short      Len;
    unsigned long       Uid;
    unsigned long       Pwd;
    unsigned short      Stat;
} cpcaSEND_PKT_HDR;

/*******************************************************
  JobStart
*******************************************************/
typedef struct {
    cpcaHANDLE          Object1;
    cpcaUBYTE           Mode;
    cpcaHANDLE          Object2;
} cpcaIN_JOBSTART;

/*******************************************************
  SetJob
*******************************************************/
typedef struct {
  cpcaATTRIB        Attribute ;
  unsigned long     Size ;
  Ptr               Data ;
} cpcaIN_SETJOB ;

/*******************************************************
  UserInfo
*******************************************************/
typedef struct _UserInfo {
    cpcaUSER            User;
    cpcaUSER_PASSWORD   Password;
} cpcaUSER_INFO;

/*******************************************************
  BinderStart
*******************************************************/
typedef struct {
  cpcaHANDLE        Object ;
} cpcaIN_BINDERSTART ;

/*******************************************************
  BinderBinInfo
*******************************************************/
typedef	struct {
  unsigned short            BinAlias ;
} BinderInfo ;

/*******************************************************
  BinderSet
*******************************************************/
typedef struct {
  cpcaATTRIB       Attribute ;
  unsigned long    Size ;
  Ptr              Data ;
} cpcaIN_SETBINDER ;

/*******************************************************
  BinderEnd
*******************************************************/
typedef struct {
  short            dummy ;
} cpcaIN_BINDEREND ;

/*******************************************************
  Date&Time
*******************************************************/
typedef struct _DateTimes {
  cpcaDATE  Date;
  cpcaTIME  Time;
} cpcaDATETIMES, *PcpcaDATETIMES;

/*******************************************************
  DocumentStart
*******************************************************/
typedef struct {
  cpcaHANDLE       Object ;
} cpcaIN_DOCUMENTSTART ;

/*******************************************************
  SetDocument
*******************************************************/
typedef struct {
  cpcaATTRIB        Attribute ;
  unsigned long     Size ;
  Ptr               Data ;
} cpcaIN_SETDOCUMENT ;

/*******************************************************
  Set LBP Document
*******************************************************/
typedef struct {
  cpcaHANDLE        Object ;
  cpcaUSHORT        Method ;
  unsigned long     Size ;
  Ptr               Data ;
} cpcaIN_SETLBPDOCUMENT ;

/*******************************************************
  SendData
*******************************************************/
typedef struct {
  cpcaUBYTE        Mode ;
  unsigned long    Size ;
  Ptr              Data ;
} cpcaIN_SEND ;

/*******************************************************
  DocumentEnd
*******************************************************/
typedef struct {
  short            dummy ;
} cpcaIN_DOCUMENTEND ;

/*******************************************************
  JobEnd
*******************************************************/
typedef struct {
  cpcaUBYTE        Code ;
} cpcaIN_JOBEND ;

/*******************************************************
  Prototype
*******************************************************/
/**********************************      Glue         ********************************************/
unsigned long
glue_cpcaJobStart              ( Ptr             p,
                                 unsigned char   jobMode ) ;

unsigned long
glue_cpcaSetJob                ( Ptr             p,
                                 cpcaATTRIB      Attribute,
                                 cpcaUSHORT      Size,
                                 Ptr             Data ) ;

unsigned long
glue_cpcaSetJobString          ( Ptr             p,
                                 cpcaATTRIB      Attribute,
                                 StringPtr       str ) ;

unsigned long
glue_cpcaSetJobTime            ( Ptr             p ) ;

unsigned long
glue_cpcaSetJobImpression      ( Ptr             p,
                                 unsigned short  impression ) ;

unsigned long
glue_cpcaBinderStart           ( Ptr             p ) ;

unsigned long
glue_cpcaSetBinder             ( Ptr             p,
                                 cpcaATTRIB      Attribute,
                                 cpcaUSHORT      Size,
                                 cpcaANY         Data ) ;

unsigned long
glue_cpcaSetBinderCopy         ( Ptr             p,
                                 unsigned short  copy ) ;

unsigned long
glue_cpcaSetBinderOutputBin    ( Ptr             p,
                                 unsigned char   bindInfo,
                                 unsigned char   bindNum ) ;

unsigned long
glue_cpcaSetBinderStapleSort   ( Ptr             p,
                                 unsigned short  staple_val ) ;

unsigned long
glue_cpcaDocumentStart         ( Ptr             p ) ;

unsigned long
glue_cpcaSetDocument           ( Ptr             p,
                                 cpcaATTRIB      Attribute,
                                 cpcaUSHORT      Size,
                                 cpcaANY         Data ) ;

unsigned long
glue_cpcaSetDocumentFormat     ( Ptr             p,
                                 unsigned char   family ) ;

unsigned long
glue_cpcaSetDocumentResolution ( Ptr             p,
                                 unsigned short  resolutionX,
                                 unsigned short  resolutionY ) ;

unsigned long
glue_cpcaSetDocumentColorMode  ( Ptr             p,
                                 unsigned char   type ) ;

unsigned long
glue_cpcaSetDocumentCopy       ( Ptr             p,
                                 unsigned short  copy ) ;

unsigned long
glue_cpcaSetDocumentPlex       ( Ptr             p,
                                 unsigned char   plex ) ;

unsigned long
glue_cpcaSetDocumentBindDir    ( Ptr             p,
                                 char            isLongEdge );

unsigned long
glue_cpcaSetDocumentBindMarg   ( Ptr             p,
                                 long            shiftSize ) ;

unsigned long
glue_cpcaSetDocumentInputTray  ( Ptr             p,
                                 Ptr             trayName ) ;

unsigned long
glue_SetLBPDocument            ( Ptr             p,
                                 StringPtr       Personality,
                                 StringPtr       envVariable,
                                 StringPtr       envValue ) ;

unsigned long
glue_SetPanelMessage           ( Ptr             p,
                                 StringPtr       MessageData ) ;

unsigned long
glue_cpcaSend                  ( Ptr             p,
                                 unsigned long   size,
                                 Ptr             data ) ;

unsigned long
glue_BinderEnd                 ( Ptr             p ) ;

unsigned long
glue_DocumentEnd               ( Ptr             p ) ;


unsigned long
glue_cpcaJobEnd                ( Ptr             p,
                                 char            isCancel ) ;


/**********************************      _CPCA         ********************************************/
long  _CPCA_JobStart       ( Ptr                   p,
                             cpcaUSER_INFO         *userInfo,
                             cpcaIN_JOBSTART       *inParam ) ;

long  _CPCA_SetJob         ( Ptr                   p,
                             cpcaUSER_INFO         *userInfo,
                             cpcaIN_SETJOB         *inParam ) ;

long  _CPCA_BinderStart    ( Ptr                   p,
                             cpcaUSER_INFO         *userInfo,
                             cpcaIN_BINDERSTART    *inParam ) ;

long  _CPCA_SetBinder      ( Ptr                   p,
                             cpcaUSER_INFO         *userInfo,
                             cpcaIN_SETBINDER      *inParam ) ;

long  _CPCA_DocumentStart  ( Ptr                   p,
                             cpcaUSER_INFO         *userInfo,
                             cpcaIN_DOCUMENTSTART  *inParam ) ;

long  _CPCA_SetDocument    ( Ptr                   p,
                             cpcaUSER_INFO         *userInfo,
                             cpcaIN_SETDOCUMENT    *inParam ) ;

long  _CPCA_SetLBPDocument ( Ptr                   p,
                             cpcaUSER_INFO         *userInfo,
                             cpcaIN_SETLBPDOCUMENT *inParam ) ;

long  _CPCA_Send           ( Ptr                   p,
                             cpcaUSER_INFO         *userInfo,
                             cpcaIN_SEND           *inParam ) ;

long  _CPCA_DocumentEnd    ( Ptr                   p,
                             cpcaUSER_INFO         *userInfo,
                             cpcaIN_DOCUMENTEND    *inParam ) ;

long  _CPCA_BinderEnd      ( Ptr                   p,
                             cpcaUSER_INFO         *userInfo,
                             cpcaIN_BINDEREND      *inParam ) ;

long  _CPCA_JobEnd         ( Ptr                   p,
                             cpcaUSER_INFO         *userInfo,
                             cpcaIN_JOBEND         *inParam ) ;

/**********************************      Utility         ********************************************/


long  make_JobStart         ( Ptr                  p,
                              unsigned short       *blockNo,
                              cpcaUSER_INFO        *userInfo,
                              cpcaIN_JOBSTART      *inParam ) ;

long  make_SetJob           ( Ptr                  p,
                              unsigned short       *blockNo,
                              cpcaUSER_INFO        *userInfo,
                              cpcaIN_SETJOB        *inParam ) ;

long  make_BinderStart      ( Ptr                  p,
                              unsigned short       *blockNo,
                              cpcaUSER_INFO        *userInfo,
                              cpcaIN_BINDERSTART   *inParam ) ;

long  make_SetBinder        ( Ptr                   p,
                              unsigned short        *blockNo,
                              cpcaUSER_INFO         *userInfo,
                              cpcaIN_SETBINDER      *inParam ) ;

short  make_OutputBin       ( Ptr                   p,
                              unsigned char         bindInfo,
                              unsigned char         bindNum ) ;

short  make_StapleSort      ( Ptr                   p,
                              unsigned short        staple_val ) ;

long  make_DocumentStart    ( Ptr                   p,
                              unsigned short        *blockNo,
                              cpcaUSER_INFO         *userInfo,
                              cpcaIN_DOCUMENTSTART  *inParam ) ;

long  make_SetDocument      ( Ptr                   p,
                              unsigned short        *blockNo,
                              cpcaUSER_INFO         *userInfo,
                              cpcaIN_SETDOCUMENT    *inParam ) ;

long  make_SetLBPDocument   ( Ptr                   p,
                              unsigned short        *blockNo,
                              cpcaUSER_INFO         *userInfo,
                              cpcaIN_SETLBPDOCUMENT *inParam ) ;

long  make_Send             ( Ptr                   p,
                              unsigned short        *blockNo,
                              cpcaUSER_INFO         *userInfo,
                              cpcaIN_SEND           *inParam ) ;

long  make_BinderEnd        ( Ptr                   p,
                              unsigned short        *blockNo,
                              cpcaUSER_INFO         *userInfo,
                              cpcaIN_BINDEREND      *inParam ) ;

long  make_DocumentEnd      ( Ptr                   p,
                              unsigned short        *blockNo,
                              cpcaUSER_INFO         *userInfo,
                              cpcaIN_DOCUMENTEND    *inParam ) ;

long  make_JobEnd           ( Ptr                   p,
                              unsigned short        *blockNo,
                              cpcaUSER_INFO         *userInfo,
                              cpcaIN_JOBEND         *inParam ) ;

static short z_make_header  ( Ptr                   p,
                              unsigned char         version,
                              unsigned char         flag,
                              unsigned short        opCode,
                              unsigned short        blockNum,
                              unsigned short        paramLen,
                              unsigned long         userID,
                              unsigned long         password ) ;

static void z_setUserInfo   ( cpcaUSER_INFO         *userInfo,
                              unsigned long         *userID,
                              unsigned long         *password ) ;

static void _makeString     ( char                  *setData,
                              const char            *inData ) ;

static void _makeNameOwner  ( PRasterArg_t          arg ) ;

static void _makeItoa       ( char                  *SetData,
                              int                   value,
                              int                   size ) ;

static void _makeDatetime   ( cpcaDATETIMES         *DateTime ) ;

/**********************************      Glue         ********************************************/

/*====================================================================
//  glue_cpcaJobStart()
====================================================================*/
unsigned long  glue_cpcaJobStart ( 
  Ptr                 p,
  unsigned char       jobMode )
{
  cpcaIN_JOBSTART     inParam ;
  cpcaUSER_INFO       userInfo ;
  unsigned long       comSize = 0 ;

  if (p) {
    inParam.Object1   =  id_oc_print_job ;
    inParam.Mode      =  jobMode ;
    inParam.Object2   =  0x00000000 ;
    userInfo.User     =  0x00000000 ;
    userInfo.Password =  0x00000000 ;
    comSize = _CPCA_JobStart( p, &userInfo, &inParam ) ;
  }
  return comSize ;
}

/*====================================================================
//  glue_cpcaSetJob()
====================================================================*/
unsigned long  glue_cpcaSetJob(
  Ptr               p,
  cpcaATTRIB        Attribute,
  cpcaUSHORT        Size,
  Ptr               Data )
{
  cpcaIN_SETJOB     inParam ;
  cpcaUSER_INFO     userInfo ;
  unsigned long     comSize = 0 ;

  if (p) {
    inParam.Attribute    = Attribute ;
    inParam.Size         = Size ;
    inParam.Data         = Data ;
    userInfo.User        = 0x00000000 ;
    userInfo.Password    = 0x00000000 ;
    comSize = _CPCA_SetJob ( p, &userInfo, &inParam ) ;
  }
  return comSize ;
}


/*====================================================================
//  glue_cpcaSetJobString()
====================================================================*/
unsigned long  glue_cpcaSetJobString(
  Ptr          p,
  cpcaATTRIB   Attribute,
  StringPtr    str )
{
  Ptr             data ;
  short           size ;
  Ptr             work ;
  unsigned long   comSize = 0 ;

  if (p && str) {
    size = sizeof( short ) + (unsigned char)str[ 0 ] + 1 ;
    data = (Ptr)malloc( size ) ;
    if (data) {
      work = data ;

      *( ( short *)work )++ = id_val_code_set_Shift_JIS ;
      memcpy( work, str, (unsigned char)str[ 0 ] + 1 ) ;
      comSize = glue_cpcaSetJob( p,
                                 Attribute,
                                 size,
                                 data ) ;
      free(data) ;
    }
  }
  return comSize ;
}

/*====================================================================
//  glue_cpcaSetJobTime()
====================================================================*/
unsigned long  glue_cpcaSetJobTime(
  Ptr              p )
{
  cpcaDATETIMES    DateTime ;
  unsigned long    comSize = 0 ;

  _makeDatetime( &DateTime);

  comSize = glue_cpcaSetJob( p,
                             id_att_job_submission_time_on_client,
                             sizeof( cpcaDATETIMES ),
                             ( Ptr )&DateTime ) ;
  return comSize ;
}

/*====================================================================
//  glue_cpcaSetJobImpression()
====================================================================*/
unsigned long  glue_cpcaSetJobImpression(
  Ptr              p,
  unsigned short   impression )
{
  unsigned long    comSize = 0 ;

  impression = impression << 8;

  if (p) {
  comSize = glue_cpcaSetJob( p,
                             id_att_impressions,
                             2,
                             ( Ptr )&impression ) ;
  }
  return comSize ;
}



/*====================================================================
//  glue_cpcaBinderStart()
====================================================================*/
unsigned long  glue_cpcaBinderStart ( Ptr p )
{
  cpcaIN_BINDERSTART          inParam ;
  cpcaUSER_INFO               userInfo ;
  unsigned long               comSize = 0 ;

  if ( p ) {
    inParam.Object      = id_oc_print_binder << 16 ;
    userInfo.User       = 0x00000000 ;
    userInfo.Password   = 0x00000000 ;
    comSize = _CPCA_BinderStart( p, &userInfo, &inParam ) ;
  }
  return comSize ;
}


/*====================================================================
//  glue_cpcaSetBinder()
====================================================================*/
unsigned long  glue_cpcaSetBinder (
  Ptr                   p,
  cpcaATTRIB            Attribute,
  cpcaUSHORT            Size,
  cpcaANY               Data )
{
  cpcaIN_SETBINDER      inParam ;
  cpcaUSER_INFO         userInfo ;
  unsigned long         comSize = 0 ;

  if (p) {
    inParam.Attribute   = Attribute ;
    inParam.Size        = Size ;
    inParam.Data        = Data ;
    userInfo.User       = 0x00000000 ;
    userInfo.Password   = 0x00000000 ;
    comSize = _CPCA_SetBinder ( p, &userInfo, &inParam ) ;
  }
  return comSize ;
}


/*====================================================================
//  glue_cpcaSetBinderCopy()
====================================================================*/
unsigned long  glue_cpcaSetBinderCopy (
  Ptr               p,
  unsigned short    copy )
{
  unsigned long     comSize = 0 ;

  if (p) {
  comSize = glue_cpcaSetBinder (  p,
                                  id_att_copies,
                                  2,
                                  ( Ptr )&copy ) ;
  }
  return comSize ;
}


/*====================================================================
//  glue_cpcaSetBinderOutputBin()
====================================================================*/
unsigned long  glue_cpcaSetBinderOutputBin (
  Ptr                   p,
  unsigned char         bindInfo,
  unsigned char         bindNum )
{
  Ptr                   data ;
  short                 setDataSize ;
  unsigned long         comSize = 0 ;
  unsigned long         allComSize = 0 ;

  if (p) {
    data = malloc( kGluePacketMemSize ) ;
    if (data) {
      /* make output bin info */
      setDataSize = make_OutputBin ( data, bindInfo, bindNum ) ;
      if (setDataSize > 0) {
        comSize = glue_cpcaSetBinder( p,
                                      id_att_output_bin,
                                      setDataSize,
                                      data ) ;
        p          += comSize ;
        allComSize += comSize ;
      }
      free(data) ;
    }
  }
  return allComSize ;
}


/*====================================================================
//  glue_cpcaSetBinderStapleSort()
====================================================================*/
unsigned long  glue_cpcaSetBinderStapleSort (
  Ptr                   p,
  unsigned short        staple_val )

{
  Ptr                   data ;
  short                 setDataSize ;
  unsigned long         comSize = 0 ;
  unsigned long         allComSize = 0 ;

  if (p) {
    data = malloc( kGluePacketMemSize ) ;
    if (data) {
      /* make output bin info command */
      setDataSize = make_StapleSort ( data, staple_val ) ;
      if (setDataSize > 0) {
        comSize = glue_cpcaSetBinder( p,
                                      id_att_finishing,
                                      setDataSize,
                                      data ) ;
        p          += comSize ;
        allComSize += comSize ;
      }

      free(data) ;
    }
  }
  return allComSize ;
}

/*====================================================================
//  glue_cpcaDocumentStart()
====================================================================*/
unsigned long  glue_cpcaDocumentStart ( Ptr p )
{
  cpcaIN_DOCUMENTSTART      inParam ;
  cpcaUSER_INFO             userInfo ;
  unsigned long             comSize = 0 ;

  if (p) {
    inParam.Object = id_oc_print_document ;
    userInfo.User     = 0x00000000 ;
    userInfo.Password = 0x00000000 ;
    comSize = _CPCA_DocumentStart ( p, &userInfo, &inParam ) ;
  }
  return comSize ;
}



/*====================================================================
//  glue_cpcaSetDocument()
====================================================================*/
unsigned long  glue_cpcaSetDocument(
  Ptr               p,
  cpcaATTRIB        Attribute,
  cpcaUSHORT        Size,
  cpcaANY           Data )
{
  cpcaIN_SETDOCUMENT     inParam ;
  cpcaUSER_INFO          userInfo ;
  unsigned long          comSize = 0 ;

  if (p) {
    inParam.Attribute = Attribute ;
    inParam.Size      = Size ;
    inParam.Data      = Data ;
    userInfo.User     = 0x00000000 ;
    userInfo.Password = 0x00000000 ;
    comSize = _CPCA_SetDocument ( p, &userInfo, &inParam ) ;
  }
  return comSize ;
}



/*====================================================================
//  glue_cpcaSetDocumentFormat()
====================================================================*/
unsigned long  glue_cpcaSetDocumentFormat(
  Ptr              p,
  unsigned char    family )
{
  Ptr              data ;
  Ptr              work ;
  unsigned long    comSize = 0 ;

  if ( p )
  {
  data = ( Ptr )malloc( 3 ) ;
    if ( data )
    {
      work = data ;

      *work++ = family ;
      *work++ = 0x00 ;
      *work++ = 0x00 ;

      comSize = glue_cpcaSetDocument ( p,
                                       id_att_document_format,
                                       3,
                                       data ) ;
      free( data ) ;
    }
  }

  return comSize ;
}



/*====================================================================
//  glue_cpcaSetDocumentResolution()
====================================================================*/
unsigned long  glue_cpcaSetDocumentResolution(
  Ptr               p,
  unsigned short    resolutionX,
  unsigned short    resolutionY )
{
  short          size ;
  Ptr            data ;
  Ptr            work ;
  unsigned long  comSize = 0 ;

  if (p) {
    size = 5 ;
    data = ( Ptr )malloc( size ) ;
    if (data) {
      work = data ;

      *work++ = id_val_resolution_type_dpi ;

      *( ( unsigned short *)work )++ = resolutionX ;
      *( ( unsigned short *)work )++ = resolutionY ;

      comSize = glue_cpcaSetDocument ( p,
                                       id_att_resolution,
                                       size,
                                       data ) ;

      free(data) ;
    }
  }
  return comSize ;
}


/*====================================================================
//  glue_cpcaSetDocumentColorMode()
====================================================================*/
unsigned long  glue_cpcaSetDocumentColorMode(
  Ptr              p,
  unsigned char    type )
{
  unsigned long    comSize = 0 ;

  if (p) {
    comSize = glue_cpcaSetDocument ( p,
                                     id_att_color_mode_select,
                                     1,
                                     &type ) ;
  }
  return comSize ;
}



/*====================================================================
//  glue_cpcaSetDocumentCopy()
//  ɥȤΥԡ
====================================================================*/
unsigned long  glue_cpcaSetDocumentCopy (
  Ptr               p,
  unsigned short    copy )
{
  unsigned long     comSize = 0 ;

  if (p) {
    comSize = glue_cpcaSetDocument ( p,
                                     id_att_copies,
                                     2,
                                     ( Ptr )&copy ) ;
  }
  return comSize ;
}






/*====================================================================
//  glue_cpcaSetDocumentPlex()
//  ξ̰λ
====================================================================*/
unsigned long  glue_cpcaSetDocumentPlex (
  Ptr               p,
  unsigned char     plex )
{
  unsigned long     comSize = 0 ;

  if (p) {
    comSize = glue_cpcaSetDocument ( p,
                                     id_att_plex,
                                     1,
                                     ( Ptr )&plex ) ;
  }
  return comSize ;
}



/*====================================================================
//  glue_cpcaSetDocumentBindDir()
//  ɥȤ֤
====================================================================*/
unsigned long  glue_cpcaSetDocumentBindDir(
  Ptr           p,
  char          isLongEdge)
{
  unsigned char    edge ;
  unsigned long    comSize = 0 ;

  edge = ( isLongEdge ) ? id_val_binding_left_edge : id_val_binding_top_edge ;
  if (p) {
    comSize = glue_cpcaSetDocument ( p,
                                     id_att_binding_edge,
                                     1,
                                     ( Ptr )&edge ) ;
  }
  return comSize ;
}


/*====================================================================
//  glue_cpcaSetDocumentBindMarg()
//  ɥȤ֤
====================================================================*/
unsigned long  glue_cpcaSetDocumentBindMarg(
  Ptr           p,
  long          shiftSize )
{
  unsigned long    comSize = 0 ;
  shiftSize = SWAP_DWORD( shiftSize );

  if (p) {
    comSize += glue_cpcaSetDocument ( p,
                                      id_att_binding_edge_image_shift,
                                      4,
                                      ( Ptr )&shiftSize ) ;

  }
  return comSize ;
}




/*====================================================================
//  glue_cpcaSetDocumentInputTray()
//  ȥ쥤
====================================================================*/
unsigned long  glue_cpcaSetDocumentInputTray(
  Ptr             p,
  Ptr             trayName )
{
  unsigned short  size ;
  Ptr             data ;
  Ptr             work ;
  unsigned long   comSize = 0 ;

  if (p) {
    size = 2 + trayName[0] + 1;
    data = (Ptr)malloc(size);
    if (data) {
      work = data;
      *work++ = id_val_input_tray_named;
      *work++ = 0x00;
      memcpy(work, trayName, trayName[0] + 1);

      comSize = glue_cpcaSetDocument ( p,
                                       id_att_input_tray,
                                       size,
                                       data ) ;
      free(data);
    }
  }
  return comSize ;
}




/*====================================================================
//  glue_SetLBPDocument()
//  LBPɥȴĶͤ
====================================================================*/
unsigned long       glue_SetLBPDocument(
  Ptr             p,
  StringPtr       Personality,
  StringPtr       envVariable,
  StringPtr       envValue )
{
  cpcaIN_SETLBPDOCUMENT       inParam ;
  cpcaUSER_INFO               userInfo ;
  Ptr                         work ;
  char                        Personal;
  unsigned long               comSize = 0 ;

  /* param check */
  if ((p) && (envVariable) && (envValue)) {

    inParam.Object =  id_oc_print_document;
    inParam.Method = 0x0020 ;
    inParam.Data = (Ptr)malloc( 256 ) ;

    if (inParam.Data) {
      work = inParam.Data ;
      if (Personality) {
        memcpy( work, Personality, Personality[ 0 ] + 1 ) ;
        work += Personality[ 0 ] + 1 ;
      } else {
        Personal = 0x00;
        memcpy( work, &Personal, 1 ) ;
        work += 1 ;
      }
      memcpy( work, envVariable, envVariable[ 0 ] + 1 ) ;
      work += envVariable[ 0 ] + 1 ;
      memcpy( work, envValue, envValue[ 0 ] + 1 ) ;
      work += envValue[ 0 ] + 1 ;

      inParam.Size = work - inParam.Data ;

      userInfo.User = 0x00000000 ;
      userInfo.Password = 0x00000000 ;

      comSize = _CPCA_SetLBPDocument( p,
                                      &userInfo,
                                      &inParam ) ;

      free(inParam.Data) ;
    }
  }
  return comSize ;
}




/*====================================================================
//  glue_SetLBPDocument()
//  LBPɥȴĶͤ
====================================================================*/
unsigned long       glue_SetPanelMessage(
  Ptr             p,
  StringPtr       MessageData )
{
  cpcaIN_SETLBPDOCUMENT       inParam ;
  cpcaUSER_INFO               userInfo ;
  Ptr                         work ;
  unsigned long               comSize = 0 ;
  int                         msgSize = 0 ;

  /* param check */
  if ((p) && (MessageData)) {

    inParam.Object =  id_oc_print_document;
    inParam.Method = 0x0120 ;
    inParam.Data = (Ptr)malloc( 20 ) ;

    if (inParam.Data) {
      work = inParam.Data ;
      if ((unsigned char)MessageData[0] > 16) {
        msgSize = 17 ;
      } else {
        msgSize = MessageData[ 0 ] + 1 ;
      }
      memcpy( work, MessageData, msgSize ) ;
      inParam.Size = (unsigned char)msgSize;

      userInfo.User = 0x00000000 ;
      userInfo.Password = 0x00000000 ;

      comSize = _CPCA_SetLBPDocument( p,
                                      &userInfo,
                                      &inParam ) ;

      free(inParam.Data) ;
    }
  }
  return comSize ;
}




/*====================================================================
//  glue_cpcaSend()
//  
====================================================================*/
unsigned long  glue_cpcaSend(
  Ptr             p,
  unsigned long   size,
  Ptr             data )
{
  cpcaIN_SEND     inParam ;
  cpcaUSER_INFO   userInfo ;
  unsigned long   comSize = 0 ;

  /* param check */
  if ((p == nil) || (data == nil) || (size == 0)) {

    inParam.Mode = id_val_send_mode_raw ;
    inParam.Size = size ;
    inParam.Data = data ;

    userInfo.User = 0x00000000 ;
    userInfo.Password = 0x00000000 ;

    comSize = _CPCA_Send ( p, &userInfo, &inParam ) ;

  }
  return comSize ;
}





/*====================================================================
//  glue_BinderEnd()
//  Хɽ
====================================================================*/
unsigned long  glue_BinderEnd ( Ptr p )
{
  cpcaIN_BINDEREND       inParam ;
  cpcaUSER_INFO          userInfo ;
  unsigned long          comSize = 0 ;

  if (p) {
    inParam.dummy       = 0 ;
    userInfo.User       = 0x00000000 ;
    userInfo.Password   = 0x00000000 ;
    comSize = _CPCA_BinderEnd ( p, &userInfo, &inParam ) ;
  }
  return comSize ;
}



/*====================================================================
//  glue_DocumentEnd()
//  ɥȥɽ
====================================================================*/
unsigned long  glue_DocumentEnd ( Ptr p )
{
  cpcaIN_DOCUMENTEND       inParam ;
  cpcaUSER_INFO            userInfo ;
  unsigned long            comSize = 0 ;

  if (p) {
    inParam.dummy = 0 ;
    userInfo.User = 0x00000000 ;
    userInfo.Password = 0x00000000 ;

    comSize = _CPCA_DocumentEnd ( p, &userInfo, &inParam ) ;
  }
  return comSize ;
}




/*====================================================================
//  glue_cpcaJobEnd()
//  ֥ɽ
====================================================================*/
unsigned long  glue_cpcaJobEnd ( Ptr p, char isCancel )
{
  cpcaIN_JOBEND         inParam ;
  cpcaUSER_INFO         userInfo ;
  unsigned long         comSize = 0 ;

  if (isCancel) {
    inParam.Code = 0x01 ;
  } else {
    inParam.Code = 0x00 ;
  }

  if (p) {
    userInfo.User = 0x00000000 ;
    userInfo.Password = 0x00000000 ;
    comSize = _CPCA_JobEnd ( p, &userInfo, &inParam ) ;
  }
  return comSize ;
}


/**********************************      _CPCA      ********************************************/


/*============================================================================
//  _CPCA_JobStart
//  JobStartѥåȺ
//
//    <input>
//    p                     (I/O)   Ǽǡݥ
//    userInfo              (I)     桼
//    inParam               (I)     ѥ᡼
============================================================================*/
long  _CPCA_JobStart(
  Ptr                   p,
  cpcaUSER_INFO         *userInfo,
  cpcaIN_JOBSTART       *inParam )
{
  long                  comSize ;
  unsigned short        blockNo ;

  blockNo = 0 ;

  /* make packet */
  comSize = make_JobStart( p,
                           &blockNo,
                           userInfo,
                           inParam ) ;
  return comSize ;
}






/*============================================================================
//  _CPCA_SetJob
//  SetJobѥåȺ
//
//    <input>
//    p                     (I/O)   Ǽǡݥ
//    userInfo              (I)     桼
//    inParam               (I)     ѥ᡼
============================================================================*/
long  _CPCA_SetJob(
  Ptr                  p,
  cpcaUSER_INFO        *userInfo,
  cpcaIN_SETJOB        *inParam )
{
  long                 comSize ;
  unsigned short       blockNo ;

  blockNo = 0 ;
  /* make packet */
  comSize = make_SetJob( p,
                         &blockNo,
                         userInfo,
                         inParam ) ;

  return comSize ;
}





/*============================================================================
//  _CPCA_BinderStart
//  BinderStart ѥåȺ
//
//    <input>
//    p                     (I/O)   Ǽǡݥ
//    userInfo              (I)     桼
//    inParam               (I)     ѥ᡼
============================================================================*/
long  _CPCA_BinderStart (
  Ptr                       p,
  cpcaUSER_INFO             *userInfo,
  cpcaIN_BINDERSTART     *inParam )
{
  long                  comSize ;
  unsigned short        blockNo ;

  blockNo = 0 ;

  /* make packet */
  comSize = make_BinderStart( p,
                              &blockNo,
                              userInfo,
                              inParam ) ;
  return comSize ;
}



/*============================================================================
//  _CPCA_SetBinder
//  SetBinder ѥåȺ
//
//    <input>
//    p                     (I/O)   Ǽǡݥ
//    userInfo              (I)     桼
//    inParam               (I)     ѥ᡼
============================================================================*/
long  _CPCA_SetBinder (
  Ptr                       p,
  cpcaUSER_INFO             *userInfo,
  cpcaIN_SETBINDER          *inParam )
{
  long                  comSize ;
  unsigned short        blockNo ;

  blockNo = 0 ;

  /* make packet */
  comSize = make_SetBinder( p,
                            &blockNo,
                            userInfo,
                            inParam ) ;
  return comSize ;
}





/*============================================================================
//  _CPCA_DocumentStart
//  JobStartѥåȺ
//
//    <input>
//    p                     (I/O)   Ǽǡݥ
//    userInfo              (I)     桼
//    inParam               (I)     ѥ᡼
============================================================================*/
long  _CPCA_DocumentStart(
  Ptr                       p,
  cpcaUSER_INFO             *userInfo,
  cpcaIN_DOCUMENTSTART   *inParam )
{
  long                  comSize ;
  unsigned short        blockNo ;


  blockNo = 0 ;
  /* make packet */
  comSize = make_DocumentStart ( p,
                                 &blockNo,
                                 userInfo,
                                 inParam ) ;
  return comSize ;
}



/*============================================================================
//  _CPCA_SetDocument
//  SetDocumentѥåȺ
//
//    <input>
//    p                     (I/O)   Ǽǡݥ
//    userInfo              (I)     桼
//    inParam               (I)     ѥ᡼
============================================================================*/
long  _CPCA_SetDocument(
  Ptr                       p,
  cpcaUSER_INFO             *userInfo,
  cpcaIN_SETDOCUMENT        *inParam )
{
  long                  comSize ;
  unsigned short        blockNo ;


  blockNo = 0 ;

  /* make packet */
  comSize = make_SetDocument( p,
                              &blockNo,
                              userInfo,
                              inParam ) ;
  return comSize ;
}



/*============================================================================
//  _CPCA_SetLBPDocument
//  SetLBPDocumentѥåȺ
//
//    <input>
//    p                     (I/O)   Ǽǡݥ
//    userInfo              (I)     桼
//    inParam               (I)     ѥ᡼
============================================================================*/
long  _CPCA_SetLBPDocument(
  Ptr                         p,
  cpcaUSER_INFO               *userInfo,
  cpcaIN_SETLBPDOCUMENT       *inParam )
{
  long                comSize ;
  unsigned short      blockNo ;


  blockNo = 0 ;

  /* make packet */
  comSize = make_SetLBPDocument( p,
                                 &blockNo,
                                 userInfo,
                                 inParam ) ;
  return comSize ;
}



/*============================================================================
//  _CPCA_Send
//  SendѥåȺ
//
//    <input>
//    p                     (I/O)   Ǽǡݥ
//    userInfo              (I)     桼
//    inParam               (I)     ѥ᡼
============================================================================*/
long  _CPCA_Send(
  Ptr                p,
  cpcaUSER_INFO      *userInfo,
  cpcaIN_SEND        *inParam )
{
  long               comSize ;
  unsigned short     blockNo ;


  blockNo = 0 ;

  /* make packet */
  comSize = make_Send ( p,
                        &blockNo,
                        userInfo,
                        inParam ) ;
  return comSize ;
}




/*============================================================================
//  _CPCA_BinderEnd
//  BinderEnd ѥåȺ
//
//    <input>
//    p                     (I/O)   Ǽǡݥ
//    userInfo              (I)     桼
//    inParam               (I)     ѥ᡼
============================================================================*/
long  _CPCA_BinderEnd (
  Ptr                      p,
  cpcaUSER_INFO            *userInfo,
  cpcaIN_BINDEREND         *inParam )
{
  long                     comSize ;
  unsigned short           blockNo ;

  blockNo = 0 ;

  /* make packet */
  comSize = make_BinderEnd( p,
                            &blockNo,
                            userInfo,
                            inParam ) ;
  return comSize ;
}




/*============================================================================
//  _CPCA_DocumentEnd
//  DocumentEndѥåȺ
//
//    <input>
//    p                     (I/O)   Ǽǡݥ
//    userInfo              (I)     桼
//    inParam               (I)     ѥ᡼
============================================================================*/
long  _CPCA_DocumentEnd(
  Ptr                       p,
  cpcaUSER_INFO             *userInfo,
  cpcaIN_DOCUMENTEND        *inParam )
{
  long                  comSize ;
  unsigned short        blockNo ;


  blockNo = 0 ;

  /* make packet */
  comSize = make_DocumentEnd( p,
                              &blockNo,
                              userInfo,
                              inParam ) ;
  return comSize ;
}




/*============================================================================
//  _CPCA_JobEnd
//  JobEndѥåȺ
//
//    <input>
//    p                     (I/O)   Ǽǡݥ
//    userInfo              (I)     桼
//    inParam               (I)     ѥ᡼
============================================================================*/
long  _CPCA_JobEnd(
  Ptr                   p,
  cpcaUSER_INFO         *userInfo,
  cpcaIN_JOBEND         *inParam )
{
  long                  comSize ;
  unsigned short        blockNo ;


  blockNo = 0 ;

  /* make packet */
  comSize = make_JobEnd( p,
                         &blockNo,
                         userInfo,
                         inParam ) ;
  return comSize ;
}




/**********************************      Utility      ********************************************/


/*============================================================================
// make_JobStart()
============================================================================*/
long make_JobStart(
  Ptr                  p,
  unsigned short       *blockNo,
  cpcaUSER_INFO        *userInfo,
  cpcaIN_JOBSTART   *inParam )
{
  short                comSize ;
  short                paramSize ;
  unsigned long        userID ;
  unsigned long        password ;


  z_setUserInfo( userInfo, &userID, &password ) ;

  paramSize = 0x0900;
  comSize = z_make_header(
                          p,
                          cpcaVERSION_V10,
                          cpcaPKTFLG_SEND_V10,
                          id_val_operation_job_start,
                          *blockNo,
                          paramSize,
                          userID,
                          password ) ;

  (*blockNo)++ ;
  p += comSize ;

  /* ͭѥ᡼ */
  *( ( cpcaHANDLE *)p )++  = inParam->Object1 ;
  *( ( cpcaUBYTE  *)p )++  = inParam->Mode ;
  *( ( cpcaHANDLE *)p )++  = inParam->Object2 ;
  comSize += 9;

  return comSize ;
}








/*============================================================================
//  make_SetJob()
============================================================================*/
long  make_SetJob(
  Ptr               p,
  unsigned short    *blockNo,
  cpcaUSER_INFO     *userInfo,
  cpcaIN_SETJOB     *inParam )
{
  short             comSize ;
  short             paramSize ;
  unsigned long     userID ;
  unsigned long     password ;

	
  z_setUserInfo( userInfo, &userID, &password ) ;
  paramSize = 2 + inParam->Size ;
  comSize = z_make_header(
                          p,
                          cpcaVERSION_V10,
                          cpcaPKTFLG_SEND_V10,
                          id_val_operation_set_job,
                          *blockNo,
                          SWAP_WORD( paramSize ),
                          userID,
                          password ) ;
  (*blockNo)++ ;
  p += comSize ;

  /*  ͭѥ᡼ */
  *( ( cpcaATTRIB *)p )++  = inParam->Attribute ;
  if ( inParam->Data && ( inParam->Size > 0 ) )
  {
    memcpy( p, inParam->Data, inParam->Size ) ;
  }
  comSize += paramSize ;

  return comSize ;
}




/*============================================================================
//  make_BinderStart()
============================================================================*/
long  make_BinderStart (
  Ptr                     p,
  unsigned short          *blockNo,
  cpcaUSER_INFO           *userInfo,
  cpcaIN_BINDERSTART      *inParam )
{
  short               comSize ;
  short               paramSize ;
  unsigned long       userID ;
  unsigned long       password ;

  z_setUserInfo( userInfo, &userID, &password ) ;
  paramSize = sizeof( cpcaHANDLE ) ;
  comSize   = z_make_header( p,
                             cpcaVERSION_V10,
                             cpcaPKTFLG_SEND_V10,
                             id_val_operation_binder_start,
                             *blockNo,
                             SWAP_WORD( paramSize ),
                             userID,
                             password ) ;

  (*blockNo)++ ;
  p += comSize ;

  /*  ͭѥ᡼ */
  *( ( cpcaHANDLE *)p )++ = inParam->Object ;
  comSize += paramSize ;

  return comSize ;
}


/*============================================================================
//  make_SetBinder()
============================================================================*/
long  make_SetBinder (
  Ptr                     p,
  unsigned short          *blockNo,
  cpcaUSER_INFO           *userInfo,
  cpcaIN_SETBINDER        *inParam )
{
  short             comSize ;
  short             paramSize ;
  unsigned long     userID ;
  unsigned long     password ;

  z_setUserInfo( userInfo, &userID, &password ) ;
  paramSize = sizeof( cpcaATTRIB ) + inParam->Size ;
  comSize   = z_make_header( p,
                             cpcaVERSION_V10,
                             cpcaPKTFLG_SEND_V10,
                             id_val_operation_set_binder,
                             *blockNo,
                             SWAP_WORD( paramSize ),
                             userID,
                             password ) ;

  (*blockNo)++ ;
  p += comSize ;

  /*  ͭѥ᡼ */
  *( ( cpcaATTRIB *)p )++ = inParam->Attribute ;
  if ( inParam->Data && ( inParam->Size > 0 ) )
  {
    memcpy( p, inParam->Data, inParam->Size ) ;
  }
  comSize += paramSize ;

  return comSize ;
}




/*============================================================================
//  make_OutputBin
//  ϥӥκ
============================================================================*/
short  make_OutputBin (
  Ptr                   p,
  unsigned char         bindInfo,
  unsigned char         bindNum )
{
  short                 comSize ;
  Ptr                   work ;
  char                  BindName;

  work = p ;

  /* ӥ󥨥ꥢ */
  *( ( unsigned char *)work )++ = bindInfo;
  /* ӥֹʤʣ) */
  *( ( unsigned char *)work )++ = bindNum ;
  /* ӥ̾λ */
  /*  nullʸ */
  BindName = 0;

  memcpy( ( StringPtr )work, &BindName, 1 ) ;
  work += 1 ;
  /* ϥӥ󡢽λӥ */
  *( ( unsigned char *)work )++ = 0 ;
  *( ( unsigned char *)work )++ = 0 ;
  /* μ */
  comSize = work - p ;

  return comSize ;
}






/*============================================================================
//  make_StapleSort
//  ƥץ륽Ȥξ
============================================================================*/
short  make_StapleSort (
  Ptr                   p,
  unsigned short        staple_val )
{
  short                 comSize ;
  Ptr                   work ;
  char                  count;
  char                  str_count;

  work = p ;
  count = 1;
  str_count = 1;

  /* Type:ե˥åˡ */
  *( ( unsigned char *)work )++ = str_count;
  /* Type:ե˥åˡ */
  *( ( unsigned char *)work )++ = id_val_finishing_type_staple_stitching;
  /* Count:ե˥å󥰤ԤĿ */
  *( ( unsigned char *)work )++ = count ;
  /* Locate:ե˥å󥰤Ԥս */
  *( ( unsigned short *)work )++ = staple_val;
  /* μ */
  comSize = work - p ;

  return comSize ;
}





/*============================================================================
//  make_DocumentStart()
============================================================================*/
long  make_DocumentStart(
  Ptr                        p,
  unsigned short             *blockNo,
  cpcaUSER_INFO              *userInfo,
  cpcaIN_DOCUMENTSTART       *inParam )
{
  short          comSize ;
  short          paramSize ;
  unsigned long  userID ;
  unsigned long  password ;

  z_setUserInfo( userInfo, &userID, &password ) ;
  paramSize = 4 ;
  comSize = z_make_header(
                    p,
                    cpcaVERSION_V10,
                    cpcaPKTFLG_SEND_V10,
                    id_val_operation_document_start,
                    *blockNo,
                    SWAP_WORD( paramSize ),
                    userID,
                    password ) ;
  (*blockNo)++ ;
  p += comSize ;

  /* ͭѥ᡼ */
  *( ( cpcaHANDLE *)p )++ = inParam->Object ;
  comSize += paramSize ;

  return comSize ;
}



/*============================================================================
//  make_SetDocument()
============================================================================*/
long  make_SetDocument(
  Ptr                       p,
  unsigned short            *blockNo,
  cpcaUSER_INFO             *userInfo,
  cpcaIN_SETDOCUMENT        *inParam )
{
  short           comSize ;
  short           paramSize ;
  unsigned long   userID ;
  unsigned long   password ;


  z_setUserInfo( userInfo, &userID, &password ) ;
  paramSize = 2 + inParam->Size ;
  comSize = z_make_header(
                    p,
                    cpcaVERSION_V10,
                    cpcaPKTFLG_SEND_V10,
                    id_val_operation_set_document,
                    *blockNo,
                    SWAP_WORD( paramSize ),
                    userID,
                    password ) ;
  (*blockNo)++ ;
  p += comSize ;

  /* ͭѥ᡼ */
  *( ( cpcaATTRIB *)p )++ = inParam->Attribute ;
  if ( inParam->Data && ( inParam->Size > 0 ) )
  {
    memcpy( p, inParam->Data, inParam->Size ) ;
  }
  comSize += paramSize ;

  return comSize ;
}



/*============================================================================
//  make_SetLBPDocument()
============================================================================*/
long    make_SetLBPDocument(
  Ptr                         p,
  unsigned short              *blockNo,
  cpcaUSER_INFO               *userInfo,
  cpcaIN_SETLBPDOCUMENT       *inParam )
{
  short           comSize ;
  short           paramSize ;
  unsigned long   userID ;
  unsigned long   password ;

  z_setUserInfo( userInfo, &userID, &password ) ;
  paramSize = 6 + inParam->Size ;
  comSize = z_make_header(
                    p,
                    cpcaVERSION_V10,
                    cpcaPKTFLG_SEND_V10,
                    id_val_operation_execute_method,
                    *blockNo,
                    SWAP_WORD( paramSize ),
                    userID,
                    password ) ;
  (*blockNo)++ ;
  p += comSize ;              

  /* ͭѥ᡼ */
  *( ( cpcaHANDLE *)p )++     = inParam->Object ;
  *( ( cpcaUSHORT *)p )++     = inParam->Method ;
  if ( inParam->Data && ( inParam->Size > 0 ) )
  {
    memcpy( p, inParam->Data, inParam->Size ) ;
  }
  comSize += paramSize ;

  return comSize ;
}



/*============================================================================
//  make_Send()
============================================================================*/
long  make_Send(
  Ptr                   p,
  unsigned short        *blockNo,
  cpcaUSER_INFO         *userInfo,
  cpcaIN_SEND           *inParam )
{
  unsigned short        comSize ;
  unsigned short        paramSize ;
  unsigned long         userID ;
  unsigned long         password ;


  z_setUserInfo( userInfo, &userID, &password ) ;
  paramSize = inParam->Size + 1 ;
  comSize = z_make_header(
                    p,
                    cpcaVERSION_V10,
                    cpcaPKTFLG_SEND_V10,
                    id_val_operation_send,
                    *blockNo,
                    SWAP_WORD( paramSize ),
                    userID,
                    password ) ;
  (*blockNo)++ ;
  p += comSize ;

  /* ͭѥ᡼ */
  *p++ = inParam->Mode ;
  comSize += 1 ;
  if ( inParam->Data && ( inParam->Size > 0 ) )
  {
    memcpy( p, inParam->Data, inParam->Size ) ;
    comSize += inParam->Size ;
  }

  return comSize ;
}



/*============================================================================
//  make_BinderEnd()
============================================================================*/
long  make_BinderEnd (
  Ptr                    p,
  unsigned short         *blockNo,
  cpcaUSER_INFO          *userInfo,
  cpcaIN_BINDEREND       *inParam )
{
  short            comSize ;
  short            paramSize ;
  unsigned long    userID ;
  unsigned long    password ;

  z_setUserInfo( userInfo, &userID, &password ) ;
  paramSize = 0 ;
  comSize   = z_make_header( p,
                             cpcaVERSION_V10,
                             cpcaPKTFLG_SEND_V10,
                             id_val_operation_binder_end,
                             *blockNo,
                             paramSize,
                             userID,
                             password ) ;

  (*blockNo)++ ;
  p += comSize ;

  /* ͭѥ᡼Ϥʤ */

  return comSize ;
}




/*============================================================================
//  make_DocumentEnd()
============================================================================*/
long  make_DocumentEnd(
  Ptr                       p,
  unsigned short            *blockNo,
  cpcaUSER_INFO             *userInfo,
  cpcaIN_DOCUMENTEND        *inParam )
{

  short             comSize ;
  short             paramSize ;
  unsigned long     userID ;
  unsigned long     password ;

  z_setUserInfo( userInfo, &userID, &password ) ;
  paramSize = 0 ;
  comSize = z_make_header(
                    p,
                    cpcaVERSION_V10,
                    cpcaPKTFLG_SEND_V10,
                    id_val_operation_document_end,
                    *blockNo,
                    paramSize,
                    userID,
                    password ) ;
  (*blockNo)++ ;
  p += comSize ;

  return comSize ;
}





/*============================================================================
//  make_JobEnd()
============================================================================*/
long  make_JobEnd(
  Ptr                   p,
  unsigned short        *blockNo,
  cpcaUSER_INFO         *userInfo,
  cpcaIN_JOBEND         *inParam )
{
  short                 comSize ;
  short                 paramSize ;
  unsigned long         userID ;
  unsigned long         password ;

  z_setUserInfo( userInfo, &userID, &password ) ;
  paramSize = 1 ;
  comSize = z_make_header(
                    p,
                    cpcaVERSION_V10,
                    cpcaPKTFLG_SEND_V10,
                    id_val_operation_job_end,
                    *blockNo,
                    SWAP_WORD( paramSize ),
                    userID,
                    password ) ;
  (*blockNo)++ ;
  p += comSize ;

  /* ͭѥ᡼ */
  *( ( cpcaUBYTE *)p )++ = inParam->Code ;
  comSize += paramSize ;

  return comSize ;
}





/*============================================================================
//  z_make_header
//  Headder packet κ
============================================================================*/
static short  z_make_header (
  Ptr                   p,
  unsigned char         version,
  unsigned char         flag,
  unsigned short        opCode,
  unsigned short        blockNum,
  unsigned short        paramLen,
  unsigned long         userID,
  unsigned long         password )
{
  short                 hedderSize ;
  cpcaSEND_PKT_HDR      *headder ;

  /* initialize */
  headder = ( cpcaSEND_PKT_HDR * )p ;
  headder->HdrID = cpcaPKT_HDR_ID ;
  headder->Ver   = version ;
  headder->Flgs  = flag ;
  headder->Oper  = opCode ;
  headder->BlkNo = blockNum ;
  headder->Len   = paramLen ;
  headder->Uid   = userID ;
  headder->Pwd   = password ;
  headder->Stat  = 0x0000 ;

  hedderSize = sizeof( cpcaSEND_PKT_HDR ) ;

  return hedderSize ;
}


/*============================================================================
//  z_setUserInfo
//  桼IDѥɤꤹ
============================================================================*/
static void  z_setUserInfo(
  cpcaUSER_INFO         *userInfo,
  unsigned long         *userID,
  unsigned long         *password )
{
  *userID = null;
  *password = null;

  if (userInfo) {
    *userID = userInfo->User ;
    *password = userInfo->Password ;
  }
}



















/*============================================================================
//  _makeString()
//  ƬХȤʸǼʸ
============================================================================*/
static void _makeString(
                  char              *SetData,
                  const char        *inData)
{
  SetData[0] = strlen(inData);
  memcpy( SetData + 1, inData, SetData[0] );
}




/*============================================================================
//  _makeItoa()
//  ʸǡѴ
============================================================================*/
static void _makeItoa(
                  char              *SetData,
                  int               value,
                  int               size)
{
  int  strValue = 0 ;
  int  cmpValue = 1 ;
  int  i = 0 ;

  if ( value == 0 ) {
    /* case 0 */
    for (i = 0; i < size; i++) {
      *SetData++ = 0x30 ;
    }
  } else {
    /* case 1 */
    for (i = 1; i < size; i++) {
      cmpValue *= 10;
    }
    for (i = cmpValue; i > 0; i = i / 10) {
      strValue = value / i ;
      *SetData++ = strValue + 0x30 ;
      value -= strValue * i ;
    }
  }
}








/*============================================================================
//  _makeNameOwner()
//  ̾ȥ桼̾Υǡ
============================================================================*/
static void _makeNameOwner( PRasterArg_t      arg )
{
  /* Get Owner Name */
  {
    char *ptr;
    if ((ptr = (char *)getenv("LOGNAME")) != NULL)
    {
      if ((strcmp(ptr, "root") == 0) || (strcmp(ptr, "lp") == 0)) {
        if (arg->PanelMsg[0] != 0) {
          memcpy(arg->UserName, arg->PanelMsg, arg->PanelMsg[0] + 1);
        } else {
          /* -sUserNameꤵƤʤ */
          arg->UserName[0] = strlen(ptr);
          memcpy(arg->UserName + 1, ptr, arg->UserName[0]);
        }
      } else {
        /* 桼"root""lp"ʳ */
        arg->UserName[0] = strlen(ptr);
        memcpy(arg->UserName + 1, ptr, arg->UserName[0]);
      }
    } else {
      arg->UserName[0] = 8;
      memcpy(arg->UserName + 1,"nudefine", 8);
    }
  }
  /* Get Job Name */
  {
    int off_size = 1;
    int size = 0;
    char data[256];

    if (arg->UserName[0] > 8) {
      size = 8;
    } else {
      size = arg->UserName[0];
    }
    memcpy(arg->JobName + off_size, arg->UserName + 1, size);
    off_size += size;
    memcpy(arg->JobName + off_size, "-", 1);
    off_size += 1;

    if (gethostname(data, sizeof(data)) > 0 ) {
      size = 8;
      memcpy(arg->JobName + off_size, "undefine", size);
    } else {
      size = strlen(data);
      if (size > 8) {
        size = 8;
      }
      memcpy(arg->JobName + off_size, data, size);
    }
    off_size += size;
    memcpy(arg->JobName + off_size, "-", 1);
    off_size += 1;

    /* make time table data */
    {
      int     time_value;
      time_t  timer;
      timeStr *tmStr;

      timer = time(NULL);
      tmStr = localtime(&timer);
      /* year */
      time_value = 1900 + tmStr->tm_year;
      _makeItoa(data, time_value, 4);
      memcpy(arg->JobName + off_size, data, 4);
      off_size += 4;
      /* month */
      time_value = 1 + tmStr->tm_mon;
      _makeItoa(data, time_value, 2);
      memcpy(arg->JobName + off_size, data, 2);
      off_size += 2;
      /* day */
      time_value = tmStr->tm_mday;
      _makeItoa(data, time_value, 2);
      memcpy(arg->JobName + off_size, data, 2);
      off_size += 2;
      /* hour */
      time_value = tmStr->tm_hour;
      _makeItoa(data, time_value, 2);
      memcpy(arg->JobName + off_size, data, 2);
      off_size += 2;
      /* min */
      time_value = tmStr->tm_min;
      _makeItoa(data, time_value, 2);
      memcpy(arg->JobName + off_size, data, 2);
      off_size += 2;
      /* sec */
      time_value = tmStr->tm_sec;
      _makeItoa(data, time_value, 2);
      memcpy(arg->JobName + off_size, data, 2);
      off_size += 2;
    }
    arg->JobName[0] = off_size - 1;
  }
}








/*============================================================================
//  _makeDatetime()
//  ա֤Υѥ᡼
============================================================================*/
static void _makeDatetime( cpcaDATETIMES   *DateTime )
{
  time_t          timer;
  timeStr         *tmStr;
  unsigned long   swapData = 0;
  unsigned long   wdayMake[7] = {7, 1, 2, 3, 4, 5, 6};

  timer = time( NULL );
  tmStr = gmtime( &timer );

  swapData  = (unsigned long)( 1900 +   tmStr->tm_year  ) << 20 |
              (unsigned long)(    1 +   tmStr->tm_mon   ) << 16 |
              (unsigned long)(          tmStr->tm_mday  ) << 11 |
              (unsigned long)( wdayMake[tmStr->tm_wday] ) << 8 ;

  DateTime->Date = SWAP_DWORD(swapData);

  swapData  = (unsigned long)(        tmStr->tm_hour ) << 24 |
              (unsigned long)(        tmStr->tm_min  ) << 16 |
              (unsigned long)( 1000 * tmStr->tm_sec  );

  DateTime->Time = SWAP_DWORD(swapData);
}








/*============================================================================
//  cpca_JobStart()
//  ֳϻCPCAѥåȤ
============================================================================*/
unsigned int cpca_JobStart(
  Ptr             outBuf,
  PRasterArg_t    arg)
{
  Ptr                  p;

  char                 Personality[100];
  char                 envVariable[100];
  char                 envValue[100];
  long                 comSize = 0;
  long                 bufOffset = 0;

  p = (Ptr)malloc(255) ;

/* make JobStart command packet */
  comSize = glue_cpcaJobStart(p, 0x01);
  memcpy(outBuf + bufOffset, p, comSize);
  bufOffset += comSize;

/* make SetJob(name & owner) command packet */
  _makeNameOwner(arg);

  comSize = glue_cpcaSetJobString(p, JOB_NAME, arg->JobName);
  memcpy(outBuf + bufOffset, p, comSize);
  bufOffset += comSize;

  comSize = glue_cpcaSetJobString(p, JOB_OWNER, arg->UserName);
  memcpy(outBuf + bufOffset, p, comSize);
  bufOffset += comSize;

/* make SetJob(time) command packet */
  comSize = glue_cpcaSetJobTime(p);
  memcpy(outBuf + bufOffset, p, comSize);
  bufOffset += comSize;

/* make BinderStart command packet */
  comSize = glue_cpcaBinderStart(p);
  memcpy(outBuf + bufOffset, p, comSize);
  bufOffset += comSize;

/* make SetBinder(Collate-copies) command packet */
  /* use sorter ? */
  {
    unsigned short       copy;
    if ((arg->Collate) || (arg->SortMode)) {
      copy = arg->Copies;
    } else {
      copy = 1;
    }
    comSize = glue_cpcaSetBinderCopy( p, SWAP_WORD(copy));
    memcpy(outBuf + bufOffset, p, comSize);
    bufOffset += comSize;
  }

/* make SetBinder(output bin) command packet */
  {
    unsigned char bindInfo = 0;

    if (arg->SortMode != 0) {
      if (arg->SortMode == 1) {
        bindInfo = id_val_output_bin_group;	/* Sort */
      } else {
        bindInfo = id_val_output_bin_stack;	/* Stack */
      }
      comSize = glue_cpcaSetBinderOutputBin(p, bindInfo, 1 );
      memcpy(outBuf + bufOffset, p, comSize);
      bufOffset += comSize;
    } else {
      if (arg->OutputBin != 0) {
        unsigned char bindNum = 0;

        switch (arg->OutputBin) {
          case 1:
            bindInfo = id_val_output_bin_face_down;
            bindNum = 1;
            break;
          case 2:
          case 3:
            bindInfo = id_val_output_bin_face_up;
            bindNum = 1;
            break;
          case 11:
          case 12:
          case 13:
          case 14:
          case 15:
          case 16:
          case 17:
            bindInfo = id_val_output_bin_numbered;
            bindNum = arg->OutputBin - 10;
            break;
        }
        comSize = glue_cpcaSetBinderOutputBin(p, bindInfo, bindNum);
        memcpy(outBuf + bufOffset, p, comSize);
        bufOffset += comSize;
      }
    }
  }

/* make SetBinder(Staple Sort) command packet */
  if (arg->StapleLocate != 0) {
    unsigned short staple_val;
    switch (arg->StapleLocate) {
      case 1:
        staple_val = id_val_finishing_location_top_left;
      break;
      case 2:
        staple_val = id_val_finishing_location_bottom_left;
      break;
      case 3:
        staple_val = id_val_finishing_location_left;
      break;
      default:
        staple_val = 0;
      break;
    }
    if (staple_val) {
      comSize = glue_cpcaSetBinderStapleSort(p, staple_val);
      memcpy(outBuf + bufOffset, p, comSize);
      bufOffset += comSize;
    }
  }

/* make DocumentStart command packet */
  comSize = glue_cpcaDocumentStart(p);
  memcpy(outBuf + bufOffset, p, comSize);
  bufOffset += comSize;

/* make SetDocument(Format) command packet */
  comSize = glue_cpcaSetDocumentFormat(p, id_LIPS);
  memcpy(outBuf + bufOffset, p, comSize);
  bufOffset += comSize;

/* make SetDocument(Resolution) command packet */
  {
    unsigned short reso;

    reso = SWAP_WORD((unsigned short)arg->Resolution);
    comSize = glue_cpcaSetDocumentResolution(p, reso, reso);
    memcpy(outBuf + bufOffset, p, comSize);
    bufOffset += comSize;
  }

/* make SetDocument(Copies) command packet */
  {
    unsigned short copy =0;

    if ((arg->Collate) || (arg->SortMode)) {
      copy = 1;
    } else {
      copy = arg->Copies;
    }
    copy = SWAP_WORD(copy);
    comSize = glue_cpcaSetDocumentCopy(p, copy);
    memcpy(outBuf + bufOffset, p, comSize);
    bufOffset += comSize;
  }

/* make SetDocument(Color Mode) command packet */
  {
    unsigned char type;

    if (arg->ColorMode == 0) {
      type = MONO;
    } else {
      type = COLOR;
    }
    comSize = glue_cpcaSetDocumentColorMode(p, type);
    memcpy(outBuf + bufOffset, p, comSize);
    bufOffset += comSize;
  }

/* make SetDocument(Duplex) command packet */
  if (arg->Duplex != -1) {
    unsigned char plex;
    if (arg->Duplex == 0) {
      plex = SIMPLEX;
    } else {
      plex = DUPLEX;
    }
    comSize = glue_cpcaSetDocumentPlex(p, plex);
    memcpy(outBuf + bufOffset, p, comSize);
    bufOffset += comSize;
  }

/* make SetDocument(Bind Direction) command packet */
  if ((arg->Bind_Dir != 0) || (arg->Bind_Marg > 0)) {
    char direction = 0;
    long shift = 0;

    if (arg->Bind_Dir == 0) {
      arg->Bind_Dir = 1;
    }

    switch (arg->Bind_Dir) {
      case 1:
        direction = 1;
        shift     = arg->Bind_Marg * 10;
        break;
      case 2:
        direction = 1;
        shift     = arg->Bind_Marg * -10;
        break;
      case 3:
        direction = 0;
        shift     = arg->Bind_Marg * 10;
        break;
      case 4:
        direction = 0;
        shift     = arg->Bind_Marg * -10;
        break;
      default:
        break;
    }
    comSize = glue_cpcaSetDocumentBindDir(p, direction);
    memcpy(outBuf + bufOffset, p, comSize);
    bufOffset += comSize;
    comSize = glue_cpcaSetDocumentBindMarg(p, shift);
    memcpy(outBuf + bufOffset, p, comSize);
    bufOffset += comSize;
  }

/* make SetDocument(In Tray Number) command packet */
  if (arg->InTrayNum != -1) {
    char trayName[20];
    trayName[0] = 0;

    switch (arg->InTrayNum) {
      case 0:
        trayName[0] = 4;
        sprintf(trayName + 1, "AUTO");
        break;
      case 1:
      case 10:
        trayName[0] = 4;
        sprintf(trayName + 1, "TRAY");
        break;
      case 2:
      case 11:
        trayName[0] = 9;
        sprintf(trayName + 1, "CASSETTE1");
        break;
      case 3:
      case 12:
        trayName[0] = 9;
        sprintf(trayName + 1, "CASSETTE2");
        break;
      case 4:
      case 13:
        trayName[0] = 9;
        sprintf(trayName + 1, "CASSETTE3");
        break;
      case 14:
        trayName[0] = 9;
        sprintf(trayName + 1, "CASSETTE4");
        break;
      default:
        break;
    }

    if (trayName[0]) {
      comSize = glue_cpcaSetDocumentInputTray( p, trayName );
      memcpy(outBuf + bufOffset, p, comSize);
      bufOffset += comSize;
    }
  }


/* make SetPanelMessage command packet */
  if (arg->PanelMsg[0] == 0) {
    arg->PanelMsg[0] = 11;
    sprintf(arg->PanelMsg + 1, GS_PANEL_MSG_DEFAULT);
  }
  comSize = glue_SetPanelMessage( p, arg->PanelMsg );
  memcpy(outBuf + bufOffset, p, comSize);
  bufOffset += comSize;

/* make SetLBPDocument(TONER-DENISTY) command packet */
  if (arg->TonerDensity != -1) {
    char value[2];

    _makeString( Personality,  "" );
    _makeString( envVariable, "TONER-DENSITY" );
    value[0] = arg->TonerDensity + 0x30;
    value[1] = 0x00;
    _makeString( envValue, value );

    comSize = glue_SetLBPDocument( p, Personality, envVariable, envValue );
    memcpy(outBuf + bufOffset, p, comSize);
    bufOffset += comSize;
  }

/* make SetLBPDocument(TONER-SAVING) command packet */
  if (arg->TonerSaving != -1) {

    _makeString( Personality,  "" );
    _makeString( envVariable, "TONER-SAVING" );
    if (arg->TonerSaving == 1) {
      _makeString( envValue, "ON" );
    } else {
      _makeString( envValue, "OFF" );
    }
    comSize = glue_SetLBPDocument( p, Personality, envVariable, envValue );
    memcpy(outBuf + bufOffset, p, comSize);
    bufOffset += comSize;
  }

/* make SetLBPDocument(SPECIAL-MODE-H) command packet */
  if (arg->SpecialModeH != 0) {

    _makeString( Personality,  "" );
    _makeString( envVariable, "SPECIAL-MODE-H" );
    _makeString( envValue, "ON" );

    comSize = glue_SetLBPDocument( p, Personality, envVariable, envValue );
    memcpy(outBuf + bufOffset, p, comSize);
    bufOffset += comSize;
  }

  /* buffer free */
  free(p);
  return(bufOffset);
}












/*============================================================================
//  cpca_JobEnd()
//  ֽλCPCAѥåȤ
============================================================================*/
unsigned int cpca_JobEnd(
  Ptr             outBuf,
  PRasterArg_t    arg,
  char            isCancel)
{
  Ptr                  p;
  long                 comSize = 0;
  long                 bufOffset = 0;

  p = ( Ptr )malloc( 255 );

/* make DocumentEnd command packet */
  comSize = glue_DocumentEnd( p );
  memcpy(outBuf + bufOffset, p, comSize);
  bufOffset += comSize;

/* make BinderEnd command packet */
  comSize = glue_BinderEnd( p );
  memcpy(outBuf + bufOffset, p, comSize);
  bufOffset += comSize;

/* make SetJob(Impression Count) command packet */
  comSize = glue_cpcaSetJobImpression( p, arg->ImpressVal );
  memcpy(outBuf + bufOffset, p, comSize);
  bufOffset += comSize;

/* make JobEnd command packet */
  comSize = glue_cpcaJobEnd( p, isCancel );
  memcpy(outBuf + bufOffset, p, comSize);
  bufOffset += comSize;

  /* buffer free */
  free ( p ) ;
  return(bufOffset);
}













/*============================================================================
//  cpca_SendData()
//  ǡCPCAѥåȲ
============================================================================*/
unsigned int  cpca_SendData(
  unsigned long   size,
  Ptr             data,
  Ptr             pp,
  int             contFlag )
{
  unsigned short    comSize ;
  cpcaIN_SEND    inParam ;
  cpcaUSER_INFO     userInfo ;
  unsigned short    blockNo ;
  unsigned short    paramSize ;
  unsigned long     userID ;
  unsigned long     password ;
  short             hedderSize ;
  cpcaSEND_PKT_HDR  *headder ;


  if ( ( pp == nil ) || ( data == nil ) || ( size == 0 ) )
  {
    comSize = 0 ;
    return(comSize) ;
  }

  inParam.Mode = id_val_send_mode_raw ;
  inParam.Size = size ;
  inParam.Data = data ;

  userInfo.User = 0x00000000 ;
  userInfo.Password = 0x00000000 ;

  blockNo = 0 ;


  z_setUserInfo( &userInfo, &userID, &password ) ;
  paramSize = inParam.Size + 1 ;

  /* init */
  headder = ( cpcaSEND_PKT_HDR * )pp ;

  headder->HdrID = cpcaPKT_HDR_ID ;
  headder->Ver   = cpcaVERSION_V10 ;
  if (contFlag == 1)
  {
    headder->Flgs  = cpcaPKTFLG_SEND_CONT ;
  } else {
    headder->Flgs  = cpcaPKTFLG_SEND_END ;
  }
  headder->Oper  = id_val_operation_send ;
  headder->BlkNo = blockNo ;
  headder->Len   = SWAP_WORD( paramSize ) ;
  headder->Uid   = userID ;
  headder->Pwd   = password ;
  headder->Stat  = 0x0000 ;

  hedderSize = sizeof( cpcaSEND_PKT_HDR ) ;

  comSize = hedderSize ;

  (blockNo)++  ;
  pp += comSize ;

  /* set param */
  *pp++ = inParam.Mode ;
  comSize += 1 ;
  if ( inParam.Data && ( inParam.Size > 0 ) )
  {
    memcpy( pp, inParam.Data, inParam.Size ) ;
    comSize += inParam.Size ;
  }

  return(comSize) ;
}



