/* Copyright (C) 1998 Sean Gabriel <gabriel@montana.com>
 
This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
 
You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */


#include <gtk/gtk.h>
#include <libicq.h>

#include "contactlist.h"
#include "message.h"
#include "rename.h"

extern GdkPixmap* message_pixmap;
extern GdkPixmap* gray_message_pixmap;
extern GdkPixmap* away_message_pixmap;
extern GdkPixmap* na_message_pixmap;
extern GdkPixmap* dnd_message_pixmap;
extern GdkPixmap* occ_message_pixmap;
extern GdkBitmap* mask;

extern GtkWidget* contact_list;

static int menu_add_id = 0;
static int menu_message_id = 0;
static int menu_url_id = 0;
static int menu_info_id = 0;
static int menu_away_id = 0;
static int menu_rename_id = 0;
static int menu_delete_id = 0;

int gtk_clist_find_row_from_uin(GtkCList* clist, unsigned long uin)
{
  GList *list;
  gpointer data;
  CLInfoPtr info;
  gint n;

  g_return_val_if_fail(clist != NULL, -1);
  g_return_val_if_fail(GTK_IS_CLIST(clist), -1);
  
  if(g_list_length(clist->row_list) == 0)
    return -1;

  for(n = 0, list = clist->row_list; list; n++, list = list->next)
    if(GTK_CLIST_ROW(list) && GTK_CLIST_ROW(list)->data &&
      ((CLInfoPtr)(GTK_CLIST_ROW(list)->data))->uin == uin)
      return n;

  return -1;
}

CLInfoPtr contact_list_find_data_from_uin(ContactList* cl, unsigned long uin)
{
  GtkCList* clist[3];
  GList *list;
  gint m, n;

  g_return_val_if_fail(cl != NULL, (CLInfoPtr)-1);
  g_return_val_if_fail(IS_CONTACT_LIST(cl), (CLInfoPtr)-1);

  clist[0] = GTK_CLIST(cl->online_list);
  clist[1] = GTK_CLIST(cl->offline_list);
  clist[2] = GTK_CLIST(cl->nil_list);

  for(m = 0; m < 3; m++)
    for(n = 0, list = clist[m]->row_list; list; n++, list = list->next)
      if(GTK_CLIST_ROW(list)->data && ((CLInfoPtr)(GTK_CLIST_ROW(list)->data))->uin == uin)
        return (CLInfoPtr)(GTK_CLIST_ROW(list)->data);

  return (CLInfoPtr)(NULL);
}

static void GetInfo(GtkWidget *widget, gpointer data)
{
  GtkWidget* infobox;
  GtkCList* cl;
  CLInfo* info;
  int row;
  gchar* nick;
  
  cl = GTK_CLIST(data);
  if(cl->selection == NULL) return;
        row = (int)(cl->selection->data);
  
  gtk_clist_get_text(cl, row, 1, &nick);
  info = (CLInfo*)gtk_clist_get_row_data(cl, row);
  
  if(info) ICQ_Get_Info(info->uin);
}


static void AddUser(GtkWidget *widget, gpointer data)
{
  GtkCList* cl;
  int row;
  CLInfo* info;
  gchar* nick;
  
  cl = GTK_CLIST(data);
  if(cl->selection == NULL) return;
  row = (int)(cl->selection->data);
  
  gtk_clist_get_text(cl, row, 1, &nick);
  info = (CLInfo*)gtk_clist_get_row_data(cl, row);

  if(info) ICQ_Add_User(info->uin, nick);
}


static void RenameUser(GtkWidget *widget, gpointer data)
{
  GtkWidget* rbox;
  GtkCList* cl;
  int row;
  CLInfo* info;
  gchar* nick;
  
  cl = GTK_CLIST(data);
  if(cl->selection == NULL) return;
  row = (int)(cl->selection->data);
  
  gtk_clist_get_text(cl, row, 1, &nick);
  info = (CLInfo*)gtk_clist_get_row_data(cl, row);

  if(info)
  {
    rbox = rename_box_new(info->uin, nick);
    gtk_widget_show(rbox);
  }
}


static void DeleteUser(GtkWidget *widget, gpointer data)
{
  CLInfo* info;
  GtkCList* cl;  // list of contacts
  int row;  // row of current contact
  gchar* nick;  // nickname of contact
  int nil;
  
  cl = GTK_CLIST(data);
  if(cl->selection == NULL) return;
  row = (int)(cl->selection->data);
  
  gtk_clist_get_text(cl, row, 1, &nick);
  info = (CLInfo*)gtk_clist_get_row_data(cl, row);

  if(!info) return;

  ICQ_Delete_User(info->uin);
  gtk_clist_remove(cl, row);

  if(cl == GTK_CLIST(CONTACT_LIST(contact_list)->nil_list) && cl->rows == 0)
  {
    gtk_widget_hide(CONTACT_LIST(contact_list)->nil_toggle);
    gtk_widget_hide(CONTACT_LIST(contact_list)->nil_scroll_win);
  }
}


static void SendMessage(GtkWidget *widget, gpointer data)
{
  GtkWidget* msgbox;
  GtkCList* cl;
  int row;
  CLInfo* info;
  gchar* nick;

  cl = GTK_CLIST(data);
  if(cl->selection == NULL) return;
  row = (int)(cl->selection->data);

  gtk_clist_get_text(cl, row, 1, &nick);
  info = (CLInfo*)gtk_clist_get_row_data(cl, row);

  if(info)
  {
    msgbox = message_new(SEND_MESSAGE, info->uin, nick);
    gtk_widget_show(msgbox);
  }
}


static void CheckDoubleClick(GtkWidget *widget, gpointer data) 
{
  GtkCList* cl;
  int row;
  gchar* nick;
  CLInfo* info;

  cl = GTK_CLIST(widget);
  if(cl->selection == NULL) return;
  row = (int)(cl->selection->data);
  gtk_clist_get_text(cl, row, 1, &nick);
  
  info = (CLInfo*)gtk_clist_get_row_data(cl, row);
  if(!info) return;

  if(info->events)
  {
    gtk_widget_show(GTK_WIDGET(g_list_first(info->events)->data));
    info->events = g_list_remove_link(info->events, g_list_first(info->events));
/*
    contact_list_update_user(CONTACT_LIST(contact_list), (unsigned long)info->uin, nick, -2);
*/
  }
  else
    SendMessage(NULL, widget);
}


static void SendURL(GtkWidget *widget, gpointer data)
{
  GtkWidget* msgbox;
  GtkCList* cl;
  int row;
  unsigned long uin;
  gchar* nick;

  cl = GTK_CLIST(data);
  if(cl->selection == NULL) return;
  row = (int)(cl->selection->data);

  gtk_clist_get_text(cl, row, 1, &nick);
  uin = ((CLInfoPtr)gtk_clist_get_row_data(cl, row))->uin;
  msgbox = GTK_WIDGET(url_new(SEND_MESSAGE, uin, nick));
  gtk_widget_show(msgbox);
}


static void GetAwayMessage(GtkWidget *widget, gpointer data)
{
  GtkCList* cl;
  int row;
  unsigned long uin;

  cl = GTK_CLIST(data);
  if(cl->selection == NULL) return;
  row = (int)(cl->selection->data);

  uin = ((CLInfoPtr)gtk_clist_get_row_data(cl, row))->uin;
  ICQ_Get_Away_Message(uin);
}


static void ClickItem(GtkWidget* widget, GdkEventButton* bevent, gpointer data)
{
  ContactList* cl = NULL;
  int sel, row, column, status;

  cl = CONTACT_LIST(data);
  g_return_if_fail(cl != NULL);
  g_return_if_fail(IS_CONTACT_LIST(cl));

  if(menu_add_id) gtk_signal_disconnect(GTK_OBJECT(cl->menu_add), menu_add_id);
  if(menu_message_id) gtk_signal_disconnect(GTK_OBJECT(cl->menu_message), menu_message_id);
  if(menu_url_id) gtk_signal_disconnect(GTK_OBJECT(cl->menu_url), menu_url_id);
  if(menu_info_id) gtk_signal_disconnect(GTK_OBJECT(cl->menu_info), menu_info_id);
  if(menu_away_id) gtk_signal_disconnect(GTK_OBJECT(cl->menu_away), menu_away_id);
  if(menu_rename_id) gtk_signal_disconnect(GTK_OBJECT(cl->menu_rename), menu_rename_id);
  if(menu_delete_id) gtk_signal_disconnect(GTK_OBJECT(cl->menu_delete), menu_delete_id);

  menu_add_id = menu_message_id = menu_url_id = menu_info_id = 
    menu_away_id = menu_rename_id = menu_delete_id = 0;

  gtk_clist_get_selection_info(GTK_CLIST(widget), bevent->x, bevent->y, &row, &column);

  gtk_clist_set_selection_mode(GTK_CLIST(widget), GTK_SELECTION_BROWSE);
  gtk_clist_select_row(GTK_CLIST(widget), row, column);

  if(widget != cl->online_list && GTK_CLIST(cl->online_list)->selection)
  {
    sel = (int)(GTK_CLIST(cl->online_list)->selection->data);
    gtk_clist_set_selection_mode(GTK_CLIST(cl->online_list), GTK_SELECTION_SINGLE);
    gtk_clist_unselect_row(GTK_CLIST(cl->online_list), sel, 0);
  }

  if(widget != cl->offline_list && GTK_CLIST(cl->offline_list)->selection)
  {
    sel = (int)(GTK_CLIST(cl->offline_list)->selection->data);
    gtk_clist_set_selection_mode(GTK_CLIST(cl->offline_list), GTK_SELECTION_SINGLE);
    gtk_clist_unselect_row(GTK_CLIST(cl->offline_list), sel, 0);
  }

  if(widget != cl->nil_list && GTK_CLIST(cl->nil_list)->selection)
  {
    sel = (int)(GTK_CLIST(cl->nil_list)->selection->data);
    gtk_clist_set_selection_mode(GTK_CLIST(cl->nil_list), GTK_SELECTION_SINGLE);
    gtk_clist_unselect_row(GTK_CLIST(cl->nil_list), sel, 0);
  }

  if(widget != cl->system && GTK_CLIST(cl->system)->selection)
  {
    sel = (int)(GTK_CLIST(cl->system)->selection->data);
    gtk_clist_set_selection_mode(GTK_CLIST(cl->system), GTK_SELECTION_SINGLE);
    gtk_clist_unselect_row(GTK_CLIST(cl->system), sel, 0);
  }


  if(bevent)
  {
    switch(bevent->type)
    {
      case GDK_BUTTON_PRESS:
      if(bevent->button == 3)
      {
        if(widget == cl->system) break;
        if(GTK_CLIST(widget)->selection == NULL) break;

        row = (int)(GTK_CLIST(widget)->selection->data);
        status = ((CLInfoPtr)gtk_clist_get_row_data(GTK_CLIST(widget), row))->status;

        menu_message_id =
          gtk_signal_connect(GTK_OBJECT(cl->menu_message),
          "activate", GTK_SIGNAL_FUNC(SendMessage), (gpointer)widget);

        menu_url_id =
          gtk_signal_connect(GTK_OBJECT(cl->menu_url), "activate",
          GTK_SIGNAL_FUNC(SendURL), (gpointer)widget);

        menu_info_id =
          gtk_signal_connect(GTK_OBJECT(cl->menu_info), "activate",
          GTK_SIGNAL_FUNC(GetInfo), (gpointer)widget);

        menu_rename_id =
          gtk_signal_connect(GTK_OBJECT(cl->menu_rename), "activate",
          GTK_SIGNAL_FUNC(RenameUser), (gpointer)widget);

        menu_delete_id =    
          gtk_signal_connect(GTK_OBJECT(cl->menu_delete), "activate",
          GTK_SIGNAL_FUNC(DeleteUser), (gpointer)widget);
          
        if(widget == cl->nil_list)
        {
          gtk_widget_show(cl->menu_add);

          menu_add_id =    
            gtk_signal_connect(GTK_OBJECT(cl->menu_add), "activate",
            GTK_SIGNAL_FUNC(AddUser), (gpointer)widget);
        }
        else
          gtk_widget_hide(cl->menu_add);

        if(status == STATUS_AWAY || 
          status == STATUS_NA || 
          status == STATUS_OCCUPIED || 
          status == STATUS_DND)
        {
          gtk_widget_show(cl->menu_away);
          menu_away_id =
            gtk_signal_connect(GTK_OBJECT(cl->menu_away),
            "activate", GTK_SIGNAL_FUNC(GetAwayMessage), (gpointer)widget);
        }
        else
          gtk_widget_hide(cl->menu_away);

        gtk_menu_popup(GTK_MENU(cl->menu), NULL, NULL,
          NULL, NULL, bevent->button, 0);
      }

      break;

      case GDK_2BUTTON_PRESS:
      if(bevent->button == 1 && widget != cl->system)
        /*        SendMessage(NULL, (gpointer)widget); */
        CheckDoubleClick(widget, data);
        
      break;

      default:
      break;
    }
  }
}


void ToggleList(GtkWidget *widget, gpointer data)
{
  if(GTK_TOGGLE_BUTTON(widget)->active)
  {
    /* If control reaches here, the toggle button is down */
    gtk_widget_hide(GTK_WIDGET(data));
  }
  else
  {
    /* If control reaches here, the toggle button is up */
    gtk_widget_show(GTK_WIDGET(data));
  }
}


static void contact_list_init(ContactList* cl)
{
  GtkWidget* separator;
  gchar* str[2];

  gtk_box_set_spacing(GTK_BOX(cl), 2);

  cl->online_list = gtk_clist_new(2);
  cl->offline_list = gtk_clist_new(2);
  cl->nil_list = gtk_clist_new(2);
  cl->system = gtk_clist_new(2);
  
  cl->online_toggle = gtk_toggle_button_new_with_label("Online");
  cl->offline_toggle = gtk_toggle_button_new_with_label("Offline");
  cl->nil_toggle = gtk_toggle_button_new_with_label("Not In List");
  
  cl->online_scroll_win = gtk_scrolled_window_new(NULL, NULL);
  cl->offline_scroll_win = gtk_scrolled_window_new(NULL, NULL);
  cl->nil_scroll_win = gtk_scrolled_window_new(NULL, NULL);

  gtk_clist_set_column_width(GTK_CLIST(cl->online_list), 0, 16);
/*  gtk_clist_set_column_width(GTK_CLIST(cl->online_list), 1, 50); */
  gtk_widget_set_usize(cl->online_list, -2, 75);

  gtk_clist_set_column_width(GTK_CLIST(cl->offline_list), 0, 16);
/*  gtk_clist_set_column_width(GTK_CLIST(cl->offline_list), 1, 50); */
  gtk_widget_set_usize(cl->offline_list, -2, 75);

  gtk_clist_set_column_width(GTK_CLIST(cl->nil_list), 0, 16);
/*  gtk_clist_set_column_width(GTK_CLIST(cl->nil_list), 1, 50); */
  gtk_widget_set_usize(cl->nil_list, -2, 42);

  gtk_widget_set_usize(cl->system, -2, 21);
  gtk_clist_set_column_width(GTK_CLIST(cl->system), 0, 16);
  

  str[0] = NULL;
  str[1] = "System";
  gtk_clist_append(GTK_CLIST(cl->system), str);

  gtk_container_add(GTK_CONTAINER(cl->online_scroll_win), cl->online_list);
        gtk_container_add(GTK_CONTAINER(cl->offline_scroll_win), cl->offline_list);
        gtk_container_add(GTK_CONTAINER(cl->nil_scroll_win), cl->nil_list);

        gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(cl->online_scroll_win),
                GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);

        gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(cl->offline_scroll_win),
                GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);

        gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(cl->nil_scroll_win),
                GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);

  gtk_box_pack_start(GTK_BOX(cl), cl->online_toggle, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(cl), cl->online_scroll_win, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(cl), cl->offline_toggle, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(cl), cl->offline_scroll_win, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(cl), cl->nil_toggle, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(cl), cl->nil_scroll_win, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(cl), cl->system, FALSE, FALSE, 0);

  gtk_widget_show(cl->online_scroll_win);
  gtk_widget_show(cl->offline_scroll_win);
  gtk_widget_show(cl->nil_list);
  gtk_widget_show(cl->online_toggle);
  gtk_widget_show(cl->online_list);
  gtk_widget_show(cl->offline_toggle);
  gtk_widget_show(cl->offline_list);
  gtk_widget_show(cl->system);

  gtk_signal_connect(GTK_OBJECT(cl->online_list), "button_press_event",
    GTK_SIGNAL_FUNC(ClickItem), (gpointer)cl);

  gtk_signal_connect(GTK_OBJECT(cl->offline_list), "button_press_event",
    GTK_SIGNAL_FUNC(ClickItem), (gpointer)cl);

  gtk_signal_connect(GTK_OBJECT(cl->nil_list), "button_press_event",
    GTK_SIGNAL_FUNC(ClickItem), (gpointer)cl);

  gtk_signal_connect(GTK_OBJECT(cl->system), "button_press_event",
    GTK_SIGNAL_FUNC(ClickItem), (gpointer)cl);


  gtk_signal_connect(GTK_OBJECT(cl->online_toggle), "toggled",
    GTK_SIGNAL_FUNC(ToggleList), (gpointer)cl->online_scroll_win);

  gtk_signal_connect(GTK_OBJECT(cl->offline_toggle), "toggled",
    GTK_SIGNAL_FUNC(ToggleList), (gpointer)cl->offline_scroll_win);

  gtk_signal_connect(GTK_OBJECT(cl->nil_toggle), "toggled",
    GTK_SIGNAL_FUNC(ToggleList), (gpointer)cl->nil_scroll_win);
  
  /* Pop-up menu */
  
  cl->menu = gtk_menu_new();
  cl->menu_add = gtk_menu_item_new_with_label("Add to Contact List");
  cl->menu_message = gtk_menu_item_new_with_label("Send Message");
  cl->menu_url = gtk_menu_item_new_with_label("Send URL");
  separator = gtk_menu_item_new();
  cl->menu_info = gtk_menu_item_new_with_label("Get Info");
  cl->menu_away = gtk_menu_item_new_with_label("Get Away Message");
  cl->menu_rename = gtk_menu_item_new_with_label("Rename");
  cl->menu_delete = gtk_menu_item_new_with_label("Delete");

  gtk_menu_append(GTK_MENU(cl->menu), cl->menu_add);
  gtk_menu_append(GTK_MENU(cl->menu), cl->menu_message);
  gtk_menu_append(GTK_MENU(cl->menu), cl->menu_url);
  gtk_menu_append(GTK_MENU(cl->menu), separator);
  gtk_menu_append(GTK_MENU(cl->menu), cl->menu_info);
  gtk_menu_append(GTK_MENU(cl->menu), cl->menu_away);
  gtk_menu_append(GTK_MENU(cl->menu), cl->menu_rename);
  gtk_menu_append(GTK_MENU(cl->menu), cl->menu_delete);
  
  gtk_widget_show(cl->menu_message);
  gtk_widget_show(cl->menu_url);
  gtk_widget_show(separator);
  gtk_widget_show(cl->menu_info);
  gtk_widget_show(cl->menu_rename);
  gtk_widget_show(cl->menu_delete);
}


guint contact_list_get_type()
{
         static guint cl_type = 0;
  
         if(!cl_type)
  {
    GtkTypeInfo cl_info =
    {
      "ContactList",
      sizeof(ContactList),
      sizeof(ContactListClass),
      (GtkClassInitFunc)NULL,
      (GtkObjectInitFunc)contact_list_init,
      (GtkArgSetFunc)NULL,
      (GtkArgGetFunc)NULL,
    };

    cl_type = gtk_type_unique(gtk_vbox_get_type(), &cl_info);
  }

  return cl_type;
}


GtkWidget* contact_list_new()
{
  return gtk_type_new(contact_list_get_type());
}


char* contact_list_get_nickname(ContactList* cl, unsigned long uin)
{
  int row;
  gchar* str;
  
  if((row = gtk_clist_find_row_from_uin(GTK_CLIST(cl->online_list), uin)) != -1)
  {
    gtk_clist_get_text(GTK_CLIST(cl->online_list), row, 1, &str);
    return str;
  }
  
  if((row = gtk_clist_find_row_from_uin(GTK_CLIST(cl->offline_list), uin)) != -1)
  {
    gtk_clist_get_text(GTK_CLIST(cl->offline_list), row, 1, &str);
    return str;
  }

  if((row = gtk_clist_find_row_from_uin(GTK_CLIST(cl->nil_list), uin)) != -1)
  {
    gtk_clist_get_text(GTK_CLIST(cl->nil_list), row, 1, &str);
    return str;
  }
  
  return "Unknown";
}


void contact_list_update_user(ContactList* cl, unsigned long uin, char* nick, long status)
{
  int x, online_row, offline_row, nil_row;
  gchar* cell_text, *str[2];
  CLInfo* tmp;

  tmp = NULL;
  status = status % 65536;

  online_row = gtk_clist_find_row_from_uin(GTK_CLIST(cl->online_list), uin);
  offline_row = gtk_clist_find_row_from_uin(GTK_CLIST(cl->offline_list), uin);
  nil_row = gtk_clist_find_row_from_uin(GTK_CLIST(cl->nil_list), uin);
  
  if(status == -2)
  {
    for(x = 0; x < Num_Contacts; x++)
      if(Contacts[x].uin == uin) break;
      
    if(x == Num_Contacts)
      status = STATUS_NOT_IN_LIST;
    else
      status = Contacts[x].status;
  }

  
  if(status == STATUS_OFFLINE)
  {
    gtk_clist_freeze(GTK_CLIST(cl->online_list));

    if(online_row != -1) 
    {
      tmp = gtk_clist_get_row_data(GTK_CLIST(cl->online_list), online_row);
      gtk_clist_remove(GTK_CLIST(cl->online_list), online_row);
    }
    else if(nil_row != -1)
    {
      tmp = gtk_clist_get_row_data(GTK_CLIST(cl->nil_list), nil_row);
      gtk_clist_remove(GTK_CLIST(cl->nil_list), nil_row);
    }
    
    if(offline_row == -1)
    {
      str[0] = NULL;
      str[1] = nick;
      offline_row = gtk_clist_append(GTK_CLIST(cl->offline_list), str);
      if(!tmp)
      {
        tmp = g_malloc(sizeof(CLInfo));
        tmp->uin = uin;
        tmp->events = NULL;
      }
      gtk_clist_set_row_data(GTK_CLIST(cl->offline_list), offline_row, (gpointer)tmp);
      gtk_clist_set_pixmap(GTK_CLIST(cl->offline_list), offline_row, 0, gray_message_pixmap, mask);
    }

    gtk_clist_get_text(GTK_CLIST(cl->offline_list), offline_row, 1, &cell_text);
    if(strcmp(cell_text, nick))
      gtk_clist_set_text(GTK_CLIST(cl->offline_list), offline_row, 1, nick);

    gtk_clist_thaw(GTK_CLIST(cl->online_list));
  }
  else if(status == STATUS_NOT_IN_LIST)
  {
    gtk_clist_freeze(GTK_CLIST(cl->nil_list));

    gtk_widget_show(cl->nil_toggle);
    gtk_widget_show(cl->nil_scroll_win);

    if(nil_row == -1)
    {
      str[0] = NULL;
      str[1] = nick;
      tmp = g_malloc(sizeof(CLInfo));
      tmp->uin = uin;
      tmp->events = NULL;
      nil_row = gtk_clist_append(GTK_CLIST(cl->nil_list), str);
      gtk_clist_set_row_data(GTK_CLIST(cl->nil_list), nil_row, (gpointer)tmp);
      gtk_clist_set_pixmap(GTK_CLIST(cl->nil_list), nil_row, 0, gray_message_pixmap, mask);
    }

    gtk_clist_get_text(GTK_CLIST(cl->nil_list), nil_row, 1, &cell_text);
    if(strcmp(cell_text, nick))
      gtk_clist_set_text(GTK_CLIST(cl->nil_list), nil_row, 1, nick);

    gtk_clist_thaw(GTK_CLIST(cl->nil_list));
  }
  else
  {
    gtk_clist_freeze(GTK_CLIST(cl->offline_list));

    if(offline_row != -1)
    {
      tmp = gtk_clist_get_row_data(GTK_CLIST(cl->offline_list), offline_row);
      gtk_clist_remove(GTK_CLIST(cl->offline_list), offline_row);
    }
    else if(nil_row != -1)
    {
      tmp = gtk_clist_get_row_data(GTK_CLIST(cl->nil_list), nil_row);
      gtk_clist_remove(GTK_CLIST(cl->nil_list), nil_row);
    }

    if(online_row == -1)
    {
      str[0] = NULL;
      str[1] = nick;
      online_row = gtk_clist_append(GTK_CLIST(cl->online_list), str);
      if(!tmp)
      {
        tmp = g_malloc(sizeof(CLInfo));
        tmp->uin = uin;
        tmp->events = NULL;
      }
      gtk_clist_set_row_data(GTK_CLIST(cl->online_list), online_row, (gpointer)tmp);
      gtk_clist_set_pixmap(GTK_CLIST(cl->online_list), online_row, 0, gray_message_pixmap, mask);
    }
    else
    {
      tmp = gtk_clist_get_row_data(GTK_CLIST(cl->online_list), online_row);
    }

    gtk_clist_get_text(GTK_CLIST(cl->online_list), online_row, 1, &cell_text);
    if(strcmp(cell_text, nick))
       gtk_clist_set_text(GTK_CLIST(cl->online_list), online_row, 1, nick);

    gtk_clist_thaw(GTK_CLIST(cl->offline_list));
  }

  if(tmp) tmp->status = status;

  
  if(GTK_CLIST(cl->nil_list)->rows == 0)
  {
    gtk_widget_hide(cl->nil_toggle);
    gtk_widget_hide(cl->nil_scroll_win);
  }
}

