/* -*- Mode: C; tab-width: 3; indent-tabs-mode: nil; c-basic-offset: 3 -*- */

/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: icon_widget.c,v 1.2 2003/06/13 09:43:16 makeinu Exp $
 */

/*
 * These codes are mostly taken from GTK See.
 * GTK See code Copyright (C) 1998 Hotaru Lee <jkhotaru@mail.sti.com.cn> <hotaru@163.net>
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "icon_widget.h"
#include "gtk2-compat.h"

static void	icon_widget_class_init	 (IconWidgetClass *klass);
static void	icon_widget_init		    (IconWidget *ii);
static gint	icon_widget_expose		 (GtkWidget *widget,
                                      GdkEventExpose *event);
static gint icon_widget_focus_in     (GtkWidget     *widget,
                                      GdkEventFocus *event);
static gint icon_widget_focus_out    (GtkWidget     *widget,
                                      GdkEventFocus *event);
static void	icon_widget_size_request (GtkWidget *widget,
                                      GtkRequisition *requisition);
static void	icon_widget_paint		    (GtkWidget *widget,
                                      GdkRectangle *area);
static void	icon_widget_realize		 (GtkWidget *widget);

#ifdef USE_GTK2
static void icon_widget_finalize     (GObject *object);
#else
static void icon_widget_finalize     (GtkObject *object);
static void	icon_widget_draw		    (GtkWidget *widget,
                                      GdkRectangle *area);
static void icon_widget_draw_focus   (GtkWidget *widget);
#endif

static GtkWidgetClass *parent_class = NULL;


static gint get_max_width    (GtkWidget   *widget,
                              const gchar *str,
                              gint        *lines);
static gint get_string_width (IconWidget  *icon,
                              const gchar *str);


static void
icon_widget_class_init(IconWidgetClass *klass)
{
   GtkObjectClass *object_class;
   GtkWidgetClass *widget_class;
	
   object_class = (GtkObjectClass *) klass;
   widget_class = (GtkWidgetClass*) klass;
	
   OBJECT_CLASS_SET_FINALIZE_FUNC (klass, icon_widget_finalize);
#ifndef USE_GTK2
   widget_class->draw       = icon_widget_draw;
   widget_class->draw_focus = icon_widget_draw_focus;
#endif /* USE_GTK2 */

   widget_class->size_request    = icon_widget_size_request;
   widget_class->realize         = icon_widget_realize;
   widget_class->expose_event    = icon_widget_expose;
   widget_class->focus_in_event  = icon_widget_focus_in;
   widget_class->focus_out_event = icon_widget_focus_out;
}


static void
icon_widget_init(IconWidget *icon)
{
   GTK_WIDGET_SET_FLAGS (icon, GTK_CAN_FOCUS);

   parent_class = gtk_type_class (GTK_TYPE_WIDGET);

   icon->pixmap = NULL;
   icon->mask = NULL;
   icon->width = 18;
   icon->height = 18;
   icon->xpad = 2;
   icon->ypad = 2;
   icon->space = 5;
   icon->label = NULL;
   icon->color_set = FALSE;
   icon->bg_gc = NULL;
}


guint
icon_widget_get_type()
{
   static guint ii_type = 0;

   if (!ii_type) {
      GtkTypeInfo ii_info = {
         "IconWidget",
         sizeof (IconWidget),
         sizeof (IconWidgetClass),
         (GtkClassInitFunc) icon_widget_class_init,
         (GtkObjectInitFunc) icon_widget_init,
         NULL,
         NULL
      };
      ii_type = gtk_type_unique (gtk_widget_get_type(), &ii_info);
   }
   return ii_type;
}


GtkWidget*
icon_widget_new (GdkPixmap *pixmap, GdkBitmap *mask,
                 const gchar *label,
                 guint width, guint height)
{
   IconWidget *icon;
   GtkWidget *widget;

   icon = ICON_WIDGET (gtk_type_new(icon_widget_get_type()));
   widget = GTK_WIDGET (icon);

   icon_widget_set_pixmap (icon, pixmap, mask);

   if (label) {
      icon->label = g_strdup (label);
   }

   icon->lines = 1;

   if (width  > 0) icon->width = width;
   if (height > 0) icon->height = height;

   return GTK_WIDGET (icon);
}


void
icon_widget_set_color(IconWidget *icon, GdkColor *color)
{
   if (color == NULL) {
      icon->color_set = FALSE;
   } else {
      icon->color_set = TRUE;
      icon->color.pixel = color->pixel;
   }
   gtk_widget_queue_draw (GTK_WIDGET (icon));
}


static void
#ifdef USE_GTK2
icon_widget_finalize (GObject *object)
#else
icon_widget_finalize (GtkObject *object)
#endif
{
   IconWidget *icon;

   g_return_if_fail (object);
   g_return_if_fail (IS_ICON_WIDGET (object));

   icon = ICON_WIDGET (object);

   icon_widget_set_pixmap (icon, NULL, NULL);
   if (icon->bg_gc) gdk_gc_destroy (icon->bg_gc);

   if (icon->label)
      g_free (icon->label);
   icon->label = NULL;

   OBJECT_CLASS_FINALIZE_SUPER (parent_class, object);
}


#ifndef USE_GTK2
static void
icon_widget_draw (GtkWidget *widget, GdkRectangle *area)
{
   if (GTK_WIDGET_DRAWABLE (widget)) {
      icon_widget_paint (widget, area);
      gtk_widget_draw_default (widget);
   }
}


static void
icon_widget_draw_focus (GtkWidget *widget)
{
   gtk_widget_draw (widget, NULL);
}
#endif


static void
icon_widget_paint (GtkWidget *widget, GdkRectangle *area)
{
   IconWidget *icon;
   GdkFont *font;
   guint width, height, iwidth, iheight, x, y;

   g_return_if_fail (widget != NULL);
   g_return_if_fail (IS_ICON_WIDGET (widget));

   icon = ICON_WIDGET (widget);

   x = icon->xpad;
   y = icon->ypad;
   iwidth = icon->width;
   iheight = icon->height;

   font = gtk_style_get_font (widget->style);

   if (GTK_WIDGET_DRAWABLE (widget)) {
      /* clear area */
      gtk_style_set_background (widget->style,
                                widget->window,
                                GTK_STATE_NORMAL);
      gdk_window_clear_area(widget->window,
                            area->x, area->y, area->width, area->height);
		
      width = icon->width + icon->space
         + get_string_width(ICON_WIDGET (widget), icon->label);
      height = icon->height;

      /* fill label with bg_gc if needed */
      if (GTK_WIDGET_STATE (widget) == GTK_STATE_SELECTED) {
         gdk_draw_rectangle (widget->window,
                             widget->style->bg_gc[GTK_STATE_SELECTED],
                             TRUE, 0, 0,
                             widget->allocation.width, widget->allocation.height);
      } else if (icon->color_set) {
         gdk_gc_set_foreground (icon->bg_gc, &icon->color);
         gdk_draw_rectangle (widget->window,
                             icon->bg_gc,
                             TRUE, 0, 0,
                             widget->allocation.width, widget->allocation.height);
      }
		
      /* draw pixmap */
      if (icon->pixmap) {
         gdk_window_get_size (icon->pixmap, &iwidth, &iheight);
         if (iheight < icon->height)
            y = (icon->height - iheight) / 2 + icon->ypad;

         if (icon->mask) {
            gdk_gc_set_clip_mask (widget->style->black_gc, icon->mask);
            gdk_gc_set_clip_origin (widget->style->black_gc, x, y);
         }

         gdk_draw_pixmap(widget->window,
                         widget->style->black_gc,
                         icon->pixmap,
                         0, 0,
                         x, y,
                         -1, -1);

         if (icon->mask) {
            gdk_gc_set_clip_mask (widget->style->black_gc, NULL);
            gdk_gc_set_clip_origin (widget->style->black_gc, 0, 0);
         }
      }

      /* draw label */
#ifdef USE_GTK2
      if (icon->label) {
         PangoLayout *layout;
         gint lwidth = 0, lheight = 0, xpos, ypos;
         layout = gtk_widget_create_pango_layout (widget, "012456789");
         pango_layout_set_text (layout, icon->label, -1);
         pango_layout_get_pixel_size (layout, &lwidth, &lheight);
         if (icon->height > lheight) {
            ypos = (icon->height - lheight) / 2;
         } else {
            ypos = 0;
         }
         xpos = icon->width + icon->xpad + icon->space;
         gdk_draw_layout (widget->window,
                          widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
                          xpos, ypos,
                          layout);

         g_object_unref (layout);
      }
#else
      if (icon->label) {
         gchar *startchar, *endchr;
         gint i, len, length, startpos = 0;
         gboolean finish = FALSE;
         gint offset = 0, strheight;

         len = strlen (icon->label);
         startchar = icon->label;
         strheight = gdk_string_height (font,
                                        icon->label)
            * icon->lines;
         if (strheight < icon->height)
            offset = (icon->height - strheight) / 2;

         for (i = 0;; i++) {
            startpos = len - strlen (startchar);
            endchr = strchr (startchar, '\n');
            if (!endchr) {
               endchr = icon->label + len - 1;
               finish = TRUE;
            }
            length = len - startpos - strlen (endchr);
            if (finish) length = length + 1;

            gdk_draw_text (widget->window,
                           font,
                           widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
                           icon->width + icon->xpad + icon->space,
                           font->ascent
                           + (gdk_string_height (font, startchar) + 2) * i
                           + icon->ypad +  offset,
                           startchar,
                           length);

            if (finish) break;
            startchar = endchr + 1;
         }
      }
#endif

      /* if focused */
      if (GTK_WIDGET_HAS_FOCUS (widget)) {
         gtk_paint_focus (widget->style, widget->window,
#ifdef USE_GTK2
                          GTK_WIDGET_STATE (widget),
#endif
                          area, widget, "button",
                          0, 0,
                          widget->allocation.width - 1,
                          widget->allocation.height - 1);
      }
   }
}

static void
icon_widget_realize (GtkWidget *widget)
{
   IconWidget *item;
   GdkWindowAttr attributes;
   gint attributes_mask;

   g_return_if_fail (widget != NULL);
   g_return_if_fail (IS_ICON_WIDGET (widget));

   item = ICON_WIDGET (widget);
   GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

   attributes.window_type = GDK_WINDOW_CHILD;
   attributes.x = widget->allocation.x;
   attributes.y = widget->allocation.y;
   attributes.width = widget->allocation.width;
   attributes.height = widget->allocation.height;
   attributes.wclass = GDK_INPUT_OUTPUT;
   attributes.visual = gtk_widget_get_visual (widget);
   attributes.colormap = gtk_widget_get_colormap (widget);
   attributes.event_mask = gtk_widget_get_events (widget);
   attributes.event_mask |= (GDK_EXPOSURE_MASK
                             | GDK_BUTTON_PRESS_MASK
                             | GDK_BUTTON_RELEASE_MASK
                             | GDK_ENTER_NOTIFY_MASK
                             | GDK_LEAVE_NOTIFY_MASK);
   attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
   widget->window = gdk_window_new (gtk_widget_get_parent_window (widget),
                                    &attributes, attributes_mask);
   gdk_window_set_user_data (widget->window, item);
	
   widget->style = gtk_style_attach (widget->style, widget->window);
   gtk_style_set_background (widget->style, widget->window, GTK_STATE_NORMAL);

   item->bg_gc = gdk_gc_new (widget->window);
}


static gint
icon_widget_expose (GtkWidget *widget, GdkEventExpose *event)
{
   IconWidget *item;
	
   g_return_val_if_fail (widget, FALSE);
   g_return_val_if_fail (IS_ICON_WIDGET(widget), FALSE);
   g_return_val_if_fail (event, FALSE);
	
   if (GTK_WIDGET_DRAWABLE(widget)) {
      item = ICON_WIDGET (widget);
      icon_widget_paint (widget, &event->area);
#ifndef USE_GTK2
      gtk_widget_draw_default (widget);
      gtk_widget_draw_focus (widget);
#endif
   }
   return FALSE;
}


static void
icon_widget_size_request (GtkWidget *widget, GtkRequisition *requisition)
{
   IconWidget *icon;

   g_return_if_fail (widget);
   g_return_if_fail (IS_ICON_WIDGET(widget));

   icon = ICON_WIDGET (widget);

   requisition->width = icon->width + icon->xpad * 2 + icon->space
      + get_max_width (widget, icon->label, &icon->lines);
   requisition->height = icon->height + icon->ypad * 2;
}


static gint
icon_widget_focus_in (GtkWidget     *widget,
                      GdkEventFocus *event)
{
   g_return_val_if_fail (widget != NULL, FALSE);
   g_return_val_if_fail (IS_ICON_WIDGET (widget), FALSE);
   g_return_val_if_fail (event != NULL, FALSE);

   GTK_WIDGET_SET_FLAGS (widget, GTK_HAS_FOCUS);
#ifdef USE_GTK2
   gtk_widget_draw (widget, NULL);
#else
   gtk_widget_draw_focus (widget);
#endif

   return FALSE;
}


static gint
icon_widget_focus_out (GtkWidget     *widget,
                       GdkEventFocus *event)
{
   g_return_val_if_fail (widget != NULL, FALSE);
   g_return_val_if_fail (IS_ICON_WIDGET (widget), FALSE);
   g_return_val_if_fail (event != NULL, FALSE);

   GTK_WIDGET_UNSET_FLAGS (widget, GTK_HAS_FOCUS);
#ifdef USE_GTK2
   gtk_widget_draw (widget, NULL);
#else
   gtk_widget_draw_focus (widget);
#endif

   return FALSE;
}


void
icon_widget_set_pixmap (IconWidget *icon,
                        GdkPixmap  *pixmap,
                        GdkBitmap  *mask)
{
   g_return_if_fail (icon);
   g_return_if_fail (IS_ICON_WIDGET (icon));

   if (icon->pixmap != pixmap) {
      if (icon->pixmap)
         gdk_pixmap_unref (icon->pixmap);
      icon->pixmap = pixmap;
      if (icon->pixmap)
         gdk_pixmap_ref (icon->pixmap);
   }

   if (icon->mask != mask) {
      if (icon->mask)
         gdk_bitmap_unref (icon->mask);
      icon->mask = mask;
      if (icon->mask)
         gdk_pixmap_ref (icon->mask);
   }

   gtk_widget_queue_draw(GTK_WIDGET(icon));
}

void
icon_widget_get_pixmap (IconWidget  *icon,
                        GdkPixmap  **pixmap_ret,
                        GdkBitmap  **mask_ret)
{
   g_return_if_fail (icon);
   g_return_if_fail (IS_ICON_WIDGET (icon));
   g_return_if_fail (pixmap_ret && mask_ret);

   *pixmap_ret = icon->pixmap;
   *mask_ret = icon->mask;
}


static gint
get_max_width (GtkWidget *widget, const gchar *str, gint *lines)
{
   gint i, strwidth, maxwidth = 0;
   gchar **strs;

   g_return_val_if_fail (widget, 0);
   g_return_val_if_fail (lines, 0);

   if (!str) return 0;

   strs = g_strsplit (str, "\n", -1);
   if (!strs) {
      *lines = 1;
      return 0;
   }

   for (i = 0; strs[i]; i++) {
      strwidth = get_string_width (ICON_WIDGET (widget), strs[i]);
      if (maxwidth < strwidth)
         maxwidth = strwidth;
   }
   *lines = i++;

   g_strfreev (strs);

   return maxwidth;
}


#ifdef USE_GTK2
static gint 
get_string_width (IconWidget *icon, const gchar *str)
{
   PangoLayout *layout;
   gint width, height;

   if (!str) return 0;

   layout = gtk_widget_create_pango_layout (GTK_WIDGET (icon), "(NULL)");
   pango_layout_set_text (layout, str, -1);
   pango_layout_get_pixel_size (layout, &width, &height);
   g_object_unref (layout);

   return width;
}
#else
static gint 
get_string_width (IconWidget *icon, const gchar *str)
{
   GdkFont *font = gtk_style_get_font (GTK_WIDGET (icon)->style);
   return gdk_string_width(font, str);
}
#endif
