/*
 * @(#)ArrayList.java	1.10 98/03/18
 *
 * Copyright 1997, 1998 by Sun Microsystems, Inc.,
 * 901 San Antonio Road, Palo Alto, California, 94303, U.S.A.
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of Sun Microsystems, Inc. ("Confidential Information").  You
 * shall not disclose such Confidential Information and shall use
 * it only in accordance with the terms of the license agreement
 * you entered into with Sun.
 */

package java.util;

/**
 * Resizable-array implementation of the List interface.  Implements all
 * optional List operations, and permits all elements, including null.  In
 * addition to implementing the List interface, ArrayList provides methods
 * to manipulate the size of the array that is used internally to store the
 * List.  (ArrayList is roughly equivalent to Vector, except that it is
 * unsynchronized.)
 * <p>
 * The size, isEmpty, get, set, clear, iterator, and listIterator
 * operations run in constant time.  The add() operation runs in constant
 * time unless it causes the ArrayList to exceed its capacity, in which case
 * it runs in linear time.  All of the other operations run in linear time
 * (roughly speaking).  The constant factor is low compared to that for
 * LinkedList.
 * <p>
 * Each ArrayList has a <em>capacity</em> and a <em>capacityIncrement</em>.
 * The <code>capacity</code> is the size of the array used to store the
 * elements in the List. It is always at least as large as the List size; it
 * is usually larger because as components are added to the ArrayList, the
 * ArrayList's storage increases in chunks the size of its
 * capacityIncrement.  (The capacityIncrement does not change over the
 * lifetime of the ArrayList.)  An application can increase the capacity of
 * an ArrayList before inserting a large number of components; this reduces
 * the amount of incremental reallocation.
 * <p>
 * <strong>Note that this implementation is not synchronized.</strong> If
 * multiple threads access an ArrayList concurrently, and at least one of
 * the threads modifies the ArrayList structurally, it <em>must</em> be
 * synchronized externally.  (A structural modification is any operation
 * that adds or deletes one or more elements, or explicitly resizes the
 * backing array; merely setting the value of an element is not a structural
 * modification.)  This is typically accomplished by synchronizing on some
 * object that naturally encapsulates the ArrayList.  If no such object
 * exists, the ArrayList should be "wrapped" using the
 * Collections.synchronizedSet method.  This is best done at creation time,
 * to prevent accidental unsynchronized access to the ArrayList:
 * <pre>
 *	List list = Collections.synchronizedList(new ArrayList(...));
 * </pre>
 * <p>
 * The Iterators returned by ArrayList's iterator and listIterator
 * methods are <em>fail-fast</em>: if the ArrayList is structurally modified
 * at any time after the Iterator is created, in any way except through the
 * Iterator's own remove or add methods, the Iterator will throw a
 * ConcurrentModificationException.  Thus, in the face of concurrent
 * modification, the Iterator fails quickly and cleanly, rather than risking
 * arbitrary, non-deterministic behavior at an undetermined time in the future.
 *
 * @author  Josh Bloch
 * @version 1.10 03/18/98
 * @see	    Collection
 * @see	    List
 * @see	    LinkedList
 * @see	    Vector
 * @see	    Collections.synchronizedList
 * @since JDK1.2
 */

public class ArrayList<A> extends AbstractList<A> implements List<A>, Cloneable,
					            java.io.Serializable {
    /**
     * The array buffer into which the components of the ArrayList are stored.
     * The capacity of the ArrayList is the length of this array buffer.
     */
    private transient A elementData[];

    /**
     * The number of valid components in the ArrayList.
     */
    private int elementCount;

    /**
     * The amount by which the capacity of the ArrayList is automatically
     * incremented when its size becomes greater than its capacity. If
     * the capacity is <code>0</code>, the capacity of the ArrayList is
     * doubled each time it needs to grow.
     */
    private int capacityIncrement;

    /**
     * Constructs an empty ArrayList with the specified initial capacity and
     * capacity increment.
     *
     * @param   initialCapacity     the initial capacity of the ArrayList.
     * @param   capacityIncrement   the amount by which the capacity is
     *                              increased when the ArrayList overflows.
     */
    public ArrayList(int initialCapacity, int capacityIncrement) {
	super();
	this.elementData = new A[initialCapacity];
	this.capacityIncrement = capacityIncrement;
    }

    /**
     * Constructs an empty ArrayList with the specified initial capacity.
     *
     * @param   initialCapacity   the initial capacity of the ArrayList.
     */
    public ArrayList(int initialCapacity) {
	this(initialCapacity, 0);
    }

    /**
     * Constructs an empty ArrayList.
     */
    public ArrayList() {
	this(10);
    }

    /**
     * Constructs an ArrayList containing the elements of the specified
     * Collection, in the order they are returned by the Collection's
     * iterator.  The ArrayList has initial capacity of 110% the size
     * of the specified Collection, and the default capacity increment.
     */
    public ArrayList(Collection<A> c) {
	this((c.size()*110)/100);	// Allow 10% room for growth

	Iterator<A> i = c.iterator();
	while (i.hasNext())
	    elementData[elementCount++] = i.next();
    }

    /**
     * Trims the capacity of this ArrayList to be the ArrayList's current
     * size. An application can use this operation to minimize the
     * storage of an ArrayList.
     */
    public void trimToSize() {
	modCount++;
	int oldCapacity = elementData.length;
	if (elementCount < oldCapacity) {
	    Object oldData[] = elementData;
	    elementData = new A[elementCount];
	    System.arraycopy(oldData, 0, elementData, 0, elementCount);
	}
    }

    /**
     * Increases the capacity of this ArrayList, if necessary, to ensure
     * that it can hold at least the number of components specified by
     * the minimum capacity argument.
     *
     * @param   minCapacity   the desired minimum capacity.
     */
    public void ensureCapacity(int minCapacity) {
	modCount++;
	int oldCapacity = elementData.length;
	if (minCapacity > oldCapacity) {
	    Object oldData[] = elementData;
	    int newCapacity = (capacityIncrement > 0) ?
		(oldCapacity + capacityIncrement) : (oldCapacity * 2);
    	    if (newCapacity < minCapacity) {
		newCapacity = minCapacity;
	    }
	    elementData = new A[newCapacity];
	    System.arraycopy(oldData, 0, elementData, 0, elementCount);
	}
    }

    /**
     * Returns the number of components in this ArrayList.
     *
     * @return  the number of components in this ArrayList.
     */
    public int size() {
	return elementCount;
    }

    /**
     * Tests if this ArrayList has no components.
     *
     * @return  <code>true</code> if this ArrayList has no components;
     *          <code>false</code> otherwise.
     */
    public boolean isEmpty() {
	return elementCount == 0;
    }

    /**
     * Returns true if this ArrayList contains the specified element.
     *
     * @param o element whose presence in this List is to be tested.
     */
    public boolean contains(A elem) {
	return indexOf(elem, 0) >= 0;
    }

    /**
     * Searches for the first occurence of the given argument, testing
     * for equality using the <code>equals</code> method.
     *
     * @param   elem   an object.
     * @return  the index of the first occurrence of the argument in this
     *          ArrayList; returns <code>-1</code> if the object is not found.
     * @see     Object#equals(Object)
     */
    public int indexOf(A elem) {
	return indexOf(elem, 0);
    }

    /**
     * Searches for the first occurence of the given argument, beginning
     * the search at <code>index</code>, and testing for equality using
     * the <code>equals</code> method.
     *
     * @param   elem    an object.
     * @param   index   the index to start searching from.
     * @return  the index of the first occurrence of the object argument in
     *          this ArrayList at position <code>index</code> or later in the
     *          ArrayList; returns <code>-1</code> if the object is not found.
     * @see     Object#equals(Object)
     */
    public int indexOf(A elem, int index) {
	if (elem == null) {
	    for (int i = index ; i < elementCount ; i++)
		if (elementData[i]==null)
		    return i;
	} else {
	    for (int i = index ; i < elementCount ; i++)
		if (elem.equals(elementData[i]))
		    return i;
	}
	return -1;
    }

    /**
     * Returns the index of the last occurrence of the specified object in
     * this ArrayList.
     *
     * @param   elem   the desired component.
     * @return  the index of the last occurrence of the specified object in
     *          this ArrayList; returns -1 if the object is not found.
     */
    public int lastIndexOf(A elem) {
	return lastIndexOf(elem, elementCount-1);
    }

    /**
     * Searches backwards for the specified object, starting from the
     * specified index, and returns an index to it.
     *
     * @param  elem    the desired component.
     * @param  index   the index to start searching from.
     * @return the index of the last occurrence of the specified object in this
     *          ArrayList at position less than index in the ArrayList;
     *          -1 if the object is not found.
     */
    public int lastIndexOf(A elem, int index) {
	if (elem == null) {
	    for (int i = index; i >= 0; i--)
		if (elementData[i]==null)
		    return i;
	} else {
	    for (int i = index; i >= 0; i--)
		if (elem.equals(elementData[i]))
		    return i;
	}
	return -1;
    }

    /**
     * Returns a shallow copy of this ArrayList.  (The elements themselves
     * are not copied.)
     *
     * @return  a clone of this ArrayList.
     */
    public Object clone() {
	try {
	    ArrayList v = (ArrayList)super.clone();
	    v.elementData = new Object[elementCount];
	    System.arraycopy(elementData, 0, v.elementData, 0, elementCount);
	    v.modCount = 0;
	    return v;
	} catch (CloneNotSupportedException e) {
	    // this shouldn't happen, since we are Cloneable
	    throw new InternalError();
	}
    }

    /**
     * Returns an array containing all of the elements in this ArrayList
     * in the correct order.
     */
    public Object[] toArray() {
	Object[] result = new Object[elementCount];
	System.arraycopy(elementData, 0, result, 0, elementCount);
	return result;
    }


    // Positional Access Operations

    /**
     * Returns the element at the specified position in this ArrayList.
     *
     * @param index index of element to return.
     * @exception IndexOutOfBoundsException index is out of range (index
     * 		  &lt; 0 || index &gt;= size()).
     */
    public A get(int index) {
	RangeCheck(index);

	return elementData[index];
    }

    /**
     * Replaces the element at the specified position in this ArrayList with
     * the specified element.
     *
     * @param index index of element to replace.
     * @param element element to be stored at the specified position.
     * @return the element previously at the specified position.
     * @exception IndexOutOfBoundsException index out of range
     *		  (index &lt; 0 || index &gt;= size()).
     */
    public A set(int index, A element) {
	RangeCheck(index);

	A oldValue = elementData[index];
	elementData[index] = element;
	return oldValue;
    }

    /**
     * Appends the specified element to the end of this ArrayList.
     *
     * @param o element to be appended to this ArrayList.
     * @return true (as per the general contract of Collection.add).
     */
    public boolean add(A o) {
	ensureCapacity(elementCount + 1);  // Increments modCount!!
	elementData[elementCount++] = o;
	return true;
    }

    /**
     * Inserts the specified element at the specified position in this
     * ArrayList. Shifts the element currently at that position (if any) and
     * any subsequent elements to the right (adds one to their indices).
     *
     * @param index index at which the specified element is to be inserted.
     * @param element element to be inserted.
     * @exception IndexOutOfBoundsException index is out of range
     *		  (index &lt; 0 || index &gt; size()).
     */
    public void add(int index, A element) {
	RangeCheck(index);

	ensureCapacity(elementCount+1);  // Increments modCount!!
	System.arraycopy(elementData, index, elementData, index + 1,
			 elementCount - index);
	elementData[index] = element;
	elementCount++;
    }

    /**
     * Removes the element at the specified position in this ArrayList.
     * Shifts any subsequent elements to the left (subtracts one from their
     * indices).  Returns the element that was removed from the ArrayList.
     *
     * @exception IndexOutOfBoundsException index out of range (index
     * 		  &lt; 0 || index &gt;= size()).
     * @param index the index of the element to removed.
     */
    public A remove(int index) {
	RangeCheck(index);

	modCount++;
	A oldValue = elementData[index];

	int numMoved = elementCount - index - 1;
	if (numMoved > 0)
	    System.arraycopy(elementData, index+1, elementData, index,
			     numMoved);
	elementData[--elementCount] = null; // Let gc do its work

	return oldValue;
    }

    /**
     * Removes all of the elements from this ArrayList.  The ArrayList will
     * be empty after this call returns, unless it throws an exception.
     *
     * @exception UnsupportedOperationException clear is not supported
     * 		  by this Set.
     */
    public void clear() {
	modCount++;

	// Let gc do its work
	for (int i = 0; i < elementCount; i++)
	    elementData[i] = null;

	elementCount = 0;
    }

    /**
     * Appends all of the elements in the specified Collection to the end of
     * this this ArrayList, in the order that they are returned by the
     * specified Collection's Iterator.  The behavior of this operation is
     * undefined if the specified Collection is modified while the operation
     * is in progress.  (This implies that the behavior of this call is
     * undefined if the the specified Collection is this ArrayList, and this
     * ArrayList is nonempty.)
     *
     * @param index index at which to insert first element
     *			  from the specified collection.
     * @param c elements to be inserted into this ArrayList.
     * @exception IndexOutOfBoundsException index out of range (index
     *		  &lt; 0 || index &gt; size()).
     */
    public boolean addAll(Collection<A> c) {
	modCount++;
	int numNew = c.size();
	ensureCapacity(elementCount + numNew);

	Iterator<A> e = c.iterator();
	for (int i=0; i<numNew; i++)
	    elementData[elementCount++] = e.next();

	return numNew != 0;
    }

    /**
     * Removes from this ArrayList all of the elements whose index is between
     * fromIndex, inclusive and toIndex, exclusive.  Shifts any succeeding
     * elements to the left (reduces their index).
     * This call shortens the ArrayList by (toIndex - fromIndex) elements.  (If
     * toIndex==fromIndex, this operation has no effect.)
     *
     * @param fromIndex index of first element to be removed.
     * @param fromIndex index after last element to be removed.
     * @exception IndexOutOfBoundsException fromIndex or toIndex out of
     *		  range (fromIndex &lt; 0 || fromIndex &gt;= size() || toIndex
     *		  &gt; size() || toIndex &lt; fromIndex).
     */
    public void removeRange(int fromIndex, int toIndex) {
	modCount++;
	if (fromIndex < 0 || fromIndex >= elementCount ||
	    toIndex > elementCount || toIndex < fromIndex)
	    throw new IndexOutOfBoundsException();

	int numMoved = elementCount - toIndex;
	if (numMoved > 0)
	    System.arraycopy(elementData, toIndex, elementData, fromIndex,
			     numMoved);

	// Let gc do its work
	int newElementCount = elementCount - (toIndex-fromIndex);
	while (elementCount != newElementCount)
	    elementData[--elementCount] = null;
    }

    /**
     * Inserts all of the elements in the specified Collection into this
     * ArrayList, starting at the specified position.  Shifts the element
     * currently at that position (if any) and any subsequent elements to
     * the right (increases their indices).  The new elements will appear
     * in the ArrayList in the order that they are returned by the
     * specified Collection's iterator.
     *
     * @param index index at which to insert first element
     *		    from the specified collection.
     * @param c elements to be inserted into this ArrayList.
     * @exception IndexOutOfBoundsException index out of range (index
     *		  &lt; 0 || index &gt; size()).
     */
    public boolean addAll(int index, Collection<A> c) {
	RangeCheck(index);

	int numNew = c.size();
	ensureCapacity(elementCount + numNew);  // Increments modCount!!

	int numMoved = elementCount - index;
	if (numMoved > 0)
	    System.arraycopy(elementData, index, elementData, index + numNew,
			     numMoved);

	Iterator<A> e = c.iterator();
	for (int i=0; i<numNew; i++)
	    elementData[index++] = e.next();

	elementCount += numNew;
	return numNew != 0;
    }

    /**
     * Check if the given index is in range.  If not, throw an appropriate
     * runtime exception.
     */
    private void RangeCheck(int index) {
	if (index >= elementCount || index < 0)
	    throw new IndexOutOfBoundsException(
		"Index: "+index+", Size: "+elementCount);
    }
    /**
     * Save the state of the ArrayList to a stream (i.e., serialize it).
     */
    private synchronized void writeObject(java.io.ObjectOutputStream s)
        throws java.io.IOException{
	// Write out element count, capacity increment, and any hidden stuff
	s.defaultWriteObject();

        // Write out array length
        s.writeInt(elementData.length);

	// Write out all elements in the proper order.
	for (int i=0; i<elementCount; i++)
            s.writeObject(elementData[i]);
    }

    /**
     * Reconstitute the ArrayList from a stream (i.e., deserialize it).
     */
    private synchronized void readObject(java.io.ObjectInputStream s)
        throws java.io.IOException, ClassNotFoundException {
	// Read in element count, capacity increment, and any hidden stuff
	s.defaultReadObject();

        // Read in array length and allocate array
        int arrayLength = s.readInt();
        elementData = new A[arrayLength];

	// Read in all elements in the proper order.
	for (int i=0; i<elementCount; i++)
            elementData[i] = (A)s.readObject();
    }


    int capacity() {
        return elementData.length;
    }

    int capacityIncrement() {
        return capacityIncrement;
    }
}
