/*
 * @(#)Properties.java	1.41 98/03/18
 *
 * Copyright 1995-1998 by Sun Microsystems, Inc.,
 * 901 San Antonio Road, Palo Alto, California, 94303, U.S.A.
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of Sun Microsystems, Inc. ("Confidential Information").  You
 * shall not disclose such Confidential Information and shall use
 * it only in accordance with the terms of the license agreement
 * you entered into with Sun.
 */

package java.util;

import java.io.IOException;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.BufferedWriter;
import java.util.Hashtable;

/**
 * The <code>Properties</code> class represents a persistent set of 
 * properties. The <code>Properties</code> can be saved to a stream 
 * or loaded from a stream. Each key and its corresponding value in 
 * the property list is a string. 
 * <p>
 * A property list can contain another property list as its 
 * "defaults"; this second property list is searched if 
 * the property key is not found in the original property list. 
 *
 * @author  Arthur van Hoff
 * @author  Michael McCloskey
 * @version 1.41, 98/03/18
 * @since   JDK1.0
 */
public
class Properties extends Hashtable {
    /**
     * A property list that contains default values for any keys not 
     * found in this property list. 
     */
    protected Properties defaults;

    /**
     * Creates an empty property list with no default values. 
     */
    public Properties() {
	this(null);
    }

    /**
     * Creates an empty property list with the specified defaults. 
     *
     * @param   defaults   the defaults.
     */
    public Properties(Properties defaults) {
	this.defaults = defaults;
    }

    /**
     * Calls the hashtable method <code>put</code>. Provided for
     * parallelism with the getProperties method. Enforces use of
     * strings for property keys and values.
     * @since    JDK1.2
     */

    public synchronized Object setProperty(String key, String value) {
        return put(key, value);
    }

    /**
     * Maps the specified <code>key</code> to the specified 
     * <code>value</code> in this Properties set. Neither the key nor the 
     * value can be <code>null</code>.
     * <p>
     *
     * @param      key     the Properties key.
     * @param      value   the value.
     * @return     the previous value of the specified key in this hashtable,
     *             or <code>null</code> if it did not have one.
     * @exception  NullPointerException  if the key or value is
     *               <code>null</code>
     * @exception  IllegalArgumentException if the key is not a String
     *                                    or if the value is not a String
     * @see     Object#equals(Object)
     * @see     #get(Object)
     * @since   JDK1.2
     */
    public synchronized Object put(Object key, Object value) {
        if (!(key instanceof String))
            throw new IllegalArgumentException("Property keys must be Strings");
        if (!(value instanceof String))
            throw new IllegalArgumentException("Property values must be Strings");

        return super.put(key, value);
    }

    private static String keyValueSeparators = "=: \t\r\n\f";

    private static String specialSaveChars = "=: \t\r\n\f#!";

    private static String whiteSpaceChars = " \t\r\n\f";

    /**
     * Reads a property list (key and element pairs) from the input stream.
     * <p>
     * Every property occupies one line of the input stream. Each line
     * is terminated by a line terminator (<code>\n</code> or <code>\r</code>
     * or <code>\r\n</code>). Lines from the input stream are processed until
     * end of file is reached on the input stream.
     * <p>
     * A line that contains only whitespace or whose first non-whitespace
     * character is an ASCII <code>#</code> or <code>!</code> is ignored
     * (thus, <code>#</code> or <code>!</code> indicate comment lines).
     * <p>
     * Every line other than a blank line or a comment line describes one
     * property to be added to the table (except that if a line ends with \,
     * then the following line, if it exists, is treated as a continuation
     * line, as described
     * below). The key consists of all the characters in the line starting
     * with the first non-whitespace character and up to, but not including,
     * the first ASCII <code>=</code>, <code>:</code>, or whitespace
     * character. All of the key termination characters may be included in
     * the key by preceding them with a \. 
     * Any whitespace after the key is skipped; if the first non-whitespace
     * character after the key is <code>=</code> or <code>:</code>, then it
     * is ignored and any whitespace characters after it are also skipped.
     * All remaining characters on the line become part of the associated
     * element string. Within the element string, the ASCII
     * escape sequences <code>\t</code>, <code>\n</code>,
     * <code>\r</code>, <code>\\</code>, <code>\"</code>, <code>\'</code>,
     * <code>\ &#32;</code> &#32;(a backslash and a space), and
     * <code>\\u</code><i>xxxx</i> are recognized and converted to single
     * characters. Moreover, if the last character on the line is
     * <code>\</code>, then the next line is treated as a continuation of the
     * current line; the <code>\</code> and line terminator are simply
     * discarded, and any leading whitespace characters on the continuation
     * line are also discarded and are not part of the element string.
     * <p>
     * As an example, each of the following four lines specifies the key
     * <code>"Truth"</code> and the associated element value
     * <code>"Beauty"</code>:
     * <p>
     * <pre>
     * Truth = Beauty
     *	Truth:Beauty
     * Truth			:Beauty
     * </pre>
     * As another example, the following three lines specify a single
     * property:
     * <p>
     * <pre>
     * fruits				apple, banana, pear, \
     *                                  cantaloupe, watermelon, \
     *                                  kiwi, mango
     * </pre>
     * The key is <code>"fruits"</code> and the associated element is:
     * <p>
     * <pre>"apple, banana, pear, cantaloupe, watermelon,kiwi, mango"</pre>
     * Note that a space appears before each <code>\</code> so that a space
     * will appear after each comma in the final result; the <code>\</code>,
     * line terminator, and leading whitespace on the continuation line are
     * merely discarded and are <i>not</i> replaced by one or more other
     * characters.
     * <p>
     * As a third example, the line:
     * <p>
     * <pre>cheeses
     * </pre>
     * specifies that the key is <code>"cheeses"</code> and the associated
     * element is the empty string.<p>
     *
     * @param      in   the input stream.
     * @exception  IOException  if an error occurred when reading from the
     *               input stream.
     */
    public synchronized void load(InputStream inStream) throws IOException {
	
        BufferedReader in = new BufferedReader(new InputStreamReader(inStream, "8859_1"));
	while (true) {
            // Get next line
            String theLine = in.readLine();
            if(theLine == null)
                return;
         
            if (theLine.length() > 0) {
                // Continue lines that end in slashes
                while (theLine.endsWith("\\")) {
                    String nextLine = in.readLine();
                    if(nextLine == null)
                        nextLine = new String("");
                    String loppedLine = theLine.substring(0, theLine.length()-1);
                    // Advance beyond whitespace on new line
                    int startIndex=0;
                    for(startIndex=0; startIndex<nextLine.length(); startIndex++)
                        if (whiteSpaceChars.indexOf(nextLine.charAt(startIndex)) == -1)
                            break;
                    nextLine = nextLine.substring(startIndex,nextLine.length());
                    theLine = new String(loppedLine+nextLine);
                }
                // Skip comments
                char firstChar = theLine.charAt(0);
                if((firstChar != '#') && (firstChar != '!')) {
                    // Find separation between key and value
                    int len = theLine.length();
                    int separatorIndex;
                    for(separatorIndex=0; separatorIndex<len; separatorIndex++) {
                        char currentChar = theLine.charAt(separatorIndex);
                        if (currentChar == '\\')
                            separatorIndex++;
                        else if(keyValueSeparators.indexOf(currentChar) != -1)
                            break;
                    }

                    // Value starts after whitespace and separators
                    int valueIndex;
                    for (valueIndex=separatorIndex+1; valueIndex<len; valueIndex++)
                       if (keyValueSeparators.indexOf(theLine.charAt(valueIndex)) == -1)
                           break;
                    String key = theLine.substring(0, separatorIndex);
                    String value = (separatorIndex < len) ? theLine.substring(valueIndex, len) : "";

                    // Convert then store key and value
                    key = loadConvert(key); 
                    value = loadConvert(value);
                    put(key, value);
                }
            }
	}
    }

    /*
     * convert encoded \\uxxxx to unicode chars
     * and change special saved chars to their original forms
     */
    private String loadConvert (String theString) {
        char aChar;
        int len = theString.length();
        StringBuffer outBuffer = new StringBuffer(len);
  
        for(int x=0; x<len; ) {
            aChar = theString.charAt(x++);
            if (aChar == '\\') {
                aChar = theString.charAt(x++);
                if(aChar == 'u') {
                    // Read the xxxx
                    int value=0;
		    for (int i=0; i<4; i++) {
		        aChar = theString.charAt(x++);
		        switch (aChar) {
		          case '0': case '1': case '2': case '3': case '4':
		          case '5': case '6': case '7': case '8': case '9':
		             value = (value << 4) + aChar - '0';
			     break;
			  case 'a': case 'b': case 'c':
                          case 'd': case 'e': case 'f':
			     value = (value << 4) + 10 + aChar - 'a';
			     break;
			  case 'A': case 'B': case 'C':
                          case 'D': case 'E': case 'F':
			     value = (value << 4) + 10 + aChar - 'A';
			     break;
			  default:
                              throw new IllegalArgumentException(
                                           "Malformed \\uxxxx encoding.");
                        }
                    }
                    outBuffer.append((char)value);
                } else {
                    if (aChar == 't') aChar = '\t';
                    else if (aChar == 'r') aChar = '\r';
                    else if (aChar == 'n') aChar = '\n';
                    else if (aChar == 'f') aChar = '\f';
                    outBuffer.append(aChar);
                }
            } else 
                outBuffer.append(aChar);
        }
        return outBuffer.toString();
    }

    /*
     * Converts unicodes to encoded \\uxxxx
     * and writes out any of the characters in specialSaveChars
     * with a preceding slash
     */
    private String saveConvert(String theString) {
        char aChar;
        int len = theString.length();
        StringBuffer outBuffer = new StringBuffer(len*2);

        for(int x=0; x<len; ) {
            aChar = theString.charAt(x++);
            switch(aChar) {
                case '\\':outBuffer.append('\\'); outBuffer.append('\\');
                          continue;
                case '\t':outBuffer.append('\\'); outBuffer.append('t');
                          continue;
                case '\n':outBuffer.append('\\'); outBuffer.append('n');
                          continue;
                case '\r':outBuffer.append('\\'); outBuffer.append('r');
                          continue;
                case '\f':outBuffer.append('\\'); outBuffer.append('f');
                          continue;
                default:
                    if ((aChar < 20) || (aChar > 127)) {
                        outBuffer.append('\\');
                        outBuffer.append('u');
                        outBuffer.append(toHex((aChar >> 12) & 0xF));
                        outBuffer.append(toHex((aChar >> 8) & 0xF));
                        outBuffer.append(toHex((aChar >> 4) & 0xF));
                        outBuffer.append(toHex((aChar >> 0) & 0xF));
                    }
                    else {
                        if (specialSaveChars.indexOf(aChar) != -1)
                            outBuffer.append('\\');      
                        outBuffer.append(aChar);
                    }
            }
        }
        return outBuffer.toString();
    }

    /**
     * Writes this property list (key and element pairs) in this 
     * <code>Properties</code> table to the output stream in a format suitable
     * for loading into a <code>Properties</code> table using the
     * <code>load</code> method.
     * <p>
     * Properties from the defaults table of this <code>Properties</code>
     * table (if any) are <i>not</i> written out by this method.
     * <p>
     * If the header argument is not null, then an ASCII <code>#</code>
     * character, the header string, and a newline are first written to the
     * output stream. Thus, the <code>header</code> can serve as an
     * identifying comment.
     * <p>
     * Next, a comment line is always written, consisting of an ASCII
     * <code>#</code> character, the current date and time (as if produced
     * by the <code>toString</code> method of <code>Date</code> for the
     * current time), and a newline.
     * <p>
     * Then every entry in this <code>Properties</code> table is written out,
     * one per line. For each entry the key string is written, then an ASCII
     * <code>=</code>, then the associated element string. Each character of
     * the element string is examined to see whether it should be rendered as
     * an escape sequence. The ASCII characters <code>\</code>, tab, newline,
     * and carriage return are written as <code>\\</code>, <code>\t</code>,
     * <code>\n</code>, and <code>\r</code>, respectively. Characters less
     * than <code>\u0020</code> and characters greater than
     * <code>\u007E</code> are written as <code>\\u</code><i>xxxx</i> for
     * the appropriate hexadecimal value <i>xxxx</i>. Space characters, but
     * not embedded or trailing space characters, are written with a preceding
     * <code>\</code>. The key and value characters <code>#</code>,
     * <code>!</code>, <code>=</code>, and <code>:</code> are written with a
     * preceding slash to ensure that they are properly loaded.
     * <p>
     * After the entries have been written, the output stream is flushed.  The
     * output stream remains open after this method returns.
     *
     * @param   out      an output stream.
     * @param   header   a description of the property list.
     */
    public synchronized void save(OutputStream out, String header)  {
        BufferedWriter awriter;
        try {
            awriter = new BufferedWriter(
                                 new OutputStreamWriter(out, "8859_1"));
            if (header != null) 
                awriter.write('#' + header + '\n');
           
            awriter.write('#' + new Date().toString() + '\n');

            for (Enumeration e = keys(); e.hasMoreElements();) {
                String key = (String)e.nextElement();
                String val = (String)get(key);
                key = saveConvert(key);
                val = saveConvert(val);
                awriter.write(key + '=' + val + '\n');
            }
            awriter.flush();
        } catch (IOException e) {
        }
    }

    /**
     * Searches for the property with the specified key in this property list.
     * If the key is not found in this property list, the default property list,
     * and its defaults, recursively, are then checked. The method returns
     * <code>null</code> if the property is not found.
     *
     * @param   key   the property key.
     * @return  the value in this property list with the specified key value.
     * @see     java.util.Properties#defaults
     */
    public String getProperty(String key) {
	Object oval = super.get(key);
	String sval = (oval instanceof String) ? (String)oval : null;
	return ((sval == null) && (defaults != null)) ? defaults.getProperty(key) : sval;
    }

    /**
     * Searches for the property with the specified key in this property list.
     * If the key is not found in this property list, the default property list,
     * and its defaults, recursively, are then checked. The method returns the
     * default value argument if the property is not found.
     *
     * @param   key            the hashtable key.
     * @param   defaultValue   a default value.
     *
     * @return  the value in this property list with the specified key value.
     * @see     java.util.Properties#defaults
     */
    public String getProperty(String key, String defaultValue) {
	String val = getProperty(key);
	return (val == null) ? defaultValue : val;
    }

    /**
     * Returns an enumeration of all the keys in this property list, including
     * the keys in the default property list.
     *
     * @return  an enumeration of all the keys in this property list, including
     *          the keys in the default property list.
     * @see     java.util.Enumeration
     * @see     java.util.Properties#defaults
     */
    public Enumeration propertyNames() {
	Hashtable h = new Hashtable();
	enumerate(h);
	return h.keys();
    }

    /**
     * Prints this property list out to the specified output stream. 
     * This method is useful for debugging. 
     *
     * @param   out   an output stream.
     */
    public void list(PrintStream out) {
	out.println("-- listing properties --");
	Hashtable h = new Hashtable();
	enumerate(h);
	for (Enumeration e = h.keys() ; e.hasMoreElements() ;) {
	    String key = (String)e.nextElement();
	    String val = (String)h.get(key);
	    if (val.length() > 40) {
                val = val.substring(0, 37) + "...";
	    }
	    out.println(key + "=" + val);
	}
    }

    /**
     * Prints this property list out to the specified output stream. 
     * This method is useful for debugging. 
     *
     * @param   out   an output stream.
     * @since   JDK1.1
     */
    /*
     * Rather than use an anonymous inner class to share common code, this
     * method is duplicated in order to ensure that a non-1.1 compiler can
     * compile this file.
     */
    public void list(PrintWriter out) {
	out.println("-- listing properties --");
	Hashtable h = new Hashtable();
	enumerate(h);
	for (Enumeration e = h.keys() ; e.hasMoreElements() ;) {
	    String key = (String)e.nextElement();
	    String val = (String)h.get(key);
	    if (val.length() > 40) {
		val = val.substring(0, 37) + "...";
	    }
	    out.println(key + "=" + val);
	}
    }

    /**
     * Enumerates all key/value pairs in the specified hastable.
     * @param h the hashtable
     */
    private synchronized void enumerate(Hashtable h) {
	if (defaults != null) {
	    defaults.enumerate(h);
	}
	for (Enumeration e = keys() ; e.hasMoreElements() ;) {
	    String key = (String)e.nextElement();
	    h.put(key, get(key));
	}
    }

    /**
     * Convert a nibble to a hex character
     * @param	nibble	the nibble to convert.
     */
    private static char toHex(int nibble) {
	return hexDigit[(nibble & 0xF)];
    }

    /** A table of hex digits */
    private static char[] hexDigit = {
	'0','1','2','3','4','5','6','7','8','9','A','B','C','D','E','F'
    };
}
