/*
 * @(#)PropertyPermission.java	1.7 98/03/18
 *
 * Copyright 1997 by Sun Microsystems, Inc.,
 * 901 San Antonio Road, Palo Alto, California, 94303, U.S.A.
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of Sun Microsystems, Inc. ("Confidential Information").  You
 * shall not disclose such Confidential Information and shall use
 * it only in accordance with the terms of the license agreement
 * you entered into with Sun.
 */
 
package java.util;

import java.io.Serializable;
import java.io.IOException;
import java.security.*;

/**
 * This class is for property permissions. 
 * 
 * <P>
 * The name is the name of the property ("java.home",
 * "os.name", etc). The naming
 * convention follows the  hierarchical property naming convention.
 * Also, an asterisk 
 * may appear at the end of the name, following a ".", or by itself, to 
 * signify a wildcard match. For example: "java.*" or "*" is valid, 
 * "*java" or "a*b" is not valid.
 * <P>
 * <P>
 * The actions to be granted are passed to the constructor in a string containing 
 * a list of zero or more comma-separated keywords. The possible keywords are
 * "read" and "write". Their meaning is defined as follows:
 * <P>
 * <DL> 
 *    <DT> read 
 *    <DD> read permission. Allows <code>System.getProperty</code> to
 *         be called.
 *    <DT> write
 *    <DD> write permission. Allows <code>System.setProperty</code> to
 *         be called. 
 * </DL>
 * <P>
 * The actions string is converted to lowercase before processing.
 * <P>
 * @see java.security.BasicPermission
 * @see java.security.Permission
 * @see java.security.Permissions
 * @see java.security.PermissionCollection
 * @see java.lang.SecurityManager
 *
 * @version 1.7 98/03/18
 *
 * @author Roland Schemers
 */

public final class PropertyPermission extends BasicPermission {

    /** use serialVersionUID from JDK 1.2 for interoperability */
    private static final long serialVersionUID = -2348415693254064675L;

    /**
     * Read action.
     */
    private final static int READ    = 0x1;

    /**
     * Write action.
     */
    private final static int WRITE   = 0x2;
    /**
     * All actions (read,write);
     */
    private final static int ALL     = READ|WRITE;
    /**
     * No actions.
     */
    private final static int NONE    = 0x0;

    // the actions mask
    private int mask;

    // the actions string. Left null as long as possible, then
    // created and re-used in the getAction function.

    private String actions;

    /**
     * initialize a PropertyPermission object. Common to all constructors.
     * Also called during de-serialization.
     *
     * @param mask the actions mask to use.
     *
     */

    private void init(int mask) 
    {

	if ((mask & ALL) != mask) 
		throw new IllegalArgumentException("invalid actions mask");

	if (mask == NONE) 
		throw new IllegalArgumentException("invalid actions mask");

	if (getName() == null) 
		throw new IllegalArgumentException("name can't be null");

	this.mask = mask;
    }

    /**
     * Creates a new PropertyPermission object with the specified name.
     * The name is the name of the system property, and 
     * <i>actions</i> contains a comma-separated list of the
     * desired actions granted on the property. Possible actions are
     * "read" and "write".
     *
     * @param name the name of the PropertyPermission.
     * @param actions the actions string.
     */

    public PropertyPermission(String name, String actions) 
    {
	super(name,actions);
	init(getMask(actions));
    }

    /**
     * Checks if this PropertyPermission object "implies" the specified
     * permission.
     * <P>
     * More specifically, this method returns true if:<p>
     * <ul>
     * <li> <i>p</i> is an instanceof PropertyPermission,<p>
     * <li> <i>p</i>'s actions are a proper subset of this
     * object's actions, and <p>
     * <li> <i>p</i>'s name is implied by this object's
     *      name. For example, "java.*" implies "java.home".
     * </ul>
     * @param p the permission to check against.
     *
     * @return true if the specified permission is implied by this object,
     * false if not.
     */
    public boolean implies(Permission p) {
	if (!(p instanceof PropertyPermission))
	    return false;

	PropertyPermission that = (PropertyPermission) p;

	// we get the effective mask. i.e., the "and" of this and that.
	// They must be equal to that.mask for implies to return true.

	return ((this.mask & that.mask) == that.mask) && super.implies(that);
    }


    /**
     * Checks two PropertyPermission objects for equality. Checks that <i>obj</i> is
     * a PropertyPermission, and has the same name and actions as this object.
     * <P>
     * @param obj the object we are testing for equality with this object.
     * @return true if obj is a PropertyPermission, and has the same name and
     * actions as this PropertyPermission object.
     */
    public boolean equals(Object obj) {
	if (obj == this)
	    return true;

	if (! (obj instanceof PropertyPermission))
	    return false;

	PropertyPermission that = (PropertyPermission) obj;

	return (this.mask == that.mask) && 
	    (this.getName().equals(that.getName()));
    }

    /**
     * Returns the hash code value for this object.
     * The hash code used is the hash code of this permissions name, that is, 
     * <code>getName().hashCode()</code>, where <code>getName</code> is
     * from the Permission superclass.
     * 
     * @return a hash code value for this object.
     */

    public int hashCode() {
	return this.getName().hashCode();
    }


    /**
     * Converts an actions String to an actions mask.
     *
     * @param action the action string.
     * @return the actions mask.
     */

    private static int getMask(String actions) {

	int mask = NONE;

	if (actions == null) {
	    return mask;
	}

	actions = actions.toLowerCase();
        StringTokenizer st = new StringTokenizer(actions,",");
        while (st.hasMoreTokens()) {
	    String token = st.nextToken().trim();
	    if (token.equals("read"))
		mask |= READ;
	    else if (token.equals("write"))
		mask |= WRITE;
	    else 
              throw new IllegalArgumentException("invalid permission: "+token);
        }
	return mask;
    }


    /**
     * Return the canonical string representation of the actions.
     * Always returns present actions in the following order: 
     * read, write.
     *
     * @return the canonical string representation of the actions.
     */
    private static String getActions(int mask)
    {
	StringBuffer sb = new StringBuffer();
        boolean comma = false;

	if ((mask & READ) == READ) {
	    comma = true;
	    sb.append("read");
	}

	if ((mask & WRITE) == WRITE) {
	    if (comma) sb.append(',');
    	    else comma = true;
	    sb.append("write");
	}
	return sb.toString();
    }

    /**
     * Returns the "canonical string representation" of the actions.
     * That is, this method always returns present actions in the following order: 
     * read, write. For example, if this PropertyPermission object
     * allows both write and read actions, a call to <code>getActions</code>
     * will return the string "read,write".
     *
     * @return the canonical string representation of the actions.
     */
    public String getActions()
    {
	if (actions == null)
	    actions = getActions(this.mask);

	return actions;
    }

    /**
     * Return the current action mask.
     * Used by the PropertyPermissionCollection
     *
     * @return the actions mask.
     */

    int getMask() {
	return mask;
    }

    /**
     * Returns a new PermissionCollection object for storing
     * PropertyPermission objects.
     * <p>
     *
     * @return a new PermissionCollection object suitable for storing 
     * PropertyPermissions.
     */

    public PermissionCollection newPermissionCollection() {
	return new PropertyPermissionCollection();
    }

    /**
     * WriteObject is called to save the state of the PropertyPermission 
     * to a stream. Only the mask is serialized since we want to
     * recalculate the other values when the contents are restored.
     */

    private synchronized void writeObject(java.io.ObjectOutputStream s)
        throws IOException
    {
	// Write out the mask. The superclass takes care of the name
	s.writeInt(mask);
    }

    /**
     * readObject is called to restore the state of the PropertyPermission from
     * a stream.
     */
    private synchronized void readObject(java.io.ObjectInputStream s)
         throws IOException, ClassNotFoundException
    {
	// Read in the mask, then restore everything else by calling init.
	mask = s.readInt();
	init(mask);
    }
}

/**
 * A PropertyPermissionCollection stores a set of PropertyPermission
 * permissions. 
 * 
 * @see java.security.Permission
 * @see java.security.Permissions
 * @see java.security.PermissionCollection
 *
 * @version 1.43 97/09/09
 *
 * @author Roland Schemers
 */

final class PropertyPermissionCollection extends PermissionCollection
implements Serializable 
{
    /** use serialVersionUID from JDK 1.2 for interoperability */
    private static final long serialVersionUID = 7015263904581634791L;


    private Hashtable permissions;
    private boolean all_allowed; // true if "*" is in the collection

    /**
     * Create an empty PropertyPermissions object.
     *
     */

    public PropertyPermissionCollection() {
	permissions = new Hashtable();
	all_allowed = false;
    }

    /**
     * Adds a permission to the PropertyPermissions. The key for the hash is
     * the name.
     *
     * @param permission the Permission object to add.
     */

    public void add(Permission permission)
    {
	if (! (permission instanceof PropertyPermission))
	    throw new IllegalArgumentException("invalid permission: "+
					       permission);
	PropertyPermission pp = (PropertyPermission) permission;

	permissions.put(pp.getName(), permission);
        if (!all_allowed) {
	    if (pp.getName().equals("*"))
		all_allowed = true;
	}
    }

    /**
     * Check and see if this set of permissions implies the permissions 
     * expressed in "permission".
     *
     * @param p the Permission object to compare
     *
     * @return true if "permission" is a proper subset of a permission in 
     * the set, false if not.
     */

    public boolean implies(Permission permission) 
    {
	if (! (permission instanceof PropertyPermission))
   		return false;

	PropertyPermission pp = (PropertyPermission) permission;
	PropertyPermission x;

	int desired = pp.getMask();
	int effective = 0;

	// short circuit if the "*" Permission was added
	if (all_allowed) {
	    x = (PropertyPermission) permissions.get("*");
	    if (x != null) {
		effective |= x.getMask();
		if ((effective & desired) == desired)
		    return true;
	    }
	}

	// strategy:
	// Check for full match first. Then work our way up the
	// name looking for matches on a.b.*

	String name = pp.getName();
	//System.out.println("check "+name);

	x = (PropertyPermission) permissions.get(name);

	if (x != null) { 
	    // we have a direct hit!
	    effective |= x.getMask();
	    if ((effective & desired) == desired)
		return true;
	}

	// work our way up the tree...
	int last, offset;

	offset = name.length()-1;

	while ((last = name.lastIndexOf(".", offset)) != -1) {

	    name = name.substring(0, last+1) + "*";
	    //System.out.println("check "+name);
	    x = (PropertyPermission) permissions.get(name);

	    if (x != null) { 
		effective |= x.getMask();
		if ((effective & desired) == desired)
		    return true;
	    }
	    offset = last -1;
	}

	// we don't have to check for "*" as it was already checked
	// at the top (all_allowed), so we just return false
	return false;
    }

    /**
     * Returns an enumeration of all the PropertyPermission objects in the 
     * container.
     *
     * @return an enumeration of all the PropertyPermission objects.
     */

    public Enumeration elements()
    {
	return permissions.elements();
    }
}


