/*
 * @(#)TreeSet.java	1.7 98/03/18
 *
 * Copyright 1998 by Sun Microsystems, Inc.,
 * 901 San Antonio Road, Palo Alto, California, 94303, U.S.A.
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of Sun Microsystems, Inc. ("Confidential Information").  You
 * shall not disclose such Confidential Information and shall use
 * it only in accordance with the terms of the license agreement
 * you entered into with Sun.
 */

package java.util;

/**
 * This class implements the Set interface, backed by a TreeMap.  This class
 * guarantees that the Map will be in ascending key order, sorted according
 * to the <i>natural sort method</i> for the key Class (see Comparable), or
 * by the Comparator provided at TreeSet creation time, depending on which
 * constructor is used.  Note that this ordering must be <em>total</em>
 * in order for the Tree to function properly.  (A total ordering is
 * an ordering for which a.compareTo(b)==0 implies that a.equals(b).)
 * <p>
 * This implementation provides guaranteed log(n) time cost for the basic
 * operations (add, remove and contains).
 * <p>
 * <strong>Note that this implementation is not synchronized.</strong> If
 * multiple threads access a TreeSet concurrently, and at least one of the
 * threads modifies the TreeSet, it <em>must</em> be synchronized externally.
 * This is typically accomplished by synchronizing on some object that
 * naturally encapsulates the TreeSet.  If no such object exists, the TreeSet
 * should be "wrapped" using the Collections.synchronizedSet method.  This is
 * best done at creation time, to prevent accidental unsynchronized access to
 * the TreeSet:
 * <pre>
 *	Set s = Collections.synchronizedSet(new TreeSet(...));
 * </pre>
 * <p>
 * The Iterators returned by TreeSet's iterator method are
 * <em>fail-fast</em>: if the HashSet is modified at any time after the
 * Iterator is created, in any way except through the Iterator's own remove
 * method, the Iterator will throw a ConcurrentModificationException.  Thus,
 * in the face of concurrent modification, the Iterator fails quickly and
 * cleanly, rather than risking arbitrary, non-deterministic behavior at an
 * undetermined time in the future.
 *
 * @author  Josh Bloch
 * @version 1.7 03/18/98
 * @see	    Collection
 * @see	    Set
 * @see	    HashSet
 * @see	    ArraySet
 * @see     Comparable
 * @see     Comparator
 * @see	    Collections.synchronizedSet
 * @see	    TreeMap
 * @since JDK1.2
 */

public class TreeSet<A> extends AbstractSet<A>
		     implements SortedSet<A>, Cloneable, java.io.Serializable
{
    private transient SortedMap<A,Object> m;	 // The backing Map
    private transient Set<A>      	 keySet; // The keySet view of the backing Map

    // Dummy value to associate with an Object in the backing Map
    private static final Object PRESENT = new Object();

    /**
     * Constructs a TreeSet backed by the given SortedMap.
     */
    private TreeSet(SortedMap<A,Object> m) {
        this.m = m;
        keySet = m.keySet();
    }

    /**
     * Constructs a new, empty TreeSet, sorted according to the elements'
     * natural sort method.  All elements inserted into the TreeSet must
     * implement the Comparable interface.  Furthermore, all such elements
     * must be <i>mutually comparable</i>: e1.compareTo(e2) must not throw a
     * typeMismatchException for any elements e1 and e2 in the TreeSet.  If
     * the user attempts to add an element to the TreeSet that violates this
     * constraint (for example, the user attempts to add a String element to
     * a TreeSet whose elements are Integers), the add(Object) call will throw
     * a ClassCastException.
     * @see Comparable
     */
    public TreeSet() {
	this(new TreeMap<A,Object>());
    }

    /**
     * Constructs a new, empty TreeSet, sorted according to the given
     * comparator.  All elements inserted into the TreeSet must be <i>mutually
     * comparable</i> by the given comparator: comparator.compare(e1, e2)
     * must not throw a typeMismatchException for any elements e1 and e2
     * in the TreeSet.  If the user attempts to add an element to the
     * TreeSet that violates this constraint, the add(Object) call will throw
     * a ClassCastException.
     */
    public TreeSet(Comparator<A> c) {
	this(new TreeMap<A,Object>(c));
    }

    /**
     * Constructs a new TreeSet containing the elements in the specified
     * Collection, sorted according to the elements' <i>natural sort
     * method</i>.  All keys inserted into the TreeSet must implement the
     * Comparable interface.  Furthermore, all such keys must be <i>mutually
     * comparable</i>: k1.compareTo(k2) must not throw a typeMismatchException
     * for any elements k1 and k2 in the TreeSet.
     *
     * @exception ClassCastException the keys in t are not Comparable, or
     *		  are not mutually comparable.
     */
    public TreeSet(Collection<A> c) {
        this();
        addAll(c);
    }

    /**
     * Constructs a new TreeSet containing the same elements as the given
     * SortedSet, sorted according to the same ordering.
     */
    public TreeSet(SortedSet<A> s) {
        this(s.comparator());
	addAll(s);  // ***COULD BE LINEAR INSTEAD OF N*LOG(N)***
    }

    /**
     * Returns an Iterator over the elements in this TreeSet.  The elements
     * are returned in ascending order.
     */
    public Iterator<A> iterator() {
	return keySet.iterator();
    }

    /**
     * Returns the number of elements in this TreeSet (its cardinality).
     */
    public int size() {
	return m.size();
    }

    /**
     * Returns true if this TreeSet contains no elements.
     */
    public boolean isEmpty() {
	return m.isEmpty();
    }

    /**
     * Returns true if this TreeSet contains the specified element.
     */
    public boolean contains(A o) {
	return m.containsKey(o);
    }

    /**
     * Adds the specified element to this Set if it is not already present.
     *
     * @param o element to be added to this Set.
     * @return true if the Set did not already contain the specified element.
     */
    public boolean add(A o) {
	return m.put(o, PRESENT)==null;
    }

    /**
     * Removes the given element from this Set if it is present.
     *
     * @param o object to be removed from this Set, if present.
     * @return true if the Set contained the specified element.
     */
    public boolean remove(A o) {
	return m.remove(o)==PRESENT;
    }

    /**
     * Removes all of the elements from this Set.
     */
    public void clear() {
	m.clear();
    }

    /**
     * Returns a view of the portion of this TreeSet whose elements range
     * from fromElement, inclusive, to toElement, exclusive.  The returned Set
     * is backed by this TreeSet, so changes in the returned Set are
     * reflected in this TreeSet, and vice-versa.  The returned Set supports
     * all optional Set operations.  It does not support the subSet
     * operation.
     * <p>
     * The Set returned by this method will throw an IllegalArgumentException
     * if the user attempts to insert an element outside the specified range.
     *
     * @param fromElement low endpoint (inclusive) of the subSet.
     * @param fromElement high endpoint (exclusive) of the subSet.
     * @exception ClassCastException fromElement or toElement cannot be
     *		  compared
     *		  with the elements currently in the TreeSet.
     * @exception NullPointerException fromElement or toElement is null and
     *		  this TreeSet uses natural sort method, or its comparator
     *		  does not tolerate null elements.
     * @exception IllegalArgumentException fromElement is greater than
     *		  toElement.
     */
    public SortedSet<A> subSet(A fromElement, A toElement) {
	return new TreeSet<A>(m.subMap(fromElement, toElement));
    }

    /**
     * Returns a view of the portion of this TreeSet whose elements are
     * strictly less than toElement.  The returned Set is backed by this
     * TreeSet, so changes in the returned Set are reflected in this TreeSet,
     * and vice-versa.  The returned Set supports all optional Set operations.
     * It does not support the subSet operation, as it is not a TreeSet.
     * <p>
     * The Set returned by this method will throw an IllegalArgumentException
     * if the user attempts to insert an element greater than or equal to
     * toElement.
     *
     * @param toElement high endpoint (exclusive) of the headSet.
     * @exception ClassCastException toElement cannot be compared
     *		  with the elements currently in the TreeSet.
     * @exception NullPointerException toElement is null and this
     *		  TreeSet uses natural sort method, or its comparator does
     *		  not tolerate null elements.
     */
    public SortedSet<A> headSet(A toElement) {
	return new TreeSet<A>(m.headMap(toElement));
    }

    /**
     * Returns a view of the portion of this TreeSet whose elements are
     * strictly less than toElement.  The returned Set is backed by this
     * TreeSet, so changes in the returned Set are reflected in this TreeSet,
     * and vice-versa.  The returned Set supports all optional Set operations.
     * It does not support the subSet operation, as it is not a TreeSet.
     * <p>
     * The Set returned by this method will throw an IllegalArgumentException
     * if the user attempts to insert an element greater than or equal to
     * toElement.
     *
     * @param toElement high endpoint (exclusive) of the headSet.
     * @exception ClassCastException toElement cannot be compared
     *		  with the elements currently in the TreeSet.
     * @exception NullPointerException toElement is null and this
     *		  TreeSet uses natural sort method, or its comparator does
     *		  not tolerate null elements.
     */
    public SortedSet<A> tailSet(A fromElement) {
	return new TreeSet<A>(m.tailMap(fromElement));
    }

    /**
     * Returns the comparator used to order this TreeMap, or null if this
     * TreeMap uses its keys natural sort method.
     */
    public Comparator<A> comparator() {
        return m.comparator();
    }

    /**
     * Returns the first (lowest) element currently in this TreeSet.
     *
     * @return the first (lowest) element currently in this TreeSet.
     * @exception NoSuchElementException Set is empty.
     */
    public A first() {
        return m.firstKey();
    }

    /**
     * Returns the last (highest) element currently in this TreeSet.
     *
     * @return the last (highest) element currently in this TreeSet.
     * @exception NoSuchElementException Set is empty.
     */
    public A last() {
        return m.lastKey();
    }

    /**
     * Returns a shallow copy of this TreeSet. (The elements themselves
     * are not cloned.)
     */
    public Object clone() {
        return new TreeSet(this);
    }

    /**
     * Save the state of the TreeSet to a stream (i.e., serialize it).
     */
    private synchronized void writeObject(java.io.ObjectOutputStream s)
        throws java.io.IOException {
	// Write out any hidden stuff
	s.defaultWriteObject();

        // Write out size
        s.writeInt(m.size());

	// Write out all elements in the proper order.
	for (Iterator<A> i=m.keySet().iterator(); i.hasNext(); )
            s.writeObject(i.next());
    }

    /**
     * Reconstitute the LinkedList from a stream (i.e., deserialize it).
     */
    private synchronized void readObject(java.io.ObjectInputStream s)
        throws java.io.IOException, ClassNotFoundException {
	// Read in any hidden stuff
	s.defaultReadObject();

        // Create backing TreeMap and keySet view
        m = new TreeMap<A,Object>();
        keySet = m.keySet();

        // Read in size
        int size = s.readInt();

	// Read in all elements in the proper order.
	for (int i=0; i<size; i++) {
            A e = (A)s.readObject();
            m.put(e, PRESENT);
        }
    }
}
