/* {{{ license
 * GKrellm Gamma
 * Copyright (C) 2002 Gregory Thiemonge <gregory.thiemonge@libertysurf.fr>
 * Copyright (C) 2003 Tomas Styblo <tripie@cpan.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 }}} */

/* includes {{{ */
#include <gkrellm2/gkrellm.h>
#include <stdlib.h>
#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/extensions/xf86vmode.h>
#include <gdk/gdkx.h>
#include <math.h>
/* }}} */

/* static data {{{ */
static GkrellmMonitor *monitor;
static GkrellmPanel   *panel;
static gint     style_id;
static GkrellmKrell   *krell;
static GkrellmDecal   *decal;
/* }}} */

static void set_gamma (float value) /* {{{ */
{
  XF86VidModeGamma gamma;
  gamma.red = value;
  gamma.blue = value;
  gamma.green = value;
  
  XF86VidModeSetGamma(GDK_DISPLAY(), gdk_x11_screen_get_screen_number(gdk_screen_get_default()), &gamma);
}
/* }}} */

static float get_gamma (void) /* {{{ */
{
  XF86VidModeGamma gamma;
  float value;
  XF86VidModeGetGamma(GDK_DISPLAY(), gdk_x11_screen_get_screen_number(gdk_screen_get_default()), &gamma);

  value = (gamma.red + gamma.blue + gamma.green) / 3.0;
  set_gamma(value);
  return value;
}
/* }}} */

static gint panel_expose_event(GtkWidget *widget, GdkEventExpose *ev) /* {{{ */
{
  gdk_draw_pixmap(widget->window,
		  widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
		  panel->pixmap,
		  ev->area.x, ev->area.y,
		  ev->area.x, ev->area.y,
		  ev->area.width, ev->area.height);
  return FALSE;
}
/* }}} */

static void slider_motion(GtkWidget *widget, GdkEventMotion *ev, /* {{{ */
        gpointer data)
{
  gint   x;
  GdkModifierType state;
  char *buf;
  float value;

  state = ev->state;
  if (!(state & GDK_BUTTON1_MASK)) {
    return;
  }
  x = ev->x * krell->full_scale / (gkrellm_chart_width() - 1);
  if(x < 1)
    x = 1;
  if(x > 100)
    x = 100;
  krell->previous = 0;

  gkrellm_update_krell(panel, krell, (gulong)x);
  
  value = (float)pow(10.0, x / 100.0) - 1.0;
  if(value < 0.1)
    value = 0.1;
  set_gamma(value);

  buf = g_strdup_printf("%1.2f", value);
  gkrellm_draw_decal_text(panel, decal, buf, -1);
  g_free(buf);

  gkrellm_draw_panel_layers(panel);
}
/* }}} */

static void panel_button_press(GtkWidget *widget, GdkEventButton *ev) /* {{{ */
{
    gint   x;
    char *buf;
    float value = get_gamma();

    switch(ev->button) {
        case 1:
            x = ev->x * krell->full_scale / (gkrellm_chart_width() - 1);
            if(x < 1) x = 1;
            if(x > 100) x = 100;
            value = (float)pow(10.0, x / 100.0) - 1.0;
            if(value < 0.1)
                value = 0.1;
            break;
        case 2:
            /* reset gamma to 1.0 by Nafees Bin Zafar */
            value = 1.0;
            krell->previous = 0;
            x = 25;
            break;
        default:
            return;
            break;
    }

    krell->previous = 0;
    gkrellm_update_krell(panel, krell, (gulong)x);

    set_gamma(value);

    buf = g_strdup_printf("%1.2f", value);
    gkrellm_draw_decal_text(panel, decal, buf, -1);
    g_free(buf);

    gkrellm_draw_panel_layers( panel );
}
/* }}} */

static void update_plugin (void) /* {{{ */
{
  float value = get_gamma();
  int x = (int)(log10(value+1)*100.0);
  char *buf;

  krell->previous = 0;
  gkrellm_update_krell(panel, krell, (gulong)x);
  
  buf = g_strdup_printf("%1.2f", value);
  gkrellm_draw_decal_text(panel, decal, buf, -1);
  g_free(buf);
 
  gkrellm_draw_panel_layers(panel);
}
/* }}} */

static gint callback_mouse_wheel (GtkWidget *widget, /* {{{ */
        GdkEventScroll *event, void *data) {
    gint   x;
    char *buf;
    float value = get_gamma();

    if (event->direction == GDK_SCROLL_UP) {
        value += 0.1;
        if(value > 9.0)
            value = 9.0;
        x = (int)(log10(value+1.0)*100.0);
    }
    else {
        value -= 0.1;
        if(value < 0.1)
            value = 0.1;
        x = (int)(log10(value+1.0)*100.0);
    }

    krell->previous = 0;
    gkrellm_update_krell(panel, krell, (gulong)x);

    set_gamma(value);

    buf = g_strdup_printf("%1.2f", value);
    gkrellm_draw_decal_text(panel, decal, buf, -1);
    g_free(buf);

    gkrellm_draw_panel_layers( panel );
    return TRUE;
}
/* }}} */

static void create_plugin (GtkWidget *vbox, gint first_create) /* {{{ */
{
  GkrellmStyle         *style;
  GkrellmPiximage *krell_image;
  GkrellmMargin   *m;
  char *buf;
  int  w;

  if (first_create) {
    panel = gkrellm_panel_new0();
  }

  style = gkrellm_krell_slider_style();
  krell_image = gkrellm_krell_slider_piximage();
  krell = gkrellm_create_krell(panel, krell_image, style);
  krell->full_scale = 100;

  style = gkrellm_meter_style(style_id);
  m = gkrellm_get_style_margins(style);

  w = gdk_string_width(gkrellm_meter_textstyle(style_id)->font, "8.88");
  decal = gkrellm_create_decal_text(panel, "Gamma",
				    gkrellm_meter_textstyle(style_id),
				    style,
				    gkrellm_chart_width() - w - m->right, -1, -1);

  gkrellm_panel_configure(panel, NULL, style);
  gkrellm_panel_create(vbox, monitor, panel);

  buf = g_strdup_printf("%1.2f", get_gamma());
  gkrellm_draw_decal_text(panel, decal, buf, -1);
  g_free(buf);

  if (first_create) {
    g_signal_connect(GTK_OBJECT(panel->drawing_area), "expose_event",
            G_CALLBACK(panel_expose_event), NULL);
    g_signal_connect(GTK_OBJECT(panel->drawing_area), "motion_notify_event",
            G_CALLBACK(slider_motion), NULL);
    g_signal_connect(GTK_OBJECT(panel->drawing_area), "button_press_event",
            G_CALLBACK(panel_button_press), NULL);
    g_signal_connect(G_OBJECT(panel->drawing_area), "scroll_event", 
            G_CALLBACK(callback_mouse_wheel), NULL);
  }
  gkrellm_update_krell(panel, krell, (gulong)(log10(get_gamma()+1)*100.0));
  gkrellm_draw_panel_layers(panel);
}
/* }}} */

static GkrellmMonitor plugin_gamma = { /* {{{ */
  "Gamma",
  0,
  create_plugin,
  update_plugin,
  NULL,
  NULL,
  
  NULL,
  NULL,
  NULL,

  NULL,
  NULL,
  NULL,
  
  MON_UPTIME,
  
  NULL,
  NULL
};
/* }}} */

GkrellmMonitor * gkrellm_init_plugin (void) /* {{{ */
{
  style_id = gkrellm_add_meter_style(&plugin_gamma, "gkrellm-gamma");
  monitor = &plugin_gamma;

  return &plugin_gamma;
}
/* }}} */
