/* gkrelltop.c    - requires GKrellM 1.2.0 or better
 * 
 *   gcc -fPIC `gtk-config --cflags` `imlib-config --cflags-gdk` -c gkrelltop.c
 *   gcc -shared -Wl -o gkrelltop.so gkrelltop.o
 *   gkrellm -p gkrelltop.so
 *
 * 
 * gkrelltop is a monitor to see the three most cpu intensive processes. It
 * shows the pid of the most active one on mouse over as a tooltip. This way
 * if mozilla is going crazy on you, you can figure it out in a split and
 * kill it if necessary. Most useful for laptops which dont want the cpu to
 * be used at all times. But useful also to know, at times, which are the
 * processes that are claiming that cpu usage you see on the cpu monitor (of
 * gkrellm or otherwise). Definitely in initial stages, even though it does
 * what i want it to do, for the moment.
 *
 * to test it out try 
 *    gkrellm -p gkrelltop.so
 *
 *
 * Display the top three processes occupying the cpu 
 *
 * based on the demos from gkrell website
 *
 * it is a module for gkrellm which uses sources
 * from Wmtop -- WindowMaker process view dock app
 *
 * Written by Adi Zaimi adizaimi-at-pegasus.rutgers.edu
 *
 * This software is licensed through the GNU General Public License.
 */


/*    
 *
 * this part of the code is borrowed from the gkrellm demos 
*/

#include "gkrelltop.h"
#include "support.h"

#define PLUGIN_CONFIG_KEYWORD  "gkrellmtop"
#define CONFIG_NAME            "gkrelltop"
#define UPDATE_TIMEFRAME        5  /* seconds */    

/* variables */
static Monitor *monitor;
static Panel *panel;
static GtkWidget   * g_threshold      = NULL;
static GtkWidget   * g_updates        = NULL;
static GtkWidget   * g_show_nice      = NULL;


static Decal *decal_text1, *decal_text2, *decal_pix1, *decal_pix2,
    *decal_text3, *decal_pix3;
/* variables */
static GtkTooltips *panel_tooltip = NULL;
static gint style_id;
static gint first_top;
static gint mouseIn = 0;
static gint display_tooltip = 0;

static int global_update_HZ=0;     
static gint threshold=0;     
static gint local_updates=1;     
/* global show_nice_processes variable */
int show_nice_processes;
static gchar tooltip_text[256];


static int    (*find_top_three)(struct process **best);
static void do_update(void);


#ifdef    GKRELLM_HAVE_CLIENT_MODE_PLUGINS
/* ---------------- gkrelltop client mode interface -------------- */

static struct process   cache_best[3];
static gint             cache_ntop;
static gboolean         gkrelltop_server_available;

static gint             gkrelltopd_version_major,
                        gkrelltopd_version_minor,
                        gkrelltopd_version_rev;

static void
gkrelltop_client_setup(gchar *line)
{
    if (!strcmp(line, "available"))
        gkrelltop_server_available = TRUE;
    else if (!strncmp(line, "version ", 8))
        sscanf(line, "%*s %d %d %d", &gkrelltopd_version_major,
             &gkrelltopd_version_minor, &gkrelltopd_version_rev);
}

static void
gkrelltop_client_data_from_server(gchar *line)
{
    gfloat    amount;
    gchar    which[32], item[128], name[128];
    gint    pid, n;

    if (sscanf(line, "%31s %127[^\n]", which, item) != 2)
       return;

    if (!strcmp(which, "ntop"))
    {
        cache_ntop = atoi(item);
        if (cache_ntop < 0 || cache_ntop > 3)
            cache_ntop = 0;
            for (n = cache_ntop; n < 3; ++n)
            {
                gkrellm_dup_string(&cache_best[n].name, "");
                cache_best[n].amount = 0;
                cache_best[n].pid = 0;
            }
        }
    else if (   !strcmp(which, "best")
          && sscanf(item, "%d %127s %d %f", &n, name, &pid, &amount) == 4
          && n >= 0 && n < 3)
    {
        gkrellm_dup_string(&cache_best[n].name, name);
        cache_best[n].pid = pid;
        cache_best[n].amount = amount;
    }
}

static gint
gkrelltop_client_process_find_top_three(struct process **best)
{
    gint    i;

    for (i = 0; i < 3; ++i)
        best[i] = &cache_best[i];
    return cache_ntop;
}

/* ----------------- end of client mode interface ---------------- */
#endif


static gint panel_expose_event(GtkWidget * widget, GdkEventExpose * ev)
{
    gdk_draw_pixmap(widget->window,
        widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
        panel->pixmap, ev->area.x, ev->area.y, ev->area.x,
        ev->area.y, ev->area.width, ev->area.height);
    return TRUE;
}

void panel_enter_notify_event(GtkWidget * widget, GdkEventCrossing * event,
        gpointer user_data)
{
    mouseIn = 1;        /* change status */
    display_tooltip=1;
//printf("mouse in\n");
}

void panel_leave_notify_event(GtkWidget * widget, GdkEventCrossing * event,
        gpointer user_data)
{
    mouseIn = 0;        /* change status */
}

/* show the setup window on right click */
void panel_click_event(GtkWidget * widget, GdkEventButton * event,
        gpointer user_data)
{
    if (event->button==3) 
        gkrellm_open_config_window(monitor);
}

static void update_plugin()
{

#ifdef GKRELLM2
    GkrellmTicks *GKp = gkrellm_ticks();
#endif

#ifdef GKRELLM2
    if (GKp->five_second_tick) {
#else
    if (GK.five_second_tick) {
#endif
        /* set the global update_HZ variable every five second ticks 
         * so if user changed the global_update_HZ, we will have 
         * the latest  value 
         * */
        global_update_HZ=gkrellm_update_HZ();
        if(global_update_HZ <=0 || global_update_HZ > 10) 
            global_update_HZ=5;
    }

#ifdef GKRELLM2
    if ( (GKp->timer_ticks % (int)(global_update_HZ*UPDATE_TIMEFRAME/local_updates)) == 0 ) { 
        //printf("timer=%d, local_updates=%d, freq=%d, UP_TIM=%d\n", GKp->timer_ticks,local_updates,(int)global_update_HZ, UPDATE_TIMEFRAME);
#else
    if ((GK.timer_ticks % (int)(global_update_HZ*UPDATE_TIMEFRAME)) == local_updates) {
#endif
        do_update();  /* do the actual update */
    }   //if gk.second_tick
}

/* do the actual update of the plugin, separate if needed to be called
 * by some other function besides update_plugin() */
static void do_update()
{
    static gint x_scroll, w;

    int i = 0, n = 0;
    struct process *best[3] = { 0, 0, 0 };
    gchar display_text[3][64];
    int ifirst_pid;


    display_text[0][0] = display_text[1][0] = display_text[2][0] = '\0';
    tooltip_text[0] = '\0';

#ifdef  GKRELLM_HAVE_CLIENT_MODE_PLUGINS
    /* If user is running a gkrelltop version that can get server data but
    |  isn't, make sure user can know displayed data is from local host.
    */
    if (!gkrelltop_server_available && gkrellm_client_mode())
        snprintf(tooltip_text, sizeof(tooltip_text), "Localhost: %s\n",
            gkrellm_get_hostname());
#endif
    
    /* make possible for the writing of the message */
    if (w == 0)
        w = gkrellm_chart_width();
    x_scroll = (x_scroll + 1) % (2 * w);
    
    /* every tick update the process lists, and get data from either
    * a gkrellmd server or locally via top_three.c
    */
    n = (*find_top_three)(best);    /* Find the top three processes */

    if (n > 0) {
        ifirst_pid = best[0]->pid;
    }

    for (i = 0; i < n; i++) {

        /* display used to depend on mouseIn: if (!mouseIn) { blah ..} */

        if(best[i]->amount >= threshold ) {
            snprintf(display_text[i], sizeof(display_text[i])," %2.0f%s %s",
                  best[i]->amount,"%",best[i]->name);
        } //if
        if(mouseIn && best[i]->amount >= threshold ) {
            char ch = (i < n-1) ? '\n' : ' ' ; /* prnt \n only for 1 and 2*/
            char tmp[256];
            snprintf(tmp,sizeof(tmp), "%4.1f%c  %6.2d  %.30s%c", 
                best[i]->amount, '%', best[i]->pid, 
                best[i]->name, ch);
            strncat(tooltip_text,tmp,sizeof(tooltip_text));
        } //if
    } // for


    /* put as a tooltip the most active's process pid, update only if 
    * the current top pid is different from the previous top pid */
    if (display_tooltip==1 ) { //ifirst_pid != first_top) {
        gtk_tooltips_set_tip(panel_tooltip, panel->drawing_area, tooltip_text, ""); //set to empty
        display_tooltip=0;
        first_top = ifirst_pid;
    }
    
    /* set the panels with the top three processes, or with blank */
    gkrellm_draw_decal_text(panel, decal_text1, 
                  display_text[0], w - x_scroll);
    gkrellm_draw_decal_text(panel, decal_text2, 
                  display_text[1], w - x_scroll);
    gkrellm_draw_decal_text(panel, decal_text3, 
                  display_text[2], w - x_scroll);
    gkrellm_draw_panel_layers(panel);
    
    /* next steps are to put the name of the process in red if > 33% of
    * the processor, yellow if > 10 and green if below. 
    */
}


static void create_plugin(GtkWidget * vbox, gint first_create)
{
    Style *style;
    TextStyle *ts, *ts_alt;
    GdkPixmap *pixmap;
    GdkBitmap *mask;
    gint y;
    gint x;

    /* See comments about first create in demo2.c
     */
    if (first_create)
        panel = gkrellm_panel_new0();   /* create new panel */

    style = gkrellm_meter_style(style_id);  /* set style of the meter */

    ts = gkrellm_meter_textstyle(style_id); //ts is in big font e darker color
    ts_alt = gkrellm_meter_alt_textstyle(style_id); //smaller font e brighter col 

    /* Create three text decal that will be used to display text.  Make it
     * the full panel width (minus the margins).  Position it at top border
     * and left margin of the style.
     */
    decal_text1 =
       gkrellm_create_decal_text(panel, "Ayl0", ts_alt, style, -1, -1, -1);

    y = decal_text1->y + decal_text1->h + 2;
    decal_text2 =
        gkrellm_create_decal_text(panel, "Ayl0", ts_alt, style, -1, y, -1);
    y = y + decal_text2->h + 2;
    decal_text3 =
        gkrellm_create_decal_text(panel, "Ayl0", ts_alt, style, -1, y, -1);
    //decal_text2 = gkrellm_create_decal_text(panel, "panel1", ts_alt, style, 2, y, 0);

    /* Create a decal pixmap and place it to the right of decal_text2.  Use
       |  decals from the builtin decal_misc.xpm.  */
    pixmap = gkrellm_decal_misc_pixmap();
    mask = gkrellm_decal_misc_mask();

    /* create the pixmap with the text decals */
    x = decal_text1->x + decal_text1->w + 1;
    decal_pix1 = gkrellm_create_decal_pixmap(panel, pixmap, mask, 
                  N_MISC_DECALS, NULL, x, y);
    decal_pix2 = gkrellm_create_decal_pixmap(panel, pixmap, mask, N_MISC_DECALS,
                NULL, x, y);
    decal_pix3 = gkrellm_create_decal_pixmap(panel, pixmap, mask, N_MISC_DECALS,
                NULL, x, y);

    /* Configure the panel to hold the above created decals, and create it.  */
    gkrellm_panel_configure(panel, NULL, style);
    gkrellm_panel_create(vbox, monitor, panel);

    /* Note: all of the above gkrellm_draw_decal_XXX() calls will not
       |  appear on the panel until a gkrellm_draw_panel_layers() call is
       |  made.  This will be done in update_plugin(), otherwise we would
       |  make the call here and anytime the decals are changed.
     */

    if (first_create) {
        gtk_signal_connect(GTK_OBJECT(panel->drawing_area), "expose_event",
            (GtkSignalFunc) panel_expose_event, NULL);
        gtk_signal_connect(GTK_OBJECT(panel->drawing_area),"enter_notify_event",
            (GtkSignalFunc) panel_enter_notify_event, NULL);
        gtk_signal_connect(GTK_OBJECT(panel->drawing_area),"leave_notify_event",
            (GtkSignalFunc) panel_leave_notify_event, NULL);
        gtk_signal_connect(GTK_OBJECT(panel->drawing_area),"button_press_event",
            (GtkSignalFunc) panel_click_event, NULL);
    }

    panel_tooltip = gtk_tooltips_new();
    gtk_tooltips_set_tip(panel_tooltip, panel->drawing_area,
            " \n \n ", "empty"); /* start with empty string */
    gtk_tooltips_set_delay(panel_tooltip, 0);
    gtk_tooltips_enable(panel_tooltip);

    gkrellm_draw_panel_layers(panel);

    /* set the global update_HZ variable every five second ticks */
    global_update_HZ=gkrellm_update_HZ();
    if(global_update_HZ <=0 || global_update_HZ > 10) 
        global_update_HZ=5;

    /* set the local_updates after reading from the config */
    if(local_updates > global_update_HZ*UPDATE_TIMEFRAME) 
        local_updates = global_update_HZ*UPDATE_TIMEFRAME;
    if(local_updates <= 0) 
        local_updates = 1; 
    return;
}

static void create_plugin_tab(GtkWidget *tab_vbox) 
{
    GtkWidget *tabs;
    //GtkWidget *table;
    GtkWidget  *vbox, *vbox1, *hbox;
    //GtkWidget *button;
    GtkWidget *label;
    GtkWidget *text;
    //GSList *group;
    gchar *about_text;
    //gint i;

    tabs = gtk_notebook_new();
    gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
    gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

    /* -- setup -- */
    vbox = gkrellm_create_framed_tab(tabs, _("Setup"));
    vbox1 = gkrellm_framed_vbox(vbox, _("Visualisation Options"), 4,FALSE,0,2);
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, FALSE, 0);

    label = gtk_label_new(_("Show only processes above:"));
    gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
    //gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
    gkrellm_spin_button(hbox, &g_threshold, (gfloat) threshold, 0.0, 100.0, 1.0, 5.0, 0, 60, NULL, NULL, FALSE, _("%"));
    //gtk_table_attach_defaults(GTK_TABLE(table), hbox, 0, 2, 3, 4);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, FALSE, 0);
    label = gtk_label_new(_("Frequency of updates (in 5 seconds):"));
    gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
    gkrellm_spin_button(hbox, &g_updates, (gfloat) local_updates, 1.0, global_update_HZ*UPDATE_TIMEFRAME, 1.0, 5.0, 0, 60, NULL, NULL, FALSE, _(""));

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, FALSE, 0);
    //gtk_table_attach_defaults(GTK_TABLE(table), hbox, 0, 1, 2, 3);
    gkrellm_check_button(hbox, &g_show_nice, show_nice_processes, TRUE, 0,
      _("Show nice processes"));


    /* -- About -- */
    about_text = g_strdup_printf(
      "gkrelltop version 2.2.3\n"
      "GKrellM Top Plugin\n"
      "Copyright (c) 2002,2003 Adi Zaimi\n"
      "zaimi-at-pegasus.rutgers.edu\n"
      "http://psychology.rutgers.edu/~zaimi/\n\n"
      "Released under the GNU Public License");
    text = gtk_label_new(about_text);
    label = gtk_label_new("About");
    gtk_notebook_append_page(GTK_NOTEBOOK(tabs),text,label);
    g_free(about_text);

    return;
}

static void apply_config() 
{
    GtkSpinButton *spin;
    spin = GTK_SPIN_BUTTON(g_threshold);
    threshold = gtk_spin_button_get_value_as_int(spin);
    show_nice_processes = GTK_TOGGLE_BUTTON(g_show_nice)->active;
    spin = GTK_SPIN_BUTTON(g_updates);
    local_updates = gtk_spin_button_get_value_as_int(spin);

    /* safety check */
    if(local_updates > global_update_HZ*UPDATE_TIMEFRAME) 
        local_updates = global_update_HZ*UPDATE_TIMEFRAME;
    if(local_updates <= 0) 
        local_updates = 1; 

    return;
}

static void save_config(FILE *f) 
{
   /* Save any configuration data we have in config lines in the format:
    PLUGIN_CONFIG_KEYWORD  config_keyword  data
    */

    fprintf(f, "%s threshold %d\n", PLUGIN_CONFIG_KEYWORD, threshold);
    fprintf(f, "%s show_nice_processes %d\n", PLUGIN_CONFIG_KEYWORD, show_nice_processes);
    fprintf(f, "%s local_updates %d\n", PLUGIN_CONFIG_KEYWORD, local_updates);
    return;
}

static void load_config(gchar *config_line) 
{
    gchar   config_keyword[32], config_data[CFG_BUFSIZE];
    gint    n;

    if ((n = sscanf(config_line, "%31s %[^\n]", config_keyword, 
                                                   config_data)) != 2) {
        return;
    }

    if ( strcmp(config_keyword, "threshold")==0 ) {
        sscanf(config_data, "%d", & threshold);
    }
    else if (strcmp(config_keyword, "show_nice_processes")==0) {
        sscanf(config_data, "%d", & show_nice_processes);
    }
    else if (strcmp(config_keyword, "local_updates")==0) {
        sscanf(config_data, "%d", & local_updates);
    }

    return;
}


/* The monitor structure tells GKrellM how to call the plugin routines.  */
static Monitor plugin_mon = {
    CONFIG_NAME,              /* Name, for config tab.                      */
    0,                        /* Id,  0 if a plugin                         */
    create_plugin,            /* The create function                        */
    update_plugin,            /* The update function                        */
    create_plugin_tab,        /* The config tab create function             */
    apply_config,             /* Apply the config function                  */
    save_config,              /* Save user config                           */
    load_config,              /* Load user config                           */
    PLUGIN_CONFIG_KEYWORD,    /* config keyword                             */

    NULL,                     /* Undefined 2                                */
    NULL,                     /* Undefined 1                                */
    NULL,                     /* private                                    */
  
    MON_MAIL,                 /* Insert plugin before this monitor          */
    NULL,                     /* Handle if a plugin, filled in by GKrellM   */
    NULL                      /* path if a plugin, filled in by GKrellM     */
};


/* initialise plugin -- required */
Monitor *init_plugin()
{
    style_id = gkrellm_add_meter_style(&plugin_mon, STYLE_NAME);

#ifdef    GKRELLM_HAVE_CLIENT_MODE_PLUGINS
    gkrellm_client_plugin_get_setup(GKRELLTOP_SERVE_NAME,
          gkrelltop_client_setup);

    if (gkrelltop_server_available)
    {
       find_top_three = gkrelltop_client_process_find_top_three;
       gkrellm_client_plugin_serve_data_connect(&plugin_mon,
       GKRELLTOP_SERVE_NAME, gkrelltop_client_data_from_server);
    }
    else
       find_top_three = gkrelltop_process_find_top_three;
#else
       find_top_three = gkrelltop_process_find_top_three;
#endif

    monitor = &plugin_mon;
    return &plugin_mon;
}
