/*
 * top_three.c a slightly modified wmtop.c -- copied from the WindowMaker 
 * process view dock app which is:
 *
 * Modified by Adi Zaimi
 *
 * Derived by Dan Piponi dan@tanelorn.demon.co.uk
 * http://www.tanelorn.demon.co.uk
 * http://wmtop.sourceforge.net 
 * from code originally contained in wmsysmon by Dave Clark 
(clarkd@skynet.ca)
 * This software is licensed through the GNU General Public License.
 */

/*
 * Ensure there's an operating system defined. There is *no* default
 * because every OS has it's own way of revealing CPU/memory usage.
 * compile with gcc -DOS ...
 */
#if defined(FREEBSD)
#define OS_DEFINED
#endif        /* defined(FREEBSD) */

#if defined(LINUX)
#define OS_DEFINED
#endif        /* defined(LINUX) */

#if !defined(OS_DEFINED)
#error No operating system selected
#endif        /* !defined(OS_DEFINED) */

/******************************************/
/* Includes                               */
/******************************************/

#include "gkrelltop.h"

/******************************************/
/* Defines                                */
/******************************************/


/*
 * XXX: I shouldn't really use this BUFFER_LEN variable but scanf is so
 * lame and it'll take me a while to write a replacement.
 */
#define BUFFER_LEN 1024

#if defined(LINUX)
#define PROCFS_TEMPLATE "/proc/%d/stat"
#define PROCFS_CMDLINE_TEMPLATE "/proc/%d/cmdline"
#endif        /* defined(LINUX) */

#if defined(FREEBSD)
#define PROCFS_TEMPLATE "/proc/%d/status"
#endif        /* defined(FREEBSD) */

/******************************************/
/* Globals                                */
/******************************************/

static regex_t *exclusion_expression = 0;
int show_nice_processes;

static int        g_time = 0;

static int previous_total = 0;



/******************************************/
/* Debug                                  */
/******************************************/

#if defined(DEBUG)
/*
 * Memory handler
 */
static int g_malloced = 0;

static void *gktop_malloc(int n)
{
    int *p = (int *) malloc(sizeof(int) + n);
    p[0] = n;
    g_malloced += n;
    return (void *) (p + 1);
}

static void gktop_free(void *n)
{
    int *p = (int *) n;
    g_malloced -= p[-1];
    free(p - 1);
}

static void show_memory()
{
    fprintf(stderr, "%d bytes allocated\n", g_malloced);
}
#else        /* defined(DEBUG) */
#define gktop_malloc malloc
#define gktop_free free
#endif        /* defined(DEBUG) */

static char *gktop_strdup(const char *s)
{
    return strcpy((char *) gktop_malloc(strlen(s) + 1), s);
}

/******************************************/
/* Process class                          */
/******************************************/

/*
 * Pointer to head of process list
 */
static struct process *first_process = 0;

static struct process *find_process(pid_t pid)
{
    struct process *p = first_process;
    while (p) {
        if (p->pid == pid)
               return p;
        p = p->next;
    }
    return 0;
}

/*
 * Create a new process object and insert it into the process list
 */
static struct process *new_process(int p)
{
    struct process *process;
    process = gktop_malloc(sizeof(struct process));

    /*
     * Do stitching necessary for doubly linked list
     */
    process->name = 0;
    process->previous = 0;
    process->next = first_process;
    if (process->next)
        process->next->previous = process;
    first_process = process;

    process->pid = p;
    process->time_stamp = 0;
    process->previous_user_time = INT_MAX;
    process->previous_kernel_time = INT_MAX;
    process->counted = 1;

/*    process_find_name(process);*/

    return process;
}

/******************************************/
/* Functions                              */
/******************************************/

static int process_parse_procfs(struct process *);
static int update_process_table(void);
static int calculate_cpu(struct process *);
static void process_cleanup(void);
static void delete_process(struct process *);
//inline void draw_processes(void);
static int calc_cpu_total(void);
static void calc_cpu_each(int);

#if 0
#if defined(LINUX)
static int calc_mem_total(void);
static void calc_mem_each(int);
#endif
#endif

/******************************************/
/* Extract information from /proc         */
/******************************************/

/*
 * These are the guts that extract information out of /proc.
 * Anyone hoping to port wmtop should look here first.
 */
static int process_parse_procfs(struct process *process)
{
    char line[BUFFER_LEN], filename[BUFFER_LEN], procname[BUFFER_LEN];
    int ps;
    int user_time, kernel_time;
    int rc;
#if defined(LINUX)
    char *r, *q;
    char deparenthesised_name[BUFFER_LEN];
    int endl;
    int nice_val;
#endif        /* defined(LINUX) */
#if defined(FREEBSD)
    int us, um, ks, km;
#endif        /* defined(FREEBSD) */

    snprintf(filename,sizeof(filename), PROCFS_TEMPLATE, process->pid);

    ps = open(filename, O_RDONLY);
    if (ps < 0)
        /*
        * The process must have finished in the last few jiffies!
        */
        return 1;

    /*
     * Mark process as up-to-date.
     */
    process->time_stamp = g_time;

    rc = read(ps, line, sizeof(line));
    close(ps);
    if (rc < 0)
        return 1;

#if defined(LINUX)
    /*
     * Extract cpu times from data in /proc filesystem
     */
    rc = sscanf(line,
               "%*s %s %*s %*s %*s %*s %*s %*s %*s %*s %*s %*s %*s %d %d %*s %*s %*s %d %*s %*s %*s %d %d",
               procname, &process->user_time, &process->kernel_time,&nice_val,
               &process->vsize, &process->rss);
    if (rc < 5)
        return 1;
    /*
     * Remove parentheses from the process name stored in /proc/ under Linux...
     */
    r = procname + 1;
    /* remove any "kdeinit: " */
    if (r == strstr(r, "kdeinit")) {
        snprintf(filename, sizeof(filename), PROCFS_CMDLINE_TEMPLATE, process->pid);

        ps = open(filename, O_RDONLY);
        if (ps < 0)
        /*
          * The process must have finished in the last few jiffies!
          */
               return 1;

        endl = read(ps, line, sizeof(line));
        close(ps);

        /* null terminate the input */
        line[endl] = 0;
        /* account for "kdeinit: " */
        if ((char *) line == strstr(line, "kdeinit: "))
               r = ((char *) line) + 9;
        else
               r = (char *) line;

        q = deparenthesised_name;
        /* stop at space */
        while (*r && *r != ' ')
               *q++ = *r++;
        *q = 0;
    } else {
        q = deparenthesised_name;
        while (*r && *r != ')')
               *q++ = *r++;
        *q = 0;
    }

    if (process->name)
        gktop_free(process->name);
    process->name = gktop_strdup(deparenthesised_name);
#endif        /* defined(LINUX) */

#if defined(FREEBSD)
    /*
     * Extract cpu times from data in /proc/<pid>/stat
     * XXX: Process name extractor for FreeBSD is untested right now.
     */
    rc = sscanf(line, "%s %*s %*s %*s %*s %*s %*s %*s %d,%d %d,%d",
        procname, &us, &um, &ks, &km);
    if (rc < 5)
        return 1;
    if (process->name)
        gktop_free(process->name);
    process->name = gktop_strdup(procname);
    process->user_time = us * 1000 + um / 1000;
    process->kernel_time = ks * 1000 + km / 1000;
#endif        /* defined(FREEBSD) */

    process->rss *= getpagesize();

    if (process->previous_user_time == INT_MAX)
        process->previous_user_time = process->user_time;
    if (process->previous_kernel_time == INT_MAX)
        process->previous_kernel_time = process->kernel_time;

    /* store the difference of the user_time */
    user_time = process->user_time - process->previous_user_time;
    kernel_time = process->kernel_time - process->previous_kernel_time;

    /* backup the process->user_time for next time around*/
    process->previous_user_time = process->user_time;
    process->previous_kernel_time = process->kernel_time;

    /* store only the difference of the user_time here...*/
    process->user_time = user_time;
    process->kernel_time = kernel_time;

    /* set it to zero for niced processes */
    if(show_nice_processes==0 && nice_val > 1) {
        process->user_time = 0;
        process->kernel_time = 0;
    }

    return 0;
}

/******************************************/
/* Update process table                   */
/******************************************/

static int update_process_table()
{
    DIR *dir;
    struct dirent *entry;

    if (!(dir = opendir("/proc")))
        return 1;

    ++g_time;

    /*
     * Get list of processes from /proc directory
     */
    while ((entry = readdir(dir))) {
        pid_t pid;

        if (!entry) {
          /*
          * Problem reading list of processes
          */
           closedir(dir);
           return 1;
        }

        if (sscanf(entry->d_name, "%d", &pid) > 0) {
               struct process *p;
               p = find_process(pid);
               if (!p)
                      p = new_process(pid);

               /* compute each process cpu usage */
               calculate_cpu(p);
        }
    }

    closedir(dir);

    return 0;
}

/******************************************/
/* Get process structure for process pid  */
/******************************************/

/*
 * This function seems to hog all of the CPU time. I can't figure out why - it
 * doesn't do much.
 */
static int calculate_cpu(struct process *process)
{
    int rc;

    /* compute each process cpu usage by reading /proc/<proc#>/stat */
    rc = process_parse_procfs(process);
    if (rc)
        return 1;

    /*
     * Check name against the exclusion list
     */
    if (process->counted && exclusion_expression
        && !regexec(exclusion_expression, process->name, 0, 0, 0))
        process->counted = 0;

    return 0;
}

/******************************************/
/* Strip dead process entries             */
/******************************************/

static void process_cleanup()
{

    struct process *p = first_process;
    while (p) {
        struct process *current = p;

#if defined(PARANOID)
        assert(p->id == 0x0badfeed);
#endif        /* defined(PARANOID) */

        p = p->next;
        /*
        * Delete processes that have died
        */
        if (current->time_stamp != g_time)
            delete_process(current);
    }
}

/******************************************/
/* Destroy and remove a process           */
/******************************************/

static void delete_process(struct process *p)
{
#if defined(PARANOID)
    assert(p->id == 0x0badfeed);

    /*
     * Ensure that deleted processes aren't reused.
     */
    p->id = 0x007babe;
#endif        /* defined(PARANOID) */

    /*
     * Maintain doubly linked list.
     */
    if (p->next)
        p->next->previous = p->previous;
    if (p->previous)
        p->previous->next = p->next;
    else
        first_process = p->next;

    if (p->name)
        gktop_free(p->name);
    gktop_free(p);
}

/******************************************/
/* Calculate cpu total                    */
/******************************************/

static int calc_cpu_total()
{
    int total, t;
#if defined(LINUX)
    int rc;
    int ps;
    char line[BUFFER_LEN];
    int cpu, nice, system, idle;

    ps = open("/proc/stat", O_RDONLY);
    rc = read(ps, line, sizeof(line));
    close(ps);
    if (rc < 0)
        return 0;
    sscanf(line, "%*s %d %d %d %d", &cpu, &nice, &system, &idle);
    total = cpu + nice + system + idle;

#endif        /* defined(LINUX) */

#if defined(FREEBSD)
    struct timeval tv;

    gettimeofday(&tv, 0);
    total = tv.tv_sec * 1000 + tv.tv_usec / 1000;
#endif        /* defined(FREEBSD) */

    t = total - previous_total;
    previous_total = total;

    if (t < 0)
        t = 0;

    return t;
}

/******************************************/
/* Calculate each processes cpu           */
/******************************************/

static void calc_cpu_each(int total)
{
    struct process *p = first_process;
    while (p) {
        p->amount = total ?  
                    ( 100 * (float) (p->user_time + p->kernel_time) / total ) 
                     : 0;

        if (p->amount > 100) p->amount=0;
            p = p->next;
    }
}

/******************************************/
/* Calculate total memory                 */
/******************************************/

#if 0
#if defined(LINUX)
static int calc_mem_total()
{
    int ps;
    char line[512];
    char *ptr;
    int rc;

    ps = open("/proc/meminfo", O_RDONLY);
    rc = read(ps, line, sizeof(line));
    close(ps);
    if (rc < 0)
        return 0;

    if ((ptr = strstr(line, "Mem:")) == NULL) {
        return 0;
    } else {
        ptr += 4;
        return atoi(ptr);
    }

}
#endif        /* defined(LINUX) */
#endif

/******************************************/
/* Calculate each processes memory        */
/******************************************/

#if 0
#if defined(LINUX)
static void calc_mem_each(int total)
{
    struct process *p = first_process;
    while (p) {
        p->amount = 100 * (float) p->rss / total;
        p = p->next;
    }
}
#endif        /* defined(LINUX) */
#endif

/******************************************/
/* Find the top three processes           */
/******************************************/

/*
 * Result is stored in decreasing order in best[0-2].
 */
int gkrelltop_process_find_top_three(struct process **best)
{
    struct process *p;
    int n = 0;
    int total;

    total = calc_cpu_total();        /* calculate the total of the processor */

    update_process_table();      /* update the table with process list */
    calc_cpu_each(total);        /* and then the percentage for each task */
    process_cleanup();    /* cleanup list from exited processes */

    /*
     * Insertion sort approach to skim top 3
     * need to make this more efficient...
     */
    p = first_process;
    while (p) {
        if (p->counted && p->amount > 0
            && (!best[0] || p->amount > best[0]->amount)) {
            best[2] = best[1];
            best[1] = best[0];
            best[0] = p;
            ++n;
        } else if (p->counted && p->amount > 0
            && (!best[1] || p->amount > best[1]->amount)) {
            best[2] = best[1];
            best[1] = p;
            ++n;
        } else if (p->counted && p->amount > 0
            && (!best[2] || p->amount > best[2]->amount)) {
            ++n;
            best[2] = p;
        }

        p = p->next;
    }

    return n > 3 ? 3 : n;
}
