/*
 * GLeyes - A GLUT version of xeyes
 * Copyright (C) 1998 Claudio Matsuoka <claudio@pos.inf.ufpr.br>
 * $Id: gleyes.c,v 1.3 1998/10/13 16:09:04 claudio Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/*
 * Thu, 8 Oct 1998 16:18:37 -0700 (PDT)  Ralph Giles <ralphbla@sfu.ca>
 *
 * I've made a patch against the version 0.1 I got this morning which I 
 * think improves the mouse-tracking math some. I didn't try to make it any 
 * more efficient, but the iris will not spin as the pointer goes by.
 *
 * Also there was a bug in the 'orbit' code--the lX,lY,lZ were wrapping at 
 * 360, but the C trig routines take their arguments in 'radians' (2*M_PI is 
 * a full revolution), so this should cause a discontinuity in the movement 
 * now and then. I patched this as well. It happens so infrequently I can't 
 * tell if it's really helping, but it's the proper thing to do and 
 * certainly doesn't look any worse.
 */

/* Tue Oct 13 13:55:55 EDT 1998  Felipe Rosinha <rosinha@dainf.cefetpr.br>
 *
 * Ported to Win32. Have fun with the Makefile <grin>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#ifdef _WIN32
#include <windows.h>
#include <winuser.h>
#include <GL/glut.h>
#else
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#define CDECL
#include "glutint.h"
#endif

#ifndef TIMER
#define TIMER 120
#endif

#ifndef SLICES
#define SLICES 20
#endif

#ifndef STACKS
#define STACKS 10
#endif

#define FLAT	1
#define SMOOTH	2
#define CYCLOPS 3
#define HUMAN	4
#define LITE	5
#define REDRAW	6
#define QUIT	9

#ifndef M_PI
#define M_PI     3.14159265358979323846		/* Yuck */
#endif

#ifdef _WIN32

typedef struct pos {
	int width;
	int height;
} POS;

#endif

static GLfloat lX, lY, lZ;
static GLfloat rX, rY, rZ;
static GLfloat lrX, lrY, rrX, rrY;

static int cyclops = 0;			/* Cyclops mode */
static int smooth = 1;			/* Smooth shading */
static int window_x = 180;		/* Default window size (X) */
static int window_y = 100;		/* Default window size (Y) */
static int lite = 1;			/* Render on move */

extern unsigned char iris[128 * 128 * 3];


static void update (int value)
{
#ifdef _WIN32
	POINT point;
	RECT  rect;
	POS   xwa;  
#else
	XWindowAttributes xwa;
	Window root, child;
	unsigned int mask;
	int wx, wy;
#endif
	float dx, dy;
	int rootx, rooty;
	static int old_rootx = -1, old_rooty = -1;

#ifdef _WIN32
	GetCursorPos(&point);
	GetWindowRect (GetDesktopWindow(), &rect);

	xwa.width  = rect.right - rect.left;
	xwa.height = rect.bottom - rect.top;

	rootx = point.x;
	rooty = point.y;       
#else
	XQueryPointer (__glutDisplay, __glutCurrentWindow->win,
		&root, &child, &rootx, &rooty, &wx, &wy, &mask);
	XGetWindowAttributes (__glutDisplay, root, &xwa);
#endif
	if (lite && rootx == old_rootx && rooty == old_rooty) {
		glutTimerFunc (TIMER, update, 0);
		return;
	}

	old_rootx = rootx;
	old_rooty = rooty;

	dy = rooty - (glutGet (GLUT_WINDOW_Y) + window_y / 2);
	dx = rootx - (glutGet (GLUT_WINDOW_X) + (cyclops ? window_x / 2 :
		window_x / 4));

	/*
	 * Angles fixed by Ralph Giles <ralphbla@sfu.ca>
	 */

	lrX = atan2 (4 * dy, xwa.height) * (120.0 / M_PI);
	lrY = atan2 (4 * dx, xwa.width) * (120.0 / M_PI);

	lX += 0.062; if (lX > 2.0 * M_PI) lX -= 2.0 * M_PI;
	lY += 0.053; if (lY > 2.0 * M_PI) lY -= 2.0 * M_PI;
	lZ += 0.087; if (lZ > 2.0 * M_PI) lZ -= 2.0 * M_PI;
	
	if (!cyclops) {
		dx = rootx - (glutGet (GLUT_WINDOW_X) + 3 * window_x / 4);
		rrX = atan2 (4 * dy, xwa.height) * (135.0 / M_PI);
		rrY = atan2 (4 * dx, xwa.width) * (135.0 / M_PI);
		
		rX += 0.066; if (rX > 2.0 * M_PI) rX -= 2.0 * M_PI;
		rY += 0.058; if (rY > 2.0 * M_PI) rY -= 2.0 * M_PI;
		rZ += 0.091; if (rZ > 2.0 * M_PI) rZ -= 2.0 * M_PI;
	}

	glutPostRedisplay ();
	glutTimerFunc (TIMER, update, 0);
}


static void display (void)
{
	glClear (GL_COLOR_BUFFER_BIT);

	glPushMatrix ();
	if (!cyclops)
		glTranslatef (-1.3, 0.0, 0.0);
	if (!lite)
		glTranslatef (0.1 * sin (lX), 0.1 * sin (lY), 0.2 * sin (lZ));	
	glRotatef (lrX, 1,0,0);
	glRotatef (lrY, 0,1,0);

	glutSolidSphere (1.1, SLICES, STACKS);
	glPopMatrix ();

	if (!cyclops) {
		glPushMatrix ();
		glTranslatef (1.3, 0.0, 0.0);
		if (!lite)
			glTranslatef (0.1 * sin (rX), 0.1 * sin (rY),
				0.2 * sin (rZ));
		glRotatef (rrX, 1, 0, 0);
		glRotatef (rrY, 0, 1, 0);

		glutSolidSphere (1.1, SLICES, STACKS);
		glPopMatrix ();
	}

	glutSwapBuffers ();
}


static void reshape (int width, int height)
{
	window_x = width;
	window_y = height;

	glViewport (0, 0, width, height);
	glMatrixMode (GL_PROJECTION);
	glLoadIdentity ();
	cyclops ? glFrustum (-0.5, 0.5, -0.5, 0.5, 5.0, 25.0) :
		glFrustum (-0.9, 0.9, -0.5, 0.5, 5.0, 25.0);
	glMatrixMode (GL_MODELVIEW);
	glLoadIdentity ();
	glTranslatef (0.0, 0.0, -15.0);
}


static void menu (int i)
{
	switch (i) {
	case FLAT:
		glShadeModel (GL_FLAT);
		break;
	case SMOOTH:
		glShadeModel (GL_SMOOTH);
		break;
	case CYCLOPS:
		if (!cyclops)
			glutReshapeWindow (window_x * 100 / 180, window_y);
		cyclops = 1;
		break;
	case HUMAN:
		if (cyclops)
			glutReshapeWindow (window_x * 180 / 100, window_y);
		cyclops = 0;
		break;
	case LITE:
		lite = 1;
		break;
	case REDRAW:
		lite = 0;
		break;
	case QUIT:
		exit (0);
	}

	glutPostRedisplay ();
}


static void help ()
{
	printf (
"Usage: gleyes [X options] [options]\n"
"Options:\n"
"	-c	Cyclops mode\n"
"	-f	Use flat shading\n"
"	-h	Print this help message\n"
"	-r	Render all frames\n"
"	-v	Print version number\n"
	);
}


int main (int argc, char **argv)
{
	GLfloat mat[4] = { 1.0, 1.0, 1.0, 1.0 };
	GLfloat pos[4] = {-1.0, 1.0, 1.0, 0.0 };
	int o, width, height;

	glutInitWindowSize (window_x, window_y);
	glutInit (&argc, argv);

	while ((o = getopt (argc, argv, "cfhrv")) != -1) {
		switch (o) {
		case 'c':
			cyclops = 1;
			break;
		case 'f':
			smooth = 0;
			break;
		case 'h':
			help ();
			exit (0);
		case 'r':
			lite = 0;
			break;
		case 'v':
			printf ("gleyes %s \n", VERSION);
			exit (0);
		}
	}

	glutInitDisplayMode (GLUT_RGB | GLUT_DOUBLE);
	glutCreateWindow ("GLeyes");

	width = glutGet (GLUT_WINDOW_WIDTH);
	height = glutGet (GLUT_WINDOW_HEIGHT);
	if (window_x != width || window_y != height) {
		window_x = width;
		window_y = height;
		if (cyclops)
			window_x = window_x * 180 / 100;
	}

	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	gluBuild2DMipmaps(GL_TEXTURE_2D, 3, 128, 128,
		GL_RGB, GL_UNSIGNED_BYTE, iris);

	glEnable (GL_LIGHTING);
	glEnable (GL_LIGHT0);
	glMaterialfv (GL_FRONT, GL_AMBIENT_AND_DIFFUSE, mat);
	glLightfv (GL_LIGHT0, GL_POSITION, pos);
	glEnable (GL_CULL_FACE);
	glShadeModel (smooth ? GL_SMOOTH : GL_FLAT);

	glTexGeni (GL_S, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);
	glTexGeni (GL_T, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);
	glEnable (GL_TEXTURE_GEN_S);
	glEnable (GL_TEXTURE_GEN_T);
	glEnable (GL_TEXTURE_2D);

	glMatrixMode (GL_TEXTURE);
	glTranslatef (0.5, 0.5, 0);
	glScalef (0.6, 0.6, 1.0);
	glMatrixMode (GL_MODELVIEW);

	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);

	glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

	glutReshapeFunc (reshape);
	glutDisplayFunc (display);
	glutTimerFunc (TIMER, update, 0);

	glutCreateMenu (menu);
	glutAddMenuEntry ("Flat shading", FLAT);
	glutAddMenuEntry ("Smooth shading", SMOOTH);
	glutAddMenuEntry ("Cyclops mode", CYCLOPS);
	glutAddMenuEntry ("Human mode", HUMAN);
	glutAddMenuEntry ("Render on move", LITE);
	glutAddMenuEntry ("Render all frames", REDRAW);
	glutAddMenuEntry ("Close", QUIT);
	glutAttachMenu (GLUT_RIGHT_BUTTON);

	if (cyclops)
		glutReshapeWindow (window_x * 100 / 180, window_y);

	glutMainLoop ();

	return 0;
}
