
#include <stdio.h>
#include <stdlib.h>

#include "doomdef.h"
#include "p_local.h"
#include "soundst.h"
#include "sounds.h"
#include "i_sound.h"

/*
 * These are not used, but should be (menu).
 * Maximum volume of a sound effect.
 * Internal default is max out of 0-15.
 */
long snd_SfxVolume = 15;

/* volume of music */
long snd_MusicVolume = 15;

#define S_MAX_VOLUME		127

/*
 * when to clip out sounds
 * Does not fit the large outdoor areas.
 */
#define S_CLIPPING_DIST		(1200*0x10000)

/*
 * Distance tp origin when sounds should be maxed out.
 * This should relate to movement clipping resolution
 * (see BLOCKMAP handling).
 * Originally: (200*0x10000).
 */
#define S_CLOSE_DIST		(160*0x10000)


#define S_ATTENUATOR		((S_CLIPPING_DIST-S_CLOSE_DIST)>>FRACBITS)

/* Adjustable by menu. */
#define NORM_VOLUME    		snd_SfxVolume

#define NORM_PITCH     		128
#define NORM_PRIORITY		32
#define NORM_SEP		128

#define S_PITCH_PERTURB		1
#define S_STEREO_SWING		(96*0x10000)

/* percent attenuation from front to back */
#define S_IFRACVOL		30

#define NA			0
#define S_NUMCHANNELS		2


/* the set of channels available */
#ifdef __DOSOUND__
static channel_t*	channels;
#endif /* __DOSOUND__ */

/* whether songs are mus_paused */
#ifdef __DOMUSIC__
static boolean		mus_paused;
#endif /* __DOMUSIC__ */

#ifdef __DOMUSIC__
static musicinfo_t*	mus_playing=0;
#endif /* __DOMUSIC__ */

/*
 * following is set
 *  by the defaults code in M_misc:
 * number of channels available
 */
long		numChannels;
#ifdef __DOSOUND__
static int		nextcleanup;
#endif /* __DOSOUND__ */


/*
 * Internals.
 */
#ifdef __DOSOUND__
static int S_getChannel( void* origin, sfxinfo_t* sfxinfo );

static int S_AdjustSoundParams( mobj_t*	listener, mobj_t* source,
				int* vol, int* sep, int* pitch );

static void S_StopChannel(int cnum);
#endif /* __DOSOUND__ */



/* ====================== Internal sound-functions ====================== */

#ifdef __DOSOUND__
static void S_StopChannel(int cnum)
{
  int		i;
  channel_t*	c = &channels[cnum];
  
  if (c->sfxinfo)
    {
      /* stop the sound playing */
      if (I_SoundIsPlaying(c->handle))
	{
	  I_StopSound(c->handle);
	}
      
      /*
       * check to see
       *  if other channels are playing the sound
       */
      for (i=0 ; i<numChannels ; i++)
	{
	  if (cnum != i
	      && c->sfxinfo == channels[i].sfxinfo)
	    {
	      break;
	    }
	}
      
      /* degrade usefulness of sound data */
      c->sfxinfo->usefulness--;
      
      c->sfxinfo = 0;
    }
}
#endif/* __DOSOUND__ */


/*
 * Changes volume, stereo-separation, and pitch variables
 *  from the norm of a sound effect to be played.
 * If the sound is not audible, returns a 0.
 * Otherwise, modifies parameters and returns 1.
 */
#ifdef __DOSOUND__
static int S_AdjustSoundParams( mobj_t*	listener, mobj_t* source, 
				int* vol, int* sep, int* pitch )
{
  fixed_t	approx_dist;
  fixed_t	adx;
  fixed_t	ady;
  angle_t	angle;

  /*
   * calculate the distance to sound origin
   *  and clip it if necessary
   */
  adx = abs(listener->x - source->x);
  ady = abs(listener->y - source->y);

  /* From _GG1_ p.428. Appox. eucledian distance fast. */
  approx_dist = adx + ady - ((adx < ady ? adx : ady)>>1);
  
  if ( approx_dist > S_CLIPPING_DIST )
    {
      return 0;
    }
  
  /* angle of source to listener */
  angle = R_PointToAngle2(listener->x, listener->y, source->x, source->y);
  
  if (angle > listener->angle)
    angle = angle - listener->angle;
  else
    angle = angle + (0xffffffff - listener->angle);
  
  angle >>= ANGLETOFINESHIFT;
  
  /* stereo separation */
  *sep = 128 - (FixedMul(S_STEREO_SWING,finesine[angle])>>FRACBITS);
  
  /* volume calculation */
  if (approx_dist < S_CLOSE_DIST)
    {
      *vol = snd_SfxVolume;
    }
  else 
    {
      /* distance effect */
      *vol = (snd_SfxVolume
	      * ((S_CLIPPING_DIST - approx_dist)>>FRACBITS)) / S_ATTENUATOR;
    }
  return (*vol > 0);
}
#endif /* __DOSOUND__ */


/*
 * S_getChannel :
 *   If none available, return -1.  Otherwise channel #.
 */
#ifdef __DOSOUND__
static int S_getChannel( void* origin, sfxinfo_t* sfxinfo )
{
  /* channel number to use */
  int		cnum;
  
  channel_t*	c;
  
  /* Find an open channel */
  for (cnum=0 ; cnum<numChannels ; cnum++)
    {
      if (!channels[cnum].sfxinfo)
	break;
      else if (origin &&  channels[cnum].origin ==  origin)
	{
	  S_StopChannel(cnum);
	  break;
	}
    }
  
  /* None available */
  if (cnum == numChannels)
    {
      /* Look for lower priority */
      for (cnum=0 ; cnum<numChannels ; cnum++)
	if (channels[cnum].sfxinfo->priority >= sfxinfo->priority) break;
      
      if (cnum == numChannels)
	{
	  /* FUCK!  No lower priority.  Sorry, Charlie. */
	  return -1;
	}
      else
	{
	  /* Otherwise, kick out lower priority. */
	  S_StopChannel(cnum);
	}
    }
  
  c = &channels[cnum];
  
  /* channel is decided to be cnum. */
  c->sfxinfo = sfxinfo;
  c->origin = origin;
  
  return cnum;
}
#endif /* __DOSOUND__ */


/* ====================== Global sound-functions ====================== */

/*
 * Initializes sound stuff, including volume
 * Sets channels, SFX and music volume,
 *  allocates channel buffer, sets S_sfx lookup.
 */
void S_Init( int sfxVolume, int	musicVolume )
{
#ifdef __DOSOUND__
  int i;
  
  /* Whatever these did with DMX, these are rather dummies now. */
  I_SetChannels();

  S_SetSfxVolume(sfxVolume);
  S_SetMusicVolume(musicVolume);
  
  /*
   * Allocating the internal channels for mixing
   * (the maximum numer of sounds rendered
   * simultaneously) within zone memory.
   */
  channels = 
    (channel_t *) Z_Malloc(numChannels*sizeof(channel_t), PU_STATIC, 0);
  
  /* Free all channels for use */
  for (i=0 ; i<numChannels ; i++)
    channels[i].sfxinfo = 0;

  /* no sounds are playing, and they are not mus_paused */
#ifdef __DOMUSIC__
  mus_paused = 0;
#endif /* __DOMUSIC__ */

  /* Note that sounds have not been cached (yet). */
  for (i=1 ; i<NUMSFX ; i++)
    S_sfx[i].lumpnum = S_sfx[i].usefulness = -1;
#endif /* __DOSOUND__ */

#ifdef _DEBUGSOUND
  printf("Done (working?): Calling S_Init...\n");
#endif
}


/*
 * Per level startup code.
 * Kills playing sounds at start of level,
 *  determines music if any, changes music.
 */
void S_Start(void)
{
#ifdef __DOSOUND__
  int cnum;
  
  /*
   * kill all playing sounds at start of level
   * (trust me - a good idea)
   */
  for (cnum=0 ; cnum<numChannels ; cnum++)
    if (channels[cnum].sfxinfo)
      S_StopChannel(cnum);

  /* start new music for the level */
#ifdef __DOMUSIC__
  {
  int mnum;
  mus_paused = 0;
  
  if (!shareware)
   {
    mnum = mus_e1m1 + (gameepisode-1)*9 + gamemap-1; /* (mus_runnin) */

#ifdef _DEBUGSOUND
  fprintf(stderr, "mnum: %d ; mus_e1m1: %d ; gameepisode: %d ; gamemap: %d\n",
                mnum, mus_e1m1, gameepisode, gamemap);
#endif
   }
   
  else
    {
      int spmus[]=
      {
	/* Song - Who? - Where? */
	
	mus_e3m4,	/* American	e4m1 */
	mus_e3m2,	/* Romero	e4m2 */
	mus_e3m3,	/* Shawn	e4m3 */
	mus_e1m5,	/* American	e4m4 */
	mus_e2m7,	/* Tim 	e4m5 */
	mus_e2m4,	/* Romero	e4m6 */
	mus_e2m6,	/* J.Anderson	e4m7 */
	mus_e2m5,	/* Shawn	e4m8 */
	mus_e1m9	/* Tim		e4m9 */
      };
      if (gameepisode < 4)
	mnum = mus_e1m1 + (gameepisode-1)*9 + gamemap-1;
      else
	mnum = spmus[gamemap-1];
    }

  S_ChangeMusic(mnum, true);
  }
#endif /* __DOMUSIC__ */
  nextcleanup = 15;  
#endif /* __DOSOUND__ */

#ifdef _DEBUGSOUND
  printf("(done?): Calling S_Start...\n");
#endif
}


void S_StartSoundAtVolume( void* origin_p, int sfx_id, int volume )
{
#ifdef __DOSOUND__
  int		rc;
  int		sep;
  int		pitch;
  int		priority;
  sfxinfo_t*	sfx;
  int		cnum;
  mobj_t*	origin = (mobj_t *) origin_p;
 
  
  /* check for bogus sound # */
  
  if (sfx_id < 1 || sfx_id > NUMSFX)
    I_Error("Bad sfx #: %d", sfx_id);
  
  sfx = &S_sfx[sfx_id];

  /* Initialize sound parameters */
  if (sfx->link)
    {
      pitch = sfx->pitch;
      priority = sfx->priority;
      volume += sfx->volume;
#ifdef _DEBUGSOUND
      fprintf(stderr, "sfx->pitch: %d\n", sfx->pitch);
      fprintf(stderr, "sfx->priority: %d\n", sfx->priority);
      fprintf(stderr, "sfx->volume: %d\n", sfx->volume);
#endif      
      if (volume < 1)
	return;
      
      if (volume > snd_SfxVolume)
	volume = snd_SfxVolume;
    }
  else
    {
      pitch = NORM_PITCH;
      priority = NORM_PRIORITY;
    }
  
  /*
   * Check to see if it is audible,
   *  and if not, modify the params
   */
  if (origin && origin != players[consoleplayer].mo)
    {
      rc = S_AdjustSoundParams(players[consoleplayer].mo, origin,
			       &volume, &sep, &pitch);
#ifdef _DEBUGSOUND
      fprintf(stderr, "rc: %d\n", rc);
#endif
      
      if ( origin->x == players[consoleplayer].mo->x
	   && origin->y == players[consoleplayer].mo->y)
	{
	  sep 	= NORM_SEP;
	}
      
      if (!rc)
	return;
    }
  else
    {
      sep = NORM_SEP;
    }

  /* hacks to vary the sfx pitches */
  if (sfx_id >= sfx_gntact /* I am right here ? (sfx_sawup) */
      && sfx_id <= sfx_phohit) /* I am right here ? (sfx_sawhit) */
    {
      pitch += 8 - (M_Random()&15);
      
      if (pitch<0)
	pitch = 0;
      else if (pitch>255)
	pitch = 255;
    }
  else if (sfx_id != sfx_itemup
	   && sfx_id != sfx_stnmov) /* I am right here ? (stx_tink) */
    {
      pitch += 16 - (M_Random()&31);
      
      if (pitch<0)
	pitch = 0;
      else if (pitch>255)
	pitch = 255;
    }

  /* kill old sound */
  S_StopSound(origin);

  /* try to find a channel */
  cnum = S_getChannel(origin, sfx);
  
#ifdef _DEBUGSOUND
  fprintf(stderr, "cnum: %d\n", cnum);
#endif

  if (cnum<0)
    return;

  /*
   * This is supposed to handle the loading/caching.
   * For some odd reason, the caching is done nearly
   *  each time the sound is needed?
   */
  
  /* get lumpnum if necessary */
  if (sfx->lumpnum < 0)
    {
#ifdef _DEBUGSOUND
      fprintf(stderr, "before getting Lumpnr. Name: %s, Number: %d\n",
	     sfx->name, sfx->lumpnum);
#endif      
      sfx->lumpnum = I_GetSfxLumpNum(sfx);
#ifdef _DEBUGSOUND
      fprintf(stderr, "after getting Lumpnr. Name: %s, Number: %d\n", 
	     sfx->name, sfx->lumpnum);
#endif
    }
  
  /* increase the usefulness */
  if (sfx->usefulness++ < 0)
    sfx->usefulness = 1;
  
  /*
   * Assigns the handle to one of the channels in the
   *  mix/output buffer.
   */
  
  /* use predefined different priority's ! */
  if (sfx->priority!=-1) priority=sfx->priority;
  
#ifdef _DEBUGSOUND
  fprintf(stderr, "Volume: %d, sep: %d, pitch: %d, priority: %d\n", 
	  volume, sep, pitch, priority );
#endif /* _DEBUGSOUND */
  
  channels[cnum].handle = I_StartSound( sfx_id, volume, sep, pitch, priority );
#endif /* __DOSOUND__ */
  
#ifdef _DEBUGSOUND
  fprintf(stderr, "(Done?): Calling S_StartSoundAtVolume...\n");
#endif
}


void S_StartSound( void* origin, int sfx_id )
{
#ifdef __DOSOUND__
  S_StartSoundAtVolume(origin, sfx_id, snd_SfxVolume);
#endif /* __DOSOUND__ */

#ifdef _DEBUGSOUND
  fprintf(stderr, "(Done?): Calling S_StartSound...\n");
#endif
}


void S_StopSound(void* origin)
{
#ifdef __DOSOUND__
  int cnum;
  
  for (cnum=0 ; cnum<numChannels ; cnum++)
    {
      if (channels[cnum].sfxinfo && channels[cnum].origin == origin)
	{
	  S_StopChannel(cnum);
	  break;
	}
    }
#endif /* __DOSOUND__ */

#ifdef _DEBUGSOUND
  fprintf(stderr, "(Done?): Calling S_StopSound...\n");
#endif
}


void S_PauseSound(void)
{
#ifdef __DOMUSIC__
  if (mus_playing && !mus_paused)
    {
      I_PauseSong(mus_playing->handle);
      mus_paused = true;
    }

#endif
  
#ifdef _DEBUGSOUND
  fprintf(stderr, "FIXME: Calling S_PauseSound...\n");
#endif
}


void S_ResumeSound(void)
{
#ifdef __DOMUSIC__
  if (mus_playing && mus_paused)
    {
      I_ResumeSong(mus_playing->handle);
      mus_paused = false;
    }

#endif

#ifdef _DEBUGSOUND
  fprintf(stderr, "FIXME: Calling S_ResumeSound...\n");
#endif
}


/*
 * Updates music & sounds
 */
void S_UpdateSounds(void* listener_p)
{
#ifdef __DOSOUND__
  int		audible;
  int		cnum;
  int		volume;
  int		sep;
  int		pitch;
  sfxinfo_t*	sfx;
  channel_t*	c;
  
  mobj_t*	listener = (mobj_t*)listener_p;
  
  for (cnum=0 ; cnum<numChannels ; cnum++)
    {
      c = &channels[cnum];
      sfx = c->sfxinfo;
      
      if (c->sfxinfo)
	{
	  if (I_SoundIsPlaying(c->handle))
	    {
	      /* initialize parameters */
	      volume = snd_SfxVolume;
	      pitch = NORM_PITCH;
	      sep = NORM_SEP;
	      
	      if (sfx->link)
		{
		  pitch = sfx->pitch;
		  volume += sfx->volume;
		  if (volume < 1)
		    {
		      S_StopChannel(cnum);
		      continue;
		    }
		  else if (volume > snd_SfxVolume)
		    {
		      volume = snd_SfxVolume;
		    }
		}

	      /*
	       * check non-local sounds for distance clipping
	       *  or modify their params
	       */
	      if (c->origin && listener_p != c->origin)
		{
		  audible = S_AdjustSoundParams(listener, c->origin,
						&volume, &sep, &pitch);
		  
		  if (!audible)
		    {
		      S_StopChannel(cnum);
		    }
		  else
		    I_UpdateSoundParams(c->handle, volume, sep, pitch);
		}
	    }
	  else
	    {
	      /*
	       * if channel is allocated but sound has stopped,
	       *  free it
	       */
	      S_StopChannel(cnum);
	    }
	}
    }
#endif /* __DOSOUND__ */  

#ifdef _DEBUGSOUND
  fprintf(stderr, "(Done?): Calling S_UpdateSounds...\n");
#endif
}


/*
 * Starts some music with the music id found in sounds.h.
 */
void S_StartMusic(int music_id)
{
#ifdef __DOMUSIC__
  S_ChangeMusic(music_id, false);
#endif

#ifdef _DEBUGSOUND
  fprintf(stderr, "FIXME: Calling S_StartMusic...\n");
#endif
}


void S_StopMusic(void)
{
#ifdef __DOMUSIC__
  if (mus_playing)
    {
      if (mus_paused)
	I_ResumeSong(mus_playing->handle);
      
      I_StopSong(mus_playing->handle);
      I_UnRegisterSong(mus_playing->handle);
      Z_ChangeTag(mus_playing->data, PU_CACHE);
      
      mus_playing->data = 0;
      mus_playing = 0;
    }

#endif

#ifdef _DEBUGSOUND
  fprintf(stderr, "FIXME: Calling S_StopMusic...\n");
#endif
}


void S_ChangeMusic( int	music_id, int looping )
{
#ifdef __DOMUSIC__
  musicinfo_t*	music;
 
  if ( (music_id <= mus_None)
       || (music_id >= NUMMUSIC) )
    {
      I_Error("Bad music number %d", music_id);
    }

  music = &S_music[music_id];
  
  if (mus_playing == music)
    return;
  
  /* shutdown old music */
  S_StopMusic();
  
  /* get lumpnum if neccessary */
  if (!music->lumpnum)
    {
      music->lumpnum = W_GetNumForName(music->name);
    }
  
  /* load & register it */
  music->data = (void *) W_CacheLumpNum(music->lumpnum, PU_MUSIC);
  music->handle = I_RegisterSong(music->data, W_LumpLength(music->lumpnum));
  
  /* play it */
  I_PlaySong(music->handle, looping);
  
  mus_playing = music;

#endif

#ifdef _DEBUGSOUND
  fprintf(stderr, "FIXME: Calling S_ChangeMusic...\n");
#endif
}


void S_SetMusicVolume(int volume)
{
#ifdef __DOMUSIC__
  if (volume < 0 || volume > 15)
    {
      I_Error("Attempt to set music volume at %d",
	      volume);
    }
  
  I_SetMusicVolume(volume);
  snd_MusicVolume = volume;

#endif

#ifdef _DEBUGSOUND
  fprintf(strerr, "FIXME: Calling S_SetMusicVolume...\n");
#endif
}


void S_SetSfxVolume(int volume)
{
#ifdef __DOSOUND__
  if (volume < 0 || volume > 127)
    I_Error("Attempt to set sfx volume at %d", volume);
  
  snd_SfxVolume = volume;
#endif /* __DOSOUND__ */
  
#ifdef _DEBUGSOUND
  fprintf(stderr, "FIXME: Calling S_SetSfxVolume...\n");
#endif
}
