/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2003 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "gnome-cmd-includes.h"
#include "gnome-cmd-chmod-dialog.h"
#include "gnome-cmd-chmod-component.h"
#include "gnome-cmd-file-funcs.h"
#include "gnome-cmd-dir-funcs.h"
#include "widget-factory.h"
#include "dir_pool.h"
#include "utils.h"


typedef enum {
	CHMOD_ALL_FILES,
	CHMOD_DIRS_ONLY,
	CHMOD_MAX
} ChmodRecursiveMode;

static gchar *recurse_opts[CHMOD_MAX] = {
	N_("All files"),
	N_("Directories only")
};

struct _GnomeCmdChmodDialogPrivate
{
	GList *files;
	GnomeCmdFile *finfo;
	GnomeVFSFilePermissions perms;

	GtkWidget *chmod_component;
	GtkWidget *recurse_check;
	GtkWidget *recurse_combo;
};


static GnomeDialogClass *parent_class = NULL;



static void do_chmod (GnomeCmdFile *in_finfo, GnomeVFSFilePermissions perm,
					  gboolean recursive, ChmodRecursiveMode mode)
{
	g_return_if_fail (in_finfo != NULL);
	g_return_if_fail (in_finfo->info != NULL);

	if (!(recursive && mode == CHMOD_DIRS_ONLY
		  && in_finfo->info->type != GNOME_VFS_FILE_TYPE_DIRECTORY)) {
		GnomeVFSResult ret = gnome_cmd_file_chmod (in_finfo, perm);

		if (ret != GNOME_VFS_OK) {
			gchar *msg = g_strdup_printf (
				_("Could not chmod %s\n%s"),
				gnome_cmd_file_get_path (in_finfo),
				gnome_vfs_result_to_string (ret));
			gnome_error_dialog (msg);
			g_free (msg);
		}
		else if (!recursive) {
			return;
		}
	}
	
	if (in_finfo->info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
		const gchar *dir_uri = gnome_cmd_file_get_uri_str (in_finfo);
		GnomeCmdDir *dir = dir_pool_get (dir_uri);
		GList *files, *tmp;

		gnome_cmd_dir_ref (dir);
		gnome_cmd_dir_list_files (dir, FALSE);
		gnome_cmd_dir_get_files (dir, &files);
		tmp = files;
			
		while (tmp) {
			GnomeCmdFile *finfo = (GnomeCmdFile*)tmp->data;
			if (strcmp (finfo->info->name, ".") != 0
				&& strcmp (finfo->info->name, "..") != 0
				&& finfo->info->flags != GNOME_VFS_FILE_FLAGS_SYMLINK) {
				do_chmod (finfo, perm, TRUE, mode);
			}
			
			tmp = tmp->next;
		}
		gnome_cmd_dir_unref (dir);
	}
}


static void do_chmod_files (GnomeCmdChmodDialog *dialog)
{
	GList *tmp = dialog->priv->files;
	
	while (tmp)
	{
		GnomeCmdFile *finfo = (GnomeCmdFile*)tmp->data;
		gboolean recursive = gtk_toggle_button_get_active (
			GTK_TOGGLE_BUTTON (dialog->priv->recurse_check));
		ChmodRecursiveMode mode;
		gchar *mode_text = gtk_entry_get_text (
			GTK_ENTRY (GTK_COMBO (dialog->priv->recurse_combo)->entry));
		if (strcmp (mode_text, recurse_opts[CHMOD_ALL_FILES]) == 0)
			mode = CHMOD_ALL_FILES;
		else
			mode = CHMOD_DIRS_ONLY;
		do_chmod (finfo, dialog->priv->perms, recursive, mode);
		tmp = tmp->next;
	}	
}


static void show_perms (GnomeCmdChmodDialog *dialog)
{
	gnome_cmd_chmod_component_set_perms (
		GNOME_CMD_CHMOD_COMPONENT (dialog->priv->chmod_component),
		dialog->priv->perms);
}


static void on_ok (GtkButton *button, GnomeCmdChmodDialog *dialog)
{
	do_chmod_files (dialog);
	gnome_cmd_file_list_free (dialog->priv->files);
	gtk_widget_hide (GTK_WIDGET (dialog));
}


static void on_cancel (GtkButton *button, GnomeCmdChmodDialog *dialog)
{
	gnome_cmd_file_list_free (dialog->priv->files);
	gtk_widget_hide (GTK_WIDGET (dialog));
}

static void on_toggle_recurse (GtkToggleButton *togglebutton, GnomeCmdChmodDialog *dialog)
{
	gtk_widget_set_sensitive (dialog->priv->recurse_combo,
							  gtk_toggle_button_get_active (togglebutton));
}


static void on_perms_changed (GnomeCmdChmodComponent *component, GnomeCmdChmodDialog *dialog)
{
	dialog->priv->perms =
		gnome_cmd_chmod_component_get_perms (
			GNOME_CMD_CHMOD_COMPONENT (dialog->priv->chmod_component));
}


/*******************************
 * Gtk class implementation
 *******************************/

static void
destroy (GtkObject *object)
{
	GnomeCmdChmodDialog *dialog = GNOME_CMD_CHMOD_DIALOG (object);

	g_free (dialog->priv);
	
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


static void
map (GtkWidget *widget)
{
	if (GTK_WIDGET_CLASS (parent_class)->map != NULL)
		GTK_WIDGET_CLASS (parent_class)->map (widget);
}


static void
class_init (GnomeCmdChmodDialogClass *class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = GTK_OBJECT_CLASS (class);
	widget_class = GTK_WIDGET_CLASS (class);

	parent_class = gtk_type_class (gnome_dialog_get_type ());
	object_class->destroy = destroy;
	widget_class->map = map;
}


static void
init (GnomeCmdChmodDialog *dialog)
{
	GtkWidget *chmod_dialog = GTK_WIDGET (dialog);
	GtkWidget *vbox;
	GtkWidget *hsep;
	GtkWidget *button;

	dialog->priv = g_new (GnomeCmdChmodDialogPrivate, 1);
	
	gtk_window_set_policy (GTK_WINDOW (chmod_dialog), FALSE, FALSE, FALSE);
	gtk_window_set_position (GTK_WINDOW (chmod_dialog), GTK_WIN_POS_CENTER);
	gtk_window_set_title (GTK_WINDOW (chmod_dialog), _("Access Permissions"));

	vbox = GNOME_DIALOG (chmod_dialog)->vbox;
	gtk_object_set_data (GTK_OBJECT (chmod_dialog), "vbox", vbox);
	gtk_widget_show (vbox);

	dialog->priv->chmod_component = gnome_cmd_chmod_component_new (0);
	gtk_widget_ref (dialog->priv->chmod_component);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog),
							  "chmod_component", dialog->priv->chmod_component,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (dialog->priv->chmod_component);
	gtk_box_pack_start (GTK_BOX (vbox), dialog->priv->chmod_component, TRUE, TRUE, 0);

	
	hsep = create_hsep (chmod_dialog);
	gtk_box_pack_start (GTK_BOX (vbox), hsep, TRUE, TRUE, 0);

	dialog->priv->recurse_check = create_check (chmod_dialog, _("Apply Recursively for"), "check");
	gtk_box_pack_start (GTK_BOX (vbox), dialog->priv->recurse_check, TRUE, TRUE, 0);
	
	dialog->priv->recurse_combo = create_combo (chmod_dialog);
	gtk_box_pack_start (GTK_BOX (vbox), dialog->priv->recurse_combo, TRUE, TRUE, 0);
	gtk_widget_set_sensitive (dialog->priv->recurse_combo, FALSE);
	
	
	gnome_dialog_append_button (GNOME_DIALOG (chmod_dialog), GNOME_STOCK_BUTTON_OK);
	button = GTK_WIDGET (g_list_last (GNOME_DIALOG (chmod_dialog)->buttons)->data);
	gtk_widget_ref (button);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "button", button,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (button);
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	gtk_signal_connect (GTK_OBJECT (button), "clicked",
						GTK_SIGNAL_FUNC (on_ok), chmod_dialog);

	gnome_dialog_append_button (GNOME_DIALOG (chmod_dialog), GNOME_STOCK_BUTTON_CANCEL);
	button = GTK_WIDGET (g_list_last (GNOME_DIALOG (chmod_dialog)->buttons)->data);
	gtk_widget_ref (button);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "button", button,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (button);
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	gtk_signal_connect (GTK_OBJECT (button), "clicked",
						GTK_SIGNAL_FUNC (on_cancel), chmod_dialog);	

	gtk_signal_connect (GTK_OBJECT (dialog->priv->recurse_check), "toggled",
						GTK_SIGNAL_FUNC (on_toggle_recurse), chmod_dialog);
	gtk_signal_connect (GTK_OBJECT (dialog->priv->chmod_component), "perms_changed",
						GTK_SIGNAL_FUNC (on_perms_changed), chmod_dialog);
}




/***********************************
 * Public functions
 ***********************************/

GtkWidget*
gnome_cmd_chmod_dialog_new (GList *files)
{
	GList *strings = NULL;
	GnomeCmdChmodDialog *dialog;

	g_return_val_if_fail (files != NULL, NULL);

	dialog = gtk_type_new (gnome_cmd_chmod_dialog_get_type ());
	dialog->priv->files = gnome_cmd_file_list_copy (files);

	dialog->priv->finfo = (GnomeCmdFile*)dialog->priv->files->data;
	g_return_val_if_fail (dialog->priv->finfo != NULL, NULL);
		
	dialog->priv->perms = dialog->priv->finfo->info->permissions;
	
	show_perms (dialog);

	strings = g_list_append (strings, recurse_opts[0]);
	strings = g_list_append (strings, recurse_opts[1]);
	
	gtk_combo_set_popdown_strings (GTK_COMBO (dialog->priv->recurse_combo), strings);
	
	return GTK_WIDGET (dialog);
}



GtkType
gnome_cmd_chmod_dialog_get_type         (void)
{
	static GtkType dlg_type = 0;

	if (dlg_type == 0)
	{
		GtkTypeInfo dlg_info =
		{
			"GnomeCmdChmodDialog",
			sizeof (GnomeCmdChmodDialog),
			sizeof (GnomeCmdChmodDialogClass),
			(GtkClassInitFunc) class_init,
			(GtkObjectInitFunc) init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		dlg_type = gtk_type_unique (gnome_dialog_get_type (), &dlg_info);
	}
	return dlg_type;
}



