/*
    goldstr, a golden string calculator
    Copyright (C) 2001  Andrea Molteni <andreamolteni@yahoo.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#define _GNU_SOURCE

#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>

#define M_PHI		1.61803398874989490252

#define REUSE_MIN	5L

int goldout(FILE *stream, long int wd, long int lp, long int rp, char zero, char one)
{
  char c;
  while(rp < wd)
  {
    if (putc(one, stream) == EOF)
      return 1;
    if (++rp >= wd)
      break;
    fseek(stream, lp++, SEEK_SET);
    c = getc(stream);
    // fseek(stream, 0L, SEEK_END) couldn't be used anymore
    // because causes troubles when reusing files
    fseek(stream, rp, SEEK_SET);
    if (c == one)
    {
      putc(zero, stream);
      rp++;
    }
  }
  return 0;
}

int main(int argc, char *argv[])
{
  FILE *fout;
  long int wd = 0;	// `digits wanted'
  long int lp = 1;	// `left pointer'
  long int rp = 2;	// `right pointer'
  char zero = '0';
  char one = '1';
  int i;
  
  int zero_flag = 0;
  int one_flag = 0;
  int reuse_flag = 0;

  for (;;)
  {
    int option_index = 0;
    static struct option long_options[] =
    {
      {"zero", 1, 0, '0'},
      {"one", 1, 0, '1'},
      {"digits", 1, 0, 'd'},
      {"help", 0, 0, 'h'},
      {"reuse", 0, 0, 'r'},
      {"version", 0, 0, 'v'},
      {0, 0, 0, 0}
    };

    i = getopt_long(argc, argv, "0:1:d:hrv",
		    long_options, &option_index);

    if (i == -1)
      break;
    switch (i)
    {
      case '0':
        if (!optarg[0] || optarg[1])
	{
	  fprintf(stderr, "%s: not a character -- `%s'\n", argv[0], optarg);
	  return 1;
	}
	zero = optarg[0];
	zero_flag = 1;
	break;
      case '1':
	if (!optarg[0] || optarg[1])
	{
	  fprintf(stderr, "%s: not a character -- `%s'\n", argv[0], optarg);
	  return 1;
	}
	one = optarg[0];
	one_flag = 1;
	break;
      case 'd':
	if ((wd = atol(optarg)) < 1)
	{
	  fprintf(stderr, "%s: bad number of digits to compute\n", argv[0]);
	  return 1;
	}
	break;
      case 'h':
	fprintf(stdout, "goldstr, version 1.6.0, Jun 14 2001\n\n"
                        "Usage: %s [options] -d digits file\n\n"
			"  -0, --zero          sets the zero digit\n"
			"  -1, --one           sets the one digit\n"
                        "  -d, --digits        specifies the number of digits\n"
                        "  -h, --help          print this help, then exit\n"
			"  -r, --reuse         reuse an already computed file\n"
	                "  -v, --version       print the program version, then exit\n"
                        "\ndigits must be a positive integer number.\n"
                        "\nReport bugs to <andreamolteni@yahoo.it>.\n",
			argv[0]);
	return 0;
      case 'r':
	reuse_flag = 1;
	break;
      case 'v':
	fprintf(stdout, "goldstr, version 1.6.0\n"
		        "\nCopyright (C) 2001 Andrea Molteni\n"
		        "This is free software; see the source for copying conditions.  There is NO\n"
		        "warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n");
        return 0;
      case '?':
	return 1;
      default:
	return 1;
    }
  }

  if ((zero_flag | one_flag) && reuse_flag)
  {
    fprintf(stderr, "%s: cannot set characters when reusing a file\n", argv[0]);
    return 1;
  }

  if (wd && optind == argc - 1)
  {
    if ((fout = fopen(argv[optind], reuse_flag ? "rb+" : "wb+")) == NULL)
    {
      fprintf(stderr, "%s: couldn't open %s\n", argv[0], argv[optind]);
      return 1;
    }
  }
  else
  {
    fprintf(stderr, "%s: bad number of arguments\n", argv[0]);
    fprintf(stdout, "%s: for help, type: `%s --help'\n", argv[0], argv[0]);
    return 1;
  }

  if (reuse_flag)
  {
    if ((one = fgetc(fout)) == EOF)
    {
      fprintf(stdout, "%s: couldn't find the one character, using `1'\n", argv[0]);
      one = '1';
      // one_flag = 0;	// it must be already clear
    }
    else
      one_flag = 1;
    if (one_flag)
      if ((zero = fgetc(fout)) == EOF)
      {
	fprintf(stdout, "%s: couldn't find the zero character, using `0'\n", argv[0]);
	zero = '0';
      }
    fseek(fout, 0L, SEEK_END);
    rp = ftell(fout);
    if (rp >= wd)
    {
      fprintf(stderr, "%s: needed digits already computed\n", argv[0]);
      fclose(fout);
      return 1;
    }
    // this is a really weird algorithm...
    lp = (double) rp / M_PHI;
    if (lp != (long int) ((double) (rp + 1) / M_PHI))
    {
      if (lp != (long int) ((double) (rp - 1) / M_PHI))
      {
        fputc(zero, fout);
	rp++;
      }
      lp++;
    }
  }

  if (rp < REUSE_MIN)
  {
    fseek(fout, 0L, SEEK_SET);
    fputc(one, fout);
    if (wd > 1)
      fputc(zero, fout);
    lp = 1;
    rp = 2;
  }
  if (goldout(fout, wd, lp, rp, zero, one))
  {
    fprintf(stderr, "%s: error while accessing %s\n", argv[0], argv[optind]);
    fclose(fout);
    return 1;
  }

  fclose(fout);

  return 0;
}

