/*-
 * Copyright (c) 2004 Patrick Mauritz <oxygene@openbios.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
#include "fpk.h"

/* internal functions */

char* fpk__readlfile(FILE* fpk) {
  unsigned char tmp;
  unsigned int len;
  char* result;

  fread(&tmp,1,1,fpk);
  len=tmp;

  result=(char*)malloc(len);
  fread(result,1,len,fpk);
  
  return result;
}

unsigned int fpk__readLE4(FILE* fpk) {
  unsigned char tmp[4];
  fread(&tmp,1,4,fpk);
  return (tmp[3]<<24)+(tmp[2]<<16)+(tmp[1]<<8)+tmp[0];
}

void fpk__clearfiles(struct fpkmeta* meta) {
  signed int c;
  if (!(meta->files)) return; /* clean already */
  for (c=0;c<meta->numfiles;c++) {
    if (meta->files[c].name) free(meta->files[c].name);
    if (meta->files[c].directory) free(meta->files[c].directory);
  }
  free(meta->files);
}

void fpk__getmeta(struct fpkmeta* meta) {
  unsigned int c;

  fseek(meta->fpk,8L,SEEK_SET);
  fread(meta->shortname,1,13,meta->fpk);
  fread(meta->longname,1,64,meta->fpk);
  fread(meta->author,1,36,meta->fpk);

  if (meta->rootpath) free(meta->rootpath);
  c=getc(meta->fpk);
  (void)getc(meta->fpk);
  meta->rootpath=(char*)malloc(c);
  fread(meta->rootpath,1,c,meta->fpk);
  
  return;
}


/* exported functions */

int fpkmagic(struct fpkmeta* meta) {
  char test[8];
  rewind(meta->fpk);
  fread(&test,1,8,meta->fpk);
  return memcmp(test,"GPKG\0\0\0\0",8);
}

void fpkgetdir(struct fpkmeta* meta) {
  unsigned int c,c2;

  fseek(meta->fpk,136L,SEEK_SET);
  c=fpk__readLE4(meta->fpk);
  
  meta->size=0;
  
  if (meta->files) fpk__clearfiles(meta);
  meta->files=(struct fpkfile*)malloc(sizeof(struct fpkfile)*c);
  meta->numfiles=c;
  
  for (c2=0;c2<c;c2++) {
    char v=getc(meta->fpk);
    char *dir, *file;
    unsigned int length;
    if (v=='2') {
      // finish
      meta->files[c2].offset=-1;
      meta->files[c2].size=-1;
      meta->files[c2].directory=0;
      meta->files[c2].name=0;
      break;
    }
    dir=fpk__readlfile(meta->fpk);
    file=fpk__readlfile(meta->fpk);
    meta->files[c2].directory=dir;
    meta->files[c2].name=file;
    if (v=='0') {
      length=0;
      meta->files[c2].offset=0;
      // directory
    }
    if (v=='1') {
      length=fpk__readLE4(meta->fpk);
      meta->files[c2].offset=ftell(meta->fpk);
      fseek(meta->fpk,length,SEEK_CUR);
      // file
    }
    meta->files[c2].size=length;
    meta->size+=length;
  }
}

signed int fpkinit(struct fpkmeta *meta, const char *filename) {
  meta->files=0;
  meta->rootpath=0;
  meta->iter=-1;
  meta->numfiles=0;
  meta->size=0;

  meta->fpk=fopen(filename,"r");
  if (!meta->fpk) return -1;
  
  if (fpkmagic(meta)) {
    fclose(meta->fpk);
    return -1;
  }
  fpk__getmeta(meta);
  return 0;
}

void fpkclose(struct fpkmeta *meta) {
  fclose(meta->fpk);
  fpk__clearfiles(meta);
}

void fpkrewind(struct fpkmeta *meta) {
  meta->iter=-1;
}

struct fpkfile* fpknextfile(struct fpkmeta *meta) {
  while ((meta->iter<meta->numfiles) && (meta->files[++meta->iter].offset<=0)) {};
  if (meta->iter>=meta->numfiles)
    return NULL;
  else
    return &meta->files[meta->iter];
}

struct fpkfile* fpknextdir(struct fpkmeta *meta) {
  while ((meta->iter<meta->numfiles) && (meta->files[++meta->iter].offset!=0)) {};
  if (meta->iter>=meta->numfiles)
    return NULL;
  else
    return &meta->files[meta->iter];
}

unsigned char* fpkgetfile(struct fpkmeta *meta, struct fpkfile *file){
  char *tmp;
  if (file->offset<=0) return NULL; /* not a file */
  fseek(meta->fpk,file->offset,SEEK_SET);
  tmp=(unsigned char*)malloc(file->size);
  fread(tmp,file->size,1,meta->fpk);
  return tmp;
}
