#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <libgen.h>
#include <fcntl.h>

#include "usb_drv.h"
#include "gp32cmds.h"

unsigned char buffer[0x100];

unsigned int readLE4(const unsigned char* data) {
	return (data[0]+(data[1]<<8)+(data[2]<<16)+(data[3]<<24));
}

struct file_info *fileinfo(const char* path) {
	signed int c=0,i=-1;
	struct file_info *data,*result=0;
	char* fullpath;
	
	fullpath=(char*)malloc(strlen(path));
	memcpy(fullpath,path,strlen(path)+1);
	
	while (c<(signed)strlen(fullpath)-1){
		if ((fullpath[c]=='/') || (fullpath[c]=='\\')) {
			i=c;
		}
		fullpath[c]=toupper(fullpath[c]);
		c++;
	}
	fullpath[c]=toupper(fullpath[c]);
	if (i>-1) {
	    fullpath[i]=0;
	    data=gp_dir(fullpath);
	} else
	    data=gp_dir(NULL);
	for(c=0; data[c].name; c++) {
		if (!strcmp(data[c].name,fullpath+1+i)) {
			result=(struct file_info*)malloc(sizeof(struct file_info));
			result->is_dir=data[c].is_dir;
			result->size=data[c].size;
			result->name=(char*)malloc(strlen(data[c].name)+1);
			strcpy(result->name,data[c].name);
		}
		free(data[c].name);
	}
	free(fullpath);
	free(data);
	return result;
}

/* convert unix-like paths to gp32 format */
/* obligation with the recipient to free the memory of
   the string. it always works as it's always a new
   string. */
char *mk_gppath(char *directory, int isdir) {
	int i=0,j;
	char *gppath;
	int path_length;


	if((directory==NULL) || (strlen(directory)==0)) {
		gppath=(char*)malloc(5);
		gppath[0]='g';
		gppath[1]='p';
		gppath[2]=':';
		gppath[3]='\\';
		gppath[4]='\0';
		return gppath;
	}

	j=((!isdir) || directory[strlen(directory)-1]=='\\' || directory[strlen(directory)-1]=='/')?0:1;

	if(!strncmp(directory, "gp:\\", 4)) {
		gppath=(char*)malloc(strlen(directory)+1+j);
		strcpy(gppath,directory);
		if (isdir) {
			gppath[strlen(gppath)+j]='\0';
			gppath[strlen(gppath)+j-1]='\\';
		}
		return gppath;
	}

	if(!strncmp(directory, "/", 1)) i=1;
	if(!strncmp(directory, "\\", 1)) i=1;

	path_length=(4+strlen(directory)+1-i+j);
	gppath=malloc(path_length+j);
	snprintf(gppath, path_length+j, "gp:\\%s%s", directory+i,(j==1)?"\\":"");
	for(i=0; gppath[i]; i++) {
		if(gppath[i]=='/') gppath[i]='\\';
	}
	return gppath;

}

int gp_connect() {
	return usb_connect();
}

void gp_disconnect() {
	usb_disconnect();
}

unsigned char* fxecheck(unsigned char* data, unsigned int *size){
	unsigned int i,keysize;
	
	if ((data[0]!='f') || (data[1]!='x') || (data[2]!='e') || (data[3]!=' '))
	    return data; /* not an fxe */
	size[0]=readLE4(data+0x45c);
	keysize=readLE4(data+0x460);
	for (i=0;i<keysize;i++){
		data[0x464+i]^=0xff;
	}
	for (i=0;i<size[0];i++) {
		data[0x464+keysize+i]^=data[0x464+(i%keysize)];
	}
	return data+0x464+keysize;
}

// run command for pc-link
// copied from Mithris' GpRun
int gp_run(char *file) {
	struct stat filestat;
	unsigned int filesize;
	int src_fd;
	unsigned char *data, *data2;

	if(stat(file, &filestat)==-1) {
		fprintf(stderr, "can't open file %s\n", file);
		return -1;
	}
	filesize=filestat.st_size;
	data=(char*)malloc(filesize);

	/* open source file */
	src_fd=open(file, O_RDONLY);
	if(src_fd==-1) {
		fprintf(stderr, "can't open file %s\n", file);
		return -1;
	}

	if (!read(src_fd, data, filesize)) {
		close(src_fd);
		return -1;
	}
	close(src_fd);

	data2=fxecheck(data,&filesize);
	if (gp_run_data(data2,filesize)) {
		free(data);
		return -1;
	}
	free(data);
	return 0;
}

int gp_run_data(unsigned char* exe, unsigned int size) {
	int result;
	unsigned int count=0;

	/* send 0xee */
	memset(buffer, 0, 0x100);
	*buffer=0xee;
	result=usb_write(buffer, 0x100);
	if(result==-1) return -1;
		//result=usb_read(buffer, 0x100);

	/* send file size */
	memset(buffer, 0, 0x100);
	(*(unsigned int *)buffer)=size;
	/* send path length and path */
	buffer[4]=4;
	buffer[5]='g';
	buffer[6]='p';
	buffer[7]=':';
	buffer[8]='/';

	result=usb_write(buffer, 0x100);
	if(result==-1) return -1;

	/* expect 0x00? seems to return 0x3 - need to do a read anyway */
	memset(buffer, 0, 0x100);
	result=usb_read(buffer, 0x100);
	if(result==-1) {
		fprintf(stderr, "error in usb read\n");
		return -1;
	}
	//if(*buffer!=0x0) fprintf(stderr, "read 0x%x\n", *buffer);

	while(count<size && memcpy(buffer,exe+count,(size-count)>0x100?0x100:(size-count))) {
		result=usb_write(buffer, 0x100);
		if(result==-1) {
			return -1;
		}
		count+=0x100;
		memset(buffer, 0, 0x100);
	}
	return 0;
}

// return an array of file_info structs for the given dir
/* obligation with the recipient to free the struct file_info*
   and every ->name member of it */
struct file_info *gp_dir(char *directory) {
	unsigned char *gpdirectory;
	int result;
	int entry_size;
	struct file_info *dir_entries;
	int num_entries;
	int reading_directory;

	memset(buffer, 0, 0x100);
	*buffer=0x95;
	result=usb_write(buffer, 0x100);
	if(result==-1) return NULL;

	gpdirectory=mk_gppath(directory,1);

	memset(buffer, 0, 0x100);
	*buffer=strlen(gpdirectory);
	strncpy(buffer+1, gpdirectory, strlen(gpdirectory));
	result=usb_write(buffer, 0x100);
	free(gpdirectory); gpdirectory=0;
	if(result==-1) return NULL;

	dir_entries=calloc(2048, sizeof(struct file_info));
	reading_directory=1;
	num_entries=0;
	while(reading_directory) {
		result=usb_read(buffer, 0x100);
		if(result==-1) return NULL;
		if((*buffer)!=0x96) {
			reading_directory=0;
		} else {
			// get entry name
			entry_size=buffer[2];
			dir_entries[num_entries].name=malloc(entry_size+1);
			strncpy(dir_entries[num_entries].name, buffer+3, entry_size);
			dir_entries[num_entries].name[entry_size]='\0';

			// get entry type
			if(buffer[1]==1) { // entry is a directory
				dir_entries[num_entries].is_dir=1;
			} else { // entry is a file
				dir_entries[num_entries].is_dir=0;
			}

			// get entry size
			dir_entries[num_entries].size=(((unsigned int)*(buffer+3+entry_size+3)<<24)
				|((unsigned int)*(buffer+3+entry_size+2)<<16)
				|((unsigned int)*(buffer+3+entry_size+1)<<8)
				|((unsigned int)*(buffer+3+entry_size)));

			num_entries++;
			if(num_entries>=2047) {
				reading_directory=0;
			} else {
				memset(buffer, 0, 0x100);
				*buffer=0x3;
				result=usb_write(buffer, 0x100);
				if(result==-1) return NULL;
			}
		}
	}
	dir_entries[num_entries].name=NULL;
	return dir_entries;
}

int gp_end_link_mode() {
	int result;

	memset(buffer, 0, 0x100);
	*buffer=0x10;
	result=usb_write(buffer, 0x100);
	if(result==-1) {
		return -1;
	} else {
		return 0;
	}
}

/* it's the obligation of the recipient to free the memory it 
   gets the pointer to! */
struct smc_info *gp_info() {
	int result;
	struct smc_info *info;

	memset(buffer, 0, 0x100);
	*buffer=0x20;
	result=usb_write(buffer, 0x100);
	if(result==-1) return NULL;
	memset(buffer, 0, 0x100);
	result=usb_read(buffer, 0x100);
	if(result==-1) return NULL;
	info=malloc(sizeof(struct smc_info));
	info->capacity=(((unsigned int)*(buffer+4)<<24)
		|((unsigned int)*(buffer+3)<<16)
		|((unsigned int)*(buffer+2)<<8)
		|((unsigned int)*(buffer+1)));
	info->bytes_used=(((unsigned int)*(buffer+8)<<24)
		|((unsigned int)*(buffer+7)<<16)
		|((unsigned int)*(buffer+6)<<8)
		|((unsigned int)*(buffer+5)));
	info->bytes_free=(((unsigned int)*(buffer+12)<<24)
		|((unsigned int)*(buffer+11)<<16)
		|((unsigned int)*(buffer+10)<<8)
		|((unsigned int)*(buffer+9)));
	return info;
}

int gp_put(char *src, char *dst, int convert) {
	struct stat srcstat;
	char *bname, *bname_tmp;
	char *srcdata;
	char bname83[13];
	int i,j;
	int gotdot;
	int src_fd;
	
	if(stat(src, &srcstat)==-1) {
		fprintf(stderr, "can't open file\n");
		return -1;
	}

	/* send filename */
	bname_tmp=malloc(strlen(src)+1);
	strcpy(bname_tmp, src);
	bname=basename(bname_tmp);
	if(convert) {
		/* convert to 8.3 filename format */
		//bname=basename(src);

		/* convert spaces */
		for(i=0; bname[i]; i++) {
			if(bname[i]==' ') bname[i]='_';
		}

		/* lose any except the last dot */
		gotdot=0;
		for(i=strlen(bname); i>0; i--) {
			if(bname[i]=='.') {
				if(gotdot==0) {
					gotdot=1;
				} else {
					bname[i]='_';
				}
			}
		}
		if(strlen(bname)>12||gotdot==0) {
			for(i=0; i<6; i++) {
				bname83[i]=bname[i];
			}
			bname83[i++]=0x30+(rand()%10);
			bname83[i++]=0x30+(rand()%10);
			for(j=0; bname[j]!='.'&&bname[j]; j++);
			while(bname[j]) bname83[i++]=bname[j++];
			bname83[i]='\0';
			bname=bname83;
		}
	//} else {
	//	bname=basename(src);
	}

	/* open source file */
	src_fd=open(src, O_RDONLY);
	if(src_fd==-1) {
		fprintf(stderr, "can't open file %s\n", src);
		return -1;
	}

	srcdata=(char*)malloc(srcstat.st_size);
	read(src_fd,srcdata,srcstat.st_size);
	close(src_fd);
	
	gp_put_data(srcdata,dst,bname,srcstat.st_size);
	free(srcdata);
	free(bname_tmp);
	
	return 0;
}

int gp_put_data(const char *srcdata, char *dst, const char* bname, unsigned int filesize) {
	int result;
	char *gppath;
	struct smc_info *info;
	unsigned int remaining, blocksize,done,progress;

	/* check for enough free space */
	info=gp_info();
	if(info==NULL) {
		fprintf(stderr, "can't get smc info\n");
		return -1;
	}

	if(info->bytes_free<filesize) {
		fprintf(stderr, "not enough space on smc\n");
		return -1;
	}
	free(info); info=0;

	memset(buffer, 0, 0x100);
	*buffer=0xd0;
	result=usb_write(buffer, 0x100);
	if(result==-1) return -1;

	gppath=mk_gppath(dst,1);

	/* send destination directory */
	memset(buffer, 0, 0x100);
	*buffer=strlen(gppath);
	strcpy(buffer+1, gppath);
	result=usb_write(buffer, 0x100);
	free(gppath); gppath=0;
	if(result==-1) return -1;


	memset(buffer, 0, 0x100);
	result=usb_read(buffer, 0x100);
	if(result==-1) return -1;
	if(*buffer!=0xd1) return -1;

	printf("%s", bname);
	memset(buffer, 0, 0x100);
	*buffer=strlen(bname);
	strcpy(buffer+1, bname);
	result=usb_write(buffer, 0x100);
	if(result==-1) return -1;

	memset(buffer, 0, 0x100);
	result=usb_read(buffer, 0x100);
	if(result==-1) return -1;
	if(*buffer!=0xd7) return -1;

	/* send filesize, TODO fix endianness? */
	memset(buffer, 0, 0x100);
	buffer[0]=(filesize&0x000000ff);
	buffer[1]=(filesize&0x0000ff00)>>8;
	buffer[2]=(filesize&0x00ff0000)>>16;
	buffer[3]=(filesize&0xff000000)>>24;
	result=usb_write(buffer, 0x100);
	if(result==-1) return -1;

	memset(buffer, 0, 0x100);
	result=usb_read(buffer, 0x100);
	if(result==-1) return -1;
	if(*buffer!=0xd2) return -1;

	progress=0;
	done=0;
	blocksize=0x100;
	remaining=filesize;
	puts("");
	while(remaining) {
		int fs;
		/* send block size */
		memset(buffer, 0, 0x100);
		fs=remaining>256*1024?256*1024:remaining;
		buffer[0]=(fs&0x000000ff);
		buffer[1]=(fs&0x0000ff00)>>8;
		buffer[2]=(fs&0x00ff0000)>>16;
		buffer[3]=(fs&0xff000000)>>24;
		result=usb_write(buffer, 0x100);
		if(result==-1) return -1;

		remaining-=fs;

		memset(buffer, 0, 0x100);
		result=usb_read(buffer, 0x100);
		if(result==-1) return -1;
		if(*buffer!=0xd3) return -1;

		/* send block */
		memset(buffer, 0, 0x100);
		while(fs>0 && memcpy(buffer, srcdata+done, blocksize)) {
			result=usb_write(buffer, 0x100);
			if(result==-1) return -1;
			memset(buffer, 0, 0x100);
			progress++;
			done+=blocksize;
			if (filesize-done<blocksize) blocksize=filesize-done;
			//if((cnt&3)==0) { 
			if((progress&255)==0) { 
				printf("%5d k sent\r",progress*0x100/1024); 
				fflush(stdout); 
			}
			fs-=0x100;
		}

		/* expect d2 */
		memset(buffer, 0, 0x100);
		result=usb_read(buffer, 0x100);
		if(result==-1) return -1;
		if(*buffer!=0xd2) {
			fprintf(stderr, "expecting d2, got %x\n", *buffer);
			return -1;
		}
	}
	puts("");

	/* send 0, expect d4 */
	memset(buffer, 0, 0x100);
	result=usb_write(buffer, 0x100);
	if(result==-1) return -1;
	memset(buffer, 0, 0x100);
	result=usb_read(buffer, 0x100);
	if(result==-1) return -1;
	if(*buffer!=0xd4) return -1;

	return 0;
}
	
int gp_get(char *src, char *dst) {
	int result;
	char *gppath;
	int filesize;
	int segmentsize;
	int fd;
	int bytes_read, total_bytes_read;
	int finished, segment_finished;
	int next_block_size;

	/* send cmd 70 */
	memset(buffer, 0, 0x100);
	*buffer=0x70;
	result=usb_write(buffer, 0x100);
	if(result==-1) return -1;

	gppath=mk_gppath(src,0);

	/* send file path */
	memset(buffer, 0, 0x100);
	*buffer=strlen(gppath);
	strcpy(buffer+1, gppath);
	result=usb_write(buffer, 0x100);
	free(gppath); gppath=0;
	if(result==-1) return -1;

	/* get file size */
	memset(buffer, 0, 0x100);
	result=usb_read(buffer, 0x100);
	if(result==-1) return -1;
	filesize=(((unsigned int)*(buffer+3)<<24)
		|((unsigned int)*(buffer+2)<<16)
		|((unsigned int)*(buffer+1)<<8)
		|((unsigned int)*(buffer)));

	/* open file for writing */
	fd=creat(dst, 0644);
	if(fd==-1) {
		fprintf(stderr, "couldn't create file %s", dst);
		return -1;
	}

	finished=0;
	total_bytes_read=0;
	while(!finished) {
		/* get file segment size */
		memset(buffer, 0, 0x100);
		*buffer=0x71;
		result=usb_write(buffer, 0x100);
		if(result==-1) {
			close(fd);
			return -1;
		}
		memset(buffer, 0, 0x100);
		result=usb_read(buffer, 0x100);
		if(result==-1) {
			close(fd);
			return -1;
		}
		segmentsize=(((unsigned int)*(buffer+3)<<24)
			|((unsigned int)*(buffer+2)<<16)
			|((unsigned int)*(buffer+1)<<8)
			|((unsigned int)*(buffer)));
		
		/* request file data */
		memset(buffer, 0, 0x100);
		*buffer=0x72;
		result=usb_write(buffer, 0x100);
		if(result==-1) {
			close(fd);
			return -1;
		}

		bytes_read=0;
		segment_finished=0;
		while(!segment_finished) {
			if((segmentsize-bytes_read)>0x100) {
				next_block_size=0x100;
			} else {
				next_block_size=segmentsize-bytes_read;
				segment_finished=1;
			}
			memset(buffer, 0, 0x100);
			result=usb_read(buffer, 0x100);
			if(result==-1) {
				close(fd);
				return -1;
			}
			bytes_read+=next_block_size;
			total_bytes_read+=next_block_size;
			write(fd, buffer, next_block_size);
			
			if(total_bytes_read>=filesize) finished=1;
			if(!finished) {
				memset(buffer, 0, 0x100);
				*buffer=0x03;
				result=usb_write(buffer, 0x100);
				if(result==-1) {
					close(fd);
					return -1;
				}
			}
		}
	}

	close(fd);
	memset(buffer, 0, 0x100);
	*buffer=0x73;
	result=usb_write(buffer, 0x100);
	if(result==-1) return -1;
	return 0;
}

int gp_mkdir(char *path) {
	int result;
	char *gppath;

	/* send cmd a0 */
	memset(buffer, 0, 0x100);
	*buffer=0xa0;
	result=usb_write(buffer, 0x100);
	if(result==-1) return -1;

	gppath=mk_gppath(path,0);

	/* send file path */
	memset(buffer, 0, 0x100);
	*buffer=strlen(gppath);
	strcpy(buffer+1, gppath);
	result=usb_write(buffer, 0x100);
	free(gppath); gppath=0;
	if(result==-1) return -1;

	/* expect response a1 */
	memset(buffer, 0, 0x100);
	result=usb_read(buffer, 0x100);
	if(result==-1) return -1;
	if(*buffer!=0xa1) return -1;
	return 0;
}

int gp_rmdir(char *path) {
	int result;
	char *gppath;

	/* send cmd b0 */
	memset(buffer, 0, 0x100);
	*buffer=0xb0;
	result=usb_write(buffer, 0x100);
	if(result==-1) return -1;

	gppath=mk_gppath(path,1);

	/* send file path */
	memset(buffer, 0, 0x100);
	*buffer=strlen(gppath);
	strcpy(buffer+1, gppath);
	result=usb_write(buffer, 0x100);
	free(gppath); gppath=0;
	if(result==-1) return -1;

	/* expect response b1 */
	memset(buffer, 0, 0x100);
	result=usb_read(buffer, 0x100);
	if(result==-1) return -1;
	if(*buffer!=0xb1) return -1;
	return 0;
}

int gp_rm(char *path) {
	int result;
	char *gppath;

	struct file_info *test=fileinfo(path);
	if (!test) return -1; // file not found
	free(test->name); test->name=0;
	if (test->is_dir) {
		free(test);
		return -1;
	}
	free(test);

	/* send cmd c0 */
	memset(buffer, 0, 0x100);
	*buffer=0xc0;
	result=usb_write(buffer, 0x100);
	if(result==-1) return -1;

	gppath=mk_gppath(path,0);

	/* send file path */
	memset(buffer, 0, 0x100);
	*buffer=strlen(gppath);
	strcpy(buffer+1, gppath);
	result=usb_write(buffer, 0x100);
	free(gppath); gppath=0;
	if(result==-1) return -1;

	/* expect response c1 */
	memset(buffer, 0, 0x100);
	result=usb_read(buffer, 0x100);
	if(result==-1) return -1;
	if(*buffer!=0xc1) return -1;
	return 0;
}

int gp_format() {
	int result;

	memset(buffer, 0, 0x100);
	buffer[0] = 0x80;
	result=usb_write(buffer, 0x100);
	if (result==-1) return -1;

	result=usb_read(buffer, 0x100);
	if (result==-1) return -1;
	if (buffer[0] != 0x81) return -1;

	result=usb_read(buffer, 0x100);
	if (result==-1) return -1;
	if (buffer[0] != 0x82) return -1;

	return 0;
}

int gp_ids() {
	int result,i;

	memset(buffer, 0, 0x100);
	buffer[0] = 0xe1;
	result=usb_write(buffer, 0x100);
	if (result==-1) return -1;

	result=usb_read(buffer, 0x100);
	if (result==-1) return -1;
	printf("PDUID: ");
	for (i=0;i<16;i++) {
	  printf("%x",buffer[i]);
	}
	printf("\n");

	memset(buffer, 0, 0x100);
	buffer[0] = 0xe0;
	result=usb_write(buffer, 0x100);
	if (result==-1) return -1;

	result=usb_read(buffer, 0x100);
	if (result==-1) return -1;
	printf("SMCID: ");
	for (i=0;i<16;i++) {
	  printf("%x",buffer[i]);
	}
	printf("\n");

	memset(buffer, 0, 0x100);
	buffer[0] = 0x42;
	result=usb_write(buffer, 0x100);
	if (result==-1) return -1;

	result=usb_read(buffer, 0x100);
	if (result==-1) return -1;
	printf("USER: %s\n",buffer);

	memset(buffer, 0, 0x100);
	buffer[0] = 0x43;
	result=usb_write(buffer, 0x100);
	if (result==-1) return -1;

	result=usb_read(buffer, 0x100);
	if (result==-1) return -1;
	printf("PASS: %s\n",buffer);

	return 0;
}

