#include <stdio.h>
#include <unistd.h>
#include <libgen.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "gp32cmds.h"
#include "fpk.h"

int end_link=0;
int convfilename=0;
int human_readable=0;
int list_sizes=0;
int recurse=0;

void usage() {
	printf(
		"\n"
		"usage: gplink [-lchx] command arguments\n"
		"gplink info                            get smc info\n"
		"gplink ids                             get ids (pduid etc)\n"
		"gplink ls [directory name]             smc directory listing\n"
		"       -l print file sizes\n"
		"gplink get source [destination dir]    download file\n"
		"	-r download files recursively\n"
		"gplink put file [file...] destination  upload file(s) to given destination\n"
		"       -c do 8.3 filename conversion\n"
		"gplink install file.fpk                install fpk archive\n"
		"gplink mkdir path                      create directory\n"
		"gplink rmdir path                      delete directory\n"
		"gplink rm file                         delete file\n"
		"gplink run file                        run .gxb or .fxe file\n"
		"       (requires pc-link/multi-fw)\n"
		//"gplink reboot                          reboots the gp32\n"
		//"       (requires pc-link/multi-fw)\n"
		//"gplink format                          format of SMC (untested!!)\n"
		"gplink end                             end gp32 link mode\n"
		"\n"
		"options:\n"
		"-h print numbers in human readable format\n"
		"-x end link mode after operation\n"
		"\n"
		);
}

void print_num(int num) {
	if(human_readable && num>=(1024*1024)) {
		printf("%dMib", num/(1024*1024));
	} else if(human_readable && num>=1024) {
		printf("%dKib", num/1024);
	} else {
		printf("%d", num);
	}
}

void ls(char **argv) {
	struct file_info *directory_list;
	int i;

	directory_list=gp_dir(*argv);
	if(directory_list==NULL) {
		fprintf(stderr, "error reading directory\n");
	} else {
		for(i=0; directory_list[i].name; i++) {
			printf("%s", directory_list[i].name);
			if(directory_list[i].is_dir) printf("/");
			if(list_sizes && !directory_list[i].is_dir) {
				printf("\t");
				print_num(directory_list[i].size);
			}
			printf("\n");
			free(directory_list[i].name);
		}
		free(directory_list);
	}
	if(end_link) gp_end_link_mode();
}

void gpinstall(char **argv) {
	struct fpkmeta meta;
	struct fpkfile *filep;
	char buffer[0x100];
	char *file;
	
	if (fpkinit(&meta,argv[0])) return;
	fpkgetdir(&meta);
	
	{
		struct smc_info *info;
		info=gp_info();
		if (!info) {
			fprintf(stderr,"can't get SMC information\n");
			return;
		}
		if (info->bytes_free<meta.size) {
			fprintf(stderr,"not enough space\n");
			return;
		}
		free(info);
	}
	
	while ((filep=fpknextdir(&meta))) {
		strncpy(buffer,meta.rootpath,strlen(meta.rootpath));
		strncpy(buffer+strlen(meta.rootpath),filep->directory,strlen(filep->directory));
		buffer[strlen(meta.rootpath)+strlen(filep->directory)]='\\';
		strncpy(buffer+strlen(meta.rootpath)+strlen(filep->directory)+1,filep->name,strlen(filep->name));
		buffer[strlen(meta.rootpath)+strlen(filep->directory)+strlen(filep->name)+1]='\0';
		printf("install: mkdir %s\n",buffer);
		if (gp_mkdir(buffer)==-1) {
			// error condition
		}
	}
	fpkrewind(&meta);
	while ((filep=fpknextfile(&meta))) {
		memset(buffer,0,0x100);
		strncpy(buffer,meta.rootpath,strlen(meta.rootpath));
		strncpy(buffer+strlen(meta.rootpath),filep->directory,strlen(filep->directory));
		buffer[strlen(meta.rootpath)+strlen(filep->directory)]='\0';
		file=fpkgetfile(&meta,filep);
		gp_put_data(file,buffer,filep->name,filep->size);
		free(file);
	}
	fpkclose(&meta);
	if(end_link) gp_end_link_mode();
}

void end() {
	gp_end_link_mode();
}

void info() {
	struct smc_info *info;

	info=gp_info();
	if(info==NULL) {
		fprintf(stderr, "error reading smc info\n");
	} else {
		printf("smc capacity: ");
		print_num(info->capacity);
		printf("\n");
		printf("bytes_used: ");
		print_num(info->bytes_used);
		printf("\n");
		printf("bytes_free: ");
		print_num(info->bytes_free);
		printf("\n");
		free(info);
	}
	if(end_link) gp_end_link_mode();
}

void putfile(char **argv) {
	int i,j;
	char *destination;
	int result;

	for(i=0; argv[i]!=NULL; i++);
	destination=argv[i-1];
	for(j=0; j<(i-1); j++) {
		printf("uploading %s to %s/",
				argv[j], destination);
		result=gp_put(argv[j], destination, convfilename);
		if(result==-1) {
			fprintf(stderr, "error in file upload\n");
			gp_disconnect();
			exit(1);
		}
		printf("...done\n");
	}
	if(end_link) gp_end_link_mode();
}

void getfile(char **argv) {
	int i, result;
	char *dest_dir;
	int dest_path_size;
	char *dest_path;
	char *srctmp;
	char *filename;

	if(argv[1]) {
		dest_dir=argv[1];
	} else {
		dest_dir=".";
	}
	srctmp=strdup(argv[0]);
	for(i=0; srctmp[i]; i++) if(srctmp[i]=='\\') srctmp[i]='/';
	filename=basename(srctmp);
	dest_path_size=strlen(dest_dir)+strlen(filename)+2;
	dest_path=malloc(dest_path_size);
	snprintf(dest_path, dest_path_size, "%s/%s", dest_dir, filename);

	printf("downloading %s to %s...",
			argv[0], dest_path);
	result=gp_get(argv[0], dest_path);
	if(result==-1) {
		fprintf(stderr, "error in file download\n");
		gp_disconnect();
		exit(1);
	}
	printf("done\n");
	if(end_link) gp_end_link_mode();
}

void gettree(char **argv) {
	char *src_base;
	char *src_path;
	int src_path_size;
	char *dest_dir;
	char *new_dir;
	int new_dir_size;
	struct file_info *directory_list;
	int i;
	char *args[2];

	src_base=argv[0];
	if(argv[1]) {
		dest_dir=argv[1];
	} else {
		dest_dir=".";
	}

	directory_list=gp_dir(argv[0]);
	if(!directory_list) return;

	for(i=0; directory_list[i].name; i++) {
		src_path_size=strlen(src_base)+strlen(directory_list[i].name)+2;
		src_path=malloc(src_path_size);
		if(src_base[strlen(src_base)-1]=='/'||src_base[strlen(src_base)-1]=='\\') {
			snprintf(src_path, src_path_size, "%s%s", src_base, directory_list[i].name);
		} else {
			snprintf(src_path, src_path_size, "%s\\%s", src_base, directory_list[i].name);
		}
		if(directory_list[i].is_dir) {
			new_dir_size=strlen(dest_dir)+strlen(directory_list[i].name)+2;
			new_dir=malloc(new_dir_size);
			snprintf(new_dir, new_dir_size, "%s/%s", dest_dir, directory_list[i].name);
			printf("creating directory %s\n", new_dir);
			mkdir(new_dir, 0777);
			args[0]=src_path;
			args[1]=new_dir;
			//printf("calling gettree %s %s\n", args[0], args[1]);
			gettree(args);
			free(new_dir);
		} else {
			//printf("copying %s to %s\n", src_path, dest_dir);
			args[0]=src_path;
			args[1]=dest_dir;
			getfile(args);
		}
		free(directory_list[i].name);
		free(src_path);
	}
	free(directory_list);
}

void get(char **argv) {
	if(recurse) {
		gettree(argv);
	} else {
		getfile(argv);
	}
}

void gpmkdir(char **argv) {
	int result;

	result=gp_mkdir(argv[0]);
	if(result==-1) {
		fprintf(stderr, "error creating directory\n");
		gp_disconnect();
		exit(1);
	} else {
		printf("%s created\n", argv[0]);
	}
	if(end_link) gp_end_link_mode();
}

void gprmdir(char **argv) {
	int result;

	result=gp_rmdir(argv[0]);
	if(result==-1) {
		fprintf(stderr, "error deleting directory\n");
		gp_disconnect();
		exit(1);
	} else {
		printf("%s deleted\n", argv[0]);
	}
	if(end_link) gp_end_link_mode();
}

void gprm(char **argv) {
	int result;

	result=gp_rm(argv[0]);
	if(result==-1) {
		fprintf(stderr, "error deleting file\n");
		gp_disconnect();
		exit(1);
	} else {
		printf("%s deleted\n", argv[0]);
	}
	if(end_link) gp_end_link_mode();
}

void gprun(char **argv) {
	int result;

	result=gp_run(argv[0]);
	if(result==-1) {
		fprintf(stderr, "error running program\n");
		gp_disconnect();
		exit(1);
	} else {
		printf("executing %s\n", argv[0]);
	}
	gp_end_link_mode();
}

void gpreboot() {
	char swi4[]={
		0x07,0x00,0x00,0xea,
		
		0x00,0x00,0x00,0xc0,
		0x24,0x00,0x00,0xc0,
		0x24,0x00,0x00,0xc0,
		0x24,0x00,0x00,0xc0,
		0x24,0x00,0x00,0xc0,
		0x24,0x00,0x00,0xc0,
		
		0x11,0x00,0x45,0x44,
		0x11,0x00,0x45,0x44,

		0x04,0x00,0x00,0xef
		};
	gp_run_data(swi4,40);
	gp_end_link_mode();
}

void gpformat() {
	gp_format();
	if (end_link) gp_end_link_mode();
}

void gpids() {
	gp_ids();
	if (end_link) gp_end_link_mode();
}

struct command {
	char *name;
	void *operation;
	int num_args;
};

struct command cmd_table[] = {
	{"ls", ls, 0},
	{"dir", ls, 0},
	{"end", end, 0},
	{"df", info, 0},
	{"info", info, 0},
	{"ids", gpids, 0},
	{"put", putfile, 2},
	{"get", get, 1},
	{"mkdir", gpmkdir, 1},
	{"rmdir", gprmdir, 1},
	{"rm", gprm, 1},
	{"install", gpinstall, 1},
	{"reboot", gpreboot, 0},
	{"format", gpformat, 0},
	{"run", gprun, 1}
};

int main(int argc, char **argv) {
	int c, i;
	int print_help=0;
	int num_cmds, num_args;
	char *cmd_name;
	void (*command)(char **);

	while((c=getopt(argc, argv, "rhxcl"))!=-1) {
		switch(c) {
			case 'r':
				recurse=1;
				break;
			case 'h':
				human_readable=1;
				break;
			case 'x':
				end_link=1;
				break;
			case 'c':
				convfilename=1;
				break;
			case 'l':
				list_sizes=1;
				break;
		}
	}

	if(print_help||(argv[optind]==NULL)) {
		usage();
		exit(0);
	}

	cmd_name=argv[optind];
	num_cmds=(sizeof(cmd_table)/sizeof(struct command));
	num_args=0;
	command=NULL;

	for(i=0; i<num_cmds; i++) {
		if(!strcmp(cmd_name, cmd_table[i].name)) {
			command=cmd_table[i].operation;
			num_args=cmd_table[i].num_args;
			break;
		}
	}

	if(command!=NULL && (argc-(optind+1))>=num_args) {
		if(gp_connect()==-1) {
			fprintf(stderr, "gp connection error\n");
			exit(1);
		}
		(*command)(argv+optind+1);
		gp_disconnect();
		exit(0);
	}

	usage();
	exit(1);
}
