/* GStreamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include <gnome.h>

#include <gst/gst.h>
#include <gst/gstutils.h>

#include "editor.h"
#include "gst-helper.h"
#include "gsteditorproperty.h"
#include "../../../pixmaps/pixmaps.h"

/* class functions */
static void 	gst_editor_element_class_init		(GstEditorElementClass *klass);
static void 	gst_editor_element_init			(GstEditorElement *element);

static void	gst_editor_element_set_property		(GObject *object, guint prop_id, 
                                                         const GValue *value, GParamSpec *pspec);
static void	gst_editor_element_get_property		(GObject *object, guint prop_id, 
                                                         GValue *value, GParamSpec *pspec);

static void 	gst_editor_element_realize		(GnomeCanvasItem *citem);
static void 	gst_editor_element_resize		(GstEditorItem *item);
static void 	gst_editor_element_repack		(GstEditorItem *item);
static gint 	gst_editor_element_event		(GnomeCanvasItem *citem,
                                     			 GdkEvent *event);
static void 	gst_editor_element_object_changed 	(GstEditorItem *bin, GstObject *object);

/* events fired by items within self */
static gint 	gst_editor_element_resizebox_event 	(GnomeCanvasItem *citem,
                                                         GdkEvent *event,
                                                         GstEditorItem *item);
static gint 	gst_editor_element_state_event		(GnomeCanvasItem *item,
                                           		 GdkEvent *event,
                                           		 gpointer data);

/* callbacks on the GstElement */
static void 	on_new_pad				(GstElement *element, GstPad *pad,
                                                         GstEditorElement *editor_element);
static void 	on_pad_removed				(GstElement *element, GstPad *pad,
                                                         GstEditorElement *editor_element);
static void	on_state_change				(GstElement *element, GstElementState old,
                                                         GstElementState state,
                                                         GstEditorElement *editor_element);
static void 	on_parent_unset				(GstElement *element, GstBin *parent,
                                                         GstEditorElement *editor_element);
static void	on_object_saved 			(GstObject *object, xmlNodePtr parent, GstEditorItem *item);

/* utility functions */
static void	gst_editor_element_add_pad		(GstEditorElement *element,
                                                         GstPad *pad);
static void	gst_editor_element_remove_pad		(GstEditorElement *element,
                                                         GstPad *pad);
static void 	gst_editor_element_add_pads		(GstEditorElement *element);
static void 	gst_editor_element_set_state		(GstEditorElement *element, gint id);
static gboolean gst_editor_element_sync_state		(GstEditorElement *element);

/* callbacks for the popup menu */
static void	on_copy					(GtkWidget *unused, GstEditorElement *element);
static void	on_cut					(GtkWidget *unused, GstEditorElement *element);
static void	on_remove				(GtkWidget *unused, GstEditorElement *element);


static GstElementState _gst_element_states[] = {
  GST_STATE_NULL,
  GST_STATE_READY,
  GST_STATE_PAUSED,
  GST_STATE_PLAYING,
};

enum {
  ARG_0,
  ARG_ACTIVE,
  ARG_RESIZEABLE,
  ARG_MOVEABLE
};

enum {
  SIZE_CHANGED,
  LAST_SIGNAL
};


static GObjectClass *parent_class;

static guint gst_editor_element_signals[LAST_SIGNAL] = { 0 };

static GnomeUIInfo menu_items[] = {
  GNOMEUIINFO_MENU_COPY_ITEM (on_copy, NULL),
  GNOMEUIINFO_MENU_CUT_ITEM (on_cut, NULL),
  GNOMEUIINFO_ITEM_STOCK ("_Remove element", "Remove element from bin", on_remove, "gtk-remove"),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_END
};


GType 
gst_editor_element_get_type(void) 
{
  static GType element_type = 0;

  if (!element_type) {
    static const GTypeInfo element_info = {
      sizeof (GstEditorElementClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) gst_editor_element_class_init,
      NULL,
      NULL,
      sizeof (GstEditorElement),
      0,
      (GInstanceInitFunc) gst_editor_element_init,
    };
      
    element_type = g_type_register_static (GST_TYPE_EDITOR_ITEM, "GstEditorElement", &element_info, 0);
  }
  return element_type;
}

static void 
gst_editor_element_class_init (GstEditorElementClass *klass) 
{
  GObjectClass *object_class;
  GstEditorItemClass *item_class;
  GnomeCanvasItemClass *citem_class;

  object_class = G_OBJECT_CLASS (klass);
  item_class = GST_EDITOR_ITEM_CLASS (klass);
  citem_class = GNOME_CANVAS_ITEM_CLASS (klass);

  parent_class = g_type_class_ref (GST_TYPE_EDITOR_ITEM);

  gst_editor_element_signals[SIZE_CHANGED] =
    g_signal_new ("size_changed", G_OBJECT_CLASS_TYPE (klass), G_SIGNAL_RUN_FIRST,
                  G_STRUCT_OFFSET (GstEditorElementClass, size_changed), 
                  NULL, NULL, gst_editor_marshal_VOID__VOID, G_TYPE_NONE, 0);

  object_class->set_property = gst_editor_element_set_property;
  object_class->get_property = gst_editor_element_get_property;

  g_object_class_install_property (object_class, ARG_ACTIVE,
       g_param_spec_boolean ("active", "active", "active",
                             FALSE, G_PARAM_READWRITE));
  g_object_class_install_property (object_class, ARG_RESIZEABLE,
       g_param_spec_boolean ("resizeable", "resizeable", "resizeable",
                             FALSE, G_PARAM_READWRITE));
  g_object_class_install_property (object_class, ARG_MOVEABLE,
       g_param_spec_boolean ("moveable", "moveable", "moveable",
                             TRUE, G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

  citem_class->realize		= gst_editor_element_realize;
  citem_class->event		= gst_editor_element_event;
  item_class->resize		= gst_editor_element_resize;
  item_class->repack		= gst_editor_element_repack;
  item_class->object_changed	= gst_editor_element_object_changed;

  GST_EDITOR_ITEM_CLASS_PREPEND_MENU_ITEMS (item_class, menu_items, 4);
}

static void 
gst_editor_element_init (GstEditorElement *element) 
{
  GstEditorItem *item = GST_EDITOR_ITEM (element);

  item->fill_color = 0xffffffff;
  item->outline_color = 0x333333ff;

  element->active = FALSE;
}

static void 
gst_editor_element_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec) 
{
  GstEditorElement *element = GST_EDITOR_ELEMENT (object);

  switch (prop_id) {
    case ARG_ACTIVE:
      element->active = g_value_get_boolean (value);
      gnome_canvas_item_set (GNOME_CANVAS_ITEM (GST_EDITOR_ITEM (element)->border),
                            "width_units", (element->active ? 2.0 : 1.0),
                             NULL);
      gnome_canvas_item_set (GNOME_CANVAS_ITEM (element->statebox),
                            "width_units", (element->active ? 2.0 : 1.0),
                             NULL);
      break;
    case ARG_RESIZEABLE:
      element->resizeable = g_value_get_boolean (value);
      if (!GST_EDITOR_ITEM (element)->realized)
        break;
      if (element->resizeable)
        gnome_canvas_item_show (element->resizebox);
      else
        gnome_canvas_item_hide (element->resizebox);
      break;
    case ARG_MOVEABLE:
      element->moveable = g_value_get_boolean (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void 
gst_editor_element_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec) 
{
  GstEditorElement *element = GST_EDITOR_ELEMENT(object);

  switch (prop_id) {
    case ARG_ACTIVE:
      g_value_set_boolean (value, element->active);
      break;
    case ARG_RESIZEABLE:
      g_value_set_boolean (value, element->resizeable);
      break;
    case ARG_MOVEABLE:
      g_value_set_boolean (value, element->moveable);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void 
gst_editor_element_realize (GnomeCanvasItem *citem)
{
  GstEditorElement *element;
  GstEditorItem *item;
  gint i;
  GdkPixbuf *pixbuf;
  static const guint8* state_icons[] = {
    off_stock_image,
    on_stock_image,
    pause_stock_image,
    play_stock_image
  };

  element = GST_EDITOR_ELEMENT (citem);
  item = GST_EDITOR_ITEM (citem);

  if (GNOME_CANVAS_ITEM_CLASS (parent_class)->realize)
    GNOME_CANVAS_ITEM_CLASS (parent_class)->realize (citem);

  /* the resize box */
  element->resizebox = gnome_canvas_item_new (GNOME_CANVAS_GROUP (citem),
                                              gnome_canvas_rect_get_type(),
                                              "width_units", 1.0,
                                              "fill_color", "white",
                                              "outline_color", "black",
                                              NULL);
  g_return_if_fail (element->resizebox != NULL);
  GST_EDITOR_SET_OBJECT (element->resizebox, item);
  g_signal_connect (G_OBJECT (element->resizebox), "event",
                    G_CALLBACK (gst_editor_element_resizebox_event), element);
  if (!element->resizeable)
    gnome_canvas_item_hide (element->resizebox);

  /* create the state boxen */
  element->statebox = gnome_canvas_item_new (GNOME_CANVAS_GROUP (item),
                                             gnome_canvas_rect_get_type(),
                                             "width_units", 1.0,
                                             "fill_color", "white",
                                             "outline_color", "black",
                                             "x1", 0.0, "y1", 0.0,
                                             "x2", 0.0, "y2", 0.0,
                                             NULL);
  g_return_if_fail (element->statebox != NULL);

  GST_EDITOR_SET_OBJECT (element->statebox, element);
    
  for (i=0;i<4;i++) {
    pixbuf = gdk_pixbuf_new_from_inline (-1, state_icons[i], FALSE, NULL);
    element->stateicons[i] = gnome_canvas_item_new (GNOME_CANVAS_GROUP (item),
                                                    gnome_canvas_pixbuf_get_type(),
                                                    "pixbuf", pixbuf,
                                                    "x", 0.0, "y", 0.0,
                                                    NULL);
    
    GST_EDITOR_SET_OBJECT (element->stateicons[i], element);
    
    g_signal_connect (element->stateicons[i], "event",
                      G_CALLBACK (gst_editor_element_state_event),
                      GINT_TO_POINTER (i));
  }
  
  EDITOR_DEBUG ("adding pads to element\n");
  gst_editor_element_add_pads (element);

  item->realized = TRUE;

  if (G_OBJECT_TYPE (item) == GST_TYPE_EDITOR_ELEMENT)
    gst_editor_item_resize (item);
}

static void 
gst_editor_element_resize (GstEditorItem *item)
{
  GstEditorItem *subitem;
  GstEditorElement *element;
  GList *l;

  element = GST_EDITOR_ELEMENT (item);

  /* a little bit of padding, eh */
  item->l.h += 4.0;
  item->r.h += 4.0;

  /* the resize box is 4x4 */
  item->b.w += 4.0;
  item->b.h = MAX (item->b.h, 4.0);

  /* state boxes are 16.0 x 16.0 + 2 px for the border */
  element->statewidth = 18.0;
  element->stateheight = 18.0;
  item->b.w += element->statewidth * 4 + 2.0;
  /* 1 px of vertical padding.. */
  item->b.h = MAX (item->b.h, element->stateheight + 1.0);

  /* now go and try to calculate necessary space for the pads */
  element->sinkwidth = 0.0;
  element->sinkheight = 0.0;
  element->sinks = 0;
  for (l = element->sinkpads; l; l=l->next) {
    subitem = GST_EDITOR_ITEM (l->data);
    element->sinkwidth = MAX (element->sinkwidth, subitem->width);
    element->sinkheight = MAX (element->sinkheight, subitem->height);
    element->sinks++;
  }
  item->l.w = MAX (item->l.w, element->sinkwidth + 12.0);
  item->l.h += element->sinkheight * element->sinks;

  element->srcwidth = 0.0;
  element->srcheight = 0.0;
  element->srcs = 0;
  for (l = element->srcpads; l; l=l->next) {
    subitem = GST_EDITOR_ITEM (l->data);
    element->srcwidth = MAX (element->srcwidth, subitem->width);
    element->srcheight = MAX (element->srcheight, subitem->height);
    element->srcs++;
  }
  item->r.w = MAX (item->r.w, element->srcwidth + 12.0);
  item->r.h += element->srcheight * element->srcs;

  GST_EDITOR_ITEM_CLASS (parent_class)->resize (item);
}

static void 
gst_editor_element_repack (GstEditorItem *item)
{
  GstEditorElement *element;
  GList *l;
  GstEditorItem *subitem;
  gint sinks;
  gint srcs;
  gdouble x1,y1,x2,y2,x,y;
  gint i;

  if (!item->realized) return;

  element = GST_EDITOR_ELEMENT (item);

  /* the resize box */
  gnome_canvas_item_set (element->resizebox,
                         "x1", item->width - 4.0,
                         "y1", item->height - 4.0,
                         "x2", item->width,
                         "y2", item->height,
                         NULL);

  /* make sure args to gnome_canvas_item_set are doubles */
  x1 = 0.0; y1 = 0.0;
  x2 = item->width; y2 = item->height;

  /* place the state boxes */
  for (i=0;i<4;i++) {
    g_return_if_fail (element->stateicons[i] != NULL);
    gnome_canvas_item_set (element->stateicons[i],
                           "x", x1 + (element->statewidth * i) + 1.0,
                           "y", y2 - element->stateheight + 1.0,
                           NULL);
  }
  gst_editor_element_sync_state(element);

  /* place the pads */
  sinks = element->sinks;
  l = element->sinkpads;
  while (l) {
    subitem = GST_EDITOR_ITEM (l->data);
    g_object_get (subitem, "x", &x, "y", &y, NULL);
    gst_editor_item_move (subitem,
                          x1 - x,
                          y2 - 2.0 - element->stateheight - (element->sinkheight * sinks) - y);
    sinks--;
    l = g_list_next(l);
  }
  srcs = element->srcs;
  l = element->srcpads;
  while (l) {
    subitem = GST_EDITOR_ITEM (l->data);
    g_object_get (subitem, "x", &x, "y", &y, NULL);
    gst_editor_item_move (subitem,
                          x2 - GST_EDITOR_ITEM (subitem)->width - x,
                          y2 - 2.0 - element->stateheight - (element->srcheight * srcs) - y);
    srcs--;
    l = g_list_next(l);
  }

  if (GST_EDITOR_ITEM_CLASS (parent_class)->repack)
    (GST_EDITOR_ITEM_CLASS (parent_class)->repack) (item);
}

static gint 
gst_editor_element_event(GnomeCanvasItem *citem,
		         GdkEvent *event)
{
  GstEditorElement *element;
  GstEditorItem *item;
  gdouble dx,dy;
  GdkCursor *fleur;

  element = GST_EDITOR_ELEMENT (citem);
  item = GST_EDITOR_ITEM (citem);

  switch(event->type) {
    case GDK_BUTTON_PRESS:
      if (event->button.button == 1) {
        g_object_set (citem->canvas, "selection", element, NULL);

        if (element->moveable) {
          /* dragxy coords are world coords of button press */
          element->dragx = event->button.x;
          element->dragy = event->button.y;
          /* set some flags */
          element->dragging = TRUE;
          element->moved = FALSE;
          fleur = gdk_cursor_new (GDK_FLEUR);
          gnome_canvas_item_grab (citem,
                                  GDK_POINTER_MOTION_MASK |
/*                             GDK_ENTER_NOTIFY_MASK | */
/*                             GDK_LEAVE_NOTIFY_MASK | */
                                  GDK_BUTTON_RELEASE_MASK,
                                  fleur, event->button.time);
        }
        
        return TRUE;
      }
      break;
    case GDK_MOTION_NOTIFY:
      if (element->dragging) {
        dx = event->button.x - element->dragx;
        dy = event->button.y - element->dragy;
        gst_editor_element_move (element,dx,dy);
        element->dragx = event->button.x;
        element->dragy = event->button.y;
        element->moved = TRUE;
      }
      return TRUE;
    case GDK_BUTTON_RELEASE:
      if (element->dragging) {
        element->dragging = FALSE;
        gnome_canvas_item_ungrab(citem,event->button.time);
        return TRUE;
      }
      break;
    default:
      break;
  }

  if (GNOME_CANVAS_ITEM_CLASS (parent_class)->event)
    return GNOME_CANVAS_ITEM_CLASS (parent_class)->event (citem, event);
  return FALSE;
}

static void
gst_editor_element_object_changed (GstEditorItem *item, GstObject *object)
{
  if (item->object) {
    g_signal_handlers_disconnect_by_func (G_OBJECT (item->object), on_state_change, item);
    g_signal_handlers_disconnect_by_func (G_OBJECT (item->object), on_new_pad, item);
    g_signal_handlers_disconnect_by_func (G_OBJECT (item->object), on_pad_removed, item);
    g_signal_handlers_disconnect_by_func (G_OBJECT (item->object), on_parent_unset, item);
    g_signal_handlers_disconnect_by_func (G_OBJECT (item->object), on_object_saved, item);
  }
  
  if (object) {
    g_signal_connect (G_OBJECT (object), "state-change", G_CALLBACK (on_state_change), item);
    g_signal_connect (G_OBJECT (object), "new-pad", G_CALLBACK (on_new_pad), item);
    g_signal_connect (G_OBJECT (object), "pad-removed", G_CALLBACK (on_pad_removed), item);
    g_signal_connect (G_OBJECT (object), "parent-unset", G_CALLBACK (on_parent_unset), item);
    g_signal_connect (G_OBJECT (object), "object-saved", G_CALLBACK (on_object_saved), item);
  }
  
  if (GST_EDITOR_ITEM_CLASS (parent_class)->object_changed)
    (GST_EDITOR_ITEM_CLASS (parent_class)->object_changed) (item, object);
}

/**********************************************************************
 * Events from canvasitems internal to the editor element
 **********************************************************************/

static gint 
gst_editor_element_resizebox_event (GnomeCanvasItem *citem,
                                    GdkEvent *event,
                                    GstEditorItem *item) 
{
  GstEditorElement *element;
  GdkCursor *bottomright;
  gdouble item_x,item_y;

  element = GST_EDITOR_ELEMENT (item);

  g_print("in resizebox_event...\n");

  /* calculate coords relative to the group, not the box */
  item_x = event->button.x;
  item_y = event->button.y;
  gnome_canvas_item_w2i(citem->parent,&item_x,&item_y);

  switch(event->type) {
    case GDK_ENTER_NOTIFY:
      gnome_canvas_item_set(GNOME_CANVAS_ITEM(element->resizebox),
                        "fill_color", "red" , NULL);
      return TRUE;
      break;
    case GDK_LEAVE_NOTIFY:
      gnome_canvas_item_set(GNOME_CANVAS_ITEM(element->resizebox),
                        "fill_color", "white" , NULL);
      element->hesitating = FALSE;
      return TRUE;
      break;
    case GDK_BUTTON_PRESS:
      element->dragx = event->button.x;
      element->dragy = event->button.y;
      element->resizing = TRUE;
      element->hesitating = TRUE;
      bottomright = gdk_cursor_new(GDK_BOTTOM_RIGHT_CORNER);
      gnome_canvas_item_grab(citem,
                             GDK_POINTER_MOTION_MASK |
                             GDK_ENTER_NOTIFY_MASK |
                             GDK_LEAVE_NOTIFY_MASK |
                             GDK_BUTTON_RELEASE_MASK,
                             bottomright,event->button.time);
      return TRUE;
      break;
    case GDK_MOTION_NOTIFY:
      if (element->resizing) {
        if (item_x > 0.0 && item_y > 0.0)
          gnome_canvas_item_set (GNOME_CANVAS_ITEM (element), "width", item_x, "height", item_y, NULL);
        return TRUE;
      }
      break;
    case GDK_BUTTON_RELEASE:
      g_print ("DEBUG: GDK_BUTTON_RELEASE\n");
      if (element->resizing) {
        element->resizing = FALSE;
        gnome_canvas_item_ungrab(citem,event->button.time);
        return TRUE;
      }
      break;
    default:
      break;
  }
  return FALSE;
}

static gint 
gst_editor_element_state_event(GnomeCanvasItem *citem,
                               GdkEvent *event,
                               gpointer data) 
{
  GstEditorElement *element;
  GstEditorItem *item;
  gint id = GPOINTER_TO_INT (data);
  GdkCursor *uparrow;
  GdkEventButton *buttonevent;
  
  element = GST_EDITOR_GET_OBJECT (citem);
  item = GST_EDITOR_ITEM (element);

  switch (event->type) {
    case GDK_ENTER_NOTIFY:
      uparrow = gdk_cursor_new (GDK_SB_UP_ARROW);
      gnome_canvas_item_grab (citem,
                              GDK_POINTER_MOTION_MASK |
                              GDK_BUTTON_RELEASE_MASK |
                              GDK_LEAVE_NOTIFY_MASK,
                              uparrow,event->button.time);
      /* NOTE: when grabbing canvas item, always get pointer_motion,
         this will allow you to actually get all the other synth events */
      break;

    case GDK_LEAVE_NOTIFY:
      gnome_canvas_item_ungrab (citem,event->button.time);
      break;

    case GDK_BUTTON_PRESS:
      buttonevent = (GdkEventButton*)event;

      if (buttonevent->button == 1)
	return TRUE;
      else
        return FALSE;

    case GDK_BUTTON_RELEASE:
      buttonevent = (GdkEventButton*)event;

      if (buttonevent->button != 1)
	return FALSE;

      if (id < 4)
        gst_editor_element_set_state (element, id);
      else
        g_warning ("Uh, shouldn't have gotten here, unknown state\n");

      return TRUE;

    default:
      break;
  }
  return FALSE;
}

/**********************************************************************
 * Callbacks from the gstelement (must be threadsafe)
 **********************************************************************/

static void
on_new_pad (GstElement *element, GstPad *pad, GstEditorElement *editor_element)
{
  g_print ("new_pad in element %s\n", GST_OBJECT_NAME (element));

  gst_editor_element_add_pad (editor_element, pad);
  gst_editor_item_resize (GST_EDITOR_ITEM (editor_element));
}

static void
on_pad_removed (GstElement *element, GstPad *pad, GstEditorElement *editor_element)
{
  g_print ("pad_removed in element %s\n", GST_OBJECT_NAME (element));

  gst_editor_element_remove_pad (editor_element, pad);
  gst_editor_item_resize (GST_EDITOR_ITEM (editor_element));
}

static void 
on_state_change (GstElement *element, GstElementState old, GstElementState state,
                 GstEditorElement *editor_element)
{
  /* g_print ("state change in element %s\n", GST_OBJECT_NAME (element)); */

  if (state == GST_STATE_PLAYING &&
      GST_IS_BIN (element) &&
      GST_FLAG_IS_SET (element, GST_BIN_FLAG_MANAGER) &&
      !GST_FLAG_IS_SET (element, GST_BIN_SELF_SCHEDULABLE)) {
    g_message ("Adding iterator for pipeline");
    if (!editor_element->source)
      editor_element->source = g_idle_add ((GSourceFunc)gst_bin_iterate, element);
  } else if (editor_element->source) {
    g_message ("Removing iterator for pipeline");
    g_source_remove (editor_element->source);
    editor_element->source = 0;
  }
  
  g_idle_add ((GSourceFunc)gst_editor_element_sync_state, editor_element);
}

static void
on_parent_unset (GstElement *element, GstBin *parent, GstEditorElement *editor_element)
{
  GstEditorBin *editor_bin;
  
  g_print ("parent unset in element %s\n", GST_OBJECT_NAME (element));

  editor_bin = GST_EDITOR_BIN (gst_editor_item_get (GST_OBJECT (parent)));
  
  /* unreffing groups does nothing to the children right now. grrrrr. just hide
   * the damn thing and hack out the editorbin internals */

  gnome_canvas_item_hide (GNOME_CANVAS_ITEM (editor_element));

  editor_bin->elements = g_list_remove (editor_bin->elements, editor_element);

  if (editor_element->active)
    g_object_set (GNOME_CANVAS_ITEM (editor_element)->canvas, "selection", NULL, NULL);
}

static void
on_object_saved (GstObject *object, xmlNodePtr parent, GstEditorItem *item)
{
  xmlNsPtr ns;
  xmlNodePtr child;
  gchar *value;
  gdouble x, y, width, height;
  
  /* first see if the namespace is already known */
  ns = xmlSearchNsByHref (parent->doc, parent,
                          "http://gstreamer.net/gst-editor/1.0/");
  if (ns == NULL) {
    xmlNodePtr root = xmlDocGetRootElement (parent->doc);
    /* add namespace to root node */
    ns = xmlNewNs (root, "http://gstreamer.net/gst-editor/1.0/", "gst-editor");
  }

  child = xmlNewChild (parent, ns, "item", NULL);
  g_object_get (G_OBJECT (item), "x", &x, "y", &y,
                "width", &width, "height", &height, NULL);
  g_print ("x: %f, y: %f\n", x, y);
  value = g_strdup_printf ("%f", x);
  xmlNewChild (child, ns, "x", value);
  g_free (value);
  value = g_strdup_printf ("%f", y);
  xmlNewChild (child, ns, "y", value);
  g_free (value);
}

/**********************************************************************
 * Utility functions
 **********************************************************************/

static void
gst_editor_element_add_pad (GstEditorElement *element, GstPad *pad)
{
  GstEditorItem *editor_pad;
  GstPadTemplate *template;
  GType pad_type;
  
  if (GST_IS_GHOST_PAD (pad))
    pad_type = gst_editor_pad_ghost_get_type ();
  else if ((template = GST_PAD_PAD_TEMPLATE (pad))
           && template->presence == GST_PAD_REQUEST)
    pad_type = gst_editor_pad_requested_get_type ();
  else
    pad_type = gst_editor_pad_always_get_type ();

  editor_pad = GST_EDITOR_ITEM (gnome_canvas_item_new (GNOME_CANVAS_GROUP (element),
                                                       pad_type,
                                                       "object", G_OBJECT (pad),
                                                       NULL));

  if (GST_PAD_DIRECTION (pad) == GST_PAD_SINK) {
    element->sinkpads = g_list_append (element->sinkpads, editor_pad);
    element->sinks++;
  } else {
    element->srcpads = g_list_append (element->srcpads, editor_pad);
    element->srcs++;
  }
}

static void
gst_editor_element_add_pads (GstEditorElement *element)
{
  GstPad *pad;
  GstPadTemplate *pad_template;
  GList *pads, *pad_templates, *l, *w;
  GstEditorItem *item = (GstEditorItem*)element, *editor_pad;
  GstElement *e;
  GType type;
  
  e = GST_ELEMENT (item->object);
  pads = g_list_copy ((GList *)gst_element_get_pad_list (e));
  pad_templates = g_list_copy (gst_element_get_pad_template_list (e));

  for (l = pads; l; l=l->next) {
    pad = (GstPad*)l->data;

    pad_template = gst_pad_get_pad_template (pad);
    /* 
     * Go through the pad_templates list and remove the pad_template for
     * this pad, rather than show both the pad and pad_template.
     */
    if (pad_template) 
    {
      w = pad_templates;
      EDITOR_LOG ("Trying to find pad template %s\n",
                  GST_OBJECT_NAME (pad_template));
      while (w) {
        if (strcmp (GST_OBJECT_NAME (w->data), GST_OBJECT_NAME (pad_template)) == 0)
          break;
        w = g_list_next (w);
      }
      if (w)
        pad_templates = g_list_remove_link (pad_templates, w);
    }
    else
    {
      EDITOR_LOG ("Element %s: pad '%s' has no pad template",
                     g_type_name (G_OBJECT_TYPE (e)), 
	  	     GST_OBJECT_NAME (pad));
    }

    
    EDITOR_DEBUG ("adding pad %s to element %s",
                  GST_OBJECT_NAME (pad), 
		  gst_element_get_name (e));
    gst_editor_element_add_pad (element, pad);
  }

  for (l = pad_templates; l; l=l->next) {
    pad_template = (GstPadTemplate*)l->data;
    EDITOR_LOG ("evaluating padtemplate %s\n",
                GST_OBJECT_NAME (pad_template));
    
    switch (pad_template->presence) {
    case GST_PAD_SOMETIMES:
      type = gst_editor_pad_sometimes_get_type ();
      break;
    case GST_PAD_REQUEST:
      type = gst_editor_pad_request_get_type ();
      break;
    case GST_PAD_ALWAYS:
      EDITOR_WARNING("Error in element %s: ALWAYS pad template '%s', but no pad provided",
                     g_type_name (G_OBJECT_TYPE (e)), 
	  	     GST_OBJECT_NAME (pad_template));
      continue;
    }
    
    editor_pad = GST_EDITOR_ITEM (gnome_canvas_item_new (GNOME_CANVAS_GROUP (element),
                                                         type,
                                                         "object", G_OBJECT (pad_template),
                                                         NULL));

    if (GST_PAD_TEMPLATE_DIRECTION (pad_template) == GST_PAD_SINK) {
      element->sinkpads = g_list_prepend (element->sinkpads, editor_pad);
      element->sinks++;
    } else {
      element->srcpads = g_list_prepend (element->srcpads, editor_pad);
      element->srcs++;
    }
  }
  
  /* the caller must repack the element */
}

static void
gst_editor_element_remove_pad (GstEditorElement *element, GstPad *pad)
{
  GstEditorItem *editor_pad;

  editor_pad = gst_editor_item_get (GST_OBJECT (pad));

  if (GST_PAD_DIRECTION (pad) == GST_PAD_SINK) {
    element->sinkpads = g_list_remove (element->sinkpads, editor_pad);
    element->sinks--;
  } else {
    element->srcpads = g_list_remove (element->srcpads, editor_pad);
    element->srcs--;
  }
}

static void 
gst_editor_element_set_state (GstEditorElement *element, gint id)
{
  GstEditorItem *item = GST_EDITOR_ITEM (element);
  
  if (item->object)
    gst_element_set_state (GST_ELEMENT (item->object), _gst_element_states[id]);
  else
    g_warning ("no item->object. wtf?");
}

/* return a bool so we can be a GSourceFunc */
static gboolean
gst_editor_element_sync_state (GstEditorElement *element) 
{
  gint id;
  GstEditorItem *item;
  GstElementState state;
  gdouble x1, x2, y1, y2;
  
  item = GST_EDITOR_ITEM (element);
  state = GST_STATE (GST_ELEMENT (item->object));

  /* make sure args to gnome_canvas_item_set are doubles */
  x1 = 0.0; y1 = 0.0;
  x2 = item->width; y2 = item->height;

  for (id=0;id<4;id++)
    if (_gst_element_states[id] == state)
      gnome_canvas_item_set (element->statebox,
                             "x1", x1 + (element->statewidth * id),
                             "y1", y2 - element->stateheight,
                             "x2", x1 + (element->statewidth * (id + 1)),
                             "y2", y2,
                             NULL);

  return FALSE;
}

/**********************************************************************
 * Popup menu calbacks
 **********************************************************************/

static void on_copy (GtkWidget *unused, GstEditorElement *element) 
{
  g_return_if_fail (GST_IS_EDITOR_ELEMENT (element));
  
  gst_editor_element_copy (element);
}

static void on_cut (GtkWidget *unused, GstEditorElement *element) 
{
  g_return_if_fail (GST_IS_EDITOR_ELEMENT (element));
  
  gst_editor_element_cut (element);
}

static void on_remove (GtkWidget *unused, GstEditorElement *element) 
{
  g_return_if_fail (GST_IS_EDITOR_ELEMENT (element));
  
  gst_editor_element_remove (element);
}

/**********************************************************************
 * Public functions
 **********************************************************************/

void 
gst_editor_element_move (GstEditorElement *element,
                         gdouble dx,gdouble dy) 
{
  GstEditorItem *parent;
  gdouble x, y, h, w;
  
  /* we must stay within the parent bin, if any */
  parent = (GstEditorItem*)GNOME_CANVAS_ITEM (element)->parent;
  if (GST_IS_EDITOR_BIN (parent)) {
    gdouble top, bottom, left, right;
    
    top = parent->t.h;
    bottom = parent->b.h;
    left = parent->l.w;
    right = parent->r.w;

    /* element x and y are with respect to the upper-left corner of the bin */
    g_object_get (element, "x", &x, "y", &y, "width", &w, "height", &h, NULL);

    if (parent->height - top - bottom < h ||
        parent->width - left - right < w) {
      g_warning ("bin is too small");
      return;
    }

    if (x + dx < left) {
      dx = left - x;
    } else if (x + dx + w > parent->width - right) {
      dx = parent->width - right - w - x;
    }

    if (y + dy < top) {
      dy = top - y;
    } else if (y + dy + h > parent->height - bottom) {
      dy = parent->height - bottom - h - y;
    }
  }

  gst_editor_item_move (GST_EDITOR_ITEM (element), dx, dy);
}

void
gst_editor_element_cut (GstEditorElement *element)
{
  gst_editor_element_copy (element);
  gst_editor_element_remove (element);
}

void
gst_editor_element_copy (GstEditorElement *element)
{
  xmlChar *mem;
  gint size = 0;
  GtkClipboard *clipboard;
  GstElement *e;

  xmlIndentTreeOutput = 1;

  e = GST_ELEMENT (GST_EDITOR_ITEM (element)->object);

  xmlDocDumpFormatMemory (gst_xml_write (e), &mem, &size, 1);

  if (!size) {
    g_warning ("copy failed");
    return;
  }
  
  clipboard = gtk_clipboard_get (GDK_NONE);
  /* should this be text/xml ? */
  gtk_clipboard_set_text (clipboard, mem, size);
}

void
gst_editor_element_remove (GstEditorElement *element)
{
  GstBin *bin;
  GstElement *e;

  e = GST_ELEMENT (GST_EDITOR_ITEM (element)->object);
  bin = (GstBin*) GST_OBJECT_PARENT (e);

  if (!bin) {
    g_object_set (GNOME_CANVAS_ITEM (element)->canvas, "status",
                  "Could not remove element: Removal of toplevel bin is not allowed.",
                  NULL);
    return;
  }

  gsth_element_unlink_all (e);
  gst_bin_remove (bin, e);
}
