/* GStreamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include "element-tree.h"


struct element_select_classlist {
  gchar *name;
  GSList *subclasses;
  GSList *factories;
};

enum 
{
  NAME_COLUMN,
  DESCRIPTION_COLUMN,
  FACTORY_COLUMN,
  NUM_COLUMNS
};

enum {
  PROP_0,
  PROP_SELECTED,
};


static void	gst_element_browser_element_tree_init		(GstElementBrowserElementTree *browser);
static void	gst_element_browser_element_tree_class_init	(GstElementBrowserElementTreeClass *browser);
static void 	gst_element_browser_element_tree_set_property	(GObject * object, guint prop_id, 
                                                                 const GValue * value, GParamSpec * pspec);
static void 	gst_element_browser_element_tree_get_property	(GObject * object, guint prop_id,
                                                                 GValue * value, GParamSpec * pspec);

static gint 	compare_name			(gconstpointer a, gconstpointer b);
static gint 	compare_str			(gconstpointer a, gconstpointer b);
static gint 	compare_classes			(gconstpointer a, gconstpointer b);

static void	populate_store			(GtkTreeStore *store,
                                                 GtkTreeIter *parent,
                                                 struct element_select_classlist *class);
static gboolean tree_select_function		(GtkTreeSelection *selection,
                                                 GtkTreeModel *model,
                                                 GtkTreePath *path, gboolean foo,
                                                 gpointer data);
static void	tree_select				(GstElementBrowserElementTree *tree);
static GSList* 	get_class_tree			(void);


static GtkScrolledWindowClass *parent_class = NULL;


GType
gst_element_browser_element_tree_get_type (void)
{
  static GType element_tree_type = 0;
  
  if (!element_tree_type) {
    static const GTypeInfo element_tree_info = {
      sizeof(GstElementBrowserElementTreeClass),
      NULL,
      NULL,
      (GClassInitFunc)gst_element_browser_element_tree_class_init,
      NULL,
      NULL,
      sizeof(GstElementBrowserElementTree),
      0,
      (GInstanceInitFunc)gst_element_browser_element_tree_init,
    };
    element_tree_type = g_type_register_static (gtk_scrolled_window_get_type (), "GstElementBrowserElementTree", &element_tree_info, 0);
  }
  return element_tree_type;
}

static void
gst_element_browser_element_tree_class_init (GstElementBrowserElementTreeClass *klass)
{
  GObjectClass *gobject_class;
  
  gobject_class = G_OBJECT_CLASS (klass);
  
  parent_class = g_type_class_ref (gtk_scrolled_window_get_type ());
  
  gobject_class->set_property = gst_element_browser_element_tree_set_property;
  gobject_class->get_property = gst_element_browser_element_tree_get_property;

  g_object_class_install_property (gobject_class, PROP_SELECTED,
    g_param_spec_object ("selected", "Selected", "Selected element factory",
                         gst_element_factory_get_type (), G_PARAM_READWRITE));
}

static void gst_element_browser_element_tree_init (GstElementBrowserElementTree *tree)
{
  GtkTreeViewColumn *column;
  GtkTreeSelection *selection;
  GSList *classtree;

  tree->store = gtk_tree_store_new (NUM_COLUMNS,
                                    G_TYPE_STRING,
                                    G_TYPE_STRING,
                                    GST_TYPE_ELEMENT_FACTORY);
  tree->view = gtk_tree_view_new_with_model (GTK_TREE_MODEL (tree->store));
  column = gtk_tree_view_column_new_with_attributes ("Element",
                                                     gtk_cell_renderer_text_new (),
                                                     "text", NAME_COLUMN,
                                                     NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree->view), column);
  column = gtk_tree_view_column_new_with_attributes ("Description",
                                                     gtk_cell_renderer_text_new (),
                                                     "text", DESCRIPTION_COLUMN,
                                                     NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree->view), column);
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree->view));
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
  gtk_tree_selection_set_select_function (selection, tree_select_function, NULL, NULL);
  gtk_widget_set_size_request (tree->view, 200, -1);
  gtk_widget_show (tree->view);
  
  for (classtree = get_class_tree (); classtree; classtree = classtree->next)
    populate_store (tree->store, NULL, 
                    (struct element_select_classlist *)(classtree->data));

  g_signal_connect_swapped (tree->view, "row-activated", G_CALLBACK (tree_select), tree);
  g_signal_connect_swapped (selection, "changed", G_CALLBACK (tree_select), tree);

  /* hack: force the scrolledwindow to realize scrollbars */
  g_object_set (tree, "hadjustment", NULL, "vadjustment", NULL, NULL);
  gtk_container_add (GTK_CONTAINER (tree), tree->view);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (tree),
                                  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
}

static void
gst_element_browser_element_tree_set_property (GObject* object, guint prop_id, 
                                            const GValue* value, GParamSpec* pspec)
{
  GstElementBrowserElementTree *tree;
	    
  tree = GST_ELEMENT_BROWSER_ELEMENT_TREE (object);

  switch (prop_id) {
    case PROP_SELECTED:
      tree->selected = (GstElementFactory*) g_value_get_object (value);
      g_warning ("I am mostly nonfunctional..");
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      return;
  }
}

static void
gst_element_browser_element_tree_get_property (GObject* object, guint prop_id, 
                                            GValue* value, GParamSpec* pspec)
{
  GstElementBrowserElementTree *tree;
	    
  tree = GST_ELEMENT_BROWSER_ELEMENT_TREE (object);

  switch (prop_id) {
    case PROP_SELECTED:
      g_value_set_object (value, (GObject*)tree->selected);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

GtkWidget *gst_element_browser_element_tree_new () 
{
  return GTK_WIDGET (g_object_new (gst_element_browser_element_tree_get_type (), NULL));
}

static void populate_store (GtkTreeStore *store, GtkTreeIter *parent,
                            struct element_select_classlist *class)
{
  GSList *traverse;
  GtkTreeIter iter, newparent;

  gtk_tree_store_append (store, &iter, parent);
  gtk_tree_store_set (store, &iter,
                      NAME_COLUMN, g_strdup (class->name),
                      DESCRIPTION_COLUMN, NULL,
                      -1);
  newparent = iter;

  /* start with a sort and save to original GSList location */
  traverse = g_slist_sort (class->subclasses, compare_classes);
  class->subclasses = traverse;
  while (traverse) {
    populate_store (store, &newparent,
                    (struct element_select_classlist *)(traverse->data));
    traverse = g_slist_next(traverse);
  }

  traverse = class->factories;
  while (traverse) {
    GstElementFactory *factory = (GstElementFactory *)(traverse->data);
    gtk_tree_store_append (store, &iter, &newparent);
    gtk_tree_store_set (store, &iter,
                        NAME_COLUMN, g_strdup (GST_OBJECT_NAME (factory)),
                        DESCRIPTION_COLUMN, g_strdup (factory->details.description),
                        FACTORY_COLUMN, factory,
                        -1);
    traverse = g_slist_next(traverse);
  }
}

static gboolean tree_select_function (GtkTreeSelection *selection, GtkTreeModel *model,
                                      GtkTreePath *path, gboolean foo, gpointer data)
{
  GtkTreeIter iter;
  GstElementFactory *factory = NULL;

  gtk_tree_model_get_iter (model, &iter, path);
  gtk_tree_model_get (model, &iter, FACTORY_COLUMN, &factory, -1);
  
  if (factory)
    return TRUE;
  else
    return FALSE;
}

static void tree_select (GstElementBrowserElementTree *tree)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  GstElementFactory *factory;

  /* if nothing is selected, ignore it */
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree->view));
  if (!gtk_tree_selection_get_selected (selection, &model, &iter))
    return;
  
  gtk_tree_model_get (model, &iter,
                      FACTORY_COLUMN, &factory,
                      -1);

  g_return_if_fail (factory != NULL);
  
  tree->selected = factory;

  g_object_notify (G_OBJECT (tree), "selected");
}

static GSList *get_class_tree (void)
{
  const GList *elements;
  GstElementFactory *element;
  gchar **classes, **class;
  GSList *classtree, *treewalk;
  GSList **curlist;
  struct element_select_classlist *branch = NULL;
  static GSList *ret = NULL;

  if (ret)
    return ret;

  /* first create a sorted (by class) tree of all the factories */
  classtree = NULL;
  elements = gst_registry_pool_feature_list (GST_TYPE_ELEMENT_FACTORY);
  while (elements) {
    element = (GstElementFactory *)(elements->data);

    /* if the class is "None", we just ignore it */
    if (strncmp ("None", element->details.klass, 4) == 0)
      goto loop;

    /* split up the factory's class */
    classes = g_strsplit(element->details.klass,"/",0);
    class = classes;
    curlist = &classtree;
    /* walk down the class tree to find where to place this element
     * the goal is for treewalk to point to the right class branch
     * when we exit this thing, branch is pointing where we want
     */
    while (*class) {
      treewalk = *curlist;
      /* walk the current level of class to see if we have the class */
      while (treewalk) {
        branch = (struct element_select_classlist *)(treewalk->data);
        /* see if this class matches what we're looking for */
        if (!strcmp(branch->name,*class)) {
          /* if so, we progress down the list into this one's list */
          curlist = &branch->subclasses;
          break;
        }
        treewalk = g_slist_next(treewalk);
      }
      /* if treewalk == NULL, it wasn't in the list. add one */
      if (treewalk == NULL) {
        /* curlist is pointer to list */
        branch = g_new0(struct element_select_classlist,1);
        branch->name = g_strdup(*class);
        *curlist = g_slist_insert_sorted(*curlist,branch,compare_str);
        curlist = &branch->subclasses;
      }
      class++;
    }
    /* theoretically branch points where we want now */
    branch->factories = g_slist_insert_sorted(branch->factories,element,
                                              compare_name);
  loop:
    elements = g_list_next(elements);
  }

  ret = g_slist_sort (classtree, compare_classes);
  return ret;
}

static gint compare_name(gconstpointer a,gconstpointer b)
{
  return (strcmp(GST_OBJECT_NAME (a),
                 GST_OBJECT_NAME (b)));
}

static gint compare_str(gconstpointer a,gconstpointer b)
{
  return (strcmp((gchar *)a,(gchar *)b));
}

static gint compare_classes (gconstpointer a, gconstpointer b)
{
  struct element_select_classlist *A = (struct element_select_classlist *) a;
  struct element_select_classlist *B = (struct element_select_classlist *) b;
  return (strcmp ((gchar *) A->name, (gchar *) B->name));
}
