/****************************\
*                            *
*  TCADGroup implementation  *
*                            *
\****************************/

#include <string.h>
#include "group.h"
#include "frame.h"
#include "text.h"
#include "simpleline.h"
#include "rightangleline.h"
#include "framedtext.h"
#include "main.h"

TCADGroup::TCADGroup(TCADSheet *Sheet) : TCADRectangleObject(Sheet)
{
  FirstChild = NULL;
  LastChild = NULL;
  BoxStyle = BS_FIXED;
}

TCADGroup::~TCADGroup()
{
  TCADObject *Obj,*Next;

  // Delete all children
  Obj = FirstChild;
  while(Obj) {
    Next = Obj->Next;
    delete Obj;
    Obj = Next;
  }
}

void
TCADGroup::Save(TCADSaveStream *Fl)
{
  Fl->SaveObjectName("TCADGroup");
  SaveProperties(Fl);
  Fl->SaveEndObject();
}

void
TCADGroup::SaveProperties(TCADSaveStream *Fl)
{
  TCADObject *Child;

  TCADObject::SaveProperties(Fl);

  Fl->SaveInt("BoxStyle",BoxStyle);

  // Save contained objects
  Child = FirstChild;
  while(Child != NULL) {
    Child->Save(Fl);
    Child = Child->Next;
  }
}

char
TCADGroup::LoadProperty(TCADLoadStream *Fl,char *ID)
{
  TCADObject *Obj;

  if (strcasecmp("BoxStyle",ID) == 0) BoxStyle = (TBoxStyle)Fl->LoadInt();
  else if (strcasecmp("TCADGroup",ID) == 0) {
    Obj = new TCADGroup(Sheet);
    Obj->Load(Fl);
    AppendChild(Obj);
  }
  else if (strcasecmp("TCADFrame",ID) == 0) {
    Obj = new TCADFrame(Sheet);
    Obj->Load(Fl);
    AppendChild(Obj);
  }
  else if (strcasecmp("TCADText",ID) == 0) {
    Obj = new TCADText(Sheet);
    Obj->Load(Fl);
    AppendChild(Obj);
  }
  else if (strcasecmp("TCADSimpleLine",ID) == 0) {
    Obj = new TCADSimpleLine(Sheet);
    Obj->Load(Fl);
    AppendChild(Obj);
  }
  else if (strcasecmp("TCADRightAngleLine",ID) == 0) {
    Obj = new TCADRightAngleLine(Sheet);
    Obj->Load(Fl);
    AppendChild(Obj);
  }
  else if (strcasecmp("TCADFramedText",ID) == 0) {
    Obj = new TCADFramedText(Sheet);
    Obj->Load(Fl);
    AppendChild(Obj);
  }
  else return TCADObject::LoadProperty(Fl,ID);

  return(1);
}

void
TCADGroup::Draw(GdkRegion *Region)
{
  GdkOverlapType Overlap;
  TCADObject *Obj;

//  printf("TCADGroup::Draw()\n");

  if (Region) {
    Overlap = gdk_region_rect_in(Region, &EncapRect);
    if (Overlap == GDK_OVERLAP_RECTANGLE_OUT) return;
    if (Overlap == GDK_OVERLAP_RECTANGLE_IN) Region = 0;  // Clear region, so childs won't do the test
  }

  // Draw from the back to the front
  Obj = LastChild;
  while(Obj) {
    Obj->Draw(Region);
    Obj = Obj->Prev;
  }
}

/*
 * Child manipulation routines
 */

void
TCADGroup::AppendChild(TCADObject *Child)
{
  Child->Prev = LastChild;
  Child->Next = NULL;

  if (FirstChild == NULL) FirstChild = Child;
  else LastChild->Next = Child;
  LastChild = Child;

  Child->Parent = this;

  AffectEncapRect(Child);
}

void
TCADGroup::PrependChild(TCADObject *Child)
{
  Child->Prev = NULL;
  Child->Next = FirstChild;

  if (LastChild == NULL) LastChild = Child;
  else FirstChild->Prev = Child;
  FirstChild = Child;

  Child->Parent = this;

  AffectEncapRect(Child);
}

void
TCADGroup::DeleteChild(TCADObject *Child)
{
  if (Child->Parent != this) return;

  if (Child->Next == NULL) LastChild = Child->Prev;
  else Child->Next->Prev = Child->Prev;
  if (Child->Prev == NULL) FirstChild = Child->Next;
  else Child->Prev->Next = Child->Next;

  Child->Next = NULL;
  Child->Prev = NULL;
  Child->Parent = NULL;

  ComputeEncapRect();
}

TCADObject *
TCADGroup::ButtonPressed(int x,int y)
{
  TCADObject *Child,*Result;

  // Pass to all childs
  Child = FirstChild;
  while(Child) {
    Result = Child->ButtonPressed(x,y);
    if (Result != NULL) return(Result);
    Child = Child->Next;
  }

  return(NULL);
}

void
TCADGroup::ComputeEncapRect()
{
  TCADObject *Child;

  if (FirstChild == NULL) {
    EncapRect.x = 0;
    EncapRect.y = 0;
    EncapRect.width = 0;
    EncapRect.height = 0;
  }
  else {
    Child = FirstChild;
    EncapRect = Child->EncapRect;
    Child = Child->Next;
    while(Child) {
      AffectEncapRect(Child);
      Child = Child->Next;
    }
  }
}

void
TCADGroup::AffectEncapRect(TCADObject *Child)
{
  System->UnionRectangles(&EncapRect,&Child->EncapRect);
}

void TCADGroup::Export(FILE *Fl,TExportFormat Format)
{
  TCADObject *Child;

  // Export contained objects
  Child = FirstChild;
  while(Child != NULL) {
    Child->Export(Fl,Format);
    Child = Child->Next;
  }
}

TCADObject *
TCADGroup::ObjectByUniq(unsigned long Uniq)
{
  TCADObject *Obj,*Result;

  Result = NULL;
  Obj = FirstChild;
  while(Obj) {
    if (Obj->Uniq == Uniq) return(Obj);
    if (Obj->IsGroup()) {
      Result = ((TCADGroup *)Obj)->ObjectByUniq(Uniq);
      if (Result != NULL) return(Result);
    }
    Obj = Obj->Next;
  }

  return(NULL);
}

gboolean
TCADGroup::CanAttach(int& x,int& y,TAttach *Attach)
{
  TCADObject *Obj;

  if ((x < (EncapRect.x-ATTACH_RANGE)) ||
      (y < (EncapRect.y-ATTACH_RANGE)) ||
      (x > (EncapRect.x+EncapRect.width+ATTACH_RANGE)) ||
      (y > (EncapRect.y+EncapRect.height+ATTACH_RANGE))) return(FALSE);  // Out of range

  Obj = FirstChild;
  while(Obj) {
    if (Obj->CanAttach(x,y,Attach)) return(TRUE);
    Obj = Obj->Next;
  }

  return(FALSE);
}

