/*****************************\
*                             *
*  Implements TCADSimpleLine  *
*                             *
\*****************************/

#include <string.h>
#include <math.h>
#include "line.h"
#include "main.h"
#include "arrowproperties.h"

// If user clicks within this distance from line, it is considered
// he/she clicked on the line.
#define SELECT_DISTANCE 4

TCADLine::TCADLine(TCADSheet *Sheet,int x=0,int y=0) : TCADObject(Sheet)
{
  First.x = x;
  First.y = y;
  Last.x = x;
  Last.y = y;

  FirstArrow = AR_NONE;
  LastArrow = AR_BLACK;
  FirstArrowSize = ARROW_SIZE_MEDIUM;
  LastArrowSize = ARROW_SIZE_MEDIUM;

  FAttach.Object = NULL;
  LAttach.Object = NULL;

  Handles = Handle;
  Handle[0].Type = HT_LINEEND;
  Handle[1].Type = HT_LINEEND;
  NumHandles = 2;
}

TCADLine::~TCADLine() {
  // Unattach
  if (FAttach.Object != NULL) FAttach.Object->UnattachLine(this);
  if (LAttach.Object != NULL) LAttach.Object->UnattachLine(this);
}

void
TCADLine::Save(TCADSaveStream *Fl)
{
  Fl->SaveObjectName("TCADLine");
  SaveProperties(Fl);
  Fl->SaveEndObject();
}

void
TCADLine::SaveProperties(TCADSaveStream *Fl)
{
  SaveLineProperties(Fl);
  Fl->SaveInt("X1",First.x);
  Fl->SaveInt("Y1",First.y);
  Fl->SaveInt("X2",Last.x);
  Fl->SaveInt("Y2",Last.y);

  Fl->SaveInt("FArrowSize",FirstArrowSize);
  Fl->SaveInt("FArrow",FirstArrow);
  Fl->SaveInt("LArrowSize",LastArrowSize);
  Fl->SaveInt("LArrow",LastArrow);

  if (LAttach.Object != NULL) Fl->SaveInt("FAttachUniq",FAttach.Object->Uniq);
  else Fl->SaveInt("FAttachUniq",0);
  Fl->SaveInt("FAttachXOff",FAttach.XOff);
  Fl->SaveInt("FAttachYOff",FAttach.YOff);
  Fl->SaveInt("FAttachFlags",FAttach.Flags);
  Fl->SaveInt("FAttachDirection",FAttach.Direction);
  Fl->SaveInt("FAttachDirectionForced",FAttach.DirectionForced);

  if (LAttach.Object != NULL) Fl->SaveInt("LAttachUniq",LAttach.Object->Uniq);
  else Fl->SaveInt("LAttachUniq",0);
  Fl->SaveInt("LAttachYOff",LAttach.YOff);
  Fl->SaveInt("LAttachXOff",LAttach.XOff);
  Fl->SaveInt("LAttachFlags",LAttach.Flags);
  Fl->SaveInt("LAttachDirection",LAttach.Direction);
  Fl->SaveInt("LAttachDirectionForced",LAttach.DirectionForced);
}

char
TCADLine::LoadProperty(TCADLoadStream *Fl,char *ID)
{
  if (strcasecmp(ID,"X1") == 0) First.x = Fl->LoadInt();
  else if (strcasecmp(ID,"Y1") == 0) First.y = Fl->LoadInt();
  else if (strcasecmp(ID,"X2") == 0) Last.x = Fl->LoadInt();
  else if (strcasecmp(ID,"Y2") == 0) Last.y = Fl->LoadInt();
  else if (strcasecmp(ID,"FArrowSize") == 0) FirstArrowSize = Fl->LoadInt();
  else if (strcasecmp(ID,"FArrow") == 0) FirstArrow = TArrowStyle(Fl->LoadInt());
  else if (strcasecmp(ID,"LArrowSize") == 0) LastArrowSize = Fl->LoadInt();
  else if (strcasecmp(ID,"LArrow") == 0) LastArrow = TArrowStyle(Fl->LoadInt());
  else if (strcasecmp(ID,"FAttachUniq") == 0)  FAttach.OUniq = Fl->LoadInt();
  else if (strcasecmp(ID,"FAttachXOff") == 0)  FAttach.XOff = Fl->LoadInt();
  else if (strcasecmp(ID,"FAttachYOff") == 0)  FAttach.YOff = Fl->LoadInt();
  else if (strcasecmp(ID,"FAttachFlags") == 0) FAttach.Flags = Fl->LoadInt();
  else if (strcasecmp(ID,"FAttachDirection") == 0) FAttach.Direction = Fl->LoadInt();
  else if (strcasecmp(ID,"FAttachDirectionForced") == 0) FAttach.DirectionForced = Fl->LoadInt();
  else if (strcasecmp(ID,"LAttachUniq") == 0)  LAttach.OUniq = Fl->LoadInt();
  else if (strcasecmp(ID,"LAttachYOff") == 0)  LAttach.YOff = Fl->LoadInt();
  else if (strcasecmp(ID,"LAttachXOff") == 0)  LAttach.XOff = Fl->LoadInt();
  else if (strcasecmp(ID,"LAttachFlags") == 0) LAttach.Flags = Fl->LoadInt();
  else if (strcasecmp(ID,"LAttachDirection") == 0) LAttach.Direction = Fl->LoadInt();
  else if (strcasecmp(ID,"LAttachDirectionForced") == 0) LAttach.DirectionForced = Fl->LoadInt();
  else if (TCADLinePropertiesI::LoadProperty(Fl,ID)) return(TRUE);
  else return TCADObject::LoadProperty(Fl,ID);

  return(TRUE);
}

void
TCADLine::PostLoad()
{
  /* Recompute rectangles */
  RecomputeEncapRect();
  RecomputeRefRect();

  /* Reattach */
  if (FAttach.OUniq != 0) FAttach.Object = Sheet->ObjectByUniq(FAttach.OUniq);
  if (LAttach.OUniq != 0) LAttach.Object = Sheet->ObjectByUniq(LAttach.OUniq);
  if (FAttach.Object != NULL) FAttach.Object->AttachLine(this);
  if (LAttach.Object != NULL) LAttach.Object->AttachLine(this);
  Handle[0].Attached = (FAttach.Object != NULL);
  Handle[1].Attached = (LAttach.Object != NULL);
/*
  printf("FAttach.OUniq = %d\n",FAttach.OUniq);
  printf("FAttach.Object = %x\n",FAttach.Object);
  printf("LAttach.OUniq = %d\n",LAttach.OUniq);
  printf("LAttach.Object = %x\n",LAttach.Object);
*/
}

TCADObject *
TCADLine::ButtonPressed(int x,int y)
{
  int MD = SELECT_DISTANCE;

  // Check if point is in rectangle
  if ((x < (EncapRect.x-MD)) || (x > (EncapRect.x+EncapRect.width+MD )) || 
      (y < (EncapRect.y-MD)) || (y > (EncapRect.y+EncapRect.height+MD))) return(NULL);  // Out of rectangle

  return(this);
}

void
TCADLine::UpdateHandlePos()
{
  int d = HANDLE_SIZE/2;

  RecomputeEncapRect();
  RecomputeRefRect();

  Handle[0].x = First.x-d;
  Handle[0].y = First.y-d;
  Handle[1].x = Last.x-d;
  Handle[1].y = Last.y-d;
}

void
TCADLine::Select(GdkRectangle *RefreshRect)
{
  TCADObject::Select(RefreshRect);
   UpdateHandlePos();
}

char
TCADLine::HandleDragged(int x,int y,GdkRectangle *RefreshRect)
{
  int T;

  if (!DHandle) return(0);

  *RefreshRect = RefRect;

  // Set new x/y
/*
  x = x-DXO;
  y = y-DYO;
*/
  if (x < 0) x = 0;
  if (y < 0) y = 0;

  T = Sheet->EncapRect.x+Sheet->EncapRect.width;
  if (x >= T) x = T-1;

  T = Sheet->EncapRect.y+Sheet->EncapRect.height;
  if (y >= T) y = T-1;

  if (DraggingHandle == 0) {
    First.x = x;
    First.y = y;
  }
  else {
    Last.x = x;
    Last.y = y;
  }

  UpdateLine((DraggingHandle >= 2));
  UpdateHandlePos();  // Also recompute RefRect

  System->UnionRectangles(RefreshRect,&RefRect);
//  printf("RefreshRect: %d,%d,%d,%d\n",RefreshRect->x,RefreshRect->y,RefreshRect->width,RefreshRect->height);

  // Enlarge RefreshRect to also contain handles
  RefreshRect->x -= HANDLE_SIZE/2;
  RefreshRect->y -= HANDLE_SIZE/2;
  RefreshRect->width += HANDLE_SIZE+1;
  RefreshRect->height += HANDLE_SIZE+1;

  return(1);
}

void
TCADLine::HandleButtonPressed(int HandleNo,int x,int y)
{
  UpdateHandlePos();

  DraggingHandle = HandleNo;
  DHandle = 1;

  // Copy attachment data to tempotary store
  if (HandleNo == 0) TmpAttach = FAttach;
  else if (HandleNo == 1) TmpAttach = LAttach;  

  if (HandleNo == 0) {
    DXO = x-First.x;
    DYO = y-First.y;
  }
  else if (HandleNo == 1) {
    DXO = x-Last.x;
    DYO = y-Last.y;
  }
}

char
TCADLine::HandleButtonReleased(GdkRectangle *RefreshRect)
{
  TAttach *ActAttach;

  RefreshRect = RefreshRect;

  if (!DHandle) return(0);
  
  /* Copy attachment */
  if (DraggingHandle == 0) ActAttach = &FAttach;
  else if (DraggingHandle == 1) ActAttach = &LAttach;

  if (DraggingHandle < 2) {
    if (ActAttach->Object != TmpAttach.Object) {
      // Reattach
      if (ActAttach->Object != NULL) ActAttach->Object->UnattachLine(this);
      if (TmpAttach.Object != NULL) TmpAttach.Object->AttachLine(this);
    }
    *ActAttach = TmpAttach;  // Copy other data
  }

  DHandle = 0;
  return(0);
}

void
TCADLine::RefreshArrows(TArrowStyle FA,TArrowStyle LA,int FS,int LS)
{
  GdkRectangle RefreshRect;

  RefreshRect = RefRect;
  FirstArrow = FA;
  LastArrow = LA;
  FirstArrowSize = FS;
  LastArrowSize = LS;
  RecomputeRefRect();
  System->UnionRectangles(&RefreshRect,&RefRect);
  Sheet->DrawRect(&RefreshRect);
}

void
TCADLine::ShowArrowProperties()
{
  ArrowPropertiesDialog(FirstArrow,LastArrow,FirstArrowSize,LastArrowSize,this);
}

void
TCADLine::MoveResize(int dx,int dy,int dw,int dh,GdkRectangle *RefreshRect)
{
  int X,Y,W,H;
  int X2,Y2,T;

  *RefreshRect = RefRect;

  // Do not move attached lines
  if ((FAttach.Object != NULL) || (LAttach.Object != NULL)) return;

  X = EncapRect.x + dx;
  First.x += dx;
  Last.x += dx;
  Y = EncapRect.y + dy;
  First.y += dy;
  Last.y += dy;
  W = EncapRect.width;
  H = EncapRect.height;
  X2 = Sheet->EncapRect.width;
  Y2 = Sheet->EncapRect.height;

  if (W < 0) W = 0;
  if (H < 0) H = 0;

  /* Make object to not exceed boundaries */

  // Right/bottom boundary
  T = (X+W)-X2+1;
  if (T > 0) {
    X -= T; // X axis. Prefer moving.
    First.x -= T;
    Last.x -= T;
  }
  T = (Y+H)-Y2+1;
  if (T > 0) {
    Y -= T; // Y axis. Prefer moving.
    First.y -= T;
    Last.y -= T;
  }

  // Top/left boundary
  if (X < 0) {
    First.x -= X;
    Last.x -= X;
    X = 0;
  }
  if (Y < 0) {
    First.y -= Y;
    Last.y -= Y;
    Y = 0;
  }

  // Check again right/bottom, but now resize
  if ((X+W) >= X2) W -= ((X+W)-X2+1);   // X axis
  if ((Y+H) >= Y2) H -= ((Y+H)-Y2+1);   // Y axis

  // Affect first/last point if line was resized
  X2 = X+W;
  Y2 = Y+H;
  if (First.x > X2) First.x = X2;
  if (Last.y > Y2) Last.y = Y2;

  UpdateLine(FALSE);
  RecomputeEncapRect();

  UpdateHandlePos();

  System->UnionRectangles(RefreshRect,&RefRect);
  RefreshRect->x -= HANDLE_SIZE/2;
  RefreshRect->y -= HANDLE_SIZE/2;
  RefreshRect->width += HANDLE_SIZE;
  RefreshRect->height += HANDLE_SIZE;
}

void
TCADLine::SetHandleAttachment(gboolean Attached,TAttach *Attach)
{
  if (!DHandle) return;

  Handles[DraggingHandle].Attached = Attached;
  TmpAttach = *Attach;
  if (!Attached) TmpAttach.Object = NULL;
}

// Tests if point x,y is near abscissa First->Last
gboolean
TCADLine::PointNearLine(GdkPoint& First,GdkPoint& Last,int x,int y)
{
  int D,MD = SELECT_DISTANCE,MD2=SELECT_DISTANCE/2;
  int A,B,C;
  int x1,y1,x2,y2;

  // Exchange points
  x1 = MIN(First.x,Last.x);
  x2 = MAX(First.x,Last.x);
  y1 = MIN(First.y,Last.y);
  y2 = MAX(First.y,Last.y);

  // Check if point is in rectangle
  if ((x < (x1-MD2)) || (x > (x2+MD2 )) ||
      (y < (y1-MD2)) || (y > (y2+MD2))) return(FALSE);  // Out of rectangle

  // Compute A,B,C;
  B =  First.x-Last.x;
  A = -First.y+Last.y;
  C = -(First.x*A + First.y*B);

  D = (int)(ABS(A*x + B*y + C)/sqrt(A*A + B*B));

  return(D <= MD);
}

void
TCADLine::RecomputeRefRect()
{
  int T = 0;
  if (FirstArrow != AR_NONE) T = FirstArrowSize;
  if (LastArrow != AR_NONE) T = MAX(T,LastArrowSize);

  RefRect.x = EncapRect.x-T;
  RefRect.y = EncapRect.y-T;
  RefRect.width = EncapRect.width+2*T+1;
  RefRect.height = EncapRect.height+2*T+1;
}

void
TCADLine::ArrowToFig(TArrowStyle Arrow,int& Type,int& Style)
{
  switch(Arrow) {
    case AR_SIMPLE:
      Type = 1;
      Style = 0;
      break;
    case AR_WHITE:
      Type = 2;
      Style = 0;
      break;
    case AR_BLACK:
      Type = 2;
      Style = 1;
      break;
    default:
      Type = 0;
      Style = 0;
  }
}

void
TCADLine::ExportFigLine(FILE *Fl,int NumPoints)
{
  int Type,Style;

  fprintf(Fl,"2 1 %d %d %d %d 0 0 %d %f %d %d %d %d %d %d\n",
    0,  // Line style
    1,  // Thickness
    0,  // Pen color
    7,  // Fill color
    -1, // Area fill
    (float)0,  // Style val
    0,  // Join style
    0,  // Cap style
    -1, // Radius
    (LastArrow != AR_NONE),
    (FirstArrow != AR_NONE),
    NumPoints   // # of points
  );

  // Export arrows
  if (LastArrow != AR_NONE) {
     ArrowToFig(LastArrow,Type,Style);
     fprintf(Fl,"%d %d 1 %d %d\n",Type,Style,LastArrowSize,LastArrowSize/2);
  }

  if (FirstArrow != AR_NONE) {
     ArrowToFig(FirstArrow,Type,Style);
    fprintf(Fl,"%d %d 1 %d %d\n",Type,Style,FirstArrowSize,FirstArrowSize/2);
  }
}

