/**************************************************\
*                                                  *
*  Implements line which connects in right angles  *
*                                                  *
\**************************************************/

#include <string.h>
#include <math.h>
#include <stdlib.h>
#include "main.h"
#include "rightangleline.h"

// Minimal length of first/last segment
#define MINLENGTH 8

// Length when we have space
#define DESIREDLENGTH 32

#define SGN(a) ((a<0)?-1:((a>0)?1:0))

int
TCADRightAngleLine::Opposite(int Dir)
{
  if (Dir & DIR_UP) return(DIR_DOWN);
  if (Dir & DIR_DOWN) return(DIR_UP);
  if (Dir & DIR_LEFT) return(DIR_RIGHT);
  if (Dir & DIR_RIGHT) return(DIR_LEFT);
  return(0);
}

gboolean
TCADRightAngleLine::IsOK(int Dir,int X1,int Y1,int X2,int Y2)
{
  if (Dir == DIR_UP) return((Y1-Y2) >= 2*MINLENGTH);
  if (Dir == DIR_DOWN) return((Y2-Y1) >= 2*MINLENGTH);
  if (Dir == DIR_LEFT) return((X1-X2) >= 2*MINLENGTH);
  if (Dir == DIR_RIGHT) return((X2-X1) >= 2*MINLENGTH);
  return(0);
}

int
TCADRightAngleLine::AddMinLength(int Dir,int X,int Y)
{
  if (Dir == DIR_UP) return(Y-MINLENGTH);
  if (Dir == DIR_DOWN) return(Y+MINLENGTH);
  if (Dir == DIR_LEFT) return(X-MINLENGTH);
  if (Dir == DIR_RIGHT) return(X+MINLENGTH);
  return(0);
}

int
TCADRightAngleLine::AddDesiredLength(int Dir,int X,int Y)
{
  if (Dir == DIR_UP) return(Y-DESIREDLENGTH);
  if (Dir == DIR_DOWN) return(Y+DESIREDLENGTH);
  if (Dir == DIR_LEFT) return(X-DESIREDLENGTH);
  if (Dir == DIR_RIGHT) return(X+DESIREDLENGTH);
  return(0);
}

void
TCADRightAngleLine::AddDesiredLength2(int Dir,int& X,int& Y)
{
  if (Dir & DIR_UP) Y -= DESIREDLENGTH/2;
  if (Dir & DIR_DOWN) Y += DESIREDLENGTH/2;
  if (Dir & DIR_LEFT) X -= DESIREDLENGTH/2;
  if (Dir & DIR_RIGHT) X += DESIREDLENGTH/2;
}

void
TCADRightAngleLine::ConnectPoints(int X1,int Y1,int D1,int X2,int Y2,int D2,int *NumPoints,GdkPoint *Points)
{
  int T,T1,T2;
  gboolean Split;
  gboolean OK1,OK2,OneLine;
  int DX,DY;

  DX = X2-X1;
  DY = Y2-Y1;
  
  /* First test if we can connect by only one line */
  OneLine = FALSE;
  if (X1 == X2) {
    if (((DY <= 0) && (D1 == DIR_UP  ) && (D2 = DIR_DOWN)) ||
        ((DY >= 0) && (D1 == DIR_DOWN) && (D2 = DIR_UP  ))) OneLine = TRUE;
  }
  if (Y1 == Y2) {
    if (((DX <= 0) && (D1 == DIR_LEFT ) && (D2 = DIR_RIGHT)) ||
        ((DX >= 0) && (D1 == DIR_RIGHT) && (D2 = DIR_LEFT ))) OneLine = TRUE;
  }
  
  if (OneLine) {
    if ((X1 != X2) || (Y1 != Y2)) *NumPoints = 2;
    else *NumPoints = 0;
  }
  else {
    /* Test same directions */
      if (D1 == D2) {
      // Same direction
      switch(D1) {
        case DIR_UP:
        case DIR_DOWN:
          if (D1 == DIR_UP) T = MIN(Y1-DESIREDLENGTH,Y2-DESIREDLENGTH);
          else T = MAX(Y1+DESIREDLENGTH,Y2+DESIREDLENGTH);
          *NumPoints = 4;
          Points[1].x = X1;
          Points[1].y = T;
          Points[2].x = X2;
          Points[2].y = T;
          break;
        case DIR_LEFT:
        case DIR_RIGHT:
          if (D1 == DIR_LEFT) T = MIN(X1-DESIREDLENGTH,X2-DESIREDLENGTH);
          else T = MAX(X1+DESIREDLENGTH,X2+DESIREDLENGTH);
          *NumPoints = 4;
          Points[1].x = T;
          Points[1].y = Y1;
          Points[2].x = T;
          Points[2].y = Y2;
          break;
      }
    }
    else if (D1 == Opposite(D2)) {
      if ((D1 == DIR_UP) || (D1 == DIR_DOWN)) {
        T = Y2-Y1;
        if (D1 == DIR_DOWN) Split = (T < (MINLENGTH*2));
        else Split = (T > (-MINLENGTH*2));
  
        if (Split) {
          T = (X1+X2)/2;
          if (D1 == DIR_UP) {
            T1 = Y1-DESIREDLENGTH;
            T2 = Y2+DESIREDLENGTH;
          }
          else {
            T1 = Y1+DESIREDLENGTH;
            T2 = Y2-DESIREDLENGTH;
          }
          *NumPoints = 6;
          Points[1].x = X1;
          Points[1].y = T1;
          Points[2].x = T;
          Points[2].y = T1;
          Points[3].x = T;
          Points[3].y = T2;
          Points[4].x = X2;
          Points[4].y = T2;
        }
        else {
          T = (Y1+Y2)/2;
          *NumPoints = 4;
          Points[1].x = X1;
          Points[1].y = T;
          Points[2].x = X2;
          Points[2].y = T;
        }
      }
      else {
        T = X2-X1;
        if (D1 == DIR_RIGHT) Split = (T < (MINLENGTH*2));
        else Split = (T > (-MINLENGTH*2));
  
        if (Split) {
          T = (Y1+Y2)/2;
          if (D1 == DIR_LEFT) {
            T1 = X1-DESIREDLENGTH;
            T2 = X2+DESIREDLENGTH;
          }
          else {
            T1 = X1+DESIREDLENGTH;
            T2 = X2-DESIREDLENGTH;
          }
          *NumPoints = 6;
          Points[1].x = T1;
          Points[1].y = Y1;
          Points[2].x = T1;
          Points[2].y = T;
          Points[3].x = T2;
          Points[3].y = T;
          Points[4].x = T2;
          Points[4].y = Y2;
        }
        else {
          T = (X1+X2)/2;
          *NumPoints = 4;
          Points[1].x = T;
          Points[1].y = Y1;
          Points[2].x = T;
          Points[2].y = Y2;
        }
      }
    }
    else {
      OK1 = IsOK(D1,X1,Y1,X2,Y2);
      OK2 = IsOK(D2,X2,Y2,X1,Y1);
  
      if (OK1 && OK2) {
        *NumPoints = 3;
        if ((D1 == DIR_UP) || (D1 == DIR_DOWN)) {
          Points[1].x = X1;
          Points[1].y = Y2;
        }
        else {
          Points[1].x = X2;
          Points[1].y = Y1;
        }
      }
      else if (OK1) {
        *NumPoints = 5;
        if ((D1 == DIR_UP) || (D1 == DIR_DOWN)) {
          T = AddDesiredLength(D2,X2,Y2);
          T2 = (Y1+Y2)/2;
          Points[1].x = X1;
          Points[1].y = T2;
          Points[2].x = T;
          Points[2].y = T2;
          Points[3].x = T;
          Points[3].y = Y2;
        }
        else {
          T = AddDesiredLength(D2,X2,Y2);
          T2 = (X1+X2)/2;
          Points[1].x = T2;
          Points[1].y = Y1;
          Points[2].x = T2;
          Points[2].y = T;
          Points[3].x = X2;
          Points[3].y = T;
        }
      }
      else if (OK2) {
        *NumPoints = 5;
        if ((D1 == DIR_UP) || (D1 == DIR_DOWN)) {
          T = AddDesiredLength(D1,X1,Y1);
          T2 = (X1+X2)/2;
          Points[1].x = X1;
          Points[1].y = T;
          Points[2].x = T2;
          Points[2].y = T;
          Points[3].x = T2;
          Points[3].y = Y2;
        }
        else {
          T = AddDesiredLength(D1,X1,Y1);
          T2 = (Y1+Y2)/2;
          Points[1].x = T;
          Points[1].y = Y1;
          Points[2].x = T;
          Points[2].y = T2;
          Points[3].x = X2;
          Points[3].y = T2;
        }
      }
      else {
        T1 = AddDesiredLength(D1,X1,Y1);
        T2 = AddDesiredLength(D2,X2,Y2);
        *NumPoints = 5;
        if ((D1 == DIR_UP) || (D1 == DIR_DOWN)) {
          Points[1].x = X1;
          Points[1].y = T1;
          Points[2].x = T2;
          Points[2].y = T1;
          Points[3].x = T2;
          Points[3].y = Y2;
        }
        else {
          Points[1].x = T1;
          Points[1].y = Y1;
          Points[2].x = T1;
          Points[2].y = T2;
          Points[3].x = X2;
          Points[3].y = T2;
        }
      }
    }
  }
  
  if (*NumPoints >= 2) {
    Points[0].x = X1;
    Points[0].y = Y1;
    Points[*NumPoints-1].x = X2;
    Points[*NumPoints-1].y = Y2;
  }
}

TCADRightAngleLine::TCADRightAngleLine(TCADSheet *Sheet,int x=0,int y=0) : TCADLine(Sheet,x,y)
{
  int i;
  
  NumPoints = 2;
  Points[0] = First;
  Points[1] = Last;

  for(i=2;i<MAXNUMHANDLES;i++) Handle[i].Type = HT_LINE_OTHER;

  RecomputeEncapRect();
  RecomputeRefRect();
}

TCADRightAngleLine::~TCADRightAngleLine() {
}

void
TCADRightAngleLine::Save(TCADSaveStream *Fl)
{
  Fl->SaveObjectName("TCADRightAngleLine");
  SaveProperties(Fl);
  Fl->SaveEndObject();
}

void
TCADRightAngleLine::SaveProperties(TCADSaveStream *Fl)
{
  int i;

  TCADLine::SaveProperties(Fl);

  Fl->SaveInt("O_X1",O_X1);
  Fl->SaveInt("O_Y1",O_Y1);
  Fl->SaveInt("O_DX",O_DX);
  Fl->SaveInt("O_DY",O_DY);
  Fl->SaveInt("O_D1",O_D1);
  Fl->SaveInt("O_D2",O_D2);

  Fl->SaveInt("NumPoints",NumPoints);
  for(i=1;i<(NumPoints-1);i++) {
    Fl->SaveInt("X",Points[i].x);
    Fl->SaveInt("Y",Points[i].y);
  }
}

char
TCADRightAngleLine::LoadProperty(TCADLoadStream *Fl,char *ID)
{
  if (strcasecmp(ID,"O_X1") == 0) O_X1 = Fl->LoadInt();
  else if (strcasecmp(ID,"O_Y1") == 0) O_Y1 = Fl->LoadInt();
  else if (strcasecmp(ID,"O_DX") == 0) O_DX = Fl->LoadInt();
  else if (strcasecmp(ID,"O_DY") == 0) O_DY = Fl->LoadInt();
  else if (strcasecmp(ID,"O_D1") == 0) O_D1 = Fl->LoadInt();
  else if (strcasecmp(ID,"O_D2") == 0) O_D2 = Fl->LoadInt();
  else if (strcasecmp(ID,"NumPoints") == 0) NumPoints = Fl->LoadInt();
  else if (strcasecmp(ID,"X") == 0) {
    if (XLPN == MAXNUMPOINTS) XLPN--;
    Points[XLPN++].x = Fl->LoadInt();
  }
  else if (strcasecmp(ID,"Y") == 0) {
    if (YLPN == MAXNUMPOINTS) YLPN--;
    Points[YLPN++].y = Fl->LoadInt();
  }
  else return(TCADLine::LoadProperty(Fl,ID));

  return(TRUE);
}

void
TCADRightAngleLine::Load(TCADLoadStream *Fl)
{
  // Initialize
  XLPN = 1;
  YLPN = 1;

  TCADLine::Load(Fl);
}

void
TCADRightAngleLine::PostLoad()
{
  if (NumPoints > MAXNUMPOINTS) NumPoints = MAXNUMPOINTS;

  // Set first and last point
  Points[0] = First;
  Points[NumPoints-1] = Last;
  NumHandles = MAX(2,NumPoints-1);

  TCADLine::PostLoad();

  UpdateHandlePos();

  // Save line state
  SaveLineState(O_X1,O_Y1,O_DX,O_DY,O_D1,O_D2);
}

void
TCADRightAngleLine::Draw(GdkRegion *Region)
{
  int i;
  int X,Y,X2,Y2;
  GdkDrawable *Drawable;

//  printf("TCADRightAngleLine(%lu)::Draw()\n",Uniq);

  if (NumPoints < 2) return;  // Should not happen, but ...

  if (Region)
   if (gdk_region_rect_in(Region, &RefRect) == GDK_OVERLAP_RECTANGLE_OUT) return;

  Drawable = Sheet->GetDrawable();

  // Draw line
  X = First.x;
  Y = First.y;
  for(i=1;i<NumPoints;i++) {
    X2 = Points[i].x;
    Y2 = Points[i].y;
    gdk_draw_line(Drawable,Sheet->GC,X,Y,X2,Y2);
    X = X2;
    Y = Y2;
  }

  // Draw arrows
  DrawArrow(Drawable,Sheet->GC,&First,&Points[1],FirstArrow,FirstArrowSize);
  DrawArrow(Drawable,Sheet->GC,&Last,&Points[NumPoints-2],LastArrow,LastArrowSize);
}

TCADObject *
TCADRightAngleLine::ButtonPressed(int x,int y)
{
  int i;
  GdkPoint *Ptr;

  if (TCADLine::ButtonPressed(x,y) == NULL) return(NULL);

  // This is more general than needed (i.e., slower), but simple.
  Ptr = Points;
  for(i=0;i<(NumPoints-1);i++) {
    if (PointNearLine(*Ptr,*(Ptr+1),x,y)) return(this);
    Ptr++;
  }

  return(NULL);
}

void
TCADRightAngleLine::Export(FILE *Fl,TExportFormat Format)
{
  int i;

  switch(Format) {
    case EF_LATEX:
      int x1,y1,x2,y2,dx,dy;

      x1 = First.x;
      y1 = First.y;
      for(i=1;i<(NumPoints-1);i++) {
        x2 = Points[i].x;
        y2 = Points[i].y;
        dx = x2-x1;
        dy = y2-y1;
        fprintf(Fl,"\\put(%d,%d){\\line(%d,%d){%f}}\n",x1,y1,dx,-dy,sqrt(dx*dx+dy*dy));
        x1 = x2;
        y1 = y2;
      }
      break;
    case EF_FIG:
      ExportFigLine(Fl,NumPoints);
      fputc('\t',Fl);
      for(i=0;i<NumPoints;i++) fprintf(Fl," %d %d",Points[i].x,Points[i].y);
      fputc('\n',Fl);
      break;
  }
}

void
TCADRightAngleLine::RecomputeEncapRect()
{
  int i;
  int x1,y1,x2,y2;
  
  x1 = x2 = First.x;
  y1 = y2 = First.y;
  for(i=1;i<NumPoints;i++) {
    x1 = MIN(x1,Points[i].x);
    x2 = MAX(x2,Points[i].x);
    y1 = MIN(y1,Points[i].y);
    y2 = MAX(y2,Points[i].y);
  }

//  printf("x1=%d, x2=%d, y1=%d, y2=%d\n",x1,x2,y1,y2);

  EncapRect.x = x1; 
  EncapRect.y = y1;
  EncapRect.width = x2-x1+1;
  EncapRect.height = y2-y1+1;
}

gboolean
TCADRightAngleLine::CanAttach(int& x,int& y,TAttach *Attach)
{
  return(FALSE);
}

int
TCADRightAngleLine::NumDirs(int Dir)
{
  int i,T;

  T = 0;
  for(i=0;i<4;i++) {
    if (Dir & 1) T++;
    Dir = (Dir>>1);
  }

  return(T);
}

void
TCADRightAngleLine::SaveLineState(int X1,int Y1,int DX,int DY,int D1,int D2)
{
  int i;

  // Save state
  S_NumPoints = NumPoints;
  S_X1 = X1;
  S_Y1 = Y1;
  S_DX = DX;
  S_DY = DY;
  S_D1 = D1;
  S_D2 = D2;
  for(i=0;i<NumPoints;i++) {
    S_Points[i].x = Points[i].x-X1;
    S_Points[i].y = Points[i].y-Y1;
  }
}

void
TCADRightAngleLine::UpdateLine(gboolean SaveState)
{
  int D1,D2;
  TAttach *FA,*LA;
  int Start,End;
  int X1,Y1,X2,Y2,DX,DY;
  gboolean Restore;
  int i;

  if (DraggingHandle == 0) FA = &TmpAttach;
  else FA = &FAttach;
  if (DraggingHandle == 1) LA = &TmpAttach;
  else LA = &LAttach;

//  printf("TCADRightAngleLine::UpdateLine()\n");

  // Determine directions
  if (First.y == Last.y) {
    if (Last.x > First.x) D1 = DIR_RIGHT;
    else D1 = DIR_LEFT;
    D2 = Opposite(D1);
  }
  else if (First.x == Last.x) {
    if (Last.y > First.y) D1 = DIR_DOWN;
    else D1 = DIR_UP;
    D2 = Opposite(D1);
  }
  else {
    if (First.y > Last.y) D1 = DIR_UP;
    else D1 = DIR_DOWN;
    if (First.x < Last.x) D2 = DIR_LEFT;
    else D2 = DIR_RIGHT;
  }
 
  // Check if these directions are possible
  if (FA->Object != NULL) {
    if (FA->DirectionForced) D1 = FA->Direction;  // Forced
    else if ((D1 & FA->Direction) == 0) {
      // Not possible
      D1 = FA->Direction;
    }
  }
  
  if (LA->Object != NULL) {
    if (LA->DirectionForced) D2 = LA->Direction;  // Forced
    else if ((D2 & LA->Direction) == 0) {
      // Not possible
      D2 = LA->Direction;
    }
  }
 
  /* Make D? point only in one direction */
  Start = 0;
  End = 0;
  X1 = First.x;
  X2 = Last.x;
  Y1 = First.y;
  Y2 = Last.y;
  
  if (NumDirs(D1) > 1) {
    Start++;
    AddDesiredLength2(D1,X1,Y1);
    DX = X2-X1;
    DY = Y2-Y1;
    if ((DX < 0) && (D1 & DIR_LEFT)) D1 = DIR_LEFT;
    else if ((DX > 0) && (D1 & DIR_RIGHT)) D1 = DIR_RIGHT;
    else if ((DY < 0) && (D1 & DIR_UP)) D1 = DIR_UP;
    else if ((DY > 0) && (D1 & DIR_DOWN)) D1 = DIR_DOWN;
    else D1 = Opposite(Opposite(D1));  // Choose one direction
  }
  
//  printf("D1=%d, D2=%d, Start=%d, End=%d\n",D1,D2,Start,End);
  if (NumDirs(D2) > 1) {
    End++;
    AddDesiredLength2(D2,X2,Y2);
    DX = X1-X2;
    DY = Y1-Y2;
    if ((DX < 0) && (D2 & DIR_LEFT)) D2 = DIR_LEFT;
    else if ((DX > 0) && (D2 & DIR_RIGHT)) D2 = DIR_RIGHT;
    else if ((DY < 0) && (D2 & DIR_UP)) D2 = DIR_UP;
    else if ((DY > 0) && (D2 & DIR_DOWN)) D2 = DIR_DOWN;
    else D2 = Opposite(Opposite(D2)); // Choose one direction
  }
  
  Restore = FALSE;
  DX = X2-X1;
  DY = Y2-Y1;
//  printf("S_NumPoints = %d\n",S_NumPoints);
  if (S_NumPoints > 0) {
    if ((SGN(S_DX) == SGN(DX)) && (SGN(S_DY) == SGN(DY)) && (S_D1 == D1) && (S_D2 == D2))
     Restore = TRUE;
  }

  if (Restore) {
    NumPoints = S_NumPoints;
    for(i=0;i<NumPoints;i++) {
      Points[i].x = X1 + DX*S_Points[i].x/S_DX;
      Points[i].y = Y1 + DY*S_Points[i].y/S_DY;
    }
  }
  else {
    // Connect
    ConnectPoints(X1,Y1,D1,X2,Y2,D2,&NumPoints,&Points[Start]);
    if (Start) NumPoints++;
    if (End) NumPoints++;
  }
  Points[0] = First;
  Points[NumPoints-1] = Last;
  NumHandles = MAX(2,NumPoints-1);
  KeepOnSheet();

  O_X1 = X1;
  O_Y1 = Y1;
  O_DX = DX;
  O_DY = DY;
  O_D1 = D1;
  O_D2 = D2;
  if ((!Restore) && (SaveState)) SaveLineState(X1,Y1,DX,DY,D1,D2);
}

void
TCADRightAngleLine::KeepOnSheet()
{
  int W,H;
  int i;

  W = Sheet->EncapRect.width;
  H = Sheet->EncapRect.height;

  for(i=0;i<NumPoints;i++) {
    if (Points[i].x < 0) Points[i].x  = 0;
    if (Points[i].x >= W) Points[i].x = W-1;
    if (Points[i].y < 0) Points[i].y  = 0;
    if (Points[i].y >= H) Points[i].y = H-1;
  }

  First = Points[0];
  Last = Points[NumPoints-1];

  UpdateHandlePos();
}

void
TCADRightAngleLine::UpdateHandlePos()
{
  int i;
  int d = HANDLE_SIZE/2;
  
//  printf("TCADRightAngleLine::UpdateHandlePos()\n");
  
  TCADLine::UpdateHandlePos();

  for(i=2;i<NumHandles;i++) {
    Handle[i].x = (Points[i-1].x+Points[i].x)/2-d;
    Handle[i].y = (Points[i-1].y+Points[i].y)/2-d;
  }
}

char
TCADRightAngleLine::HandleDragged(int x,int y,GdkRectangle *RefreshRect)
{
  int DX,DY,T,T2;

  if (!DHandle) return(0);

  *RefreshRect = RefRect;

  if (DraggingHandle < 2) TCADLine::HandleDragged(x,y,RefreshRect);
  else {
    DX = x-DXO;
    DY = y-DYO;
    if (Points[DraggingHandle-1].x == Points[DraggingHandle].x) DY = 0;
    else DX = 0;

    if ((DraggingHandle == 2) || (DraggingHandle == (NumHandles-1))) {
      // Dragging first-line handle
      if ((DY == 0) && (DX != 0)) {
        if (DraggingHandle == 2) {
	  T = Points[1].x-Points[0].x;
  	  T2 = T+DX; 
	  if ((abs(T2) < MINLENGTH) || (SGN(T2) != SGN(T))) DX = 0;
	}

        if (DraggingHandle == (NumHandles-1)) {
	  T = Points[NumHandles-1].x-Points[NumHandles].x;
	  T2 = T+DX; 
	  if ((abs(T2) < MINLENGTH) || (SGN(T2) != SGN(T))) DX = 0;
	}
      }
      else if ((DX == 0) && (DY != 0)) {
        if (DraggingHandle == 2) {
	  T = Points[1].y-Points[0].y;
  	  T2 = T+DY; 
	  if ((abs(T2) < MINLENGTH) || (SGN(T2) != SGN(T))) DY = 0;
	}

        if (DraggingHandle == (NumHandles-1)) {
	  T = Points[NumHandles-1].y-Points[NumHandles].y;
	  T2 = T+DY; 
	  if ((abs(T2) < MINLENGTH) || (SGN(T2) != SGN(T))) DY = 0;
	}
      }      
    }
    
    Points[DraggingHandle-1].x += DX;
    Points[DraggingHandle-1].y += DY;
    Points[DraggingHandle].x += DX;
    Points[DraggingHandle].y += DY;
    DXO = DXO+DX;
    DYO = DYO+DY;
    UpdateHandlePos();

    RecomputeEncapRect();
    RecomputeRefRect();
    
    System->UnionRectangles(RefreshRect,&RefRect);

    // User forced configuration
    if (S_NumPoints > 0) SaveLineState(S_X1,S_Y1,S_DX,S_DY,S_D1,S_D2);
    else SaveLineState(O_X1,O_Y1,O_DX,O_DY,O_D1,O_D2);
  }
}

void
TCADRightAngleLine::HandleButtonPressed(int HandleNo,int x,int y)
{
  TCADLine::HandleButtonPressed(HandleNo,x,y);

  if (HandleNo >= 2) {
    // Store X and Y
    DXO = x;
    DYO = y;
  }
}


