/************************\
*                        *
*  Implements TCADSheet  *
*                        *
\************************/

#include <string.h>
#include <gtk/gtk.h>
#include <stdlib.h>
#include <errno.h>
#include "sheet.h"
#include "texteditbox.h"
#include "messagebox.h"
#include "main.h"
#include "rightangleline.h"
#include "simpleline.h"
#include "text.h"
#include "frame.h"
#include "framedtext.h"
#include "gridoptions.h"

/*
 * Initial size of canvas
 */
#define INITIAL_WIDTH 1000
#define INITIAL_HEIGHT 1000

// Alignment macro
#define ALIGN(x,gran,Force) \
if ((UseGrid) || (Force)) { \
  int _T = x % gran; \
  if (_T < (gran/2)) x -= _T; /* Round down */ \
  else x += (gran-_T); /* Round up  */\
}


/*
 * Event functions
 */

static gboolean
window_delete_event (GtkWidget       *widget,
                     GdkEvent        *event,
                     gpointer         user_data)
{
  TCADSheet *Sheet;
  widget = widget; event = event;

  if (gtk_grab_get_current() != NULL) return(TRUE);  // Modal dialog is showing

  Sheet = (TCADSheet *)user_data;
  if (Sheet->Close() == FALSE) return(TRUE);
  else {
    System->SheetClosed(Sheet);
    return(FALSE);
  }
}

static gboolean
visibility_notify_event (GtkWidget          *widget,
                         GdkEventVisibility *event,
                         gpointer            user_data)
{

  TCADSheet *Sheet;
  widget = widget;
  event = event;

//  printf("visibility_notify_event()\n");
  Sheet = (TCADSheet *)user_data;
  Sheet->SetVisibilityState(event->state);
  return TRUE;
}

static gboolean
focus_in_event (GtkWidget     *widget,
                GdkEventFocus *event,
                gpointer       user_data)
{
  TCADSheet *Sheet;
  widget = widget;
  event = event;

//  printf("focus_in_event()\n");
  Sheet = (TCADSheet *)user_data;
  Sheet->GotFocus();
  return TRUE;
}

static gboolean
fixed_expose_event (GtkWidget       *widget,
                    GdkEventExpose  *event,
                    gpointer         user_data)
{
  TCADSheet *Sheet;
  widget = widget;

//  printf("fixed_expose_event()\n");
  Sheet = (TCADSheet *)user_data;
  Sheet->DrawRect(&event->area);
  return TRUE;
}

static void
fixed_draw  (GtkWidget       *widget,
             GdkRectangle    *area,
             gpointer         user_data)
{
  TCADSheet *Sheet;
  widget = widget;

//  printf("fixed_draw()\n");
  Sheet = (TCADSheet *)user_data;
//  Sheet = (TCADSheet *)gtk_object_get_data((GtkObject *)widget,"Sheet");
  Sheet->DrawRect(area);
}

static gboolean
fixed_button_press_event (GtkWidget      *widget,
                          GdkEventButton *event,
                          gpointer       user_data)
{
  TCADSheet *Sheet;
  widget = widget;

  if (event->button != 1) return FALSE;

  Sheet = (TCADSheet *)user_data;
//  printf("event->state = %x\n",event->state);
//  Sheet = (TCADSheet *)gtk_object_get_data((GtkObject *)widget,"Sheet");
  Sheet->ButtonPressed((int)event->x,(int)event->y,(event->state)&GDK_SHIFT_MASK);

  return TRUE;
}

static gboolean
fixed_button_release_event (GtkWidget      *widget,
                            GdkEventButton *event,
                            gpointer       user_data)
{
  TCADSheet *Sheet;
  widget = widget;

  if (event->button != 1) return FALSE;

  Sheet = (TCADSheet *)user_data;
//  Sheet = (TCADSheet *)gtk_object_get_data((GtkObject *)widget,"Sheet");
  Sheet->ButtonReleased();

  return TRUE;
}

static gboolean
fixed_motion_notify_event (GtkWidget      *widget,
                           GdkEventMotion *event,
                           gpointer       user_data)
{
  TCADSheet *Sheet;
  widget = widget;

  Sheet = (TCADSheet *)user_data;
//  Sheet = (TCADSheet *)gtk_object_get_data((GtkObject *)widget,"Sheet");
  Sheet->MouseMoved((int)event->x,(int)event->y);

  return FALSE;
}

/******************************************************************************/

/*
 * Class implementation
 */

#define STD_MASK GDK_EXPOSURE_MASK+GDK_BUTTON_PRESS_MASK+GDK_BUTTON_RELEASE_MASK+GDK_VISIBILITY_NOTIFY_MASK

TCADSheet::TCADSheet(char Visible) : TCADGroup(this)
{
  GtkTable *Table;

  Uniq = System->NextUniq();

  Sheet = this;
  FirstSelected = NULL;
  VisibilityState = GDK_VISIBILITY_UNOBSCURED;
  GC = NULL;  // No GC for now
  FileName = NULL;
  LastUniq = 0;

  ButtonDown = 0;
  Dragging = 0;
  NewRect = 0;
  ResetOnUp = 0;

  VRulerSettings.Visible = TRUE;
  VRulerSettings.Metric = GTK_CENTIMETERS;
  HRulerSettings.Visible = TRUE;
  HRulerSettings.Metric = GTK_CENTIMETERS;

  // Init grid
  UseGrid = TRUE;
  ShowGrid = FALSE;
  GX_eq_GY = TRUE;
  GridX = 8;
  GridY = 8;

  /* Create window */
  Window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_signal_connect ((GtkObject *)(Window), "delete_event",
                      GTK_SIGNAL_FUNC (window_delete_event),
                      this);
  gtk_signal_connect ((GtkObject *)(Window), "visibility_notify_event",
                      GTK_SIGNAL_FUNC (visibility_notify_event),
                      this);
  gtk_signal_connect ((GtkObject *)(Window), "focus_in_event",
                      GTK_SIGNAL_FUNC (focus_in_event),
                      this);
  gtk_object_set_data((GtkObject *)(Window), "Sheet", this);
  SetTitle();
  gtk_window_position (GTK_WINDOW (Window), GTK_WIN_POS_NONE);
  gtk_window_set_policy (GTK_WINDOW (Window), TRUE, TRUE, TRUE);
  gtk_widget_set_usize (Window, 640, 480);

  // Create scrolled window
  ScrolledWindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_object_set_data ((GtkObject *) (Window), "ScrolledWindow", ScrolledWindow);
  gtk_object_set_data((GtkObject *)(ScrolledWindow), "Sheet", this);
  gtk_widget_show (ScrolledWindow);
  gtk_container_add (GTK_CONTAINER (Window), ScrolledWindow);

  // Create table
  Table = GTK_TABLE(gtk_table_new(2,2,FALSE));
  gtk_table_set_col_spacing(Table,0,1);
  gtk_table_set_row_spacing(Table,0,1);
  gtk_container_border_width(GTK_CONTAINER(Table),2);
  gtk_container_add(GTK_CONTAINER(ScrolledWindow),GTK_WIDGET(Table));

  /* Create table and rulers with frame */
  Frame = GTK_FRAME(gtk_frame_new(NULL));
  gtk_frame_set_shadow_type(Frame,GTK_SHADOW_OUT);
  VRuler = GTK_VRULER(gtk_vruler_new());
  gtk_ruler_set_range (GTK_RULER(VRuler), 0, INITIAL_WIDTH, 0, INITIAL_WIDTH);
  HRuler = GTK_HRULER(gtk_hruler_new());
  gtk_ruler_set_range (GTK_RULER(HRuler), 0, INITIAL_HEIGHT, 0, INITIAL_HEIGHT);

  // Create fixed
  Fixed = gtk_fixed_new();
  gtk_object_set_data((GtkObject *)(Window), "Fixed", Fixed);
  gtk_object_set_data((GtkObject *)(Fixed), "Sheet", this);
  gtk_widget_set_usize (Fixed, INITIAL_WIDTH, INITIAL_HEIGHT);
  gtk_signal_connect ((GtkObject *) (Fixed), "draw",
                      GTK_SIGNAL_FUNC (fixed_draw),
                      this);
  gtk_signal_connect ((GtkObject *) (Fixed), "expose_event",
                      GTK_SIGNAL_FUNC (fixed_expose_event),
                      this);
  gtk_signal_connect ((GtkObject *) (Fixed), "button_press_event",
                      GTK_SIGNAL_FUNC (fixed_button_press_event),
                      this);
  gtk_signal_connect ((GtkObject *) (Fixed), "button_release_event",
                      GTK_SIGNAL_FUNC (fixed_button_release_event),
                      this);
  gtk_signal_connect ((GtkObject *) (Fixed), "motion_notify_event",
                      GTK_SIGNAL_FUNC (fixed_motion_notify_event),
                      this);
  gtk_widget_set_events(Fixed,STD_MASK+GDK_POINTER_MOTION_MASK);

  // Insert widgets to table
  gtk_table_attach(Table,GTK_WIDGET(Frame), 0, 1, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(Table,GTK_WIDGET(HRuler), 1, 2, 0, 1, ((GtkAttachOptions)(GTK_EXPAND + GTK_SHRINK + GTK_FILL)), GTK_FILL, 0, 0);
  gtk_table_attach (Table, GTK_WIDGET(VRuler), 0, 1, 1, 2,
   GTK_FILL, ((GtkAttachOptions)(GTK_EXPAND + GTK_SHRINK + GTK_FILL)), 0, 0);
  gtk_table_attach (Table, GTK_WIDGET(Fixed), 1, 2, 1, 2,
   ((GtkAttachOptions)(GTK_EXPAND + GTK_SHRINK + GTK_FILL)), ((GtkAttachOptions)(GTK_EXPAND + GTK_SHRINK + GTK_FILL)), 0, 0);
  gtk_widget_show(Fixed);
  gtk_widget_show(GTK_WIDGET(Table));
  ApplyRulerSettings();

  // Set EncapRect
  EncapRect.x = 0;
  EncapRect.y = 0;
  EncapRect.width = INITIAL_WIDTH;
  EncapRect.height = INITIAL_HEIGHT;

//  printf("Fixed->window=%x\n",(int)Fixed->window);

  if (Visible) Show();
}

TCADSheet::~TCADSheet()
{
  // Destroy window
//  printf("Closing!\n");
  Hide();
  gtk_widget_destroy(Fixed);
  gtk_widget_destroy(ScrolledWindow);
  gtk_widget_destroy(Window);

  if (GC != NULL) gdk_gc_destroy(GC);
  if (FileName != NULL) free(FileName);
}

void
TCADSheet::Save(TCADSaveStream *Fl)
{
  Fl->SaveObjectName("TCADSheet");
  SaveProperties(Fl);
  Fl->SaveEndObject();
}

void
TCADSheet::SaveProperties(TCADSaveStream *Fl)
{
  TCADGroup::SaveProperties(Fl);

  Fl->SaveInt("WndX",Window->allocation.x);
  Fl->SaveInt("WndY",Window->allocation.y);
  Fl->SaveInt("WndW",Window->allocation.width);
  Fl->SaveInt("WndH",Window->allocation.height);

  Fl->SaveInt("LastUniq",LastUniq);

  Fl->SaveInt("UseGrid",UseGrid);
  Fl->SaveInt("ShowGrid",ShowGrid);
  Fl->SaveInt("GX_eq_GY",GX_eq_GY);
  Fl->SaveInt("GridX",GridX);
  Fl->SaveInt("GridY",GridY);

  Fl->SaveInt("HRulerVisible",HRulerSettings.Visible);
  Fl->SaveInt("VRulerVisible",VRulerSettings.Visible);
  Fl->SaveInt("HRulerMetric",HRulerSettings.Metric);
  Fl->SaveInt("VRulerMetric",VRulerSettings.Metric);
}

void
TCADSheet::PostLoad()
{
  TCADGroup::PostLoad();
  gtk_widget_set_usize(Fixed, EncapRect.width, EncapRect.height);
  gtk_widget_set_uposition(Window, WS.x, WS.y);
  gtk_widget_set_usize(Window, WS.width, WS.height);
  ApplyRulerSettings();
}

char
TCADSheet::LoadProperty(TCADLoadStream *Fl,char *ID)
{
  char Result=1;

  if (strcasecmp("WndX",ID) == 0) WS.x = Fl->LoadInt();
  else if (strcasecmp("WndY",ID) == 0) WS.y = Fl->LoadInt();
  else if (strcasecmp("WndW",ID) == 0) WS.width = Fl->LoadInt();
  else if (strcasecmp("WndH",ID) == 0) WS.height = Fl->LoadInt();
  else if (strcasecmp("LastUniq",ID) == 0) LastUniq = Fl->LoadInt();
  else if (strcasecmp("UseGrid",ID) == 0) UseGrid = Fl->LoadInt();
  else if (strcasecmp("ShowGrid",ID) == 0) ShowGrid = Fl->LoadInt();
  else if (strcasecmp("GX_eq_GY",ID) == 0) GX_eq_GY = Fl->LoadInt();
  else if (strcasecmp("GridX",ID) == 0) GridX = Fl->LoadInt();
  else if (strcasecmp("GridY",ID) == 0) GridY = Fl->LoadInt();
  else if (strcasecmp("HRulerVisible",ID) == 0) HRulerSettings.Visible = Fl->LoadInt();
  else if (strcasecmp("VRulerVisible",ID) == 0) VRulerSettings.Visible = Fl->LoadInt();
  else if (strcasecmp("HRulerMetric",ID) == 0) HRulerSettings.Metric = (GtkMetricType)Fl->LoadInt();
  else if (strcasecmp("VRulerMetric",ID) == 0) VRulerSettings.Metric = (GtkMetricType)Fl->LoadInt();
  else Result = TCADGroup::LoadProperty(Fl,ID);
  
  return(Result);
}

gboolean
TCADSheet::Close()
{
  return(TRUE);
}

void
TCADSheet::Show()
{
  GdkColormap *ColorMap;

  gtk_widget_show(Window);

  if (GC == NULL) {
    GC = gdk_gc_new(Fixed->window);

    // Get colors
    ColorMap = gdk_window_get_colormap(Fixed->window);
    gdk_color_white(ColorMap,&WhiteColor);
    gdk_color_black(ColorMap,&BlackColor);
  }
}

void
TCADSheet::Hide()
{
  gtk_widget_hide(Window);
}

void
TCADSheet::DrawRect(GdkRectangle *Rect)
{
  GdkRegion *Region;
  GdkPoint Points[5];
  TCADObject *Child;
  TCADHandle *Handle;
  int i,x,y,x2,y2;

/*
  if (Rect != NULL) printf("TCADSheet::DrawRect(%d,%d,%d,%d)\n",Rect->x,Rect->y,Rect->width,Rect->height);
  else printf("DrawRect:NULL\n");
*/

  /* Create region */
  if (Rect == NULL) {
    Region = NULL;
    Rect = &EncapRect;

    // Set clipping
    gdk_gc_set_clip_rectangle(GC,&EncapRect);

    // Clear area
    gdk_gc_set_background(GC,&WhiteColor);
    gdk_gc_set_foreground(GC,&WhiteColor);
    gdk_draw_rectangle(Fixed->window,GC,1,0,0,EncapRect.width,EncapRect.height);
    gdk_gc_set_foreground(GC,&BlackColor);
  }
  else {
    // Limit rectangle
    x = Rect->x;
    y = Rect->y;
    x2 = Rect->x+Rect->width;
    y2 = Rect->y+Rect->height;

    if (x < 0) x = 0;
    if (y < 0) y = 0;
    if (x2 < 0) x2 = 0;
    if (y2 < 0) y2 = 0;

    if (x > EncapRect.width) x = EncapRect.width;
    if (y > EncapRect.height) y = EncapRect.height;
    if (x2 > EncapRect.width) x2 = EncapRect.width;
    if (y2 > EncapRect.height) y2 = EncapRect.height;
    
    Rect->x = x;
    Rect->y = y;
    Rect->width = x2-x;
    Rect->height = y2-y;
    
    // Fill points
    Points[0].x = Rect->x;
    Points[0].y = Rect->y;
    Points[1].x = Rect->x+Rect->width;
    Points[1].y = Rect->y;
    Points[2].x = Rect->x+Rect->width;
    Points[2].y = Rect->y+Rect->height;
    Points[3].x = Rect->x;
    Points[3].y = Rect->y+Rect->height;
    Points[4].x = Rect->x;
    Points[4].y = Rect->y;

    Region = gdk_region_polygon(Points,5,GDK_EVEN_ODD_RULE);

    // Set clipping
    gdk_gc_set_clip_rectangle(GC,Rect);
    
    // Clear area
    gdk_gc_set_background(GC,&WhiteColor);
    gdk_gc_set_foreground(GC,&WhiteColor);
    gdk_draw_rectangle(Fixed->window,GC,1,Rect->x,Rect->y,Rect->width,Rect->height);
    gdk_gc_set_foreground(GC,&BlackColor);
  }


  // Draw grid
  if (ShowGrid) {
    x = Rect->x;
    i = x % GridX;
    if (i > 0) x += (GridX - i);

    y = Rect->y;
    i = y % GridY;
    if (i > 0) y += (GridY - i);

    i = y;
    x2 = Rect->x+Rect->width;
    y2 = Rect->y+Rect->height;
    while(x<=x2) {
      y = i;
      while(y<=y2) {
        gdk_draw_point(Fixed->window,GC,x,y);
        y += GridY;
      }
      x += GridX;
    }
  }

  Draw(Region);

  // Draw handles (if any)
  Child = FirstSelected;
  while(Child) {
    Handle = Child->Handles;
    if (Handle) {
      for(i=Child->NumHandles;i>0;i--) {
        Handle->Draw(Fixed,GC,(Child==FirstSelected));
        Handle++;
      }
    }
     
    Child = Child->NextSelected;
  }

  // Draw focus rectangle (if necessary)
  if (Dragging) DrawFocusRect();

  if (Region != NULL) gdk_region_destroy(Region);
}

GdkDrawable *
TCADSheet::GetDrawable()
{
  if (Fixed == NULL) return(NULL);
  else return(Fixed->window);
}

GdkWindow *
TCADSheet::GetWindow()
{
  if (Fixed == NULL) return(NULL);
  else return(Fixed->window);
}

unsigned long
TCADSheet::NextUniq()
{
  return(++LastUniq);
}

void
TCADSheet::SelectObject(TCADObject *Object,GdkRectangle *RefreshRect,GdkRectangle *RefreshRect2)
{
  TCADObject *PrevFirst;

  RefreshRect->x = 0;
  RefreshRect->y = 0;
  RefreshRect->width = 0;
  RefreshRect->height = 0;
  RefreshRect2->x = 0;
  RefreshRect2->y = 0;
  RefreshRect2->width = 0;
  RefreshRect2->height = 0;
  
  if (Object->Selected) {
    return;
  }
  // Enqueue currently selected object
  PrevFirst = FirstSelected;
  Object->PrevSelected = NULL;
  Object->NextSelected = FirstSelected;
  if (FirstSelected) FirstSelected->PrevSelected = Object;

  FirstSelected = Object;
  Object->Select(RefreshRect);

  // Do OtherSelected()
  if (PrevFirst != NULL) PrevFirst->OtherSelected(RefreshRect2);
}


void
TCADSheet::ButtonPressed(int x,int y,gboolean Shift)
{
  TCADObject *Selected,*Child;
  GdkRectangle RefreshRect,RefreshRect2;
  char WasFirst;
  char UnsAll;
  TCADHandle *Handle;
  int i;
  TCADLine *Line;
  int MinX,MinY;

  // Make sheet active
  System->SheetActivated(this);

  ButtonDown = 1;
  Dragging = 0;
  DHType = HT_NORMAL;
  ResetOnUp = 0;
  NewRect = 0;
  SelectRect = 0;
  DHandle = 0;
  DownX = x;
  DownY = y;

  if (System->GetState() != ST_PICK) {
    /* We will create object */

    // Align x,y,DownX,DownY
    ALIGN(DownX,GridX,FALSE);
    ALIGN(DownY,GridY,FALSE);
    XCorr = x-DownX;
    YCorr = y-DownY;
    ALIGN(x,GridX,FALSE);
    ALIGN(y,GridY,FALSE);
    OldX = x;
    OldY = y;
    
    // Eventually create objects
    switch(System->GetState()) {
      case ST_LINE:
      case ST_ANGLELINE:
        UnselectAll();  // Unselect objects
      
        if (System->GetState() == ST_LINE) Line = new TCADSimpleLine(this,x,y);
	else Line = new TCADRightAngleLine(this,x,y);
        Line->Uniq = NextUniq();

	// Make line selected
	Line->PrevSelected = NULL;
        Line->NextSelected = NULL;
	FirstSelected = Line;
	Line->Selected = TRUE;

	AppendChild(Line);
	DHandle = 1;
	DHType = HT_LINEEND;
	Line->HandleButtonPressed(0,x,y);
	MouseMoved(x,y);  // Eventually attach
	Line->HandleButtonReleased(&RefreshRect);
	Line->HandleButtonPressed(1,x,y);
	MouseMoved(x,y);  // Eventually attach

	// Redraw
	RefreshRect.x = x-HANDLE_SIZE/2;
	RefreshRect.y = y-HANDLE_SIZE/2;
	RefreshRect.width = HANDLE_SIZE+1;
	RefreshRect.height = HANDLE_SIZE+1;
        DrawRect(&RefreshRect);
	ResetOnUp = 1;
	return;
      case ST_FRAME:
      case ST_TEXT:
      case ST_FRAMED_TEXT:
        UnselectAll();  // Unselect objects
        NewRect = TRUE;
	ResetOnUp = TRUE;
	XCorr = 0;
	YCorr = 0;
    
        UpdateFocusRect(x,y);
        return;
      default: ;
    }
  }

  // Check handles
  Handle = NULL;
  i = 0;
  if (FirstSelected) {
    Handle = FirstSelected->Handles;
    for(i=0;i<FirstSelected->NumHandles;i++) {
      if ((x >= Handle->x) && (x < (Handle->x+HANDLE_SIZE)) &&
          (y >= Handle->y) && (y < (Handle->y+HANDLE_SIZE))) break;
      Handle++;
    }
    if (i == FirstSelected->NumHandles) Handle = NULL;
  }

  if (Handle != NULL) {
    // Compute corrections
    XCorr = (Handle->x)+HANDLE_SIZE/2-x;
    YCorr = (Handle->y)+HANDLE_SIZE/2-y;
    
    // Handle clicked
    FirstSelected->HandleButtonPressed(i,x,y);
    DHandle = 1;
    DHType = Handle->Type;
    return;
  }

  Selected = TCADGroup::ButtonPressed(x,y);

  if (Selected == NULL) {
    if (System->GetState() == ST_PICK) {
      if (!Shift) UnselectAll();
    
      XCorr = 0;
      YCorr = 0;
      OldX = 0;
      OldY = 0;
      SelectRect = TRUE;
      UpdateFocusRect(x,y);
      return;
    }

  }

//  printf("Selected=%x, Shift=%d\n",(int)Selected,Shift);
  UnsAll = !Shift;
  if ((UnsAll) && (Selected != NULL))
   if (Selected->Selected) UnsAll = 0;

  if (UnsAll) UnselectAll();

  if (Selected) {
    if ((Selected->Selected) && (Shift)) {
      // Remove object from list
      WasFirst = (Selected == FirstSelected);
      if (Selected->PrevSelected == NULL) FirstSelected = Selected->NextSelected;
      else Selected->PrevSelected->NextSelected = Selected->NextSelected;
      if (Selected->NextSelected != NULL)
       Selected->NextSelected->PrevSelected = Selected->PrevSelected;

      // Unselect current object
      Selected->Unselect(&RefreshRect);

      // Redraw
      DrawRect(&RefreshRect);

      // Re-select first item
      if ((FirstSelected) && (WasFirst)) {
        FirstSelected->Select(&RefreshRect);

        // Redraw
        DrawRect(&RefreshRect);
      }
    }
    else if (Selected != FirstSelected) {
      SelectObject(Selected,&RefreshRect,&RefreshRect2);

      // Redraw
      DrawRect(&RefreshRect);
      if (RefreshRect2.width > 0) DrawRect(&RefreshRect2);
    }
  }
  
  if (FirstSelected) {
    // Compute minimal X
    MinX = FirstSelected->EncapRect.x;
    MinY = FirstSelected->EncapRect.y;
    Child = FirstSelected->NextSelected;
    while(Child) {
      MinX = MIN(Child->EncapRect.x,MinX);
      MinY = MIN(Child->EncapRect.y,MinY);
      Child=Child->NextSelected;
    }
  
    // Compute corrections
    XCorr = MinX-x;
    YCorr = MinY-y;
  }
}

void
TCADSheet::ButtonReleased()
{
  GdkRectangle Rect1,Rect2,Rect3,Rect4;
  GdkRectangle *ER;
  char First;
  TCADObject *Obj;
  int dx,dy;
  TState State;
  int x1,y1,x2,y2;

  ButtonDown = 0;

  State = System->GetState();
  if (ResetOnUp) System->ResetMode(FALSE);
  ResetOnUp = 0;

  if (SelectRect) {
    /* Select all objects in the rectangle */
    First = 1;
    x1 = FocusRect.x;
    x2 = FocusRect.x+FocusRect.width;
    y1 = FocusRect.y;
    y2 = FocusRect.y+FocusRect.height;
    Rect1 = FocusRect;
    Rect1.width++;
    Rect1.height++;

    Obj = FirstChild;
    while(Obj) {
      if (!Obj->Selected) {
        ER = &Obj->EncapRect;
        if ((ER->x >= x1) && ((ER->x+ER->width ) <= x2) &&
	    (ER->y >= y1) && ((ER->y+ER->height) <= y2)) {
          if (First) SelectObject(Obj,&Rect3,&Rect2);
          else SelectObject(Obj,&Rect3,&Rect4);
	  System->UnionRectangles(&Rect1,&Rect3);
	  First = 0;
	}
      }
      Obj = Obj->Next;
    }

    // Redraw
    DrawRect(&Rect1);
    if (!First) {
      if (Rect2.width > 0) DrawRect(&Rect2);
    }
    return;
  }

  if (NewRect) {
    /* Really create object */
 
    // Clear clipping
    gdk_gc_set_clip_rectangle(GC,NULL);
    // Clear rectangle
    DrawFocusRect();
  
    if ((FocusRect.width < 4) && (FocusRect.height < 4)) return; // Too small - ignore

    // Enlarge FocusRect so that right/bottom boundary is also on grid
    FocusRect.width++;
    FocusRect.height++;

    dx = 0;
    dy = 0;
    switch(State) {
      case ST_FRAME:
        Obj = new TCADFrame(this);
        Obj->Uniq = NextUniq();
        AppendChild(Obj);
	Obj->EncapRect = FocusRect;
	Obj->MoveResize(dx,dy,0,0,&Rect1);
	DrawRect(&Rect1);
	SelectObject(Obj,&Rect1,&Rect2);
	DrawRect(&Rect1);
        break;
      case ST_TEXT:
        Obj = new TCADText(this);
        Obj->Uniq = NextUniq();
        AppendChild(Obj);
	Obj->EncapRect = FocusRect;
	Obj->MoveResize(dx,dy,0,0,&Rect1);
	DrawRect(&Rect1);
	SelectObject(Obj,&Rect1,&Rect2);
	DrawRect(&Rect1);
        EditText();
        break;
      case ST_FRAMED_TEXT:
        Obj = new TCADFramedText(this);
        Obj->Uniq = NextUniq();
        AppendChild(Obj);
	Obj->EncapRect = FocusRect;
	Obj->MoveResize(dx,dy,0,0,&Rect1);
	DrawRect(&Rect1);
	SelectObject(Obj,&Rect1,&Rect2);
	DrawRect(&Rect1);
        EditText();
        break;
      default: ;
    }
  }

  if ((DHandle) && (FirstSelected != NULL)) {
    DHandle = 0;
    if (FirstSelected->HandleButtonReleased(&Rect1)) {
      // Redraw
      DrawRect(&Rect1);
    }
    return;
  }

  if (Dragging) {
    // Clear clipping
    gdk_gc_set_clip_rectangle(GC,NULL);
    Dragging = 0;
    
    // Clear rectangle
    DrawFocusRect();

    // Move dragged objects
    Obj = FirstSelected;
    First = 1;
    dx = FocusRect.x-FocusX;
    dy = FocusRect.y-FocusY;
    while(Obj) {
      if (First) FirstSelected->MoveResize(dx,dy,0,0,&Rect1);
      else {
        Obj->MoveResize(dx,dy,0,0,&Rect2);
        System->UnionRectangles(&Rect1,&Rect2);
      }

      First = 0;
      Obj = Obj->NextSelected;
    }
    
    if (!First) {
      // Redraw
      DrawRect(&Rect1);
    }
  }

}

void
TCADSheet::MouseMoved(int x,int y)
{
  TCADObject *Obj;
  GdkRectangle Rect;
  char First;
  int T1,T2;
  int XOff,YOff;
  gboolean Attached;
  TAttach Attach;

  GTK_RULER(VRuler)->position = y;
  gtk_ruler_draw_pos(GTK_RULER(VRuler));
  GTK_RULER(HRuler)->position = x;
  gtk_ruler_draw_pos(GTK_RULER(HRuler));

  if (!ButtonDown) return;

  if (!SelectRect) {
    Attached = FALSE;
    if ((DHandle) && (DHType == HT_LINEEND)) {
      if (CanAttach(x,y,&Attach)) {
        Attached = TRUE;
      }
      ((TCADLine *)(FirstSelected))->SetHandleAttachment(Attached,&Attach);
    }
    if (!Attached) {
      if (DHandle) {
      }
      x += XCorr;
      y += YCorr;
      ALIGN(x,GridX,FALSE);
      ALIGN(y,GridY,FALSE);
    }
  }
  
  if ((OldX == x) && (OldY == y)) return;  // Nothing changed
  OldX = x;
  OldY = y;

  if (NewRect || SelectRect) {
    // Clear clipping
    gdk_gc_set_clip_rectangle(GC,NULL);

    DrawFocusRect();
    UpdateFocusRect(x,y);
    return;
  }
  
  if (FirstSelected == NULL) return;

  if (DHandle) {
    if (FirstSelected->HandleDragged(x,y,&Rect)) {
      // Redraw
      DrawRect(&Rect);
    }
    return;
  }

  // Clear clipping
  gdk_gc_set_clip_rectangle(GC,NULL);

  if (!Dragging) {
    /* Start dragging */

    // Get rectangle
    Obj = FirstSelected;
    First = 1;
    while(Obj) {
      if (First) FocusRect = Obj->EncapRect;
      else System->UnionRectangles(&FocusRect,&Obj->EncapRect);
      
      First = 0;
      Obj = Obj->NextSelected;

    }
    FocusX = FocusRect.x;
    FocusY = FocusRect.y;

    Dragging = 1;
  }
  else DrawFocusRect();  // Clear old rect

  /* Compute focus rect position */
  FocusRect.x = x;
  FocusRect.y = y;

  // Position inside sheet
  if (FocusRect.x < 0) FocusRect.x = 0;
  if (FocusRect.y < 0) FocusRect.y = 0;
  
  T1 = FocusRect.x+FocusRect.width-1;
  T2 = EncapRect.x+EncapRect.width-1;
  if (T1 > T2) FocusRect.x -= (T1-T2);
 
  T1 = FocusRect.y+FocusRect.height-1;
  T2 = EncapRect.y+EncapRect.height-1;
  if (T1 > T2) FocusRect.y -= (T1-T2);

  DrawFocusRect();
}

void
TCADSheet::DrawFocusRect()
{
  gdk_gc_set_foreground(GC,&BlackColor);
  gdk_gc_set_function(GC,GDK_INVERT);
  gdk_gc_set_line_attributes(GC,1,GDK_LINE_ON_OFF_DASH,GDK_CAP_NOT_LAST,GDK_JOIN_MITER);

//  printf("DrawFocusRect(%d,%d,%d,%d)\n",FocusRect.x,FocusRect.y,FocusRect.width,FocusRect.height);
  gdk_draw_rectangle(Fixed->window,GC,0,FocusRect.x,FocusRect.y,FocusRect.width,FocusRect.height);
  
  gdk_gc_set_function(GC,GDK_COPY);
  gdk_gc_set_line_attributes(GC,1,GDK_LINE_SOLID,GDK_CAP_NOT_LAST,GDK_JOIN_MITER);
}

void
TCADSheet::SetVisibilityState(GdkVisibilityState State)
{
  VisibilityState = State;

  if ((State == GDK_VISIBILITY_FULLY_OBSCURED) && (GTK_WIDGET_HAS_FOCUS(Window)))
   System->SheetActivated(this);
}

void
TCADSheet::GotFocus()
{
//  if (VisibilityState == GDK_VISIBILITY_FULLY_OBSCURED)
   System->SheetActivated(this);
}

gboolean
TCADSheet::HasFocus()
{
  return(GTK_WIDGET_HAS_FOCUS(Window));
}

void
TCADSheet::BringToFront()
{
  gdk_window_raise(Window->window);
}

void
TCADSheet::AddAccelGroup(GtkAccelGroup *AccelGroup)
{
  gtk_window_add_accel_group (GTK_WINDOW(Window), AccelGroup);
}

void
TCADSheet::UpdateFocusRect(int x,int y)
{
  FocusRect.x = MIN(DownX,x);
  FocusRect.y = MIN(DownY,y);
  FocusRect.width = ABS(DownX - x);
  FocusRect.height = ABS(DownY - y);

  // Clear clipping
  gdk_gc_set_clip_rectangle(GC,NULL);

  DrawFocusRect();
}

void
TCADSheet::UnselectAll()
{
  TCADObject *Child;
  char First;
  GdkRectangle RefreshRect,RefreshRect2;

  // Unselect all objects
  First = 1;
  Child = FirstSelected;
  while(Child) {
    if (First) Child->Unselect(&RefreshRect);
    else {
      Child->Unselect(&RefreshRect2);
	
      // Union rectangles
      System->UnionRectangles(&RefreshRect,&RefreshRect2);
    }
    First = 0;
    Child=Child->NextSelected;
  }
  FirstSelected = NULL;

  if (!First) DrawRect(&RefreshRect);
}

void
TCADSheet::DeleteSelected()
{
  GdkRectangle Rect;
  TCADObject *Obj,*Next;
  gboolean First = TRUE;

  Obj = FirstSelected;
  while(Obj) {
    if (First) Rect = Obj->EncapRect;
    else System->UnionRectangles(&Rect,&Obj->EncapRect);
    First = FALSE;
    
    Next = Obj->NextSelected;
    DeleteChild(Obj);
    delete Obj;
    Obj = Next;
  }
  FirstSelected = NULL;

  if (!First) {
    // Enlarge Rect to contain handles and arrows
    Rect.x -= HANDLE_SIZE/2+ARROW_SIZE_BIG/2;
    Rect.y -= HANDLE_SIZE/2+ARROW_SIZE_BIG/2;;
    Rect.width += HANDLE_SIZE+ARROW_SIZE_BIG+1;
    Rect.height += HANDLE_SIZE+ARROW_SIZE_BIG+1;
    DrawRect(&Rect);
  }
}

static void
EditTextCallback(char *NewText,TCADSheet *Sheet)
{
  TCADObject *Obj;

  Obj = Sheet->GetFirstSelected();
  if (Obj == NULL) return;
  if (!Obj->HasText()) return;

  if (NewText != NULL) {
    ((TCADTextObject *)Obj)->SetText(NewText);

    // Redraw object
    Sheet->DrawRect(&Obj->EncapRect);
  }
}

void
TCADSheet::EditText()
{
  TCADObject *Obj;
  char *OldText;

  Obj = FirstSelected;
  if (Obj == NULL) return;
  if (!Obj->HasText()) return;

  OldText = ((TCADTextObject *)Obj)->GetText();
  TextEditBox(OldText,NULL,EditTextCallback,this);
  free(OldText);
}

TCADObject *
TCADSheet::GetFirstSelected()
{
  return(FirstSelected);
}

void
TCADSheet::Export(char *FileName,TExportFormat Format)
{
  FILE *Fl;

  Fl = fopen(FileName,"w");
  if (Fl == 0) {
    char Msg[1024];
    sprintf(Msg,"Cannot open file: %s",strerror(errno));

    MessageBox(Msg,"Error");
  }
  else {
    // Write header
    switch(Format) {
      case EF_LATEX:
        fprintf(Fl,"{%% Picture saved by %s %s\n",PACKAGE,VERSION);
        fprintf(Fl,"\\unitlength=1.000000pt\n");
        fprintf(Fl,"\\begin{picture}(%d,%d)(0.00,0.00)\n",EncapRect.width,EncapRect.height);
	break;
      case EF_FIG:
        fprintf(Fl,"#FIG 3.2\n");
	fprintf(Fl,"Portrait\n");	// Orientation
	fprintf(Fl,"Center\n");		// Justification
	fprintf(Fl,"Metric\n");		// Units
	fprintf(Fl,"A4\n");		// Paper size
	fprintf(Fl,"72.00\n");		// Magnification
	fprintf(Fl,"Single\n");		// Multiple page
	fprintf(Fl,"0\n");		// Transparent color
	fprintf(Fl,"100 2\n");		// Resolution / origin UL
//	fprintf(Fl,"\n");		// 
        break;
    }

    TCADGroup::Export(Fl,Format);

    if (Format == EF_LATEX) {
      // Write footer
      fprintf(Fl,"\\end{picture}}\n");
    }
    fclose(Fl);
  }
}

void
TCADSheet::ShowArrowProperties()
{
  TCADObject *Obj;

  Obj = FirstSelected;
  while(Obj) {
    if (Obj->IsLine()) {
      ((TCADLine *)Obj)->ShowArrowProperties();
      return;
    }
    Obj = Obj->NextSelected;
  }
}

void
TCADSheet::ShowTextProperties()
{
  TCADObject *Obj;

  Obj = FirstSelected;
  while(Obj) {
    if (Obj->HasText()) {
      ((TCADTextObject *)Obj)->ShowTextProperties(Obj);
      return;
    }
    Obj = Obj->NextSelected;
  }
}

void
TCADSheet::SetShowGrid(gboolean Value)
{
  ShowGrid = Value;
//  printf("TCADSheet::SetShowGrid(%d)\n",Value);
  DrawRect(NULL);
  System->SheetActivated(this);
}

void
TCADSheet::SetUseGrid(gboolean Value)
{
  UseGrid = Value;
  System->SheetActivated(this);
}

void
TCADSheet::ShowGridProperties()
{
  ShowGridOptionsDialog(this);
}

void
TCADSheet::GetGridOptions(gboolean& X_eq_Y,int& GX,int& GY)
{
  X_eq_Y = GX_eq_GY;
  GX = GridX;
  GY = GridY;
}

void
TCADSheet::SetGridOptions(gboolean X_eq_Y,int GX,int GY)
{
  GX_eq_GY = X_eq_Y;
  GridX = GX;
  GridY = GY;
  DrawRect(NULL);
}

void
TCADSheet::AlignObjects(THorizAlign HA, TVertAlign VA)
{
  int Val1,Val2;
  TCADObject *Obj;
  int DX,DY,DW,DH,T;
  GdkRectangle Rect;

  if (FirstSelected == NULL) return;

  switch(HA) {
    case HAL_LEFTS:
      Val1 = FirstSelected->EncapRect.x;
      break;
    case HAL_RIGHTS:
      Val1 = FirstSelected->EncapRect.x+FirstSelected->EncapRect.width;
      break;
    case HAL_SAMEWIDTH:
      Val1 = FirstSelected->EncapRect.width;
      break;
    default: ;
  }
  
  switch(VA) {
    case VAL_TOPS:
      Val2 = FirstSelected->EncapRect.y;
      break;
    case VAL_BOTTOMS:
      Val2 = FirstSelected->EncapRect.y+FirstSelected->EncapRect.height;
      break;
    case VAL_SAMEHEIGHT:
      Val2 = FirstSelected->EncapRect.height;
      break;
    default: ;
  }

  Obj = FirstSelected;
  while(Obj) {
    DY = 0;
    DX = 0;
    DW = 0;
    DH = 0;
    if (!Obj->IsLine()) {
      switch(HA) {
        case HAL_GRID:
          T = Obj->EncapRect.x;
          ALIGN(T,GridX,TRUE);
          DX += T-Obj->EncapRect.x;
          break;
        case HAL_LEFTS:
	  DX = Val1 - Obj->EncapRect.x;
          break;
	case HAL_RIGHTS:
	  DX = Val1 - (Obj->EncapRect.x+Obj->EncapRect.width);
	  break;
	case HAL_SAMEWIDTH:
          DW = Val1-EncapRect.width;
	  break;	  
	default: ;
      }
      
      switch(VA) {
        case VAL_GRID:
          T = Obj->EncapRect.y;
          ALIGN(T,GridY,TRUE);
          DY += T-Obj->EncapRect.y;
          break;
        case VAL_TOPS:
	  DY = Val2 - Obj->EncapRect.y;
          break;
	case VAL_BOTTOMS:
	  DY = Val2 - (Obj->EncapRect.y+Obj->EncapRect.height);
	  break;
	case VAL_SAMEHEIGHT:
          DH = Val2-EncapRect.height;
	  break;	  
	default: ;
      }

      if ((DX != 0) || (DY != 0) || (DW != 0) || (DH != 0)) Obj->MoveResize(DX,DY,DW,DH,&Rect);
    }
    
    Obj = Obj->NextSelected;
  }

  // Refresh
  DrawRect(NULL);
}

void
TCADSheet::SelectAll()
{
  GdkRectangle Rect1;
  TCADObject *Obj;

  Obj = FirstChild;
  while(Obj) {
    if (!Obj->Selected) SelectObject(Obj,&Rect1,&Rect1);
    Obj = Obj->Next;
  }

  // Refresh all
  DrawRect(NULL);
}

void
TCADSheet::ApplyRulerSettings()
{
  if (VRulerSettings.Visible) gtk_widget_show(GTK_WIDGET(VRuler));
  else gtk_widget_hide(GTK_WIDGET(VRuler));

  if (HRulerSettings.Visible) gtk_widget_show(GTK_WIDGET(HRuler));
  else gtk_widget_hide(GTK_WIDGET(HRuler));

  if (HRulerSettings.Visible && VRulerSettings.Visible) gtk_widget_show(GTK_WIDGET(Frame));
  else gtk_widget_hide(GTK_WIDGET(Frame));

  gtk_ruler_set_metric(GTK_RULER(VRuler),VRulerSettings.Metric);
  gtk_ruler_set_metric(GTK_RULER(HRuler),HRulerSettings.Metric);
}

void
TCADSheet::SetTitle()
{
  char *Title;
  
  if (FileName == NULL) {
    char Buf[256];
    sprintf(Buf,"Untitled %d",Uniq);

    Title = Buf;
  }
  else Title = FileName;
  gtk_window_set_title(GTK_WINDOW (Window), Title);
}

