/*
 * Copyright (C) 1998 Nathan Neulinger <nneul@umr.edu>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include "gtkyahoo.h"

struct gtkyahoo_event
{
	int type;					/* What event type */
	char *config;				/* Restrict to a particular config */
	char *id;					/* Restrict to a particular id */
	char *action;				/* which action to run */
};

static GSList *event_list = NULL;

/* global used to pass info to the event execute routine,
	this info will also be passed to the action for it's use */
struct gtkyahoo_event_info *event_info = NULL;

void event_clearinfo(void)
{
	if (event_info)
	{
		memset(event_info, 0, sizeof(*event_info));
	}
	else
	{
		event_info =
			(struct gtkyahoo_event_info *) calloc(1, sizeof(*event_info));
	}
}

static struct gtkyahoo_event *event_alloc(void)
{
	struct gtkyahoo_event *tmp;

	tmp = (struct gtkyahoo_event *) calloc(1, sizeof(*tmp));
	return tmp;
}

#if 0							/* eventually use these */
void event_clearall(void)
{
	DBG_Print("events", "Clearing all events.\n");
}

static void event_free(struct gtkyahoo_event *tmp)
{
	if (tmp)
	{
		if (tmp->id)
		{
			free(tmp->id);
		}
		if (tmp->config)
		{
			free(tmp->config);
		}
		if (tmp->action)
		{
			free(tmp->action);
		}
		free(tmp);
	}
}
#endif

void event_set(char *type, char *id, char *config, char *action)
{
	int itype = GTKYAHOO_EVENT_UNKNOWN;
	struct gtkyahoo_event *event;

	if (!strcasecmp(type, "logon"))
	{
		itype = GTKYAHOO_EVENT_LOGON;
	}
	else if (!strcasecmp(type, "logoff"))
	{
		itype = GTKYAHOO_EVENT_LOGOFF;
	}
	else if (!strcasecmp(type, "mylogoff"))
	{
		itype = GTKYAHOO_EVENT_MYLOGON;
	}
	else if (!strcasecmp(type, "mylogoff"))
	{
		itype = GTKYAHOO_EVENT_MYLOGOFF;
	}
	else if (!strcasecmp(type, "mail-new"))
	{
		itype = GTKYAHOO_EVENT_MAIL_NEW;
	}
	else if (!strcasecmp(type, "mail-none"))
	{
		itype = GTKYAHOO_EVENT_MAIL_NONE;
	}
	else if (!strcasecmp(type, "message-no-focus"))
	{
		itype = GTKYAHOO_EVENT_MESSAGE_NO_FOCUS;
	}
	else if (!strcasecmp(type, "message-focus"))
	{
		itype = GTKYAHOO_EVENT_MESSAGE_FOCUS;
	}
	else if (!strcasecmp(type, "message"))
	{
		itype = GTKYAHOO_EVENT_MESSAGE;
	}
	else if (!strcasecmp(type, "new-chat"))
	{
		itype = GTKYAHOO_EVENT_NEWCHAT;
	}
	else if (!strcasecmp(type, "status"))
	{
		itype = GTKYAHOO_EVENT_STATUS;
	}
	else if (!strcasecmp(type, "status-idle"))
	{
		itype = GTKYAHOO_EVENT_STATUS_IDLE;
	}
	else if (!strcasecmp(type, "all"))
	{
		itype = GTKYAHOO_EVENT_ALL;
	}
	else if (!strcasecmp(type, ""))
	{
		itype = GTKYAHOO_EVENT_ALL;
	}
	else if (!strcasecmp(type, "*"))
	{
		itype = GTKYAHOO_EVENT_ALL;
	}

	/* convert the type to integer first though */
	DBG_Print("events",
		"Setting event: type=%s itype=%d id=%s config=%s action=%s\n",
		DBG_NullCheck(type), itype, DBG_NullCheck(id), DBG_NullCheck(config),
		DBG_NullCheck(action));

	event = event_alloc();

	/* convert strings to lowercase for efficiency in later comparisons */
	/* strlower also handles null strings passed in */
	event->type = itype;
	event->id = strlower(id);
	event->config = strlower(config);
	event->action = strlower(action);

	event_list = g_slist_append(event_list, event);
}

void event_execute(int type)
{
	GSList *tmpnode = event_list;
	struct gtkyahoo_event *tmprec;

	/* make sure the event info is valid */
	if (!event_info)
	{
		event_clearinfo();
	}

	/* loop through all of the events */
	/* execute each one */
	while (tmpnode && tmpnode->data)
	{
		int matched = 1;

		tmprec = tmpnode->data;

		DBG_Print("vverbose-events",
			"Checking event: type=%d id=%s config=%s action=%s ei.id=%s\n",
			type,
			DBG_NullCheck(tmprec->id),
			DBG_NullCheck(tmprec->config),
			DBG_NullCheck(tmprec->action), DBG_NullCheck(event_info->id));

		/* only execute matching types and ALL types */
		if (tmprec->type != type && tmprec->type != GTKYAHOO_EVENT_ALL)
		{
			DBG_Print("verbose-events", "\ttype doesn't match\n");
			matched = 0;
		}

		/* check if id matches */
		if (matched && tmprec->id)
		{
			if (event_info->id && strcmp(tmprec->id, event_info->id))
			{
				DBG_Print("events", "\tcurrent id doesn't match\n");
				matched = 0;
			}
			else if (!event_info->id)
			{
				DBG_Print("events", "\tcurrent id null\n");
				matched = 0;
			}
		}

		/* check if config matches */
		if (matched && tmprec->config)
		{
			if (current_config && strcasecmp(tmprec->config, current_config))
			{
				DBG_Print("events", "\tcurrent_config (\"%s\") doesn't match event config (\"%s\")\n", current_config, tmprec->config);
				matched = 0;
			}
			else if (!current_config)
			{
				DBG_Print("events", "\tcurrent_config null\n");
				matched = 0;
			}
		}

		if (matched)
		{
			action_execute(tmprec->action);
		}

		tmpnode = g_slist_next(tmpnode);
	}
}
