/*
 * Copyright (C) 1998 Nathan Neulinger <nneul@umr.edu>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include "gtkyahoo.h"
#include "window-login.h"
#include <unistd.h>
#include <sys/stat.h>
#include <errno.h>
extern FILE *yyin;
extern int yyparse(void);

static void free_string (gpointer data, gpointer user_data)
{
	free (data);
}

void rcfile_init(void)
{
	struct passwd *pw;
	FILE *conf;
	FILE *sampleconf;
	char *confpath;
	char *gtkconfpath;
	int len;
	struct stat statbuf;

	g_list_foreach (id_list, free_string, NULL);
	id_list = NULL;
	g_list_foreach (name_list, free_string, NULL);
	name_list = NULL;

	umask(077);

	pw = getpwuid(getuid());
	if (!pw)
	{
		printf("Couldn't look up userid.\n");
		exit(1);
	}

	len = strlen(pw->pw_dir) + 50;
	confpath = malloc(len);
	gtkconfpath = malloc(len);

	sprintf(confpath, "%s/.gtkyahoo", pw->pw_dir);
	if (stat(confpath, &statbuf))
	{
		if (errno == ENOENT)
		{
			printf
				("Warning: ~/.gtkyahoo directory non found, creating it.\n");
			if ((mkdir(confpath, S_IRWXU)))
			{
				printf("Error: ~/.gtkyahoo could not be created. (%s)\n",
					strerror(errno));
				exit(1);
			}
		}
		else
		{
			printf("Error: ~/.gtkyahoo not found (%s).\n", strerror(errno));
			exit(1);
		}
	}

	/* Switch to ~/.gtkyahoo directory */
	chdir(confpath);

	if (stat(confpath, &statbuf))
	{
		printf("Error: ~/.gtkyahoo not found. (%s)\n", strerror(errno));
		exit(1);
	}

	if (!(S_ISDIR(statbuf.st_mode)))
	{
		printf("Error: ~/.gtkyahoo is not a directory.\n");
		exit(1);
	}

	if (getenv("TESTGTKYAHOORC"))
	{
		sprintf(confpath, "%s/.gtkyahoo/testgtkyahoorc", pw->pw_dir);
		sprintf(gtkconfpath, "%s/.gtkyahoo/testgtkrc", pw->pw_dir);
	}
	else
	{
		sprintf(confpath, "%s/.gtkyahoo/gtkyahoorc", pw->pw_dir);
		sprintf(gtkconfpath, "%s/.gtkyahoo/gtkrc", pw->pw_dir);
	}
	if (stat(confpath, &statbuf))
	{
		if (errno == ENOENT)
		{
			printf("Warning: %s not found, creating it.\n", confpath);

			if ((conf = fopen(confpath, "w")) == NULL)
			{
				printf("Error: Could not create %s. (%s)\n",
					confpath, strerror(errno));
				exit(1);
			}

			if ((sampleconf =
					fopen(PKGDATADIR "/simple.gtkyahoorc", "r")) == NULL)
			{
				printf("Error: Couldn't open '%s' (%s).\n",
					PKGDATADIR "/simple.gtkyahoorc", strerror(errno));
				printf("\tUsing internal sample config.\n");

				fprintf(conf, "# GTKYahoo Configuration File\n");
				fprintf(conf, "#\n");
				fprintf(conf, "# Global Options\n");
				fprintf(conf, "# show-online-only is ignore for now\n");
				fprintf(conf, "#\n");
				fprintf(conf, "\n");
				fprintf(conf, "Global\n");
				fprintf(conf, "\tshow-online-only no\n");
				fprintf(conf, "End-Global\n");
				fprintf(conf, "\n");
				fprintf(conf, "# Per-Config Options\n");
				fprintf(conf, "# Config name is ignored for now\n");
				fprintf(conf,
					"# Multiple users can be defined, the last one is taken for now.\n");
				fprintf(conf, "\n");
				fprintf(conf, "Config \"config name\"\n");
				fprintf(conf, "\tUserID \"userid\"\n");
				fprintf(conf, "\tPassword \"password\"\n");
				fprintf(conf, "End-Config\n");
			}
			else
			{
				char tmp[250];

				while (fgets(tmp, 250, sampleconf))
				{
					fputs(tmp, conf);
				}
				fclose(sampleconf);
			}
			fclose(conf);

			printf("\nA sample configuration file has been created.\n");
			printf("Please edit it before proceeding.\n");
			exit(1);
		}
		else
		{
			printf("Warning: %s not found. (%s)\n",
				confpath, strerror(errno));
			exit(1);
		}
	}

	if (stat(confpath, &statbuf))
	{
		printf("Error: %s not found. (%s)\n", confpath, strerror(errno));
		exit(1);
	}

	if (!(S_ISREG(statbuf.st_mode)))
	{
		printf("Error: %s is not a file.\n", confpath);
		exit(1);
	}

	rcfile_parse(confpath);
	free(confpath);
	gtk_rc_parse(gtkconfpath);
	free(gtkconfpath);
}

int rcfile_parse(char *filename)
{
	int res;

	if (!filename)
	{
		fprintf(stderr, "No rc filename passed. Parse failed.\n");
		return (1);
	}

	if ((yyin = fopen(filename, "r")) == NULL)
	{
		fprintf(stderr, "Failed to open rcfile. Parse failed.\n");
		return (1);
	}

	res = yyparse();
	fclose(yyin);
	return res;
}
