/*
 * Copyright (C) 1998 Nathan Neulinger <nneul@umr.edu>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include "gtkyahoo.h"

/* structure for holding a users status, used in this file only */
struct statusrec
{
	char *id;
	int status;
	int in_pager;
	int in_chat;
	int in_game;
	char *msg;
        time_t status_change_time; /* time the status last changed */
        int    status_time_valid;  /* true if the above is valid */
};

/* status globals - static so this file only */
static GSList *status_list = NULL;
struct statusrec *lastrec = NULL;

static void set_status_time(struct statusrec *status)
{
    if (!initial_warmup)
    {
        time(&status->status_change_time);
        status->status_time_valid = 1;
    }
}

/* allocate a new status structure */
static struct statusrec *alloc_statusrec(void)
{
	struct statusrec *tmpstatus;

	tmpstatus = (struct statusrec *) calloc(1, sizeof(*tmpstatus));
        set_status_time(tmpstatus);
	return tmpstatus;
}

/* Routines for fetching status attributes */
/* retrieve a list of users */
char **status_fetchlist(void)
{
	GSList *tmpnode;
	struct statusrec *status;
	char **tmp;
	int i;

	tmp = (char **) calloc(g_slist_length(status_list) + 1, sizeof(char *));

	i = 0;
	tmpnode = status_list;

	while (tmpnode)
	{
		status = tmpnode->data;

                printf("line %d for %s\n", __LINE__, status->id);

		tmp[i++] = strdup(status->id);
		tmpnode = g_slist_next(tmpnode);
	}
	tmp[i] = 0;

	return tmp;
}

/* retrieve a users status */
static struct statusrec *status_fetch(char *id)
{
	GSList *tmpnode;
	char *tmpid;
	struct statusrec *res = NULL;
	struct statusrec *tmprec = NULL;

	if (!id)
	{
		DBG_Print("status", "status_fetch called with null id\n");
		return NULL;
	}

	tmpid = strlower(id);

	/* short circuit if it was the same as last time */
	if (lastrec)
	{
		if (!strcmp(lastrec->id, tmpid))
		{
			free(tmpid);
			return lastrec;
		}
	}

	tmpnode = status_list;		/* get the first node if any */
	while (tmpnode && tmpnode->data && !res)
	{
		tmprec = tmpnode->data;
		if (!strcmp(tmpid, tmprec->id))
		{
			res = tmprec;
		}

		tmpnode = g_slist_next(tmpnode);
	}

	free(tmpid);
	lastrec = res;
	return res;
}

/* Fetch a status record, creating and insert a new one if not found */
static struct statusrec *status_fetchcreate(char *id)
{
	char *tmpid;
	struct statusrec *tmprec;

	if (!id)
	{
		DBG_Print("status", "status_fetchcreate called with null id\n");
		return NULL;
	}

	tmpid = strlower(id);

	tmprec = status_fetch(tmpid);
	if (!tmprec)
	{
		tmprec = alloc_statusrec();
		tmprec->id = strdup(tmpid);

		status_list = g_slist_prepend(status_list, tmprec);
	}

	free(tmpid);
	return tmprec;
}

/* indicate if user can receive messages */
int status_get_avail(char *id)
{
	return status_get_in_game(id) ||
		status_get_in_pager(id) || status_get_in_chat(id);
}

int status_get_in_game(char *id)
{
	struct statusrec *tmprec;

	tmprec = status_fetch(id);
	if (tmprec)
	{
		return tmprec->in_game;
	}
	else
	{
		return 0;
	}
}

int status_get_in_chat(char *id)
{
	struct statusrec *tmprec;

	tmprec = status_fetch(id);
	if (tmprec)
	{
		return tmprec->in_chat;
	}
	else
	{
		return 0;
	}
}

int status_get_known(char *id)
{
	struct statusrec *tmprec;

	tmprec = status_fetch(id);
	if (tmprec)
	{
		return 1;
	}
	else
	{
		return 0;
	}
}

int status_get_in_pager(char *id)
{
	struct statusrec *tmprec;

	tmprec = status_fetch(id);
	if (tmprec)
	{
		return tmprec->in_pager;
	}
	else
	{
		return 0;
	}

}

int status_get_status(char *id)
{
	struct statusrec *tmprec;

	tmprec = status_fetch(id);
	if (tmprec)
	{
		return tmprec->status;
	}
	else
	{
		return 0;
	}
}

time_t *status_get_changetime(char *id)
{
	struct statusrec *tmprec;

	tmprec = status_fetch(id);
	if (tmprec && tmprec->status_time_valid)
	{
		return &tmprec->status_change_time;
	}
	else
	{
		return NULL;
	}
}


char *status_get_msg(char *id)
{
	struct statusrec *tmprec;

	tmprec = status_fetch(id);
	if (tmprec)
	{
		return tmprec->msg;
	}
	else
	{
		return NULL;
	}
}

void status_set_in_game(char *id, int val)
{
	struct statusrec *tmprec;

	tmprec = status_fetchcreate(id);
	if (tmprec)
	{
		tmprec->in_game = val;
	}
}

void status_set_in_chat(char *id, int val)
{
	struct statusrec *tmprec;

	tmprec = status_fetchcreate(id);
	if (tmprec)
	{
		tmprec->in_chat = val;
	}
}

void status_set_in_pager(char *id, int val)
{
	struct statusrec *tmprec;

	tmprec = status_fetchcreate(id);
	if (tmprec)
	{
		tmprec->in_pager = val;
	}
}

void status_set_status(char *id, int val)
{
	struct statusrec *tmprec;

	tmprec = status_fetchcreate(id);
	if (tmprec)
	{
            if (tmprec->status != val) {
		tmprec->status = val;
                set_status_time(tmprec);
            }
	}
}

void status_set_msg(char *id, char *msg)
{
	struct statusrec *tmprec;

	tmprec = status_fetchcreate(id);
	if (!tmprec)
	{
		return;
	}
	if (tmprec->msg)
	{
		free(tmprec->msg);
	}
	tmprec->msg = NULL;
	if (msg)
	{
		tmprec->msg = strdup(msg);
	}
}
