/*
  Guifications - The end all, be all, toaster popup plugin
  Copyright (C) 2003-2004 Gary Kramlich

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#include <stdlib.h>
#include <time.h>
#include <glib.h>
#include <gdk/gdk.h>

#include <debug.h>
#include <xmlnode.h>

#include "gf_event_info.h"
#include "gf_item.h"
#include "gf_notification.h"
#include "gf_preferences.h"
#include "gf_theme.h"
#include "gf_theme_ops.h"

struct _GfNotification {
	GfTheme *theme;
	gchar *n_type;

	gchar *background;
	GList *items;
};

/*******************************************************************************
 * API
 ******************************************************************************/
GfNotification *
gf_notification_new(GfTheme *theme) {
	GfNotification *notification;

	g_return_val_if_fail(theme, NULL);

	notification = g_new0(GfNotification, 1);
	notification->theme = theme;

	return notification;
}

GfNotification *
gf_notification_new_from_xmlnode(GfTheme *theme, xmlnode *node) {
	GfNotification *notification;
	GfItem *item;
	xmlnode *child;
	const gchar *data;

	g_return_val_if_fail(theme, NULL);
	g_return_val_if_fail(node, NULL);

	notification = gf_notification_new(theme);

	notification->n_type = g_strdup(xmlnode_get_attrib(node, "type"));
	if(!notification->n_type) {
		gaim_debug_info("Guifications", "** Error: Notification type unknown\n");
		gf_notification_destroy(notification);
		return NULL;
	}

	/* clean this up later to allow no background to use the default theme colors */
	if(!(data = xmlnode_get_attrib(node, "background"))) {
		gaim_debug_info("Guifications", "** Error: could not find a background attribute\n");
		gf_notification_destroy(notification);
		return NULL;
	}
	notification->background = g_strdup(data);

	child = xmlnode_get_child(node, "item");

	while(child) {
		item = gf_item_new_from_xmlnode(notification, child);

		if(item)
			gf_notification_add_item(notification, item);

		child = xmlnode_get_next_twin(child);
	}

	return notification;
}

GfNotification *
gf_notification_copy(GfNotification *notification) {
	GfNotification *new_notification;
	GList *l;

	g_return_val_if_fail(notification, NULL);

	new_notification = gf_notification_new(notification->theme);

	if(notification->n_type)
		new_notification->n_type = g_strdup(notification->n_type);

	if(notification->background)
		new_notification->background = g_strdup(notification->background);

	for(l = notification->items; l; l = l->next) {
		GfItem *item;

		item = gf_item_copy(GF_ITEM(l->data));
		new_notification->items = g_list_append(new_notification->items, item);
	}

	return new_notification;
}

xmlnode *
gf_notification_to_xmlnode(GfNotification *notification) {
	GList *l;
	xmlnode *parent, *child;

	parent = xmlnode_new("notification");
	xmlnode_set_attrib(parent, "type", notification->n_type);

	if(notification->background)
		xmlnode_set_attrib(parent, "background", notification->background);

	for(l = notification->items; l; l = l->next) {
		if((child = gf_item_to_xmlnode(GF_ITEM(l->data))))
			xmlnode_insert_child(parent, child);
	}

	return parent;
}

void
gf_notification_destroy(GfNotification *notification) {
	GfItem *item;
	GList *l;

	g_return_if_fail(notification);

	if(notification->n_type) {
		g_free(notification->n_type);
		notification->n_type = NULL;
	}

	if(notification->background) {
		g_free(notification->background);
		notification->background = NULL;
	}

	if(notification->items) {
		for(l = notification->items; l; l = l->next) {
			item = GF_ITEM(l->data);
			gf_item_destroy(item);
		}

		g_list_free(notification->items);
		notification->items = NULL;
	}

	g_free(notification);
}

void
gf_notification_set_type(GfNotification *notification, const gchar *n_type) {
	g_return_if_fail(notification);
	g_return_if_fail(n_type);

	if(notification->n_type)
		g_free(notification->n_type);

	notification->n_type = g_strdup(n_type);
}

const gchar *
gf_notification_get_type(GfNotification *notification) {
	g_return_val_if_fail(notification, NULL);

	return notification->n_type;
}

void
gf_notification_set_background(GfNotification *notification,
							   const gchar *background)
{
	g_return_if_fail(notification);

	if(notification->background)
		g_free(notification->background);

	notification->background = g_strdup(background);
}

const gchar *
gf_notification_get_background(GfNotification *notification) {
	g_return_val_if_fail(notification, NULL);

	return notification->background;
}

void
gf_notification_add_item(GfNotification *notification, GfItem *item) {
	g_return_if_fail(notification);
	g_return_if_fail(item);

	notification->items = g_list_append(notification->items, item);
}

void
gf_notification_remove_item(GfNotification *notification, GfItem *item) {
	g_return_if_fail(notification);
	g_return_if_fail(item);

	notification->items = g_list_remove(notification->items, item);
}

GList *
gf_notification_get_items(GfNotification *notification) {
	g_return_val_if_fail(notification, NULL);

	return notification->items;
}

/*******************************************************************************
 * Finding, rendering, all that fun stuff...
 ******************************************************************************/
GList *
gf_notifications_for_event(const gchar *n_type) {
	GfTheme *theme;
	GfNotification *notification;
	GList *l = NULL, *t, *n;

	g_return_val_if_fail(n_type, NULL);

	for(t = gf_themes_get_loaded(); t; t = t->next) {
		theme = GF_THEME(t->data);

		for(n = gf_theme_get_notifications(theme); n; n = n->next) {
			notification = GF_NOTIFICATION(n->data);

			if(!g_ascii_strcasecmp(notification->n_type, n_type))
				l = g_list_append(l, notification);
		}
	}

	return l;
}

GfNotification *
gf_notification_find_for_event(const gchar *n_type) {
	GfNotification *notification = NULL;
	GList *n = NULL;
	gint c;
	time_t t;

	g_return_val_if_fail(n_type, NULL);

	n = gf_notifications_for_event(n_type);
	if(!n)
		return NULL;

	t = time(NULL);
	srand(t);
	c = rand() % g_list_length(n);

	notification = GF_NOTIFICATION(g_list_nth_data(n, c));

	return notification;
}

GdkPixbuf *
gf_notification_render(GfNotification *notification, GfEventInfo *info) {
	GfItem *item = NULL;
	GdkPixbuf *pixbuf = NULL;
	GList *l = NULL;
	gchar *filename;
	const gchar *path;

	g_return_val_if_fail(notification, NULL);
	g_return_val_if_fail(info, NULL);

	/* create the pixbuf, return if it failed */
	path = gf_theme_get_path(notification->theme);
	filename = g_build_filename(path, notification->background, NULL);
	pixbuf = gdk_pixbuf_new_from_file(filename, NULL);
	g_free(filename);

	if(!pixbuf) {
		gaim_debug_info("Guifications", "Couldn't not load notification background\n");
		return NULL;
	}

	/* render the items */
	for(l = notification->items; l; l = l->next) {
		item = GF_ITEM(l->data);

		gf_item_render(item, pixbuf, info);
	}

	/* display it already!! */
	return pixbuf;
}

GfTheme *
gf_notification_get_theme(GfNotification *notification) {
	g_return_val_if_fail(notification, NULL);

	return notification->theme;
}
