/*
  Guifications - The end all, be all, toaster popup plugin
  Copyright (C) 2003-2004 Gary Kramlich

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#include <gtk/gtk.h>

#include <account.h>
#include <blist.h>
#include <debug.h>
#include <gtkblist.h>
#include <gtkconv.h>
#include <gtklog.h>

#include "gf_action.h"
#include "gf_display.h"
#include "gf_event.h"
#include "gf_event_info.h"
#include "gf_intl.h"
#include "gf_notification.h"

struct _GfAction {
	gchar *name;
	gchar *i18n;
	GfActionFunc func;
};

static GList *actions = NULL;

/*******************************************************************************
 * API
 ******************************************************************************/
GfAction *
gf_action_new() {
	GfAction *action;

	action = g_new0(GfAction, 1);

	return action;
}

void
gf_action_destroy(GfAction *action) {
	g_return_if_fail(action);

	if(action->name)
		g_free(action->name);

	g_free(action);
	action = NULL;
}

void
gf_action_set_name(GfAction *action, const gchar *name) {
	g_return_if_fail(action);
	g_return_if_fail(name);

	if(action->name)
		g_free(action->name);

	action->name = g_strdup(name);
}

const gchar *
gf_action_get_name(GfAction *action) {
	g_return_val_if_fail(action, NULL);

	return action->name;
}

void
gf_action_set_i18n(GfAction *action, const gchar *i18n) {
	g_return_if_fail(action);
	g_return_if_fail(i18n);

	if(action->i18n)
		g_free(action->i18n);

	action->i18n = g_strdup(i18n);
}

const gchar *
gf_action_get_i18n(GfAction *action) {
	g_return_val_if_fail(action, NULL);

	return action->i18n;
}

void
gf_action_set_func(GfAction *action, GfActionFunc func) {
	g_return_if_fail(action);
	g_return_if_fail(func);

	action->func = func;
}

GfActionFunc
gf_action_get_func(GfAction *action) {
	g_return_val_if_fail(action, NULL);

	return action->func;
}

void
gf_action_execute(GfAction *action, GfDisplay *display, GdkEventButton *event) {
	g_return_if_fail(action);
	g_return_if_fail(display);

	action->func(display, event);
}

GfAction *
gf_action_find_with_name(const gchar *name) {
	GfAction *action;
	GList *l;

	g_return_val_if_fail(name, NULL);

	for(l = actions; l; l = l->next) {
		action = GF_ACTION(l->data);

		if(!g_ascii_strcasecmp(name, action->name))
			return action;
	}

	return NULL;
}

GfAction *
gf_action_find_with_i18n(const gchar *i18n) {
	GfAction *action;
	GList *l;

	g_return_val_if_fail(i18n, NULL);

	for(l = actions; l; l = l->next) {
		action = GF_ACTION(l->data);

		if(!g_ascii_strcasecmp(i18n, action->i18n))
			return action;
	}

	return NULL;
}

gint
gf_action_get_position(GfAction *action) {
	g_return_val_if_fail(action, -1);

	return g_list_index(actions, action);
}

/*******************************************************************************
 * Sub System
 ******************************************************************************/
static void
gf_action_add_default(const gchar *name, const gchar *i18n, GfActionFunc func) {
	GfAction *action;

	g_return_if_fail(name);
	g_return_if_fail(func);

	action = gf_action_new();
	gf_action_set_name(action, name);
	gf_action_set_i18n(action, i18n);
	gf_action_set_func(action, func);

	gf_actions_add_action(action);
}

void
gf_actions_init() {
	gf_action_add_default("close", _("Close"), gf_action_execute_close);
	gf_action_add_default("open", _("Open Conversation"),
						  gf_action_execute_open_conv);
	gf_action_add_default("context", _("Context Menu"),
						  gf_action_execute_context);
	gf_action_add_default("info", _("Get Info"), gf_action_execute_info);
	gf_action_add_default("log", _("Display Log"), gf_action_execute_log);
}

void
gf_actions_uninit() {
	GList *l, *ll;

	for(l = actions; l; l = ll) {
		ll = l->next;

		gf_actions_remove_action(GF_ACTION(l->data));
	}

	g_list_free(actions);
	actions = NULL;
}

void
gf_actions_add_action(GfAction *action) {
	g_return_if_fail(action);

	actions = g_list_append(actions, action);
}

void
gf_actions_remove_action(GfAction *action) {
	g_return_if_fail(action);

	actions = g_list_remove(actions, action);
}

gint
gf_actions_count() {
	return g_list_length(actions);
}

const gchar *
gf_actions_get_nth_name(gint nth) {
	GfAction *action;

	action = GF_ACTION(g_list_nth_data(actions, nth));

	return action->name;
}

const gchar *
gf_actions_get_nth_i18n(gint nth) {
	GfAction *action;

	action = GF_ACTION(g_list_nth_data(actions, nth));

	return action->i18n;
}

/*******************************************************************************
 * Action Functions
 ******************************************************************************/
void
gf_action_execute_close(GfDisplay *display, GdkEventButton *gdk_event) {
	g_return_if_fail(display);

	gf_display_destroy(display);
}

void
gf_action_execute_open_conv(GfDisplay *display, GdkEventButton *gdk_event) {
	GfEvent *event;
	GfEventInfo *info;
	const gchar *e_type, *n_type;
	GaimAccount *account = NULL;
	GaimConversation *conv = NULL;
	GaimConvWindow *win = NULL;

	g_return_if_fail(display);

	info = gf_display_get_event_info(display);
	event = gf_event_info_get_event(info);
	e_type = gf_event_get_event_type(event);
	n_type = gf_event_get_notification_type(event);

	if(!g_ascii_strcasecmp(e_type, "conversation")) {
		if((conv = gf_event_info_get_conversation(info))) {
			win = gaim_conversation_get_window(conv);
		} else if(!g_ascii_strcasecmp(n_type, "chat-invite")) {
			const GHashTable *components = gf_event_info_get_components(info);

			account = gf_event_info_get_account(info);
			conv = gaim_find_conversation_with_account(gf_event_info_get_extra(info),
													   account);

			if(!conv && components)
				serv_join_chat(account->gc, (GHashTable *)components);

			conv = NULL;
			gf_display_destroy(display);
		}
	} else if(!g_ascii_strcasecmp(e_type, "buddy")) {
		GaimBuddy *buddy;

		buddy = gf_event_info_get_buddy(info);
		conv = gaim_conversation_new(GAIM_CONV_IM,
									 gf_event_info_get_account(info),
									 buddy->name);
		win = gaim_conversation_get_window(conv);
	} else if(!g_ascii_strcasecmp(e_type, "account")) {
		if(!g_ascii_strcasecmp(n_type, "warned")) {
			const gchar *target = NULL;

			if((target = gf_event_info_get_target(info))) {
				conv = gaim_conversation_new(GAIM_CONV_IM,
											 gf_event_info_get_account(info),
											 target);
				win = gaim_conversation_get_window(conv);
			}
		}
	}

	if(win && conv) {
		gaim_conv_window_raise(win);
		gaim_conv_window_switch_conversation(win, gaim_conversation_get_index(conv));

		if(GAIM_IS_GTK_WINDOW(win))
			gtk_window_present(GTK_WINDOW(GAIM_GTK_WINDOW(win)->window));

		gf_display_destroy(display);
	}
}

void
gf_action_execute_context(GfDisplay *display, GdkEventButton *gdk_event) {
	GfEvent *event;
	GfEventInfo *info;
	const gchar *e_type;

	g_return_if_fail(display);

	info = gf_display_get_event_info(display);
	event = gf_event_info_get_event(info);
	e_type = gf_event_get_event_type(event);

	if(!g_ascii_strcasecmp(e_type, "buddy")) {
		GtkWidget *menu;
		GaimBuddy *buddy = gf_event_info_get_buddy(info);

		menu = gtk_menu_new();
		gaim_gtk_blist_make_buddy_menu(menu, buddy);
		gtk_widget_show_all(menu);
		gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
					   gdk_event->button, gdk_event->time);
	}

	gf_display_destroy(display);
}

void
gf_action_execute_info(GfDisplay *display, GdkEventButton *gdk_event) {
	GfEvent *event;
	GfEventInfo *info;
	const gchar *e_type, *n_type;
	GaimAccount *account;

	g_return_if_fail(display);

	info = gf_display_get_event_info(display);
	account = gf_event_info_get_account(info);
	event = gf_event_info_get_event(info);

	e_type = gf_event_get_event_type(event);
	n_type = gf_event_get_notification_type(event);

	if(!g_ascii_strcasecmp(e_type, "account")) {
		if(!g_ascii_strcasecmp(n_type, "warned")) {
			serv_get_info(account->gc, gf_event_info_get_target(info));

			gf_display_destroy(display);
		}
	} else if(!g_ascii_strcasecmp(e_type, "buddy")) {
		GaimBuddy *buddy;

		if((buddy = gf_event_info_get_buddy(info)))
			serv_get_info(account->gc, buddy->name);

		gf_display_destroy(display);
	} else if(!g_ascii_strcasecmp(e_type, "conversation")) {
		GaimConversation *conv;

		if((conv = gf_event_info_get_conversation(info)))
			if(gaim_conversation_get_type(conv) == GAIM_CONV_IM)
				serv_get_info(account->gc, gf_event_info_get_target(info));

		/* theres no way to get info about a chat so we just destroy */
		gf_display_destroy(display);
	}
}

void
gf_action_execute_log(GfDisplay *display, GdkEventButton *gdk_event) {
	GfEvent *event;
	GfEventInfo *info;
	GaimAccount *account;
	const gchar *e_type, *n_type;

	g_return_if_fail(display);

	info = gf_display_get_event_info(display);
	account = gf_event_info_get_account(info);
	event = gf_event_info_get_event(info);
	e_type = gf_event_get_event_type(event);
	n_type = gf_event_get_notification_type(event);

	if(!g_ascii_strcasecmp(e_type, "account")) {
		if(!g_ascii_strcasecmp(n_type, "warned")) {
			gaim_gtk_log_show(GAIM_LOG_IM, gf_event_info_get_target(info), account);

			gf_display_destroy(display);
		}
	} else if(!g_ascii_strcasecmp(e_type, "buddy")) {
		GaimBuddy *buddy;

		if((buddy = gf_event_info_get_buddy(info)))
			gaim_gtk_log_show(GAIM_LOG_IM, buddy->name, account);

		gf_display_destroy(display);
	} else if(!g_ascii_strcasecmp(e_type, "conversation")) {
		GaimConversation *conv;

		if((conv = gf_event_info_get_conversation(info))) {
			if(gaim_conversation_get_type(conv) == GAIM_CONV_IM)
				gaim_gtk_log_show(GAIM_LOG_IM, gf_event_info_get_target(info), account);
			else if(gaim_conversation_get_type(conv) == GAIM_CONV_CHAT)
				gaim_gtk_log_show(GAIM_LOG_CHAT, conv->name, account);
		}

		gf_display_destroy(display);
	}
}
