/*
  Guifications - The end all, be all, toaster popup plugin
  Copyright (C) 2003-2004 Gary Kramlich

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#include <glib.h>

#include <debug.h>
#include <account.h>
#include <blist.h>
#include <connection.h>
#include <conversation.h>
#include <prefs.h>

#include "gf_event.h"
#include "gf_event_info.h"

struct _GfEventInfo {
	GfEvent *event;
	GaimAccount *account;

	gint timeout_id;

	GaimBuddy *buddy;
	GaimConversation *conv;
	GaimConvChatBuddyFlags flags;
	gboolean contact;

	gchar *target;
	gchar *message;
	gchar *extra;
	const GHashTable *components;
};

/*******************************************************************************
 * API
 ******************************************************************************/
static void
gf_event_info_free_string(gchar *string) {
	if(string) {
		g_free(string);
		string = NULL;
	}
}

GfEventInfo *
gf_event_info_new(const gchar *notification_type) {
	GfEvent *event;
	GfEventInfo *info;

	g_return_val_if_fail(notification_type, NULL);

	event = gf_event_find_for_notification(notification_type);
	g_return_val_if_fail(event, NULL);

	info = g_new0(GfEventInfo, 1);
	info->event = event;

	return info;
}

void
gf_event_info_destroy(GfEventInfo *info) {
	g_return_if_fail(info);

	info->event = NULL;
	info->account = NULL;
	info->buddy = NULL;
	info->conv= NULL;

	gf_event_info_free_string(info->target);
	gf_event_info_free_string(info->message);
	gf_event_info_free_string(info->extra);

	info->components = NULL;

	if(info->timeout_id)
		g_source_remove(info->timeout_id);

	g_free(info);
	info = NULL;
}

GfEvent *
gf_event_info_get_event(GfEventInfo *info) {
	g_return_val_if_fail(info, GF_EVENT_TYPE_UNKNOWN);

	return info->event;
}

void
gf_event_info_set_account(GfEventInfo *info, GaimAccount *account) {
	g_return_if_fail(info);
	g_return_if_fail(account);

	info->account = account;
}

GaimAccount *
gf_event_info_get_account(GfEventInfo *info) {
	g_return_val_if_fail(info, NULL);

	return info->account;
}

void
gf_event_info_set_buddy(GfEventInfo *info, GaimBuddy *buddy) {
	g_return_if_fail(info);
	g_return_if_fail(buddy);

	info->buddy = buddy;
}

GaimBuddy *
gf_event_info_get_buddy(GfEventInfo *info) {
	g_return_val_if_fail(info, NULL);

	return info->buddy;
}

void
gf_event_info_set_conversation(GfEventInfo *info, GaimConversation *conv) {
	g_return_if_fail(info);
	g_return_if_fail(conv);

	info->conv = conv;
}

GaimConversation *
gf_event_info_get_conversation(GfEventInfo *info) {
	g_return_val_if_fail(info, NULL);

	return info->conv;
}

void
gf_event_info_set_target(GfEventInfo *info, const gchar *target) {
	g_return_if_fail(info);
	g_return_if_fail(target);

	gf_event_info_free_string(info->target);

	info->target = g_strdup(target);
}

const gchar *
gf_event_info_get_target(GfEventInfo *info) {
	g_return_val_if_fail(info, NULL);

	return info->target;
}

void
gf_event_info_set_message(GfEventInfo *info, const gchar *message) {
	g_return_if_fail(info);
	g_return_if_fail(message);

	gf_event_info_free_string(info->message);

	info->message = g_strdup(message);
}

const gchar *
gf_event_info_get_message(GfEventInfo *info) {
	g_return_val_if_fail(info, NULL);

	return info->message;
}

void
gf_event_info_set_extra(GfEventInfo *info, const gchar *extra) {
	g_return_if_fail(info);
	g_return_if_fail(extra);

	gf_event_info_free_string(info->extra);
	info->extra = g_strdup(extra);
}

const gchar *
gf_event_info_get_extra(GfEventInfo *info) {
	g_return_val_if_fail(info, NULL);

	return info->extra;
}

void
gf_event_info_set_components(GfEventInfo *info, const GHashTable *components) {
	g_return_if_fail(info);
	g_return_if_fail(components);

	info->components = components;
}

const GHashTable *
gf_event_info_get_components(GfEventInfo *info) {
	g_return_val_if_fail(info, NULL);

	return info->components;
}

void
gf_event_info_set_timeout_id(GfEventInfo *info, gint timeout_id) {
	g_return_if_fail(info);

	info->timeout_id = timeout_id;
}

gint
gf_event_info_get_timeout_id(GfEventInfo *info) {
	g_return_val_if_fail(info, -1);

	return info->timeout_id;
}

void
gf_event_info_set_conv_chat_buddy_flags(GfEventInfo *info, GaimConvChatBuddyFlags flags) {
	g_return_if_fail(info);

	info->flags = flags;
}

GaimConvChatBuddyFlags
gf_event_info_get_chat_buddy_flags(GfEventInfo *info) {
	g_return_val_if_fail(info, GAIM_CBFLAGS_NONE);

	return info->flags;
}

void
gf_event_info_set_is_contact(GfEventInfo *info, gboolean value) {
	g_return_if_fail(info);

	info->contact = value;
}

gboolean
gf_event_info_get_is_contact(GfEventInfo *info) {
	g_return_val_if_fail(info, FALSE);

	return info->contact;
}
