/*****************************************************************************
 * aliases.c
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * Copyright (C) 2000-2002 Chris Pinkham
 * Released under the terms of the GPL.
 * *NO WARRANTY*
 *
 * cpinkham@corp.infi.net, cpinkham@bc2va.org
 * http://www4.infi.net/~cpinkham/gyach/
 *****************************************************************************/

#include "config.h"

#include <sys/stat.h>

#include "gyach.h"
#include "aliases.h"
#include "commands.h"
#include "interface.h"
#include "main.h"
#include "support.h"
#include "users.h"
#include "util.h"
#include "yahoochat.h"

int  alias_count;
int  alias_selected = -1;

GtkWidget *chat_aliases;
GtkWidget *alias_menu;
GtkWidget *alias_window;
GHashTable *comment_hash = NULL;

#define GYACH_CMD_SEP "<;>"

/*********************************/
/* Default aliases list */
/*********************************/
struct ALIAS {
	char *alias_name;
	char *cmd_value1;
	char *cmd_value2;
};

static struct ALIAS aliases[] = {
	{ "Agree",	"agrees whole-heartedly.", "agrees with $0." },
	{ "Apologize",	"apologizes.", "apologizes to $0." },
	{ "Applaud",	"applauds fervently.", "applauds $0 fervently hmmm." },
	{ "Beam",	"beams.", "beams at $0." },
	{ "Beg",	"begs like a dog.", "begs $0." },
	{ "Blush",	"blushes in embarassment.", "blushes at $0 in embarassment." },
	{ "Bow",	"bows gracefully.", "bows to $0 gracefully." },
	{ "Cheer",	"cheers enthusiastically.", "cheers $0 enthusiastically." },
	{ "Clap",	"claps briefly.", "claps for $0 briefly." },
	{ "Disagree",	"disagrees.", "disagrees with $0." },
	{ "Faint",	"faints dead away.", "faints dead away at the sight of $0's." },
	{ "Flirt",	"flirts.", "flirts with $0." },
	{ "Grin",	"grins evilly.", "grins at $0 evilly." },
	{ "Grovel",	"grovels.", "grovels at $0's feet." },
	{ "Hug",	"waves Hi to everyone.", "gives $0 a great big bear hug." },
	{ "Kiss",	"kisses.", "kisses $0." },
	{ "Laugh",	"laughs hysterically.", "laughs at $0." },
	{ "Listen",	"listens.", "listens to $0 hanging on her every word." },
	{ "Melt",	"melts.", "melts in $0's arms." },
	{ "Misgrin",	"grins mischievously.", "grins at $0 mischievously." },
	{ "Nod",	"nods solemnly.", "nods to $0 approvingly." },
	{ "Nudge",	"nudges everyone in the side.", "nudges $0 in the side." },
	{ "Pout",	"pouts.", "pouts at $0." },
	{ "Sit",	"sits down.", "sits down next to $0." },
	{ "Smile",	"smiles.", "smiles at $0." },
	{ "Stand",	"stands up.", "stands up next to $0." },
	{ "Wave",	"waves at everyone.", "waves at $0." },
	{ NULL, NULL, NULL }
};

/* function to remove comment from comment hash */
static gboolean uncomment(gpointer key, gpointer value,
		gpointer user_data) {
	g_free( key );
	g_free( value );

	return( TRUE );
}

/*
 * alias datafile fmt is "alias:everyone|individual"
 * see http://www.edge-op.org/files/chatstandard for more info
 */
void read_aliases() {
	gchar *listitem[3];
	FILE *fp;
	struct ALIAS *alias;
	char line[1024];
	char tmp_line[2048];
	char comment[2048] = "";
	char *alias_name = NULL;
	char *cmd_value1 = NULL;
	char *cmd_value2 = NULL;
	char *ptr = NULL;
	char *ptr2 = NULL;
	char alias_filename[1024] = "";
	char empty_value[1] = "";
	int row;
	int old_format = 0;

	DBG( 1, "read_aliases()\n" );

	if ( comment_hash ) {
		g_hash_table_foreach_remove( comment_hash, uncomment, NULL );
	} else {
		comment_hash = g_hash_table_new( g_str_hash, g_str_equal );
	}

	/*
	gtk_clist_set_sort_column(GTK_CLIST(chat_aliases), 0 );
	gtk_clist_set_auto_sort(GTK_CLIST(chat_aliases), 1 );
	*/
	gtk_clist_set_auto_sort(GTK_CLIST(chat_aliases), 0 );

	gtk_clist_freeze(GTK_CLIST(chat_aliases));

	/* remove any items already in the list */
	while(1) {
		if ( !gtk_clist_get_text(GTK_CLIST(chat_aliases), 0, 0, &alias_name )) {
			break;
		}
		gtk_clist_remove( GTK_CLIST(chat_aliases), 0 );
	}

	listitem[0] = alias_name;
	listitem[1] = cmd_value1;
	listitem[2] = cmd_value2;
	alias_name = line;

	sprintf( alias_filename, "%s/commands", GYACH_CFG_COMMON_DIR );

	fp = fopen( alias_filename, "r" );

	alias_count = 0;

	if ( fp ) {
		while( fgets( line, 1023, fp )) {
			ptr = skip_whitespace( line );
			if ( *ptr == '#' ) {
				strcat( comment, line );
				continue;
			}

			if (( ! strchr( line, ']' )) &&
				( ! strchr( line, ':' ))) {
				continue;
			}

			while(( line[strlen(line)-1] == '\r' ) ||
				  ( line[strlen(line)-1] == '\n' )) {
				line[strlen(line)-1] = '\0';
			}

			/* stuff to convert from cheetachat format I think */
			/* should be able to delete this */
			while(( ptr = strstr( line, "%N" )) != NULL ) {
				*ptr = '$';
				*(ptr+1) = '0';
			}

			alias_count++;

			ptr = line;
			while(( *ptr != ']' ) &&
				( *ptr != ':' )) {
				*ptr = tolower( *ptr );
				ptr++;
			}

			if ( ! old_format ) {
				ptr = strchr( line, ':' );
				ptr2 = strchr( line, ']' );
				if (( ptr ) &&
					( ptr2 ) &&
					( ptr2 < ptr )) {
					/* a line with a ] separator that also contains a : */
					ptr = NULL;
				}
			} else {
				ptr = NULL;
			}
			if ( ! ptr ) {
				/* assume old "command]value|value" format */
				/* convert to new format (ie, in order to emote need a ':') */
				old_format = 1;
				ptr = strchr( line, ']' );
				if ( ! ptr ) {
					continue;
				}
				strncpy( tmp_line, line, ptr - line );
				tmp_line[ptr - line] = '\0';
				if (( *(ptr+1) == '!' ) ||
					( *(ptr+1) == ':' ) ||
					( *(ptr+1) == '/' )) {
					strcat( tmp_line, ":" );
				} else {
					strcat( tmp_line, "::" );
				}
				strcat( tmp_line, ptr + 1 );
				strcpy( line, tmp_line );
				ptr = strchr( line, ':' );
			}
			*ptr = '\0';

			if (( comment[0] ) &&
				( comment_hash )) {
				if ( ! g_hash_table_lookup( comment_hash, alias_name )) {
					g_hash_table_insert( comment_hash, g_strdup( alias_name ),
						g_strdup( comment ));
				}
				comment[0] = '\0';
			}

			cmd_value1 = ptr + 1;
			cmd_value1 = skip_whitespace( cmd_value1 );
			ptr++;

			ptr = strchr( cmd_value1, '|' );
			if ( ptr ) {
				*ptr = '\0';
				if ( old_format ) {
					strncpy( tmp_line, line, ptr - line );
					tmp_line[ptr - line] = '\0';
					strcat( tmp_line, "|:" );
					strcat( tmp_line, ptr + 1 );
					ptr = strchr( tmp_line, '|' );
				}
				cmd_value2 = ptr + 1;
				cmd_value2 = skip_whitespace( cmd_value2 );
			} else {
				/* leave 2nd part value empty */
				cmd_value2 = empty_value;
			}

			if (( ! strlen( cmd_value1 )) &&
				( strlen( cmd_value2 ))) {
				cmd_value1 = cmd_value2;
				cmd_value2 = empty_value;
			}

			if ( strlen( cmd_value1 ) > 511 ) {
				cmd_value1[511] = '\0';
			}

			if ( strlen( cmd_value2 ) > 511 ) {
				cmd_value2[511] = '\0';
			}

			listitem[0] = alias_name;
			row = gtk_clist_append(GTK_CLIST(chat_aliases), listitem );
			gtk_clist_set_text(GTK_CLIST(chat_aliases), row, 1, cmd_value1 );
			gtk_clist_set_text(GTK_CLIST(chat_aliases), row, 2, cmd_value2 );
		}
		fclose( fp );
	} else {
		/* no alias file so set defaults */
		alias = &aliases[0];
		while( alias->alias_name ) {
			listitem[0] = alias->alias_name;
			row = gtk_clist_append(GTK_CLIST(chat_aliases), listitem );
			gtk_clist_set_text(GTK_CLIST(chat_aliases), row, 1,
				alias->cmd_value1 );
			gtk_clist_set_text(GTK_CLIST(chat_aliases), row, 2,
				alias->cmd_value2 );
			
			alias++;
		}

		/* save our default set to the file */
		save_aliases();
	}

	gtk_clist_thaw(GTK_CLIST(chat_aliases));
}

void save_aliases() {
	gchar *alias_name;
	gchar *cmd_value1;
	gchar *cmd_value2;
	char *comment;
	char alias_filename[1024] = "";
	int i = 0;
	FILE *fp;

	DBG( 1, "save_aliases()\n" );

	sprintf( alias_filename, "%s/commands", GYACH_CFG_COMMON_DIR );

	fp = fopen( alias_filename, "wb" );

	if ( fp ) {
		while(1) {
			if ( !gtk_clist_get_text(GTK_CLIST(chat_aliases), i, 0,
					&alias_name)) {
				break;
			}

			gtk_clist_get_text(GTK_CLIST(chat_aliases), i, 1, &cmd_value1 );
			gtk_clist_get_text(GTK_CLIST(chat_aliases), i, 2, &cmd_value2 );

			if (( comment_hash ) &&
				(( comment = g_hash_table_lookup( comment_hash,
					alias_name )) != NULL )) {
				fprintf( fp, "%s", comment );
			}
			fprintf( fp, "%s:", alias_name );
			if ( strlen( cmd_value1 )) {
				fprintf( fp, "%s", cmd_value1 );
			}
			if (( strlen( cmd_value2 )) &&
				(  strcmp( cmd_value1, cmd_value2 ))) {
				if ( strlen( cmd_value1 ))
					fprintf( fp, "|" );
				fprintf( fp, "%s", cmd_value2 );
			}
			fprintf( fp, "\n" );
			
			i++;
		}
		fclose( fp );
	}
}

int alias_find( char *alias ) {
	gchar *alias_ptr;
	int i;
	int found = -1;

	DBG( 11, "alias_find( '%s' )\n", alias );

	i = 0;
	while( 1 ) {
		if ( !gtk_clist_get_text(GTK_CLIST(chat_aliases), i, 0, &alias_ptr ))
			break;

		if ( ! strcasecmp( alias, alias_ptr )) {
			found = i;
			break;
		}

		i++;
	}

	DBG( 12, "RETURN alias_find() == %d\n", found );

	return( found );
}

void edit_alias() {
	gchar *listitem[3];
	gchar alias_name[128] = "";
	gchar cmd_value1[512] = "";
	gchar cmd_value2[512] = "";
	GtkWidget *w_alias_name;
	GtkWidget *w_cmd_value1;
	GtkWidget *w_cmd_value2;

	listitem[0] = alias_name;
	listitem[1] = cmd_value1;
	listitem[2] = cmd_value2;

	DBG( 11, "edit_alias()\n" );

	if ( alias_selected > -1 ) {
		gtk_clist_get_text( GTK_CLIST(chat_aliases), alias_selected, 0,
			listitem );
		gtk_clist_get_text( GTK_CLIST(chat_aliases), alias_selected, 1,
			listitem+1 );
		gtk_clist_get_text( GTK_CLIST(chat_aliases), alias_selected, 2,
			listitem+2 );
	}

    if ( ! alias_window ) {
		alias_window = create_alias_window();
		gtk_widget_show( alias_window );
	}

	w_alias_name = lookup_widget( alias_window, "alias_name" );
	w_cmd_value1 = lookup_widget( alias_window, "cmd_value1" );
	w_cmd_value2 = lookup_widget( alias_window, "cmd_value2" );

	gtk_entry_set_text( GTK_ENTRY(w_alias_name), listitem[0] );
	gtk_entry_set_text( GTK_ENTRY(w_cmd_value1), listitem[1] );
	gtk_entry_set_text( GTK_ENTRY(w_cmd_value2), listitem[2] );
}

int check_alias( char *alias, char *args )
{
	gchar *listitem[3];
	gchar *alias_name;
	gchar *cmd_value_to_use;
	gchar *cmd_value1;
	gchar *cmd_value2;
	u_char  tmp[4096] = "";
	u_char  tmp_alias[21];
	char *ptr;
	char *end;
	int i, x;
	int found = -1;
	regex_t test_regex;

	DBG( 21, "check_alias( %p, '%s', '%s' )\n", sess, alias, args );

	listitem[0] = alias_name;
	listitem[1] = cmd_value1;
	listitem[2] = cmd_value2;

	for( i = 0; i < strlen( alias ); i++ ) {
		alias[i] = tolower( alias[i] );
	}

	i = 0;
	while(1) {
		if ( !gtk_clist_get_text(GTK_CLIST(chat_aliases), i, 0, &alias_name ))
			break;

		strcpy( tmp_alias, alias_name );
		for( x = 0; x < strlen( tmp_alias ); x++ ) {
			tmp_alias[x] = tolower( tmp_alias[x] );
		}

		if ( ! strcmp( alias, tmp_alias )) {
			alias_name = tmp_alias;
			found = i;
			break;
		}

		i++;
	}

	if ( found >= 0 ) {
		gtk_clist_get_text(GTK_CLIST(chat_aliases), found, 1,
			&cmd_value1 );
		gtk_clist_get_text(GTK_CLIST(chat_aliases), found, 2,
			&cmd_value2 );

		regcomp( &test_regex, "(^%s|[^\\]%s|^\\$\\*|[^\\]\\$\\*|^\\$[0-9]|[^\\]\\$[0-9])",
			REG_EXTENDED | REG_ICASE | REG_NOSUB );

		cmd_value_to_use = cmd_value1;
		if ( args[0] ) {
			if ( regexec( &test_regex, cmd_value1, 0, NULL, 0 )) {
				cmd_value_to_use = cmd_value2;
			}
		} else {
			if ( ! regexec( &test_regex, cmd_value1, 0, NULL, 0 )) {
				cmd_value_to_use = cmd_value2;
			}
		}
		regfree( &test_regex );

		if ( args[0] ) {
			strcpy( tmp, cmd_value_to_use );

			if (( strchr( tmp, '$' )) ||
				( strstr( tmp, "%s" ))) {
				strcpy( tmp, replace_args( tmp, args ));
				args[0] = '\0';
			}

			cmd_value_to_use = tmp;
		}

		ptr = cmd_value_to_use;
		end = strstr( cmd_value_to_use, GYACH_CMD_SEP );
		if ( end ) {
			while( end ) {
				*end = '\0';
				chat_command( ptr );

				ptr = end + strlen( GYACH_CMD_SEP );
				end = strstr( ptr, GYACH_CMD_SEP );
			}
		}
		chat_command( ptr );

		DBG( 22, "RETURN check_alias() == TRUE\n" );

		return( TRUE );
	} else {
		DBG( 22, "RETURN check_alias() == FALSE\n" );

		return( FALSE );
	}
}

void append_aliases_to_list( GList *list ) {
	gchar *alias_name;
	u_char  tmp_alias[21];
	int i;
	int x;

	i = 0;
	while(1) {
		if ( !gtk_clist_get_text(GTK_CLIST(chat_aliases), i, 0, &alias_name ))
			break;

		strcpy( tmp_alias, "/" );
		strcat( tmp_alias, alias_name );
		for( x = 0; x < strlen( tmp_alias ); x++ ) {
			tmp_alias[x] = tolower( tmp_alias[x] );
		}

		g_list_append( list, strdup( tmp_alias ));
		i++;
	}
}
