/*****************************************************************************
 * commands.c
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * Copyright (C) 2000-2002 Chris Pinkham
 * Released under the terms of the GPL.
 * *NO WARRANTY*
 *
 * cpinkham@corp.infi.net, cpinkham@bc2va.org
 * http://www4.infi.net/~cpinkham/gyach/
 *****************************************************************************/

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/utsname.h>
#include <sys/wait.h>
#include <signal.h>
#include <time.h>

#include "config.h"

#ifdef USE_PTHREAD_CREATE
#include <pthread.h>
#endif

#include "gyach.h"
#include "aliases.h"
#include "callbacks.h"
#include "commands.h"
#include "friends.h"
#include "gyach_int.h"
#include "history.h"
#include "ignore.h"
#include "images.h"
#include "interface.h"
#include "main.h"
#include "support.h"
#include "users.h"
#include "util.h"
#include "yahoochat.h"

char *find_whitespace( char *str ) {
	char *ptr = str;

	while(( *ptr ) && ( ! isspace( *ptr )))
		ptr++;

	return( ptr );
}

char *skip_whitespace( char *str ) {
	char *ptr = str;

	while(( *ptr ) && ( isspace( *ptr )))
		ptr++;

	return( ptr );
}

void cmd_join( char *room ) {
	my_strncpy( ymsg_sess->room, room, YMSG9_ROOM_SIZE );
	ymsg9_join( ymsg_sess );
}

void cmd_goto( char *user ) {
	ymsg9_goto( ymsg_sess, user );
}

void cmd_follow( char *user ) {
	char buf[129] = "";
	int r;

	if ( *user ) {
		/* check to make sure we don't want to display who we're following */
		if ( strcmp( "?", user )) {
			if ( !strcmp( user, ymsg_sess->user )) {
				snprintf( buf, sizeof(buf),
					"%sYou can't follow yourself!%s\n",
					YAHOO_COLOR_RED, YAHOO_COLOR_BLACK );
				append_to_textbox( chat_window, NULL, buf );
				return;
			}

			r = find_user_row( user );
			if ( r < 0 ) {
				snprintf( buf, sizeof(buf),
					"User %s%s%s is not in this room, can not follow.\n",
					YAHOO_COLOR_BLUE, user, YAHOO_COLOR_BLACK );
				append_to_textbox( chat_window, NULL, buf );
				return;
			}

			if ( follow_user ) {
				free( follow_user );
			}
			follow_user = strdup( user );
		}
	} else {
		if ( follow_user ) {
			free( follow_user );
		}
		follow_user = NULL;
	}

	if ( follow_user ) {
		snprintf( buf, sizeof(buf), "Following user: %s%s%s\n",
			YAHOO_COLOR_BLUE, follow_user, YAHOO_COLOR_BLACK );
	} else {
		snprintf( buf, sizeof(buf), "%sNot currently following anyone.%s\n",
			YAHOO_COLOR_RED, YAHOO_COLOR_BLACK );
	}
	append_to_textbox( chat_window, NULL, buf );
}

void cmd_invite( char *user ) {
	ymsg9_invite( ymsg_sess, user, ymsg_sess->room );
}

void cmd_reject_buddy( char *buddy ) {
	char buf[512] = "";
	ymsg9_reject_buddy( ymsg_sess, buddy );
	snprintf( buf, 511, "%s%sRevoking buddy list priviledges for %s%s%s.",
		YAHOO_STYLE_ITALICON, YAHOO_COLOR_RED, buddy,
		YAHOO_STYLE_ITALICOFF, YAHOO_COLOR_BLACK );
}

void cmd_cls() {
	ct_clear();
}

void cmd_tell( char *args ) {
	char buf[128];
	char *ptr;

	args = skip_whitespace( args );
	ptr = args;

	if ( strlen( ptr )) {
		/* they gave a user & message so send message */
		/* format is "tell dummytext\001user\001message" */
		ptr = find_whitespace( ptr );
		*ptr = '\0';
		ptr++;

		ymsg9_pm( ymsg_sess, args, ptr );
	} else {
		/* no user or message supplied so open window */
		/* FIXME, should we delete this?? */
		if ( ! pm_window ) {
			pm_window = create_pm_window();
			gtk_widget_show( pm_window );
		} else {
			strcpy( buf, "***Unable to open PM window, already open.***\n" );
			append_to_textbox( chat_window, NULL, buf );
		}
	}
}

void cmd_away( char *msg ) {
	GtkWidget *tmp_widget;
	GtkWidget *cw = chat_window;
	char buf[128];
	int len;

	len = strlen( msg );

	if ( len == 0 ) {
		if (( custom_away_message ) &&
			( strlen( custom_away_message )))
			msg = custom_away_message;
		else
			msg = "Be Right Back";
		my_status = 10;
	} else if (( len == 1 ) &&
		( isdigit( msg[0] ))) {
		my_status = atoi( msg );
	} else if (( len == 2 ) &&
		( isdigit( msg[0] )) &&
		( isdigit( msg[1] ))) {
		my_status = atoi( msg );
	} else {
		my_status = 10;
	}

	if ( my_status ) {
		chatter_list_status( ymsg_sess->user, pixmap_status_away, "AW" );
	} else {
		chatter_list_status( ymsg_sess->user, pixmap_status_here, "" );
	}

	if (( my_status <= 9 ) ||
		( my_status == 11 ) ||
		( my_status == 12 )) {
		if ( my_status ) {
			snprintf( buf, sizeof(buf), "%02d:", my_status );
			gyach_away( ymsg_sess, buf );
		} else {
			gyach_back( ymsg_sess );
		}

		switch( my_status ) {
			case 0: tmp_widget = lookup_widget( cw, "status_here" );
					break;
			case 1: tmp_widget = lookup_widget( cw, "status_be_right_back" );
					break;
			case 2: tmp_widget = lookup_widget( cw, "status_busy" );
					break;
			case 3: tmp_widget = lookup_widget( cw, "status_not_at_home" );
					break;
			case 4: tmp_widget = lookup_widget( cw, "status_not_at_my_desk" );
					break;
			case 5: tmp_widget = lookup_widget( cw, "status_not_in_the_office");
					break;
			case 6: tmp_widget = lookup_widget( cw, "status_on_the_phone" );
					break;
			case 7: tmp_widget = lookup_widget( cw, "status_on_vacation" );
					break;
			case 8: tmp_widget = lookup_widget( cw, "status_out_to_lunch" );
					break;
			case 9: tmp_widget = lookup_widget( cw, "status_stepped_out" );
					break;
			case 11: tmp_widget = lookup_widget( cw, "status_autoaway" );
					break;
			case 12: tmp_widget = lookup_widget( cw, "status_invisible" );
					break;
			default: tmp_widget = lookup_widget( cw, "status_here" );
		}

		gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(tmp_widget), 1 );
	} else {
		snprintf( buf, sizeof(buf), "10:%s", msg );
		gyach_away( ymsg_sess, buf );

		tmp_widget = lookup_widget( cw, "status_custom" );
		gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(tmp_widget), 1 );
	}
}

void cmd_auto_away( char *value ) {
	char buf[129];

	if ( ! value[0] )
		return;

	auto_away= atoi( value );

	if ( ! auto_away)
		auto_away = 15;

	auto_away_time = time(NULL) + ( auto_away * 60 );

	write_config();

	sprintf( buf, "*** Auto-Away timer set to %d minute(s) ***\n", auto_away );
	append_to_textbox( chat_window, NULL, buf );
}

void cmd_think( char *msg ) {
	char buf[2048];

	my_strncpy( buf, msg, 2048 );
	if( send_avatar ) {
		my_strncat( buf, AVATAR_START, 2048 );
		my_strncat( buf, send_avatar, 2048 );
		my_strncat( buf, AVATAR_SEND_END, 2048 );
#ifndef OS_WINDOWS
		if ( show_avatars ) {
			display_avatar( ymsg_sess->user, send_avatar );
		}
#endif
	}
	gyach_think( ymsg_sess, buf );
}

void cmd_emote( char *msg ) {
	char buf[2048];

	my_strncpy( buf, msg, 2048 );
	if( send_avatar ) {
		my_strncat( buf, AVATAR_START, 2048 );
		my_strncat( buf, send_avatar, 2048 );
		my_strncat( buf, AVATAR_SEND_END, 2048 );
#ifndef OS_WINDOWS
		if ( show_avatars ) {
			display_avatar( ymsg_sess->user, send_avatar );
		}
#endif
	}
	
	if ( my_status )
		cmd_away( "0" );
	gyach_emote( ymsg_sess, buf );
}

void cmd_roll( char *arg ) {
	char buf[129];

	/* FIXME */
	strcpy( buf,
		"*** /roll is unimplemented in gyach since switching to YMSG ***\n" );
	append_to_textbox( chat_window, NULL, buf );
}

void cmd_connect() {
	on_connect_activate( NULL, NULL );
}

void cmd_disconnect() {
	on_disconnect_activate( NULL, NULL );
}

void cmd_reconnect() {
	on_disconnect_activate( NULL, NULL );
	login_to_yahoo_chat();
}

void cmd_help( int full ) {
	char buf[4096] = "";

	strcat( buf,
		"\n"
		"-----------------------------------------------"
			"---------------\n"
		YAHOO_STYLE_BOLDON "Gyach v" VERSION YAHOO_STYLE_BOLDOFF
			" - Copyright 2000-2002 Chris T. Pinkham, ALL RIGHTS RESERVED\n"
		"    " YAHOO_STYLE_URLON GYACH_URL YAHOO_STYLE_URLOFF "\n"
		"\n"
		"Chat Commands:\n"
		);
	append_to_textbox_color( chat_window, NULL, buf );

	strcpy( buf, "" );
	if ( full )
		strcat( buf,
		"  /autoaway [min]             set auto-away timer to 'min' minutes\n"
		"  /alias [alias] [text]       create an alias to specified text\n"
		);

	strcat( buf, 
		"  /away [msg]                 go away with custom auto-away message\n"
		"  /back                       come back from away status\n"
		"  /clear                      clear chat text window\n"
		);

	
	if ( full )
		strcat( buf,
		"  /connect                    open login window\n"
		"  /disconnect                 disconnect from chat\n"
		);

	strcat( buf, 
		"  /find [text]                search in chat window for text\n"
		"  /follow [user]              follow user\n"
		"  /follow                     stop following\n"
		"  /friends                    list yahoo buddy list\n"
		"  /friend [user]              add/remove user to buddy/friend list\n"
		"  /goto [user]                goto the user\n"
		"  /help                       short command help screen\n"
		"  /help_full                  full command help screen\n"
		"  /history                    display command history\n"
		"  /ignore [user]              ignore/un-ignore user\n"
		"  /invite [user]              invite the user to the current room\n"
		);
	
	if ( full )
		strcat( buf,
		"  /join [room]                join the room\n"
		"  /last                       list last 10 users to leave room\n"
		"  /last_comments              list last comment/action for each user\n"
		);

	strcat( buf,
		"  /mail                       open browser window to Yahoo! Mail\n"
		"  /mute [user]                mute/un-mute user for this session\n"
		);

	if ( full )
		strcat( buf,
		"  /online                     print time online\n"
		);

	strcat( buf,
		"  /profile [user]             show user's profile\n"
		"  /quit\n"
		);
	
	if ( full )
		strcat( buf,
		"  /reconnect                  disconnect then auto-reconnect\n"
		);

	strcat( buf,
		"  /regex [regex]              add/remove regex to auto-ignore list\n"
		"  /regex_test \"text\" [regex]  test the specified text (quotes\n"
		"                                  required) against given regex\n"
		);

	if ( full )
		strcat( buf,
		"  /reload                     reload ignore and alias lists\n"
		"  /revoke [user]              revoke buddy list priviledges for user\n"
		);

	strcat( buf,
		"  /tell [user] [text]         send PM to user\n"
		"  /think [text]               . o O( think text )\n"
		);

	if ( full )
		strcat( buf,
		"  /unalias [alias]            delete alias\n"
		"  /unignore [user]            un-ignore user\n"
		"  /url [url]                  paste url as a clickable link\n"
		);

	strcat( buf,
		"  /ver                        paste Gyach version (" VERSION ") and url\n"
		"  : [text]                    send text as an emote\n"
		"  ! [command]                 execute command and post results\n"
		"  !! [command]                execute command and displayresults locally\n"
		);

	if ( ! full ) {
		strcat( buf,
		"\n"
		"   *** For a complete list of commands, type /help_full\n"
		);
	} else {
		strcat( buf,
		"\n"
		"  Shortcuts for commonly used commands:\n"
		"  /aw  - /away              /inv  - /invite /call\n"
		"  /al  - /alias             /j    - /join\n"
		"  /cl  - /clear /cls        /l    - /last\n"
		"  /f   - /find              /pm   - /tell\n"
		"  /fr  - /friend /buddy     /pr   - /profile\n"
		"  /g   - /goto              /q    - /quit /exit\n"
		"  /ig  - /ignore            /v    - /ver\n"
		);
	}

	strcat( buf,
		"-----------------------------------------------------------\n"
		"\n"
		);

	ct_append_fixed( buf, strlen( buf ));
	ct_scroll_to_bottom();
}

/* fixme, fill in the rest of this function */
void cmd_friend( char *arg ) {
	char *ptr;
	char *end;
	char buf[128];

	ptr = skip_whitespace( arg );
	end = find_whitespace( ptr );

	while( *end ) {
		*end = '\0';
		if ( find_friend( ptr )) {
			remove_friend( ptr );
			snprintf( buf, 128, "%sBuddy:%s %s%s removed as friend.\n",
				YAHOO_COLOR_BLUE, YAHOO_COLOR_RED, ptr, YAHOO_COLOR_BLACK );
			append_to_textbox_color( chat_window, NULL, buf );
		} else {
			add_friend( ptr );
			snprintf( buf, 128, "%sBuddy:%s %s%s added as friend.\n",
				YAHOO_COLOR_BLUE, YAHOO_COLOR_RED, ptr, YAHOO_COLOR_BLACK );
			append_to_textbox_color( chat_window, NULL, buf );
		}

		ptr = end + 1;
		ptr = skip_whitespace( ptr );
		end = find_whitespace( ptr );
	}

	if ( find_friend( ptr )) {
		remove_friend( ptr );
		snprintf( buf, 128, "%sBuddy:%s %s%s removed as friend.\n",
			YAHOO_COLOR_BLUE, YAHOO_COLOR_RED, ptr, YAHOO_COLOR_BLACK );
		append_to_textbox_color( chat_window, NULL, buf );
	} else {
		add_friend( ptr );
		snprintf( buf, 128, "%sBuddy:%s %s%s added as friend.\n",
			YAHOO_COLOR_BLUE, YAHOO_COLOR_RED, ptr, YAHOO_COLOR_BLACK );
		append_to_textbox_color( chat_window, NULL, buf );
	}
}

void cmd_mail() {
	char buf[1024] = "";
	char *ptr;
#ifdef USE_PTHREAD_CREATE
	pthread_t prof_thread;
#endif

	if ( strcmp( mail_user, ymsg_sess->user )) {
		my_strncpy( mail_user, ymsg_sess->user, 64 );
		snprintf( buf, 1024,
			"https://login.yahoo.com/config/login?.src=ym&.v=0"
			"&.u=30d2s70uso5fa&.last=&promo=&.intl=us&.bypass=&.partner="
			"&.done=&login=%s&passwd=%s",
			ymsg_sess->user, ymsg_sess->password );
		ptr = strdup( buf );
	} else {
		ptr = strdup( "http://mail.yahoo.com" );
	}

#ifdef USE_PTHREAD_CREATE
	pthread_create( &prof_thread, NULL, display_url, (void *)ptr );
#else
	display_url( (void *)ptr );
#endif
}

void cmd_profile( char *user ) {
	gtk_timeout_add( 5, (void *)view_profile, (gpointer)user );
}

void cmd_online( void ) {
	char buf[1025] = "";
	time_t current_time;
	int days = 0;
	int hours = 0;
	int minutes = 0;
	int seconds = 0;

	if ( connect_time ) {
		current_time = time(NULL);

		hours = (int)(( current_time - connect_time ) / 3600);
		minutes = (int)(( current_time - connect_time ) % 3600 ) / 60;
		seconds = (int)(( current_time - connect_time ) % 60 );
		if (hours) {
			days = (int)(hours / 24);
			hours -= (days * 24);
		}

		sprintf( buf, "*** online for %d days, %d:%02d:%02d ***\n",
			days, hours, minutes, seconds );
		append_to_textbox_color( chat_window, NULL, buf );
	} else {
		strcpy( buf, "*** You don't appear to be online at the moment ***\n" );
		append_to_textbox_color( chat_window, NULL, buf );
	}
}

void cmd_url( char *url ) {
	char buf[1025] = "";
	time_t current_time;
	int days = 0;
	int hours = 0;
	int minutes = 0;
	struct utsname un;
	char beta[6] = "beta/";

	if ( *url ) {
		my_strncat( buf, YAHOO_STYLE_URLON, 1024 );
		my_strncat( buf, url, 1024 );
		my_strncat( buf, YAHOO_STYLE_URLOFF, 1024 );
		chat_command( buf );
	} else {
		if ( ! strstr( VERSION, "-pre" )) {
			beta[0] = '\0';
		}
		current_time = time(NULL);

		hours = (int)(( current_time - connect_time ) / 3600);
		minutes = (int)(( current_time - connect_time ) % 3600 ) / 60;
		if (hours) {
			days = (int)(hours / 24);
			hours -= (days * 24);
		}

		uname( &un );

		if ( connect_time ) {
			snprintf( buf, 1024,
				"%s%sGyach%s v%s %s%s%s%s online for %d days, %d:%02d under %s %s [%s]",
				YAHOO_COLOR_BLACK,
				YAHOO_STYLE_BOLDON, YAHOO_STYLE_BOLDOFF, VERSION,
				YAHOO_STYLE_URLON, GYACH_URL, beta, YAHOO_STYLE_URLOFF,
				days, hours, minutes, un.sysname, un.release, un.machine );
			chat_command( buf );
		} else {
			snprintf( buf, 1024,
				"%s%sGyach%s v%s%s %s%s%s (currently OFFline) under %s %s [%s]\n",
				YAHOO_COLOR_BLACK,
				YAHOO_STYLE_BOLDON, YAHOO_STYLE_BOLDOFF, VERSION,
				YAHOO_STYLE_URLON, GYACH_URL, beta, YAHOO_STYLE_URLOFF,
				un.sysname, un.release, un.machine );
			append_to_textbox( chat_window, NULL, buf );
		}

	}
}

void cmd_alias( char *args ) {
	char tmp_str[257];
	char *ptr;
	char *alias;
	char *def;
	int found;
	gchar *listitem[3];
	gchar tmp_item[12] = "tmp";
	gchar tmp_item2[12] = "tmp";
	gchar tmp_item3[12] = "tmp";

	listitem[0] = tmp_item;
	listitem[1] = tmp_item2;
	listitem[2] = tmp_item3;

	DBG( 11, "cmd_alias( '%s' )\n", args );

	my_strncpy( tmp_str, args, sizeof( tmp_str ));
	ptr = tmp_str;
	ptr = skip_whitespace( ptr );

	/* find alias string and null-terminate it */
	alias = ptr;
	ptr = find_whitespace( ptr );
	*ptr = '\0';

	/* find args string */
	ptr++;
	ptr = skip_whitespace( ptr );
	def = ptr;

	found = alias_find( alias );

	if ( found == -1 ) {
		/* new alias so add item*/
		found = gtk_clist_append(GTK_CLIST(chat_aliases), listitem );

		my_strncpy( tmp_item, alias, sizeof( tmp_item ));
		gtk_clist_set_text(GTK_CLIST(chat_aliases), found, 0, tmp_item );
	}

	/* now just set data for item */
	gtk_clist_set_text(GTK_CLIST(chat_aliases), found, 1, def );
	gtk_clist_set_text(GTK_CLIST(chat_aliases), found, 2, def );
	gtk_clist_sort(GTK_CLIST(chat_aliases));

	/* now save list */
	save_aliases();

	DBG( 12, "RETURN cmd_alias()\n" );
}

void cmd_unalias( char *args ) {
	char tmp_str[257];
	char *ptr;
	char *alias;
	int found;

	DBG( 11, "cmd_unalias( '%s' )\n", args );

	strcpy( tmp_str, args );
	ptr = tmp_str;
	ptr = skip_whitespace( ptr );

	/* find alias string and null-terminate it */
	alias = ptr;
	ptr = find_whitespace( ptr );
	*ptr = '\0';

	found = alias_find( alias );

	if ( found != -1 ) {
		/* delete the alias */
		gtk_clist_remove( GTK_CLIST(chat_aliases), found );

		/* now save list */
		save_aliases();
	}

	DBG( 12, "RETURN cmd_unalias()\n" );
}

void cmd_last() {
	char buf[128];
	GList *this_user;

	strcpy( buf, "--- Last users to leave the room ---\n" );
	append_to_textbox( chat_window, NULL, buf );

	this_user = left_list;
	while( this_user ) {
		append_to_textbox( chat_window, NULL, this_user->data );
		strcpy( buf, "\n" );
		append_to_textbox( chat_window, NULL, buf );
		this_user = g_list_next( this_user );
	}
	strcpy( buf, "-------------------------------\n" );
	append_to_textbox( chat_window, NULL, buf );

	strcpy( buf, "--- Full Tab Completion List ---n" );
	append_to_textbox( chat_window, NULL, buf );

	this_user = full_list;
	while( this_user ) {
		append_to_textbox( chat_window, NULL, this_user->data );
		if ( tab_user == this_user ) {
			strcpy( buf, " <-- current position" );
			append_to_textbox( chat_window, NULL, buf );
		}
		strcpy( buf, "\n" );
		append_to_textbox( chat_window, NULL, buf );
		this_user = g_list_next( this_user );
	}
	strcpy( buf, "-------------------------------\n" );
	append_to_textbox( chat_window, NULL, buf );
}

void cmd_last_comments() {
	show_last_comments();
}

void cmd_send_file( char *args ) {
	char tmp[257];
	char msg[256];
	char *ptr;

	my_strncpy( tmp, args, sizeof( tmp ));
	ptr = strchr( tmp, ' ' );
	if ( *ptr ) {
		*ptr = '\0';
		ptr++;
	} else {
		strcpy( tmp, "ERROR, usage: /file <file> <username>\n" );
		append_to_textbox( chat_window, NULL, tmp );
		return;
	}

	snprintf( msg, sizeof(msg), "Filename: %s", tmp ); 
	ymsg9_send_file( ymsg_sess, ptr, msg, tmp );
}

void cmd_ignore( char *user ) {
	char *ptr;
	char *end;

	ptr = skip_whitespace( user );
	end = find_whitespace( ptr );

	while( *end ) {
		*end = '\0';
		ignore_toggle( ptr );

		ptr = end + 1;
		ptr = skip_whitespace( ptr );
		end = find_whitespace( ptr );
	}

	ignore_toggle( ptr );
}

void cmd_unignore( char *user ) {
	char *ptr;
	char *end;

	ptr = skip_whitespace( user );
	end = find_whitespace( ptr );

	while( *end ) {
		*end = '\0';

		if ( ignore_check( ptr ))
			ignore_toggle( ptr );

		ptr = end + 1;
		ptr = skip_whitespace( ptr );
		end = find_whitespace( ptr );
	}

	if ( ignore_check( ptr ))
		ignore_toggle( ptr );
}

void cmd_reload() {
	char buf[128];

	ignore_load();
	read_aliases();

	sprintf( buf, "*** Loaded files: %d aliases and %d ignored users ***\n",
		alias_count, ignore_count());

	append_to_textbox( chat_window, NULL, buf );
}

void cmd_mute( char *user ) {
	char *ptr;
	char *end;

	ptr = skip_whitespace( user );
	end = find_whitespace( ptr );

	while( *end ) {
		*end = '\0';
		mute_toggle( ptr );

		ptr = end + 1;
		ptr = skip_whitespace( ptr );
		end = find_whitespace( ptr );
	}

	mute_toggle( ptr );
}

void cmd_unmute( char *user ) {
	char *ptr;
	char *end;

	ptr = skip_whitespace( user );
	end = find_whitespace( ptr );

	while( *end ) {
		*end = '\0';

		if ( mute_check( ptr ))
			mute_toggle( ptr );

		ptr = end + 1;
		ptr = skip_whitespace( ptr );
		end = find_whitespace( ptr );
	}

	if ( mute_check( ptr ))
		mute_toggle( ptr );
}

void cmd_regex( char *regex ) {
	char *ptr;
	char *end;

	ptr = skip_whitespace( regex );
	end = find_whitespace( ptr );

/* new way, only assume 1 regex per line, so allow whitespace & skip quotes */

	if ( *ptr == '"' ) {
		ptr++;
	}
	if ( ptr[strlen(ptr)-1] == '"' ) {
		ptr[strlen(ptr)-1] = '\0';
	}
	regex_toggle( ptr );


/* old way, where it was assumed to be a line of whitespace separated regex
	while( *end ) {
		*end = '\0';
		regex_toggle( ptr );

		ptr = end + 1;
		ptr = skip_whitespace( ptr );
		end = find_whitespace( ptr );
	}

	regex_toggle( ptr );
*/

}

void cmd_regex_test( char *args ) {
	char tmp[257];
	char msg[256];
	char *ptr;
	char *text;
	char *match;
	regex_t test_regex;

	my_strncpy( tmp, args, sizeof( tmp ));
	ptr = strchr( tmp, '"' );
	if ( ! ptr ) {
		strcpy( tmp, "USAGE: /test_regex \"text to test match on\" <regex>\n" );
		append_to_textbox( chat_window, NULL, tmp );
		return;
	}
	text = ptr + 1;
	ptr = strchr( text, '"' );
	if ( *ptr ) {
		*ptr = '\0';
		ptr++;
	} else {
		strcpy( tmp, 
			"USAGE: /test_regex \"text to test match on\" <regex>\n" );
		append_to_textbox( chat_window, NULL, tmp );
	}

	if ( ! *ptr ) {
		/* no regex specified so check against all current regex */
		if (( match = auto_ignore_check( text )) != NULL ) {
			snprintf( msg, sizeof(msg),
				"*** Gyach regex test: '%s' MATCHES using regex '%s'\n",
				text, match );
		} else {
			snprintf( msg, sizeof(msg),
				"*** Gyach regex test: '%s' does NOT match any current regex\n",
				text );
		}
		append_to_textbox( chat_window, NULL, msg );
		return;
	}

	while( *ptr && *ptr == ' ' ) {
		ptr++;
	}

	regcomp( &test_regex, ptr, REG_EXTENDED | REG_ICASE | REG_NOSUB );
	if ( ! regexec( &test_regex, text, 0, NULL, 0 )) {
		/* matched */
		snprintf( msg, sizeof(msg),
			"*** Gyach regex test: '%s' matches using regex '%s'\n",
			text, ptr );
	} else {
		/* didn't match */
		snprintf( msg, sizeof(msg),
			"*** Gyach regex test: '%s' does NOT match using regex '%s'\n",
			text, ptr );
	}
	append_to_textbox( chat_window, NULL, msg );
	regfree( &test_regex );
}

void cmd_packet( char *filename ) {
	FILE *fp;
	char line[81];
	char *ptr;
	char *data = NULL;
	int type;
	int size;
	int first = 1;
	int value = 0;
	
	fp = fopen( filename, "rt" );

	if ( ! fp ) {
		strcpy( line, "***Unable to open packet datafile!***\n" );
		append_to_textbox( chat_window, NULL, line );
		return;
	}

	while( fgets( line, 80, fp )) {
		line[strlen(line)-1] = '\0';

		if ( strstr( line, "sess id   : " )) {
		} else if ( strstr( line, "data len  : " )) {
			/* don't show yet if this is the first header */
			if ( ! first ) {
				show_yahoo_packet();
			}

			first = 0;

			memset( &(ymsg_sess->pkt), 0, sizeof( ymsg_sess->pkt ));

			sscanf( line, "data len  : %d", &size );
			ymsg_sess->pkt.size = size;

			data = ymsg_sess->pkt.data;
		} else if ( strstr( line, "pkt type  : " )) {
			sscanf( line, "pkt type  : %x", &type );
			ymsg_sess->pkt.type = type;

			/* read 2 lines to skip header info */
			fgets( line, 80, fp );
			fgets( line, 80, fp );
		} else if (( ! strstr( line, " = '" )) &&
				( ! strstr( line, "--" ))) {
			ptr = line;
			while(( *ptr ) &&
				( *ptr != ' ' ) &&
				( *ptr != '\n' ) &&
				( *ptr != '\r' ) &&
				( *ptr != '#' ) &&
				( ptr != strstr( ptr, "  " ))) {
				sscanf( ptr, "%x ", &value );
				*(data++) = value;
				ptr += 3;
			}
			*data = '\0';
		}
	}
	fclose( fp );

	if ( ! first ) {
		show_yahoo_packet();
	}
}

void cmd_find( char *text ) {
	GtkWidget *tmp_widget;

	search_pos = -1;
	search_case_sensitive = 0;
	if ( search_text ) {
		free( search_text );
		search_text = NULL;
	}

	if ( strlen( text )) {
		search_text = strdup( text );

		search_chat_text( search_case_sensitive );
	} else {
		if ( ! find_window ) {
			find_window = create_find_window();
		} else {
			if ( search_text ) {
				tmp_widget = lookup_widget( find_window, "search_text" );
				gtk_entry_set_text(GTK_ENTRY(tmp_widget), search_text );

				tmp_widget = lookup_widget( find_window, "case_sensitive" );
				gtk_toggle_button_set_active(
					&(GTK_CHECK_BUTTON(tmp_widget)->toggle_button),
					search_case_sensitive );
			}
		}
		gtk_widget_show( find_window );
	}
}

void cmd_quit() {
	ymsg9_logout( ymsg_sess );

	if ( ymsg_sess->io_callback_tag != 0) {
		gdk_input_remove( ymsg_sess->io_callback_tag );
		ymsg_sess->io_callback_tag = 0;
	}

	if ( ymsg_sess->ping_callback_tag != 0) {
		gdk_input_remove( ymsg_sess->ping_callback_tag );
		ymsg_sess->ping_callback_tag = 0;
	}

	close( ymsg_sess->sock ); 
	ymsg_sess->sock = -1; 
	ymsg_sess->quit = 1; 

	history_save();

	gdk_threads_leave();

#ifndef OS_WINDOWS
	gtk_main_quit();
#endif
}

void cmd_exec( GtkWidget *window, GtkWidget *textbox, char *cmd, int local,
		char *user, int emote ) {
	char buf[601] = "";
#ifndef OS_WINDOWS
	int p[2];
	int pid;
	int chars;
	char new_cmd[513];
	char *ptr;
	int chars_read = 0;
	int elapsed = 0;
	int timed_out = 0;
	int flags;


	pipe( p );
	my_strncpy( buf, cmd, sizeof(buf));
	my_strncat( buf, ":\n", sizeof(buf));
	chars = strlen( buf );

	snprintf( new_cmd, sizeof(new_cmd), "%s 2>&1", cmd );

	if (( pid = fork()) == -1 ) {
		my_strncat( buf, "couldn't fork!\n", sizeof(buf));
	} else if ( pid ) {
		close( p[1] );
		flags = fcntl( p[0], F_GETFL, 0 );
		fcntl( p[0], F_SETFL, flags | O_NONBLOCK );
		ptr = buf + chars;
		while( chars < 512 ) {
			if ( elapsed > ( 3 * 1000 * 1000 )) {
				timed_out = 1;
				break;
			}

			if (( chars_read = read( p[0], ptr, (512 - chars ))) < 0 ) {
				usleep( 50000 );
				elapsed += 50000;
				continue;
			} else if ( chars_read == 0 ) {
				break;
			}
			chars += chars_read;
			ptr += chars_read;
			*ptr = 0;
		}

		if ( kill( pid, 0 ) == 0 ) {
			kill( pid, SIGKILL );
		}

		if ( chars > 511 ) {
			buf[509] = '.';
			buf[510] = '.';
			buf[511] = '.';
		}

		waitpid( pid, NULL, 0 );
	} else {
		close( p[0] );
		if ( p[1] != STDOUT_FILENO ) {
			dup2( p[1], STDOUT_FILENO );
			close( p[1] );
		}
		if ( STDERR_FILENO != STDOUT_FILENO ) {
			dup2(STDOUT_FILENO, STDERR_FILENO);
		}

		execlp( "sh", "sh", "-c", new_cmd, NULL );
	}

	if ( ! textbox ) {
		/* ! or !! in chat window */
		if ( local ) {
			my_strncat( buf, "\n", sizeof(buf));
			append_to_textbox( chat_window, NULL, buf );
		} else {
			if ( emote ) {
				gyach_emote( ymsg_sess, buf );
			} else {
				gyach_comment( ymsg_sess, buf );
			}
		}
	} else {
		/* ! or !! in PM window */
		if ( ! local ) {
			ymsg9_pm( ymsg_sess, user, buf );
		}
		my_strncat( buf, "\n", sizeof(buf));
		append_to_textbox( window, textbox, buf );
	}

#else
	sprintf( buf, "Sorry, /exec (!) and /execlocal (!!) are not "
		"implemented on this platform as of this Gyach version." );
	append_to_textbox( chat_window, NULL, buf );
#endif
}


void cmd_pepsi() {
	char *ptr;
#ifdef USE_PTHREAD_CREATE
	pthread_t prof_thread;
#endif

	ptr = strdup( "http://finance.yahoo.com/q?s=pep&d=c" );

#ifdef USE_PTHREAD_CREATE
	pthread_create( &prof_thread, NULL, display_url, (void *)ptr );
#else
	display_url( (void *)ptr );
#endif
}

int try_command( char *input ) {
	char *action = NULL;
	char *args = NULL;
	char tmp_input[ 512 ];

	my_strncpy( tmp_input, input, sizeof(tmp_input));

	action = tmp_input + 1; /* skip the '/' */

	action = skip_whitespace( action ); /* find head of command */
	args = action;
	args = find_whitespace( args ); /* find whitespace after command */
	if ( *args ) {
		*args = '\0';
		args++;
		args = skip_whitespace( args ); /* find text after whitespace */
	}

	if ( action[0] == '\0' )
		return( 0 );

	if ( regex_match( "^autoaway$", action )) {
		cmd_auto_away( args );
	} else if ( regex_match( "^al(ias)?$", action )) {
		cmd_alias( args );
	} else if ( regex_match( "^aw(ay)?$", action )) {
		cmd_away( args );
	} else if ( regex_match( "^back$", action )) {
		cmd_away( "0" );
	} else if ( regex_match( "^cl(s|ear)?$", action )) {
		cmd_cls( args );
	} else if ( regex_match( "^connect$", action )) {
		cmd_connect();
	} else if ( regex_match( "^disconnect)$", action )) {
		cmd_disconnect();
	} else if (( regex_match( "^emote$", action )) ||
			   ( regex_match( "^me$", action ))) {
		cmd_emote( args );
	} else if ( regex_match( "^exec$", action )) {
		cmd_exec( chat_window, NULL, args, 0, NULL, 0 );
	} else if ( regex_match( "^execlocal$", action )) {
		cmd_exec( chat_window, NULL, args, 1, NULL, 0 );
	} else if ( regex_match( "^exec_e$", action )) {
		cmd_exec( chat_window, NULL, args, 0, NULL, 1 );
	} else if ( regex_match( "^execlocal_e$", action )) {
		cmd_exec( chat_window, NULL, args, 1, NULL, 1 );
	} else if ( regex_match( "^file$", action )) {
		cmd_send_file( args );
	} else if ( regex_match( "^f(ind)?$", action )) {
		cmd_find( args );
	} else if ( regex_match( "^follow$", action )) {
		cmd_follow( args );
	} else if (( regex_match( "^friends$", action )) ||
			   ( regex_match( "^buddies$", action ))) {
		show_friends();
	} else if (( regex_match( "^fr(iend)?$", action )) ||
			   ( regex_match( "^b(uddy)?$", action ))) {
		cmd_friend( args );
	} else if ( regex_match( "^g(oto)?$", action )) {
		cmd_goto( args );
	} else if ( regex_match( "^help$", action )) {
		cmd_help( 0 );
	} else if ( regex_match( "^help_full$", action )) {
		cmd_help( 1 );
	} else if ( regex_match( "^history$", action )) {
		history_print();
	} else if ( regex_match( "^ig(nore)?$", action )) {
		cmd_ignore( args );
	} else if (( regex_match( "^call$", action )) ||
			   ( regex_match( "^inv(ite)?$", action ))) {
		cmd_invite( args );
	} else if ( regex_match( "^j(oin)?$", action )) {
		cmd_join( args );
	} else if ( regex_match( "^l(ast)?$", action )) {
		cmd_last();
	} else if ( regex_match( "^l(c|ast_comments)?$", action )) {
		cmd_last_comments();
	} else if ( regex_match( "^mail$", action )) {
		cmd_mail( args );
	} else if ( regex_match( "^mute$", action )) {
		cmd_mute( args );
	} else if ( regex_match( "^online$", action )) {
		cmd_online();
	} else if ( regex_match( "^packet$", action )) {
		cmd_packet( args );
	} else if ( regex_match( "^pepsi$", action )) {
		cmd_pepsi();
	} else if ( regex_match( "^pr(ofile)?$", action )) {
		cmd_profile( args );
	} else if (( regex_match( "^q(uit)?$", action )) ||
			   ( regex_match( "^exit$", action )) ||
			   ( regex_match( "^logo(ff|ut)$", action ))) {
		cmd_quit();
	} else if ( regex_match( "^reconnect$", action )) {
		cmd_reconnect();
	} else if ( regex_match( "^regex$", action )) {
		cmd_regex( args );
	} else if ( regex_match( "^regex_test$", action )) {
		cmd_regex_test( args );
	} else if ( regex_match( "^reload$", action )) {
		cmd_reload();
	} else if ( regex_match( "^revoke$", action )) {
		cmd_reject_buddy( args );
	} else if (( regex_match( "^tell$", action )) ||
			   ( regex_match( "^pm$", action ))) {
		cmd_tell( args );
	} else if ( regex_match( "^think$", action )) {
		cmd_think( args );
	} else if ( regex_match( "^unalias$", action )) {
		cmd_unalias( args );
	} else if ( regex_match( "^unignore$", action )) {
		cmd_unignore( args );
	} else if ( regex_match( "^unmute$", action )) {
		cmd_unmute( args );
	} else if ( regex_match( "^url$", action )) {
		cmd_url( args );
	} else if ( regex_match( "^v(er)?$", action )) {
		cmd_url( "" );
	} else {
		return( 0 );
	}

	return( 1 );
}

#define APP_CMD( list, cmd )	g_list_append( list, strdup( cmd ))
void append_commands_to_list( GList *list ) {
	/* fixme, make this into an array of structs and redo commands.c to use */
	APP_CMD( list, "/autoaway" );
	APP_CMD( list, "/alias" );
	APP_CMD( list, "/away" );
	APP_CMD( list, "/back" );
	APP_CMD( list, "/clear" );
	APP_CMD( list, "/connect" );
	APP_CMD( list, "/disconnect" );
	APP_CMD( list, "/find" );
	APP_CMD( list, "/follow" );
	APP_CMD( list, "/friend" );
	APP_CMD( list, "/friends" );
	APP_CMD( list, "/goto" );
	APP_CMD( list, "/history" );
	APP_CMD( list, "/help" );
	APP_CMD( list, "/help_full" );
	APP_CMD( list, "/ignore" );
	APP_CMD( list, "/invite" );
	APP_CMD( list, "/join" );
	APP_CMD( list, "/last" );
	APP_CMD( list, "/last_comments" );
	APP_CMD( list, "/mute" );
	APP_CMD( list, "/me" );
	APP_CMD( list, "/online" );
	APP_CMD( list, "/pepsi" );
	APP_CMD( list, "/profile" );
	APP_CMD( list, "/quit" );
	APP_CMD( list, "/regex" );
	APP_CMD( list, "/regex_test" );
	APP_CMD( list, "/tell" );
	APP_CMD( list, "/pm" );
	APP_CMD( list, "/think" );
	APP_CMD( list, "/ver" );
}
