/***************************************************************************
                          CursesScreen.cxx - a curses based Screen implementation
                             -------------------
    begin                : Thu Mar  1 13:15:18 IST 2001
    copyright            : (C) 2001 by Arie Tal
    email                : tal_arie@yahoo.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CursesScreen.h"
#include <curses.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <signal.h>

static int isVtTerminal = 0 ;

static WINDOW *
windowHandles[MAX_WINDOWS] ;

static WINDOW *
borderHandles[MAX_WINDOWS] ;

static CursesWindow *windowRecords[MAX_WINDOWS] ;

#define TO_WIN(x) (x)

static int
nextWindowToAllocate = 0 ;

#define MAIN 1
#define BOLD 2
#define BOLD2 3
#define BOLD3 4

#define MAIN_COLOR COLOR_WHITE
#define BOLD_COLOR COLOR_WHITE
#define BOLD2_COLOR COLOR_YELLOW
#define BOLD3_COLOR COLOR_GREEN

#define DEFAULT_BACKGROUND_COLOR COLOR_BLACK
#define REVERSE_BACKGROUND_COLOR COLOR_WHITE

CursesScreen::CursesScreen(int hebrew_location,
			   int alternate_hebrew_location) :
			   Screen(hebrew_location, alternate_hebrew_location) {
   initscr() ;
   start_color() ;
   init_pair(MAIN,MAIN_COLOR,DEFAULT_BACKGROUND_COLOR) ;
   init_pair(BOLD,BOLD_COLOR,DEFAULT_BACKGROUND_COLOR) ;
   init_pair(BOLD2,BOLD2_COLOR,DEFAULT_BACKGROUND_COLOR) ;
   init_pair(BOLD3,BOLD3_COLOR,DEFAULT_BACKGROUND_COLOR) ;
//   cbreak() ;
//Just returned it:
   cbreak() ;
   raw() ;
   noecho() ;
   nonl() ;
   intrflush(stdscr, FALSE); keypad(stdscr,TRUE); 
   meta(stdscr,TRUE) ;
   colors_enabled = has_colors() ;

   windowHandles[nextWindowToAllocate] = stdscr ;
   windowRecords[nextWindowToAllocate] = new CursesWindow(0,0,0,0,0) ;
   borderHandles[nextWindowToAllocate++] = 0 ;
   
   ENTER=KEY_ENTER ; REFRESH=KEY_REFRESH ; 
   F1=KEY_F(1) ; HELP=KEY_HELP ;
   F2=KEY_F(2) ; SAVE=KEY_SAVE ;
   F6=KEY_F(6) ; GOTO_LINE=F6 ;
   F4=KEY_F(4) ;
   F5=KEY_F(5) ;
   F7=KEY_F(7) ;
   F8=KEY_F(8) ;
   F9=KEY_F(9) ;
   F10=KEY_F(10) ;
   F11=KEY_F(11) ;
   F12=KEY_F(12) ;
   TAB = 9 ;
   LEFT=KEY_LEFT ; RIGHT=KEY_RIGHT ;
   UP=KEY_UP; DOWN=KEY_DOWN; SUSPEND=KEY_SUSPEND ; F3=KEY_F(3) ; EXIT=KEY_EXIT ;
   BACKSPACE=KEY_BACKSPACE ; DC=KEY_DC ; DL=KEY_DL ; NPAGE=KEY_NPAGE ;
   PPAGE=KEY_PPAGE ; END=KEY_END ; HOME=KEY_HOME ;
   special_separator = SPECIAL_SEPARATOR ;
   left_end_mark = LEFT_END_MARK ;
   right_end_mark = RIGHT_END_MARK ;
   if (getenv("TERM"))
     if (!strncmp(getenv("TERM"),"vt",2)) isVtTerminal=1 ;
}

CursesScreen::~CursesScreen()
{
   refresh() ;
   endwin() ;
}

void
CursesScreen::TemporaryExit(int justrefresh, char *cmd, int background)
{
   Erase() ;
   if ((!justrefresh) && (!cmd))
      MVAddStr(5,3,"Type  exit  to return to editing") ;
     refresh() ;
     endwin() ;
   if (!justrefresh) {
     if (!cmd)
      system("$SHELL") ;
     else if (!background) {
       fprintf(stderr,"Executing %s:\n\n",cmd) ;
       system(cmd) ;
       fprintf(stderr,"\n\nPress Enter to return to editing...\n") ;
       getchar() ;
     } else {
        char *buf = new char[strlen(cmd)+10] ;
        sprintf(buf,"%s &", cmd) ;
        system(buf) ;
        delete buf ;
     }
   }
     initscr() ;
     //   cbreak() ;
     start_color() ;
   init_pair(MAIN,MAIN_COLOR,DEFAULT_BACKGROUND_COLOR) ;
   init_pair(BOLD,BOLD_COLOR,DEFAULT_BACKGROUND_COLOR) ;
   init_pair(BOLD2,BOLD2_COLOR,DEFAULT_BACKGROUND_COLOR) ;
   init_pair(BOLD3,BOLD3_COLOR,DEFAULT_BACKGROUND_COLOR) ;
     raw() ;
     noecho() ;
     nonl() ;
     intrflush(stdscr, FALSE); keypad(stdscr,TRUE);
     // HERE WE HAVE TO RECONSTRUCT ALL THE WINDOW STRUCTURES!!!!!
     
     windowHandles[0] = stdscr ;
     Erase() ;
     int tmp = nextWindowToAllocate ;
     nextWindowToAllocate = 1 ;
     for (int i=1 ; i<tmp; i++) 
       if (windowRecords[i]) {
	 allocateNewWindow(windowRecords[i]->nlines,
			   windowRecords[i]->ncols,
			   windowRecords[i]->begin_y,
			   windowRecords[i]->begin_x,
			   windowRecords[i]->border) ;
         Erase(i) ;
       }
       else
	 nextWindowToAllocate++ ;
}

void
CursesScreen::Suspend()
{
   Erase() ;
   refresh() ;
   endwin() ;
   printf("Suspended\n") ;
   kill(getpid(), SIGSTOP) ;
   initscr() ;
     //   cbreak() ;
    start_color() ;
   init_pair(MAIN,MAIN_COLOR,DEFAULT_BACKGROUND_COLOR) ;
   init_pair(BOLD,BOLD_COLOR,DEFAULT_BACKGROUND_COLOR) ;
   init_pair(BOLD2,BOLD2_COLOR,DEFAULT_BACKGROUND_COLOR) ;
   init_pair(BOLD3,BOLD3_COLOR,DEFAULT_BACKGROUND_COLOR) ;
     raw() ;
     noecho() ;
     nonl() ;
     intrflush(stdscr, FALSE); keypad(stdscr,TRUE);
     // HERE WE HAVE TO RECONSTRUCT ALL THE WINDOW STRUCTURES!!!!!
     
     windowHandles[0] = stdscr ;
     Erase() ;
     int tmp = nextWindowToAllocate ;
     nextWindowToAllocate = 1 ;
     for (int i=1 ; i<tmp; i++) 
       if (windowRecords[i]) {
	 allocateNewWindow(windowRecords[i]->nlines,
			   windowRecords[i]->ncols,
			   windowRecords[i]->begin_y,
			   windowRecords[i]->begin_x,
			   windowRecords[i]->border) ;
         Erase(i) ;
       }
       else
	 nextWindowToAllocate++ ;
}

void CursesScreen::GetMaxYX(int *y, int *x, int wh)
{
   // in curses getmaxyx is a macro which sets y and x to the size of the 
   // screen
   getmaxyx(stdscr,*y, *x) ;
   rows = *y ; cols = *x ;
   getmaxyx(TO_WIN(windowHandles[wh]),*y, *x) ;
//   if (wh) (*y)++ ;
}

void CursesScreen::Erase(int wh, int flip_reverse)
{
   werase(TO_WIN(windowHandles[wh])) ;
}

void CursesScreen::Refresh(int wh)
{
//   touchwin(stdscr) ;
   wrefresh(TO_WIN(windowHandles[wh])) ;
}

void CursesScreen::MVAddStr(int y, int x, char *str,int wh)
{
   if (colors_enabled) {
   if (current_attr & REVERSE_ATTR) wattron(TO_WIN(windowHandles[wh]),A_REVERSE) ;
     else wattroff(TO_WIN(windowHandles[wh]),A_REVERSE) ;
     
     if ((current_attr == NORMAL_ATTR) || (current_attr == REVERSE_ATTR)) wattrset(TO_WIN(windowHandles[wh]),COLOR_PAIR(MAIN)) ;
     if (current_attr & BOLD_ATTR) wattrset(TO_WIN(windowHandles[wh]),COLOR_PAIR(BOLD)) ;
     if (current_attr & BOLD2_ATTR) wattrset(TO_WIN(windowHandles[wh]),COLOR_PAIR(BOLD2)) ;
     if (current_attr & BOLD3_ATTR) wattrset(TO_WIN(windowHandles[wh]),COLOR_PAIR(BOLD3)) ;
   if (current_attr & REVERSE_ATTR) wattron(TO_WIN(windowHandles[wh]),A_REVERSE) ;
     else wattroff(TO_WIN(windowHandles[wh]),A_REVERSE) ;
     wattron(TO_WIN(windowHandles[wh]),A_BOLD) ;
   if ((current_attr & BOLD_ATTR) || (current_attr == NORMAL_ATTR)) wattron(TO_WIN(windowHandles[wh]),A_BOLD) ;
//     else wattroff(TO_WIN(windowHandles[wh]),A_BOLD) ;
   } else {
   if (current_attr & REVERSE_ATTR) wattron(TO_WIN(windowHandles[wh]),A_REVERSE) ;
     else wattroff(TO_WIN(windowHandles[wh]),A_REVERSE) ;
     if (isVtTerminal) {
       if (current_attr & BOLD2_ATTR) wattron(TO_WIN(windowHandles[wh]),A_UNDERLINE) ;
         else wattroff(TO_WIN(windowHandles[wh]),A_UNDERLINE) ;
     } else {
       if (current_attr & BOLD2_ATTR) wattron(TO_WIN(windowHandles[wh]),A_UNDERLINE) ;
         else wattroff(TO_WIN(windowHandles[wh]),A_UNDERLINE) ;
     }
     if (current_attr == NORMAL_ATTR) wattroff(TO_WIN(windowHandles[wh]),A_UNDERLINE | A_REVERSE) ;
   }
   mvwaddstr(TO_WIN(windowHandles[wh]), y, x, str) ;
   if (wh==0)
      wmove(TO_WIN(windowHandles[wh]), hardy, hardx) ;
   if (colors_enabled) {
       wattrset(TO_WIN(windowHandles[wh]),PAIR_NUMBER(MAIN)) ;
       wattroff(TO_WIN(windowHandles[wh]),PAIR_NUMBER(BOLD)) ;
       wattroff(TO_WIN(windowHandles[wh]),PAIR_NUMBER(BOLD2)) ;
       wattroff(TO_WIN(windowHandles[wh]),PAIR_NUMBER(BOLD3)) ;
       wattroff(TO_WIN(windowHandles[wh]),A_REVERSE) ;
   }
}

void CursesScreen::Move(int y, int x, int wh)
{
   wmove(TO_WIN(windowHandles[wh]),y,x) ;
   hardy = y + windowRecords[wh]->begin_y ;
   hardx = x + windowRecords[wh]->begin_x ;
}

void CursesScreen::DeleteLn(int wh)
{
   wdeleteln(TO_WIN(windowHandles[wh])) ;
}

void CursesScreen::InsertLn(int wh)
{
   winsertln(TO_WIN(windowHandles[wh])) ;
}

int CursesScreen::GetCh()
{
   int c=getch() ;
   if ((c=='') && isVtTerminal) c='' ;
   return c ;
}

char *CursesScreen::KeyName(int c)
{
   static char buff[20] ;
   if ((c>0) && (c < 27)) {
     sprintf(buff,"^%c",'@'+c) ;
     return buff ;
   } else if (c == 27) {
      sprintf(buff,"ESC-") ;
      return buff ;
   } else 
    if (c == ENTER)
        sprintf(buff,"ENTER") ; 
    else
    if (c == REFRESH)
        sprintf(buff,"REFRESH") ; 
    else
    if (c == F1)
        sprintf(buff,"F1") ; 
    else
    if (c == HELP)
        sprintf(buff,"HELP") ; 
    else
    if (c == F2)
        sprintf(buff,"F2") ; 
    else
    if (c == LEFT)
        sprintf(buff,"LEFT") ; 
    else
    if (c == RIGHT)
        sprintf(buff,"RIGHT") ; 
    else
    if (c == SAVE)
        sprintf(buff,"SAVE") ; 
    else
    if (c == F3)
        sprintf(buff,"F3") ; 
    else
    if (c == EXIT)
        sprintf(buff,"EXIT") ; 
    else
    if (c == F4)
        sprintf(buff,"F4") ; 
    else
    if (c == F5)
        sprintf(buff,"F5") ; 
    else
    if (c == F6)
        sprintf(buff,"F6") ; 
    else
    if (c == GOTO_LINE)
        sprintf(buff,"GOTO_LINE") ; 
    else
    if (c == F7)
        sprintf(buff,"F7") ; 
    else
    if (c == F8)
        sprintf(buff,"F8") ; 
    else
    if (c == F9)
        sprintf(buff,"F9") ; 
    else
    if (c == F10)
        sprintf(buff,"F10") ; 
    else
    if (c == F11)
        sprintf(buff,"F11") ; 
    else
    if (c == F12)
        sprintf(buff,"F12") ; 
    else
    if (c == UP)
        sprintf(buff,"UP") ; 
    else
    if (c == DOWN)
        sprintf(buff,"DOWN") ; 
    else
    if (c == SUSPEND)
        sprintf(buff,"SUSPEND") ; 
    else
    if (c == BACKSPACE)
        sprintf(buff,"BACKSPACE") ; 
    else
    if (c == DC)
        sprintf(buff,"DEL_CHAR") ; 
    else
    if (c == DL)
        sprintf(buff,"DEL_LINE") ; 
    else
    if (c == NPAGE)
        sprintf(buff,"NEXT_PAGE") ; 
    else
    if (c == PPAGE)
        sprintf(buff,"PREVIOUS_PAGE") ; 
    else
    if (c == END)
        sprintf(buff,"END") ; 
    else
    if (c == HOME)
        sprintf(buff,"HOME") ; 
    else
        sprintf(buff,"%c", c) ;
   return buff ;
}

int
CursesScreen::allocateNewWindow(int nlines, int ncols, int begin_y, int begin_x, int brdr)
{
  windowRecords[nextWindowToAllocate] =
    new CursesWindow(nlines,ncols,begin_y, begin_x, brdr) ;

  int sizey, sizex ;
  GetMaxYX(&sizey, &sizex) ;
  if (nlines <= 0) nlines = sizey+nlines ;
  if (ncols <=0) ncols = sizex+ncols ;
  
  if (!brdr) {
    windowHandles[nextWindowToAllocate] = subwin(stdscr,nlines,ncols,begin_y,begin_x) ;
    borderHandles[nextWindowToAllocate] = 0 ;
  } else {
    borderHandles[nextWindowToAllocate] =
      subwin(stdscr,nlines+2,ncols+2,begin_y-1, begin_x-1) ;
    wattrset(borderHandles[nextWindowToAllocate], A_REVERSE) ;
    wborder(TO_WIN(borderHandles[nextWindowToAllocate]),0,0,0,0,0,0,0,0) ;
//    touchwin(stdscr) ;
    wrefresh(borderHandles[nextWindowToAllocate]) ;
    windowHandles[nextWindowToAllocate] = 
      subwin(stdscr,nlines,ncols,begin_y, begin_x) ;
  }
  return nextWindowToAllocate++ ;
}

void
CursesScreen::closeAllocatedWindow(int wh)
{
  if (TO_WIN(windowHandles[wh])) {
    delwin(TO_WIN(windowHandles[wh])) ;
    if (borderHandles[wh])
      delwin(borderHandles[wh]) ;
    windowHandles[wh] = 0 ;
    borderHandles[wh] = 0 ;
    windowRecords[wh]=0 ;
  }
}


void
CursesScreen::setWindowTopTitle(char *str, int wh)
{
   if (!wh) return ;

   if (!borderHandles[wh]) return ;
   
   int curattr = current_attr ;
   wattrset(borderHandles[wh],A_REVERSE) ;
   int x, y ;
   GetMaxYX(&y,&x,wh) ;
   mvwaddstr(borderHandles[wh],0,
         (x - strlen(str)) / 2,
            str) ;
//   touchwin(stdscr) ;
   wrefresh(borderHandles[wh]) ;
   SetAttr(curattr) ;
}
