/***************************************************************************
                          HebrewEditor.cxx  -  description
                             -------------------
    begin                : Sun Mar 4 2001
    copyright            : (C) 2001 by Arie Tal
    email                : tal_arie@yahoo.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "HebrewEditor.h"

#include "Edit.h"
#include "Help.h"
#include "LineEdit.h"
#include "aSelectionBox.h"
#include "aString.h"
#include "File.h"
#include "aDictionary.h"

#include "sysdep.h"

#include "aWindow.h"
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>

#define MAX_BUFFER_COUNT 100

#define HEBREW_DICTIONARY_FILE_NAME "hspell.wor"
#define ENGLISH_DICTIONARY_FILE_NAME "espell.wor"
#define LAXET_DICTIONARY_FILE_NAME "laxet.wor"


#define BUFFER_LIST_TITLE "Buffer List"


int
HebrewEditor::list_buffers(Edit **buffers, int buffer_count, Screen* scr,
                 int current_buffer)
{
  aSelectionBox<int> sb(this, -1,BUFFER_LIST_TITLE, scr) ;
  for (int i=0 ; i< buffer_count; i++) {
    if (!buffers[i]->finished()) {
      aString name = buffers[i]->getName() ;
      if (buffers[i]->getModified())
        name += " (mod.)" ;
      sb.addMember(name, i, (i==current_buffer)) ;
    }
  }

  return sb.select() ;
}




Action::Actions
HebrewEditor::MenuSelect(char *title, menuItem *menu,
       int count, Screen *scr, Action &act)
{
  aSelectionBox<Action::Actions> sb(this, Action::Null, title, scr) ;
  for (int i =0 ; i<count ; i++) {
    aString namebar = menu[i].name ;
    namebar += "  (" ;
    aString tmp = act.getActionKeyBinding(menu[i].act) ;
    namebar += tmp ;
    namebar += ") " ;
    if (tmp == "") namebar = menu[i].name ;
    else {
      aString tmp1 = act.getActionKeyBinding(menu[i].act,1) ;
      if (!(tmp1 == "")) {
        namebar = menu[i].name ;
        namebar += "  (" ;
        namebar += tmp ;
        namebar += "  or  " ;
        namebar += tmp1 ;
        namebar += ")  " ;
      }
    }
    sb.addMember(namebar, menu[i].act,0, menu[i].comment) ;
  }
  return sb.select() ;
}


Action::Actions
HebrewEditor::latex_menu(Screen *scr,Action &act)
{
  static menuItem menu[] = {
    {"LaTeX...", Action::LaTeX, "(Run LaTeX/LaXeT on a named primary file)"},
    {"View DVI(xdvi)...", Action::DviView, "(View the DVI output of LaTeX/LaXeT)"},
    {"View Available Symbols in Math-mode(xdvi)...", Action::MathSymbolsView,
                                       "(View a DVI file with tables of math symbols and commands)"}} ;

  return MenuSelect("LaTeX Menu", menu, sizeof(menu)/sizeof(menuItem),
                       scr, act) ;
}


Action::Actions
HebrewEditor::file_menu(Screen *scr,Action &act)
{
  static menuItem menu[] = {
    {"Open File...", Action::LoadFile, "(Open a file for editing)"},
    {"Save File", Action::SaveFile, "(Save the current file to disk)"},
    {"Save File As...", Action::SaveFileAs, "(Save the current file by a different name)"},
    {"Exit File...", Action::Exit, "(Exit the current file, if no more files are open, exits Hebrew Editor)"}} ;

  return MenuSelect("File Menu", menu, sizeof(menu)/sizeof(menuItem),
                       scr, act) ;
}


Action::Actions
HebrewEditor::edit_menu(Screen *scr,Action &act)
{
  static menuItem menu[] = {
    {"Undo Previous", Action::Undo, "(Undo the previous action)"},
    {"Redo Next", Action::Redo, "(Redo the previous undo)"},
    {"Mark Block Begin", Action::MarkBlockBegin, "(Mark the beginning of a block)"},
    {"MarkBlockEnd", Action::MarkBlockEnd, "(Mark the end of a block)"},
    {"Unmark Block", Action::UnmarkBlock, "(Unmark the current marked block)"},
    {"Copy Block", Action::CopyBlock, "(Copy the currently marked block to the location of the cursur)"},
    {"Delete Block", Action::DeleteBlock, "(Delete the currently marked block)"},
    {"Move Block", Action::MoveBlock, "(Move the currently marked block to the location of the cursur)"}} ;

  return MenuSelect("Edit Menu", menu, sizeof(menu)/sizeof(menuItem), scr,act) ;
}



Action::Actions
HebrewEditor::search_menu(Screen *scr,Action &act)
{
  static menuItem menu[] = {
    {"Search For...", Action::Find, "(Search for strings in either Hebrew or English)"},
    {"Search & Replace...", Action::FindReplace, "(Semi-Automatic Search and Replace of strings)"},
    {"Goto Line...", Action::GotoLine, "(Move the cursor to a given line)"}} ;

  return MenuSelect("Search Menu", menu, sizeof(menu)/sizeof(menuItem), scr,act) ;
}


Action::Actions
HebrewEditor::main_menu(Screen *scr,Action &act)
{
  static menuItem menu[] = {
    {"File menu...", Action::FileMenu, "(Open File, Save File, Save File As, Exit File,Online Tutorial)"},
    {"Edit menu...", Action::EditMenu, "(Undo, Redo, Block Operations)"},
    {"Search menu...", Action::SearchMenu, "(Search, Search&Replace, Goto Line)"},
    {"LaTeX menu...", Action::LaTeXMenu, "(Run LaTeX/LaXeT on a primary file, View DVI file)" },
//    {"Options...", Action::OptionsMenu},
    {"Buffer List...", Action::BufferList, "(Select one of the currently open buffers for editing)"},
    {"Online Tutorial...", Action::Help,"(View the online tutorial)"}} ;
//    {"Help...", Action::Help}} ;

  return MenuSelect("Main Menu", menu, sizeof(menu)/sizeof(menuItem), scr,act) ;
}




HebrewEditor::HebrewEditor(int argc, char **argv){
   if ((programPrefix=getenv(PROGRAM_PREFIX_VAR)) == NULL) {
      programPrefix = "/usr/local" ;
   }

   char *home=strdup(getenv("HOME")) ;

   aString helpTutorialName = programPrefix ;
           helpTutorialName+= "/" ;
           helpTutorialName+= HE_TUTORIAL_FILE_NAME ;

   File helpTutorialFile(helpTutorialName) ;
   if (!helpTutorialFile.exists()) {
      printf("%s environment variable is not set, unable to find some needed files\n",
               PROGRAM_PREFIX_VAR) ;
      printf("be sure to set %s to the location of the program files\n",
               PROGRAM_PREFIX_VAR) ;
      getchar() ;
   } /* endif */

   printf("Loading dictionaries\n") ;
   aString hebrewDictionaryLocation = programPrefix ;
   aString englishDictionaryLocation = programPrefix ;
   aString laxetDictionaryLocation = programPrefix ;
   hebrewDictionaryLocation += "/" ;
   englishDictionaryLocation += "/" ;
   laxetDictionaryLocation += "/" ;
   hebrewDictionaryLocation += HEBREW_DICTIONARY_FILE_NAME ;
   englishDictionaryLocation += ENGLISH_DICTIONARY_FILE_NAME ;
   laxetDictionaryLocation += LAXET_DICTIONARY_FILE_NAME ;

   aString laxetAdd((home) ? home : ".") ;
   aString espellAdd((home) ? home : ".") ;
   aString hspellAdd((home) ? home : ".") ;

   laxetAdd += "/.laxet.add" ;
   espellAdd += "/.espell.add" ;
   hspellAdd += "/.hspell.add" ;


   hebrewDictionary = new aHebrewDictionary(this, (char *)(const char *)hebrewDictionaryLocation, (char *)(const char *)hspellAdd) ;
   englishDictionary = new aDictionary(this, (char *)(const char *)englishDictionaryLocation, (char *)(const char *)espellAdd) ;
   laxetDictionary = new aHebrewDictionary(this, (char *)(const char *)laxetDictionaryLocation, (char *)(const char *)(laxetAdd)) ;
   screen = new SCREEN(HEBREW_NEWCODE, HEBREW_ALTERNATE_CODE) ;

   globalActionManager = new Action(*screen) ;

   aWindow *scratchWindow = new aWindow(screen) ;
   scratch = new Edit(this, SCRATCH_BUFFER_NAME, scratchWindow) ;
   scratch->processFile(0) ;

   screen->Erase() ;
   screen->Refresh() ;
   screen->GetMaxYX(&sizey, &sizex) ;

   buffers = new (Edit *)[MAX_BUFFER_COUNT] ;
   aString filename ;
   bufferCount = 0 ;
   if (argc > 1) {
      for (int i=1 ; i<argc ; i++) {
         filename = argv[i] ;
         buffers[bufferCount] = new Edit(this, filename, new aWindow(screen,-2, 0, 0,0,0) ) ;
         buffers[bufferCount++]->processFile() ;
       }
    } else {
      buffers[bufferCount] = new Edit(this, "Untitled.tex", new aWindow(screen,-2, 0, 0,0,0)) ;
      buffers[bufferCount++]->processFile() ;
    }
}


HebrewEditor::~HebrewEditor(){
}


void
HebrewEditor::appMain()
{
   int current_buffer = 0 ; currentEditBuffer = buffers[0] ;
   int finished = 0 ;
   buffers[current_buffer]->refresh() ;
   int last_buffer = 0 ;

   while (!finished) {
     Action::Actions act = buffers[current_buffer]->getNextAction() ;
     if (act == Action::MainMenu) {
          if (buffers[current_buffer]->file.markActive())
              buffers[current_buffer]->_markBlock(1) ;
          act=main_menu(screen,buffers[current_buffer]->action) ;
          buffers[current_buffer]->refresh() ;
     }
     if (act == Action::LaTeXMenu) {
          if (buffers[current_buffer]->file.markActive())
              buffers[current_buffer]->_markBlock(1) ;
          act=latex_menu(screen,buffers[current_buffer]->action) ;
          buffers[current_buffer]->refresh() ;
     }
     if (act == Action::FileMenu) {
          if (buffers[current_buffer]->file.markActive())
              buffers[current_buffer]->_markBlock(1) ;
          act=file_menu(screen,buffers[current_buffer]->action) ;
          buffers[current_buffer]->refresh() ;
     }
     if (act == Action::EditMenu) {
          if (buffers[current_buffer]->file.markActive())
              buffers[current_buffer]->_markBlock(1) ;
          act=edit_menu(screen,buffers[current_buffer]->action) ;
          buffers[current_buffer]->refresh() ;
     }
     if (act == Action::SearchMenu) {
          if (buffers[current_buffer]->file.markActive())
              buffers[current_buffer]->_markBlock(1) ;
          act=search_menu(screen,buffers[current_buffer]->action) ;
          buffers[current_buffer]->refresh() ;
     }
     switch (act) {
       case Action::BufferList: {
           if (buffers[current_buffer]->file.markActive())
                buffers[current_buffer]->_markBlock(1) ;
           int tmp = list_buffers(buffers, bufferCount, screen, current_buffer) ;
         if (tmp >= 0) {
           current_buffer = tmp ;
           currentEditBuffer = buffers[current_buffer] ;
         }
         buffers[current_buffer]->refresh() ;
       }
       break ;

        case Action::NextBuffer:
        case Action::Exit:
           last_buffer = current_buffer ;
           if (buffers[current_buffer]->file.markActive())
                buffers[current_buffer]->_markBlock(1) ;
           if (act == Action::Exit) {
             buffers[current_buffer]->processAction(act) ;
             if (buffers[current_buffer]->finished())
               last_buffer = current_buffer++ ;
           } else
             last_buffer = current_buffer++ ;
          if (current_buffer >= bufferCount) {
            current_buffer = 0 ;
            currentEditBuffer = buffers[current_buffer] ;
          }

          while ((buffers[current_buffer]->finished()) && (current_buffer != last_buffer)) {
            current_buffer++ ;
            currentEditBuffer = buffers[current_buffer] ;
            if (current_buffer >= bufferCount) {
              current_buffer = 0 ;
              currentEditBuffer = buffers[current_buffer] ;
            }

          }
          finished = buffers[current_buffer]->finished() ;
       // just to play with scratch
       // if ((last_buffer == current_buffer) && (current_buffer == bufferCount-1)) finished = 1 ;

           screen->MessageClear() ;
          if (!finished)
             buffers[current_buffer]->refresh() ;
      break ;


      case Action::PreviousBuffer:
          if (buffers[current_buffer]->file.markActive())
             buffers[current_buffer]->_markBlock(1) ;
          last_buffer = current_buffer-- ;
          currentEditBuffer = buffers[current_buffer] ;

         if (current_buffer <0 ) {
           current_buffer = bufferCount-1  ;
           currentEditBuffer = buffers[current_buffer] ;
         }

         while ((buffers[current_buffer]->finished()) &&
                (current_buffer != last_buffer)) {
           current_buffer-- ;
           currentEditBuffer = buffers[current_buffer] ;

           if (current_buffer < 0) {
             current_buffer = bufferCount - 1 ;
             currentEditBuffer = buffers[current_buffer] ;
           }

         }
         finished = buffers[current_buffer]->finished() ;
         // just to play with scratch
         //                   if ((last_buffer == current_buffer) && (current_buffer == bufferCount-1)) finished = 1 ;

         screen->MessageClear() ;
         if (!finished)
           buffers[current_buffer]->refresh() ;
       break ;



       case Action::LoadFile: {
          if (buffers[current_buffer]->file.markActive())
              buffers[current_buffer]->_markBlock(1) ;
             Action::Actions act ;
              screen->GetMaxYX(&sizey,&sizex) ;
              aWindow *linewin =new aWindow(screen,1,sizex,sizey-1,0,0) ;

             LineEdit *input_line = new GetFileName(this, "Open File: ",linewin) ;
             input_line->refresh() ;

             while (!input_line->finished())
                 input_line->processAction(act=input_line->getNextAction()) ;
             screen->MessageClear(1) ;
             if (act == Action::NewLine) {
                   buffers[bufferCount] = new Edit(this, input_line->getText(),
                                         new aWindow(screen,-2, 0, 0,0,0)) ;
                   buffers[bufferCount++]->processFile() ;
                   current_buffer = bufferCount-1 ;
                   currentEditBuffer = buffers[current_buffer] ;

             }
             buffers[current_buffer]->refresh() ;
             delete input_line ;
             delete linewin ;
             }
       break ;


       case Action::SaveFileAs: {
             Action::Actions act ;
              screen->GetMaxYX(&sizey,&sizex) ;
              aWindow *linewin=new aWindow(screen,1,sizex,sizey-1,0,0) ;

             LineEdit *input_line = new GetFileName(this, "Save File As: ",linewin) ;
             input_line->refresh() ;

             while (!input_line->finished())
                 input_line->processAction(act=input_line->getNextAction()) ;
             screen->MessageClear(1) ;
             if (act == Action::NewLine) {
                   FILE *fp = fopen(input_line->getText(), "r") ;
                   int flag = 1 ;
                   if (fp) {
                         fclose(fp) ;
                         buffers[current_buffer]->refresh() ;
                         int c =screen->Question("File exists, are you sure you want to write over it (y/n)? ") ;
                        while ((c != 'y')&& (c!='Y') && (c!= 'n') && (c!='N'))
                               c =screen->Question("File exists, are you sure you want to write over it (y/n)? ") ;
                        if ((c == 'n') || (c == 'N'))
                               flag = 0 ;
                   }
                   screen->MessageClear() ;
                   if (flag) {
                      buffers[current_buffer]->setName(input_line->getText()) ;
                     buffers[current_buffer]->_SaveFile(act) ;
                   }
             }
             buffers[current_buffer]->refresh() ;
             delete input_line ;
             delete linewin ;
             }
       break ;

       case Action::GotoLine: {
           if (buffers[current_buffer]->file.markActive())
                buffers[current_buffer]->_markBlock(1) ;
             Action::Actions act ;
              screen->GetMaxYX(&sizey,&sizex) ;
              aWindow *linewin=new aWindow(screen,1,sizex,sizey-1,0,0) ;

             LineEdit *input_line = new GetNumber(this, "Goto Line: ",linewin) ;
             input_line->refresh() ;

             while (!input_line->finished())
                 input_line->processAction(act=input_line->getNextAction()) ;
             screen->MessageClear(1) ;
             if (act == Action::NewLine) {
                   int num = atoi(input_line->getText()) ;
                   buffers[current_buffer]->goto_line(num) ;
             }
             delete input_line ;
             delete linewin ;
             }
       break ;

       case Action::LaTeX: {
           if (buffers[current_buffer]->file.markActive())
                buffers[current_buffer]->_markBlock(1) ;
             Action::Actions act ;
             aString title = "run LaTeX(" ;
             title+=DEFAULT_LATEX ;
             title+= ") on primary file: " ;

             if (!strcmp((char *)(const char *)primaryFile,""))
               primaryFile = buffers[current_buffer]->getName() ;

              screen->GetMaxYX(&sizey,&sizex) ;
              aWindow *linewin=new aWindow(screen,1,sizex,sizey-1,0,0) ;

             LineEdit *input_line = new GetFileName(this, (char *)(const char*)title,
                                          linewin,
                                          (char *)(const char*)primaryFile) ;
             input_line->refresh() ;

             while (!input_line->finished())
                 input_line->processAction(act=input_line->getNextAction()) ;
             screen->MessageClear(1) ;
             if (act == Action::NewLine) {
                   screen->MessageClear() ;
                   aString cmd = DEFAULT_LATEX ;
                   cmd += " " ;
                   cmd += input_line->getText() ;
                   primaryFile = input_line->getText() ;
                   screen->TemporaryExit(0,(char *)(const char*)cmd) ;
             }
             buffers[current_buffer]->refresh() ;
             delete input_line ;
             delete linewin ;
             }
       break ;

       case Action::DviView: {
           if (buffers[current_buffer]->file.markActive())
                buffers[current_buffer]->_markBlock(1) ;
             Action::Actions act ;
             aString title = "View DVI output file: " ;

              screen->GetMaxYX(&sizey,&sizex) ;
              aWindow *linewin=new aWindow(screen,1,sizex,sizey-1,0,0) ;

             if (!strcmp((char *)(const char *)lastDviFile,""))
               if (strcmp((char *)(const char *)primaryFile,"")) {
                  char *ptr = (char *)(const char *)primaryFile ;
                  char *dot = strrchr(ptr,'.') ;
                  if (dot) {
                    lastDviFile=primaryFile.at(0,dot-ptr) ;
                    lastDviFile+=".dvi" ;
                  } else {
                    lastDviFile = primaryFile + ".dvi" ;
                  }
               }

             LineEdit *input_line = new GetFileName(this, (char *)(const char*)title,
                                          linewin,
                                          (char *)(const char*)lastDviFile) ;
             input_line->refresh() ;

             while (!input_line->finished())
                 input_line->processAction(act=input_line->getNextAction()) ;
             screen->MessageClear(1) ;
             if (act == Action::NewLine) {
                   screen->MessageClear() ;
                   aString cmd = DEFAULT_DVIVIEW;
                   cmd += " " ;
                   cmd += input_line->getText() ;
                   lastDviFile = input_line->getText() ;
                   cmd += " " ;
                   cmd += DISCARD_OUTPUT ;
                   screen->TemporaryExit(0,(char *)(const char*)cmd,1) ;
             }
             buffers[current_buffer]->refresh() ;
             delete input_line ;
             delete linewin ;
             }
       break ;
       case Action::MathSymbolsView: {
           if (buffers[current_buffer]->file.markActive())
                buffers[current_buffer]->_markBlock(1) ;
             aString title = "View Math Symbols: " ;

             aString cmd = DEFAULT_DVIVIEW;
             cmd += " " ;
             cmd += programPrefix ;
             cmd += "/symbols.dvi" ;
             cmd += " " ;
             cmd += DISCARD_OUTPUT ;
             screen->TemporaryExit(0,(char *)(const char*)cmd,1) ;

             buffers[current_buffer]->refresh() ;
             }
       break ;



       default:
         buffers[current_buffer]->processAction(act) ;
     } /* end of switch */
   }
   screen->Erase() ;
   screen->Refresh() ;
}


int main(int argc, char **argv) {
   HebrewEditor *he = new HebrewEditor(argc, argv) ;

   he->appMain() ;
}


