/*-
 * Copyright (c) 1999-2000 James E. Housley <jim@thehousleys.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: getMB-smb.c,v 1.1.1.17 2004/08/17 01:08:20 housley Exp $
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_SMBUS
#include <fcntl.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <unistd.h>
#if __FreeBSD__ >= 5
#include <dev/smbus/smb.h>
#else
#include <machine/smb.h>
#endif
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <sysexits.h>

#include "methods.h"
#include "healthd.h"

#ifdef __OpenBSD__
#include <i386/pio.h>
#include <sys/types.h>
#include <machine/sysarch.h>
#endif /* __OpenBSD__ */

static int iosmb;
static u_char smb_addr;

static int
OpenIO(void) {
  static const u_char addrs[] = { 0x5a, 0x2d, 0x50 };	/* SMBus addresses to try */
  char byte;
  struct smbcmd cmd;
  int i;
  struct passwd* pwd;

#ifdef __OpenBSD__
  /* get the necessary I/O permissions */
  if (i386_iopl(3)) {
    return -1;
  }
#endif /* __OpenBSD__ */

  if ((iosmb = open("/dev/smb0", 000)) < 0){
    perror("/dev/smb0");
    return -1;
  }
  if ((pwd = getpwnam("daemon")) == NULL) {
    syslog(hdCRITICAL, "Unable to drop root permisions to \"daemon\".  \"daemon\" does not exist");
  } else {
    if (setgid(pwd->pw_gid)) {
      syslog(hdCRITICAL, "setgid(): %s", strerror(EX_NOPERM));
    }
    if (setuid( pwd->pw_uid)) {
      syslog(hdCRITICAL, "setuid(): %s", strerror(EX_NOPERM));
    }
  }

  cmd.cmd = 0x47;
  cmd.data.byte_ptr = &byte;
  for (i = 0; i < sizeof(addrs); i++) {
    cmd.slave = addrs[i];
    if (ioctl(iosmb, SMB_READB, (caddr_t)&cmd) != -1) {
      smb_addr = addrs[i];
      return 0;
    }
  }
  perror("ioctl(SMB_READB)");
  return -1;
}

static int 
CloseIO(void) {
  return close(iosmb);
}

static int
WriteByte(int addr,int value) {
  unsigned char count;
  struct smbcmd cmd;

  count = 0;
  cmd.slave = smb_addr;
  cmd.cmd = addr;
  cmd.data.byte = value;
  while (ioctl(iosmb, SMB_WRITEB, &cmd) == -1) {
    if (++count < 3) {
      sleep(1);
      continue;
    }
    syslog(LOG_ERR, "WriteByte failed 3 times");
    perror("ioctl(SMB_WRITEB)");
    exit(-1);
  }
  return 0;
}

static int
ReadByte(int addr) {
  unsigned char count = 0;
  struct smbcmd cmd;
  char ret;

  cmd.slave = smb_addr;
  cmd.cmd = addr;
  cmd.data.byte_ptr = &ret;
  while (ioctl(iosmb, SMB_READB, &cmd) == -1) {
    if (++count < 3) {
      sleep(1);
      continue;
    }
    syslog(LOG_ERR, "ReadByte failed 3 times");
    perror("ioctl(SMB_READB)");
    exit(-1);
  }
  return (int)ret;
}

struct lm_methods method_smb = {
  OpenIO, 
  CloseIO, 
  ReadByte, 
  WriteByte
};
#endif /* HAVE_SMBUS */
