;;; hnf-mode.el --- major mode for editing hnf.

;; Copyright (C) 1998-2000 by Akihiro Arisawa <ari@nijino.com>

;; Author: Akihiro Arisawa <ari@nijino.com>
;; Version: $Id: hnf-mode.el,v 3.20 2000/08/15 08:04:04 ari Exp $
;; Keywords: hnf nikki hns

;; This file is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This file is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; $B!Z$G$-$k$3$H![(B
;; 1. hnf $B$rJT=8;~$K(B TAB $B$G$"$i$f$kJd40$,$G$-$^$9!#(B
;; 2. font-lock $B$r;H$C$F(B hnf $B$r%+%i!<I=<($7$^$9!#(B
;; 3. M-x hnf $B$G(B ~/diary/1999/d19990107.hnf $BEy!"$=$NF|$N(B hnf $B$r3+$-$^$9!#(B
;;    C-u numeric M-x hnf $B$G!"(Bnumeric $BF|A0$N(B hnf $B$r3+$-$^$9!#(B
;; 4. namazu, calendar, speedbar $B$J$I$HO"7H$,$G$-$^$9!#(B
;; 5. $B$=$NB>!"$$$m$$$m!#(B
;;
;; $B!Z%$%s%9%H!<%k![(B
;; 1. APEL$B$r%$%s%9%H!<%k$7$^$9!#(B
;; 2. hnf-mode.el $B$r(B load-path $B$NDL$C$?$H$3$m$KCV$-$^$9!#(B
;; 
;; $B!Z@_Dj$J$I$J$I![(B
;; $BI,?\$N@_Dj$O(B 1$B!A(B2 $B$@$1$G$9!#$=$l0J30$O$*9%$_$G$I$&$>!#(B
;;
;; 1. $B3HD%;R$,(B .hnf $B$J%U%!%$%k$r3+$$$?;~$K(B hnf-mode $B$H$J$k$h$&$K(B ~/.emacs $BEy$G(B
;;    $B@_Dj$7$^$9!#(B
;;    (autoload 'hnf "hnf-mode" nil t)
;;    (autoload 'hnf-mode "hnf-mode" nil t)
;;    (setq auto-mode-alist (cons '("\\.hnf$" . hnf-mode) auto-mode-alist))
;;
;; 2. hnf-diary-dir, hnf-html-dir, hnf-diary-year-directory-flag $B$rI,MW$K1~$8$F(B
;;    $B@_Dj$7$^$9!#@_Dj$9$Y$-CM$OJQ?t$N%3%a%s%H$r;2>H$7$F$/$@$5$$!#(B
;;    $BNc(B) (setq hnf-diary-dir "~/diary")
;;	  (setq hnf-html-dir "~/public_html/diary")
;;	  (setq hnf-diary-year-directory-flag t)
;;
;; 3. $BJQ?t$rJd40$7$?$$>l9g$O(B hnf-variable $B$r@_Dj$7$^$9!#(B
;;    $BNc(B) (setq hnf-variable '(("TENKI") ("BASHO") ("TAIJU") ("TAION")))
;;
;; 4. $B%^!<%/$NJd40$r$7$?$$>l9g$O(B hnf-marks $B$r@_Dj$7$^$9!#(B
;;    $BNc(B) (setq hnf-marks '(("(^^)") ("(-_-)") ("(^^;") ("(;_;)") ("(T_T)"))
;;
;; 5. RLINK $B$rJd40$7$?$$>l9g$O(B hnf-rlink $B$r@_Dj$7$^$9!#(B
;;    $BNc(B) (setq hnf-rlink '(("h14m") ("ari")))
;;
;; 6. CAT $B$rJd40$7$?$$>l9g$O(B hnf-cat $B$r@_Dj$7$^$9!#(B
;;    $BNc(B) (setq hnf-cat '(("Linux") ("Nikki")))
;;
;; 7. ALIAS $B$rJd40$7$?$$>l9g$O(B hnf-alias $B$r@_Dj$7$^$9!#(B
;;    $BNc(B) (setq hnf-alias '(("Be") ("hns")))
;;
;; 8. $B?75,(B hnf $B$r3+$$$?;~$K!"<B9T$7$?$$4X?t$r(B hnf-initial-function
;;    $B$K@_Dj$7$^$9!#(B"OK"$B$r<+F0A^F~$7$?$1$l$P0J2<$N$h$&$K$7$^$9!#(B
;;    $BNc(B) (setq hnf-initial-function '(lambda () (insert "OK\n\n")))
;;
;; 9. $B0lF|$,(B28:00$B$^$G$"$k@83h$r$7$F$$$k>l9g$O!"0J2<$N@_Dj$r$7$F$_$F$/$@$5$$!#(B
;;         (setq hnf-hour-not-today 4)
;;     $B>\:Y$O(B hnf-hour-not-today $B$N@bL@$r;2>H$7$F$/$@$5$$!#(B
;;
;; 10. NEW $B$d(B SUB $B$N9T$G!"%j%s%/$N(B name $B$rI=<($7$?$$$J$i$P(B hnf-get-link(C-c=)
;;     $B$r!"%j%s%/@h$N%U%!%$%k$r%*!<%W%s$7$?$$$J$i$P(B hnf-link-find-file(C-cC-f)
;;     $B$r;H$$$^$9!#$^$?!"(BC-uC-c= $B$G%j%s%/$N(B name $B$,(B kill-ring $B$KA^F~$5$l$^$9!#(B
;;
;;    ($BCm(B1) hns-1.03pl0 $B0JA0$G$O%G%U%)%k%H$N%j%s%/$N(B name $B$,0[$J$j$^$9!#(B
;;          $B$=$N7A<0$GI=<($7$?$$;~$O0J2<$N@_Dj$r9T$C$F$/$@$5$$!#(B
;;          (setq hnf-get-link-name-function 'hnf-get-link-name)
;;    ($BCm(B2) hns-2.1 $B0J9_(B(?)$B$G$O%G%U%)%k%H$N%j%s%/$NL>A0$,0[$J$j$^$9!#(B
;;          $B$=$N7A<0$GI=<($7$?$$;~$O0J2<$N@_Dj$r9T$C$F$/$@$5$$!#(B
;;          (setq hnf-get-link-name-function 'hnf-get-link-name-with-abc)
;;
;; 11. $B;~4V$rA^F~$7$?$$$J$i$P(B hnf-insert-time(C-cC-t) $B$r;H$$$^$9!#(B
;;     $B%U%)!<%^%C%H$O(B hnf-time-format $B$GJQ99$G$-$^$9!#(B
;;     C-uC-cC-t $B$G%+!<%=%k0J9_$K4{$KA^F~$5$l$F$$$k;~4V$r8=:_$N;~4V$K(B
;;     $BJQ99$7$^$9!#(B
;;
;;     $B$^$?(B hnf-write-file-insert-time(C-cC-s) $B$r<B9T$9$k$H!"%U%!%$%k$NKvHx$K(B
;;     $B;~4V$rA^F~$7%U%!%$%k$rJ]B8$7$^$9!#(B
;;
;; 12. hnf-newline(C-cC-m) $B$G!"2~9T$HF1;~$KJD$8$k%?%0$NA^F~$J$I$,9T$o$l$^$9!#(B
;;     "\C-m"$B$K%P%$%s%I$7$F$bLdBjL5$$$H;W$$$^$9!#(B
;;     $BNc(B) (add-hook 'hnf-mode-load-hook
;;                   '(lambda ()
;;                      (define-key hnf-mode-map "\C-m" 'hnf-newline)))
;;
;; 13. namazu.el $B$K$F(B namazu_for_hns $B$N8!:w7k2L$+$i%U%!%$%k$r3+$-$?$$$J$i$P!"(B
;;     hnf-namazu-find-file $B$r(B namazu-mode-map $B$KE,59@_Dj$9$k$3$H$r$9$9$a$^$9!#(B
;;     $BNc(B) (setq namazu-command (expand-file-name "~/diary/namazu/bin/namazu"))
;;	   (add-hook 'namazu-mode-hook
;;		     '(lambda ()
;;		       (define-key namazu-mode-map "f" 'hnf-namazu-find-file)))
;;	   (autoload 'hnf-namazu-find-file "hnf-mode" nil t)
;;
;;     $B$^$?!"(Bhnf-mode $B$+$i(B C-cC-n $B$G(B namazu.el $B$r<B9T$G$-$^$9!#(B
;;
;; 14. M-x calendar $B$HO"7H$7$?$$$J$i$P!"(Bhnf-insert-diary-entry $B$rE,59%P%$%s%I(B
;;     $B$7$^$7$g$&!#(B
;;     $BNc(B) (autoload 'hnf-insert-diary-entry "hnf-mode" nil t)
;;         (add-hook 'calendar-load-hook
;;             '(lambda ()
;;                (define-key calendar-mode-map "iD" 'hnf-insert-diary-entry)))
;;     $B$^$?!"(Bhnf-mark-diary-entries $B$r<B9T$9$k$H!"F|5-$rF~NO$7$F$"$kF|$O(B
;;     $B%^!<%/$,IU$-$^$9!#(B
;;     $BNc(B) (autoload 'hnf-mark-diary-entries "hnf-mode" nil t)
;;         (add-hook 'today-visible-calendar-hook 'hnf-mark-diary-entries)
;;         (add-hook 'today-invisible-calendar-hook 'hnf-mark-diary-entries)
;;
;; 15. Web Browser $B$HO"7H$7$?$$>l9g$O!"(Bhnf-diary-url $B$KF|5-$N(BURL$B$r@_Dj$7$F(B
;;     $B2<$5$$!#(B
;;     hnf-browse-recent-diary(C-cC-b) $B$^$?$O(B hnf-browse-diary $B$G(B
;;     Web Browser $B$,5/F0$7$^$9!#(B
;;     $BNc(B) (setq hnf-diary-url "http://www.nijino.com/ari/diary/")
;;
;;     $B$^$?!"(Bhnf-preview-diary(C-cC-p) $B$G$O!"(Bhttpd $B$r7PM3$;$:$KI=<($G$-$^$9!#(B
;;
;; 2$B!A(B7,9$B$O(B custom $B$r;H$C$F@_Dj$b$G$-$k$O$:$G$9!#(B($BF0:nL$3NG'(B(^^;)
;;
;; $B!Z(BEmacsen$B![(B
;; $BF0:n3NG'$O<g$K(B emacs-20 $B$G9T$C$F$$$^$9!#(B
;; XEmacs-20/21 $B$bB?J,LdBjL5$$$G$7$g$&!#(B
;; mule-2.3@emacs-19.34 $B$G$O(B custom-1.9962 $B$r%$%s%9%H!<%k$9$k$3$H$r(B
;; $B$*A&$a$7$^$9!#(B
;; NEmacs $B$G$OF0$-$^$;$s!#(B
;;
;; $B$J$*!"(Bmule-2.3@19.34 $B$G(B custom-1.9962 $B$r;H$&>l9g$O!"(B
;; http://www.jpl.org/elips/ $B$+$i<hF@$G$-$k%Q%C%A$rEv$F$kI,MW$,$"$j$^$9!#(B
;;
;; $B!Z(BTo do$B![(B
;; 1. calendar $B$+$i$b(B browse-url $B$r8F$Y$k$h$&$K$7$h$&$+$J!#(B
;; 2. $BA4HLE*$K%(%i!<=hM}$,4E$$$N$rD>$9$Y$-$@$m$&$J!#(B
;;
;; $B!Z<U<-![(B
;; $B%O%$%Q!<F|5-%7%9%F%`$N:n<T$NJ}!9$K46<U$7$^$9!#%O%$%Q!<F|5-%7%9%F%`L5$/$7$F$O(B
;; hnf-mode $B$O@8$^$l$J$+$C$?$G$7$g$&(B:-)$B!#(B
;; $BF1N=$NJ!5o7/$K46<U$7$^$9!#Jd40$N2<0L4X?t$OH`$N(B toali.el $B$+$iC%$C$F$$$^$9(B(^^;
;; $B66K\8-0l$5$s(B(http://i21x12.fuis.fukui-u.ac.jp/~ken/diary/)$B$K46<U$7$^$9!#(B
;; $B$*$+$2$G(BM-x hnf $B$H$$$&5!G=$,DI2C$5$l$^$7$?(B(^^)$B!#(B
;; $B$^$?!"(Bhnf-mode $B$K6(NO$7$F$$$?$@$$$?0J2<$N3'$5$s$K46<U$7$^$9!#(B
;;	$B$d$^$0$A$5$s(B	http://symphony.sone.riec.tohoku.ac.jp/~yamaya/diary/
;;	$B5\:j$5$s(B	http://zak.interwave.or.jp/diary/
;;	JUN$B$5$s(B		http://jun.misao.gr.jp/diary/
;;	saw$B$5$s(B		http://www.be-in.org/~mas/diary/
;;	ysjj$B$5$s(B	http://www.unixuser.org/~ysjj/diary/
;;	kaoru$B$5$s(B	http://triaez.kaisei.org/~kaoru/diary/

;;; Code:

(defconst hnf-mode-version "2.2")

(eval-when-compile (require 'cl))
(require 'poe)
(require 'pcustom)

(defgroup hnf nil
  "Hyper Nikki File"
  :group 'hypermedia)

(defcustom hnf-diary-dir "~/diary"
  "$BF|5-%G%#%l%/%H%j!#(B"
  :group 'hnf
  :type 'directory)

(defcustom hnf-html-dir "~/public_html/diary"
  "hns $B$N%&%'%V!&%G%#%l%/%H%j!#(B"
  :group 'hnf
  :type 'directory)

(defcustom hnf-hns-program (concat hnf-html-dir "/index.cgi")
  "hns $B$N%W%m%0%i%`L>!#(B"
  :group 'hnf
  :type 'file)

(defcustom hnf-document-root "/usr/local/apache/htdocs"
  "Web $B%5!<%P$N(B DocumentRoot$B!#(B"
  :group 'hnf
  :type 'directory)

(defcustom hnf-diary-year-directory-flag t
  "nil $B$N>l9g$O!"(Bhnf $B$rF|5-%G%#%l%/%H%j$ND>2<$KCV$-$^$9!#(Bnon-nil $B$N>l9g$O(B
$BG/Kh$K%G%#%l%/%H%j$rJ,$1$FCV$-$^$9!#(B"
  :group 'hnf
  :type 'boolean)

(defcustom hnf-diary-url (concat "http://" (system-name) "/"
				 "~" (user-login-name) "/diary/")
  "$BF|5-$N(B URL$B!#(B"
  :group 'hnf
  :type 'string)

(defcustom hnf-index-name-list '("index.html" "index.shtml" "index.phtml")
  "index $B$H$7$F3+$/%U%!%$%kL>!#(B"
  :group 'hnf
  :type 'list)

(defcustom hnf-mode-hook nil
  "hnf-mode $B$N(B hook$B!#(B"
  :group 'hnf
  :type 'hook)

(defcustom hnf-mode-load-hook nil
  "hnf-mode $B$N(B load $B;~$N(B hook$B!#(B"
  :group 'hnf
  :type 'hook)

(defcustom hnf-initial-function nil
  "$B?75,%U%!%$%k$r%*!<%W%s$7$?;~$K8F$P$l$k4X?t!#(B"
  :group 'hnf
  :type 'function)

(defcustom hnf-variable nil
  "hnf $B$G;HMQ$9$kJQ?t!#(Bnil $B$r;XDj$9$k$HJQ?t$NJd40$O9T$$$^$;$s!#(B"
  :group 'hnf
  :type 'list)

(defcustom hnf-marks nil
  "hnf $B$G;HMQ$9$k%^!<%/!#(Bnil $B$r;XDj$9$k$H%^!<%/$NJd40$O9T$$$^$;$s!#(B"
  :group 'hnf
  :type 'list)

(defcustom hnf-alias nil
  "hnf $B$G;HMQ$9$k(B ALIAS$B!#(Bnil $B$r;XDj$9$k$H(B ALIAS $B$NJd40$O9T$$$^$;$s!#(B"
  :group 'hnf
  :type 'list)

(defcustom hnf-commands
  '(("NEW"	1	"NEW $B%;%/%7%g%s$N%?%$%H%k(B")
    ("SUB"	1	"SUB $B%;%/%7%g%sFb$N%5%V!&%;%/%7%g%s$N%?%$%H%k(B")
    ("CAT"	1	"CAT $B%+%F%4%j(B1 [$B%+%F%4%j(B2 ...]")

    ("LNEW"	2	"LNEW url $B%;%/%7%g%s$N%?%$%H%k(B")
    ("RLNEW"	3	"RLNEW arg1 arg2 $B@bL@J8>O(B") ; hns-2.10
    ("LSUB"	2	"LSUB url $B%;%/%7%g%sFb$N%5%V!&%;%/%7%g%s$N%?%$%H%k(B")
    ("RLSUB"	3	"RLSUB arg1 arg2 $B@bL@J8>O(B") ; hns-2.10

    ("P"	0	"P") ; hns-2.10
    ("/P"	0	"/P") ; hns-2.10
    ("GRP"	1	"GRP $B%0%k!<%W(B1 [$B%0%k!<%W(B2") ; hns-2.10

    ("LINK"	2	"LINK url $B@bL@J8>O(B")
    ("URL"	2	"URL url $B@bL@J8>O(B")
    ("RLINK"	3	"RLINK arg1 arg2 $B@bL@J8>O(B")

    ("FONT"	3	"FONT arg1 arg2 $BJ8>O(B")
    ("STRIKE"	1	"STRIKE $BJ8>O(B")
    ("LSTRIKE"	2	"STRIKE url $BJ8>O(B")
    ("STRONG"	1	"STRONG $BJ8>O(B")
    ("SPAN"	2	"SPAN class_name $BJ8>O(B") ; hns-2.10

    ("IMG"	3	"IMG {r|l|n} $B%U%!%$%kL>(B $BJ8;zNs(B")
    ("LIMG"	4	"LIMG url {r|l|n} $B%U%!%$%kL>(B $BJ8;zNs(B") ; hns-2.10
    ("MARK"	1	"MARK $B0z?t(B")

    ("UL"	0	"UL")
    ("/UL"	0	"/UL")
    ("OL"	0	"OL")
    ("/OL"	0	"/OL")
    ("LI"	1	"LI $BJ8>O(B")
    ("DL"	0	"DL") ; hns-2.10
    ("DT"	0	"DT") ; hns-2.10
    ("DD"	0	"DD") ; hns-2.10
    ("/DL"	0	"/DL") ; hns-2.10

    ("PRE"	0	"PRE")
    ("/PRE"	0	"/PRE")
    ("CITE"	0	"CITE")
    ("/CITE"	0	"/CITE")

    ("!"	1	"! $BJ8>O(B")
    ("!#"	1	"!# $BJ8>O(B")
    ("FN"	0	"FN")
    ("/FN"	0	"/FN")

    ("ALIAS"	1	"ALIAS arg")
    )
  "hnf $B$G;HMQ$9$k%3%^%s%I$r;XDj$7$^$9!#(B"
  :group 'hnf
  :type 'list)

(defcustom hnf-close-command-list
  '("P" "UL" "OL" "PRE" "CITE" "FN" "DL")
  "hnf $B$GJD$8$kI,MW$N$"$k%3%^%s%I!#(B"
  :group 'hnf
  :type 'list)

(defcustom hnf-rlink nil
  "hnf $B$G;HMQ$9$k(B RLINK$B!#(B"
  :group 'hnf
  :type 'list)

(defcustom hnf-cat nil
  "hnf $B$G;HMQ$9$k(B CAT$B!#(B"
  :group 'hnf
  :type 'list)

(defcustom hnf-font-lock-flag t
  "font-lock $B$r;H$&$+$I$&$+$N%U%i%0!#(B"
  :group 'hnf
  :type 'boolean)

(defcustom hnf-complete-command-insert-space-flag t
  "non-nil $B$N>l9g!"%3%^%s%I$NJd40$,40N;$7$?;~$K%9%Z!<%9$rA^F~$7$^$9!#(B"
  :group 'hnf
  :type 'boolean)

(defcustom hnf-complete-command-insert-newline-function 'hnf-newline
  "$B0z?t$rI,MW$H$7$J$$%3%^%s%I$NJd40$,40N;$7$?;~$K!"<B9T$9$k4X?t!#(B"
  :group 'hnf
  :type 'function)

(defcustom hnf-time-format "(%H:%M)"
  "hnf-insert-time $B$N%U%)!<%^%C%H!#(B
`format-time-string'$B$r;2>H$7$F$/$@$5$$!#(B"
  :group 'hnf
  :type 'string)

(defcustom hnf-time-regexp "([0-9][0-9]:[0-9][0-9])"
  "`hnf-insert-time' $B$GA^F~$5$l$kJ8;zNs$N@55,I=8=$G$9!#(B"
  :group 'hnf
  :type 'regexp)

(defcustom hnf-namazu-index-dir (concat hnf-diary-dir "/namazu/index")
  "namazu $B$N%$%s%G%C%/%9$NCV$$$F$"$k%G%#%l%/%H%jL>$G$9!#(B"
  :group 'hnf
  :type 'directory)

(defcustom hnf-tab-command 'tab-to-tab-stop
  "`hnf-tab-complete' $B$GJd40!?E83+$,<B9T$5$l$J$+$C$?;~$K<B9T$5$l$k4X?t$G$9!#(B"
  :group 'hnf
  :type 'function)

(defcustom hnf-header-p-function 'hnf-header-p
  "$B%X%C%@It$NH=Dj$r9T$&4X?t$G$9!#(B"
  :group 'hnf
  :type 'function)

(defcustom hnf-font-lock-keywords
  '(("^CAT.*" . hnf-cat-face)
    ("^L?NEW.*" . hnf-new-face)
    ("^L?SUB.*" . hnf-sub-face)
    ("^\\(!\\|!#\\).*" . hnf-comment-face)
    ("^\\(RLINK\\|RLNEW\\|RLSUB\\)[ \t]+\\([^ \t]+[ \t]+[^ \t]+\\)"
     1 hnf-link-face)
    ("^\\(URL\\|LINK\\|LNEW\\|LSUB\\|LIMG\\)[ \t]+\\([^ \t\n]+\\)"
     2 hnf-link-face)
    ("^IMG[ \t]+[^ \t]+[ \t]+\\([^ \t]+\\)" 1 hnf-image-face)
    ("^LIMG[ \t]+[^ \t]+[ \t]+[^ \t]+[ \t]+\\([^ \t]+\\)" 1 hnf-image-face)
    (eval . (cons (concat "^\\("
			  (mapconcat (function car) hnf-commands "\\|")
			  "\\)\\>")
		  hnf-command-face))
    (eval . (list (concat "^\\("
			  (mapconcat (function car) hnf-variable "\\|")
			  "\\)\\>")
		  '(0 (if (hnf-header-p) hnf-variable-face))))
    ("~$" . hnf-tilde-face)
    )
  "hnf$B%b!<%I$G$N(Bfont-lock$B$N%-!<%o!<%I!#(B"
  :group 'hnf
  :type 'list)

(defcustom hnf-get-link-name-function 'hnf-get-link-name-with-to
  "(hnf-get-link) $B$G%j%s%/$NL>A0$r:n@.$9$k4X?t$G$9!#(B
hns-1.03pl0 $B$^$G$N$O(B `hnf-get-link-name' $B$r!"(Bhns-1.03pl1 $B0J9_$O(B
`hnf-get-link-name-with-to' $B$r;XDj$7$F$/$@$5$$!#(B"
  :group 'hnf
  :type 'function)

(defcustom hnf-hour-not-today 0
  "$B;XDj$N;~4V$^$G$rA0F|$H$_$J$7$^$9!#(B0$B!A(B23$B$N@0?t$r;XDj$7$^$9!#(B

$BNc(B: 6$B$r;XDj$9$k$H!"(B0:00$B!A(B5:59$B$^$G$OA0F|$G$"$k$H$_$J$7!"(B
    M-x hnf$B$r<B9T;~$K$OA0F|$N(Bhnf$B$r3+$-$^$9!#$^$?!"(B(hnf-insert-time)$B$O(B
    \"25:00\" $B$H$$$C$?I=<($K$J$j$^$9!#(B"
  :group 'hnf
  :type 'integer)

(defcustom hnf-outline-level-table
  '((CAT . 1) (GRP . 1) (NEW . 1) (LNEW . 1)
    (SUB . 2) (LSUB . 2))
  "Alist that each element has the form `(KEYWORD . LEVEL)'.
KEYWORD is a hnf keyword, and LEVEL is a headings level."
  :group 'hnf
  :type 'list)

(defcustom hnf-outline-regexp
  (mapconcat (function (lambda (x) (symbol-name (car x))))
	     hnf-outline-level-table "\\|")
  "Regular expression to match the beginning of heading."
  :group 'hnf
  :type 'regexp)

(defcustom hnf-imenu-generic-expression
  '((nil "^NEW \\(.*\\)" 1)
    (nil "^LNEW \\([^ ]+\\) \\(.*\\)" 2))
  "Imenu generic expression for hnf-mode. See `imenu-generic-expression'."
  :group 'hnf
  :type 'regexp)

(defface hnf-cat-face '((t (:foreground "purple" :bold t)))
  "hnf $B$N(B CAT $B$N9T$N%U%'%$%9!#(B"
  :group 'hnf)
(defface hnf-new-face '((t (:foreground "purple" :bold t)))
  "hnf $B$N(B NEW $B$N9T$N%U%'%$%9!#(B"
  :group 'hnf)
(defface hnf-sub-face '((t (:foreground "purple")))
  "hnf $B$N(B NEW $B$N9T$N%U%'%$%9!#(B"
  :group 'hnf)
(defface hnf-link-face '((t (:foreground "Blue")))
  "hnf $B$N%j%s%/$N9T$N%U%'%$%9!#(B"
  :group 'hnf)
(defface hnf-image-face '((t (:foreground "Blue")))
  "hnf $B$N%j%s%/$N9T$N%U%'%$%9!#(B"
  :group 'hnf)
(defface hnf-comment-face '((t (:foreground "Red")))
  "hnf $B$N%3%a%s%H9T$N%U%'%$%9!#(B"
  :group 'hnf)
(defface hnf-command-face '((t (:foreground "firebrick")))
  "hnf $B$N%3%^%s%I$N%U%'%$%9!#(B"
  :group 'hnf)
(defface hnf-tilde-face '((t (:foreground "orange")))
  "hnf $B$N(B \"~\" $B$N%U%'%$%9!#(B"
  :group 'hnf)
(defface hnf-variable-face '((t (:foreground "DarkGoldenrod")))
  "hnf $B$NJQ?t$N%U%'%$%9!#(B"
  :group 'hnf)

(defvar hnf-mode-map nil)

(defvar hnf-cat-face 'hnf-cat-face)
(defvar hnf-new-face 'hnf-new-face)
(defvar hnf-sub-face 'hnf-sub-face)
(defvar hnf-link-face 'hnf-link-face)
(defvar hnf-image-face 'hnf-image-face)
(defvar hnf-comment-face 'hnf-comment-face)
(defvar hnf-command-face 'hnf-command-face)
(defvar hnf-tilde-face 'hnf-tilde-face)
(defvar hnf-variable-face 'hnf-variable-face)

(defvar hnf-complete-ok t
  "non-nil $B$N>l9g!"(BOK $B$rJd40BP>]$K4^$a$^$9!#(B")

(defconst hnf-completion-buffer-name "*HNF Completions*")

(if hnf-mode-map nil
  (setq hnf-mode-map (make-sparse-keymap))
  (define-key hnf-mode-map "\t" 'hnf-tab-complete)
  (define-key hnf-mode-map "\C-c\C-m" 'hnf-newline)
  (define-key hnf-mode-map "\C-c?" 'hnf-command-help)
  (define-key hnf-mode-map "\C-c=" 'hnf-get-link)
  (define-key hnf-mode-map "\C-c\C-f" 'hnf-link-find-file)
  (define-key hnf-mode-map "\C-c\C-t" 'hnf-insert-time)
  (define-key hnf-mode-map "\C-c\C-s" 'hnf-write-file-insert-time)
  (define-key hnf-mode-map "\C-c\C-b" 'hnf-browse-recent-diary)
  (define-key hnf-mode-map "\C-c\C-p" 'hnf-preview-diary)
  (define-key hnf-mode-map "\C-c\C-n" 'hnf-namazu)
  )

;;;###autoload
(defun hnf-mode ()
  "hnf $B$N%b!<%I!#(B"
  (interactive)
  (use-local-map hnf-mode-map)
  (setq mode-name "HNF")
  (setq major-mode 'hnf-mode)
  ;; font-lock
  (if hnf-font-lock-flag
      (progn
	(require 'font-lock)
	(make-local-variable 'font-lock-defaults)
	(setq font-lock-defaults '(hnf-font-lock-keywords t))
	(font-lock-mode t)))
  ;; outline
  (make-local-variable 'outline-regexp)
  (setq outline-regexp hnf-outline-regexp)
  (make-local-variable 'outline-level)
  (setq outline-level (function hnf-outline-level))
  (outline-minor-mode)
  ;; imenu
  (make-local-variable 'imenu-generic-expression)
  (setq imenu-generic-expression hnf-imenu-generic-expression)
  ;; hook
  (and (functionp hnf-initial-function)
       (= (buffer-size) 0)
       (buffer-file-name)
       (not (file-exists-p (buffer-file-name)))
       (funcall hnf-initial-function))
  (run-hooks 'hnf-mode-hook))

;;;###autoload
(defun hnf (&optional arg)
  "$B:#F|$N(B hnf $B$r3+$/!#(B"
  (interactive "P")
  (let* ((days-ago (if (listp arg) nil arg))
	 (now (hnf-current-time days-ago))
	 (dir (concat hnf-diary-dir "/" 
		      (and hnf-diary-year-directory-flag
			   (hnf-format-time-string "%Y/" now))))
	 (name (concat dir (unless (and arg (listp arg))
			     (hnf-format-time-string "d%Y%m%d.hnf" now)))))
    (if (and arg (listp arg)) (setq name (read-file-name "Find file: " name)))
    (or (file-directory-p (file-name-directory name))
	(make-directory  dir))
    (find-file name)
    (hnf-mode)))

(defun hnf-tab-complete ()
  "TAB $B$r2!$7$?;~$NJd40(B/$BE83+$r9T$&!#(B"
  (interactive)
  (let ((spaces (hnf-count-spaces))
	(command-word (hnf-command-word)))
    (cond ((funcall hnf-header-p-function)
	   (or (and (eq spaces 0)
		    (or (and hnf-variable
			     (not (eq (hnf-complete hnf-variable) 'no-match)))
			(and hnf-complete-ok
			     (not (eq (hnf-complete '(("OK"))) 'no-match)))))
	       (funcall hnf-tab-command)))
	  ((string= "CAT" command-word)	; CAT $B$G$OJ#?t8D$N5-=R$,2DG=(B
	   (and hnf-cat (hnf-complete hnf-cat)))
	  ((and (string= "MARK" command-word) (= spaces 1))
	   (and hnf-marks (hnf-complete hnf-marks)))
	  ((and (or (string= "RLINK" command-word)
		    (string= "RLNEW" command-word)
		    (string= "RLSUB" command-word))
		(= spaces 1))
	   (and hnf-rlink (hnf-complete hnf-rlink)))
	  ((and (string= "ALIAS" command-word) (= spaces 1))
	   (and hnf-alias (hnf-complete hnf-alias)))
	  ((and (or (string= "LINK" command-word)
		    (string= "LNEW" command-word)
		    (string= "LSUB" command-word)
		    (string= "LSTRIKE" command-word)
		    (string= "LIMG" command-word))
		(= spaces 1))
	   (hnf-complete-link))
	  ((or (and (string= "IMG" command-word) (= spaces 2))
	       (and (string= "LIMG" command-word) (= spaces 3)))
	   (hnf-complete-image))
	  ((or (and (string= "IMG" command-word) (= spaces 1))
	       (and (string= "LIMG" command-word) (= spaces 2)))
	   (hnf-complete '(("r") ("l") ("n"))))
	  ((and (string= "/" command-word) (= spaces 0))
	   (and hnf-close-command-list (hnf-close-command)))
	  ((and (= spaces 0)
		(not (eq (hnf-complete-command command-word) 'no-match))))
	  (t
	   (funcall hnf-tab-command)))))

;;; $B3F<o%3%^%s%I(B
(defun hnf-newline ()
  "return $B;~$K$$$m$$$mA^F~$7$^$9!#(B"
  (interactive)
  (let ((command-word (hnf-command-word))
	(close-command-regexp
	 (concat "^\\("
		 (mapconcat (function identity) hnf-close-command-list "\\|")
		 "\\)$"))
	(case-fold-search nil))
    (cond ((string-match close-command-regexp command-word) ; $BJD$8$k%?%0A^F~(B
	   (save-excursion (insert (concat "\n/" command-word "\n")))
	   (cond ((or (string= "UL" command-word) (string= "OL" command-word))
		  (insert "\nLI "))
		 (t
		  (insert "\n"))))
	  ((string= "CAT" command-word)
	   (insert "\nNEW "))
	  (t
	   (newline)))))

(defun hnf-command-help ()
  "$B%3%^%s%I$N4JC1$J%X%k%W$rI=<($7$^$9!#(B"
  (interactive)
  (message (nth 2 (assoc (hnf-command-word) hnf-commands))))

(defun hnf-link-find-file ()
  "LINK $B$d(B LSUB $B$K$FAjBP%Q%9$G;XDj$5$l$?%U%!%$%k$r%*!<%W%s$7$^$9!#(B"
  (interactive)
  (let ((command-word (hnf-command-word)))
    (if (or (string= "LINK" command-word)
	    (string= "LNEW" command-word)
	    (string= "LSUB" command-word))
	(let (p fname)
	  (save-excursion
	    (beginning-of-line)
	    (skip-chars-forward "^ \t\n")
	    (skip-chars-forward " ")
	    (setq p (point))
	    (skip-chars-forward "^ \t\n#")
	    (setq fname (buffer-substring-no-properties p (point))))
	  (cond ((string-match "^\\(http\\|ftp\\)://" fname)
		 (error "Absolute-URL is not supported"))
		((string-match "^/" fname)
		 (setq fname (concat hnf-document-root fname)))
		(t
		 (setq fname (concat hnf-html-dir "/" fname))))
	  (if (file-directory-p fname)
	      (progn
		(if (not (string-match "/$" fname))
		    (setq fname (concat fname "/")))
		(let ((index-list hnf-index-name-list))
		  (while (and index-list
			      (not (file-exists-p
				    (concat fname (car index-list)))))
		    (setq index-list (cdr index-list)))
		  (if index-list
		      (setq fname (concat fname (car index-list)))
		    (setq fname
			  (read-file-name "Find file: " fname))))))
	  (find-file fname)))))

(defun hnf-get-link (&optional arg)
  "NEW $B$d(B SUB $B$N%j%s%/$N(B name $B$r<hF@$7$^$9!#(B"
  (interactive "P")
  (let ((date-list (hnf-buffer-hnf-p))
	year mon day day-num dayhi abc cmd-word new-cnt sub-cnt link-name)
    (if (null date-list)
	(error "hnf $B$G$O$"$j$^$;$s(B")
      (setq year (nth 0 date-list)
	    mon (nth 1 date-list)
	    day (nth 2 date-list)
	    day-num (string-to-number day)
	    dayhi (/ day-num 10)
	    abc (cond ((< day-num 11) "a") ((< day-num 21) "b") (t "c"))
	    cmd-word (hnf-command-word))
      (save-excursion
	(if (string-match "^L?SUB$" cmd-word)
	    (progn
	      (setq sub-cnt 1)
	      (while (and (= (forward-line -1) 0)
			  (not (string-match
				"^L?NEW$"
				(setq cmd-word (hnf-command-word)))))
		(if (string-match "^L?SUB$" cmd-word)
		    (setq sub-cnt (1+ sub-cnt))))))
	(if (string-match "^L?NEW$" cmd-word)
	    (progn
	      (setq new-cnt 1)
	      (while (= (forward-line -1) 0)
		(if (string-match "^L?NEW$" (hnf-command-word))
		    (setq new-cnt (1+ new-cnt)))))))
      (if new-cnt
	  (progn 
	    (setq link-name (funcall hnf-get-link-name-function))
	    (message link-name)
	    (if arg (kill-new link-name)))))))

(defun hnf-get-link-name ()
  "hns-1.03pl0 $B0JA0$N(B NEW $B$N%j%s%/$NL>A0$r:n@.$9$k4X?t$G$9!#(B"
  (let ((link-name (format "?%s%s%d#%s%s%s%d" 
			   year mon dayhi year mon day new-cnt)))
    (if sub-cnt
	(concat link-name "S" (number-to-string sub-cnt))
      link-name)))

(defun hnf-get-link-name-with-to ()
  "hns-1.03pl1 $B0J9_$N(B NEW $B$N%j%s%/$NL>A0$r:n@.$9$k4X?t$G$9!#(B"
  (let ((link-name (format "%s%s%s%d" year mon day new-cnt)))
    (if sub-cnt
	(setq link-name (concat link-name "S" (number-to-string sub-cnt))))
    (setq link-name (format "?%s%s%d&to=%s#%s" 
			    year mon dayhi link-name link-name))))

(defun hnf-get-link-name-with-to-abc ()
  "hns-2.1 $B0J9_(B(?) $B$N(B NEW $B$N%j%s%/$NL>A0$r:n@.$9$k4X?t$G$9!#(B"
  (let ((link-name (format "%s%s%s%d" year mon day new-cnt)))
    (if sub-cnt
	(setq link-name (concat link-name "S" (number-to-string sub-cnt))))
    (setq link-name (format "?%s%s%s&to=%s#%s" 
			    year mon abc link-name link-name))))

(defun hnf-get-link-name-with-abc ()
  "hns-2.1 $B0J9_(B(?) $B$N(B NEW $B$N%j%s%/$NL>A0$r:n@.$9$k4X?t$G$9!#(B"
  (let ((link-name (format "%s%s%s%d" year mon day new-cnt)))
    (if sub-cnt
	(setq link-name (concat link-name "S" (number-to-string sub-cnt))))
    (setq link-name (format "?%s%s%s#%s" 
			    year mon abc link-name link-name))))

(defun hnf-insert-time (&optional arg)
  "$B;~4V$rA^F~$7$^$9!#(B"
  (interactive "P")
  (if (and arg
	   (re-search-forward hnf-time-regexp nil t))
      (replace-match
       (save-match-data
	 (hnf-format-time-string hnf-time-format (hnf-current-time))))
    (insert (hnf-format-time-string hnf-time-format (hnf-current-time)))))

(defun hnf-write-file-insert-time ()
  "`(hnf-insert-time)' $B$r<B9T$7!"%U%!%$%k$rJ]B8$7$^$9!#(B"
  (interactive)
  (save-excursion
    (goto-char (point-max))
    (hnf-insert-time))
  (save-buffer))

;;; for namazu.el
(defun hnf-namazu (key)
  (interactive
   (progn
     (require 'namazu)
     (list (read-from-minibuffer "Enter Keyword: " nil
				 (if (boundp 'namazu-minibuffer-field-map)
				     namazu-minibuffer-field-map)
				 nil 'namazu-history))))
  (namazu 0 hnf-namazu-index-dir key))

(defun hnf-namazu-find-file ()
  "\*namazu\* $B$K$F%]%$%s%H$,0LCV$9$k9`L\$r%(%G%#%H$7$^$9!#(B"
  (interactive)
  (save-excursion
    (if (re-search-forward 
	 "#\\([1-9][0-9][0-9][0-9]\\)\\([0-1][0-9][0-3][0-9]\\)0" nil t)
	(find-file (concat hnf-diary-dir "/"
			   (and hnf-diary-year-directory-flag 
				(concat (match-string 1) "/"))
			   "d" (match-string 1) (match-string 2) ".hnf")))))

;;; for calendar.el
(eval-when-compile (require 'calendar))

(defun hnf-filename-from-date (date)
  (concat hnf-diary-dir "/"
	  (and hnf-diary-year-directory-flag
	       (format "%d/" (extract-calendar-year date)))
	  (format "d%04d%02d%02d.hnf"
		  (extract-calendar-year date)
		  (extract-calendar-month date)
		  (extract-calendar-day date))))

;;;###autoload
(defun hnf-insert-diary-entry ()
  "$B%+%l%s%@!<$N%]%$%s%H$,$"$kF|IU$1$N(B hnf $B$r3+$-$^$9!#(B"
  (interactive)
  (find-file-other-window
   (hnf-filename-from-date (calendar-cursor-to-date t))))

;;;###autoload
(defun hnf-mark-diary-entries ()
  "$B%+%l%s%@!<Cf$KF|5-$,=q$$$F$"$kF|$r%^!<%/$7$^$9!#(B"
  (interactive)
  (let (y m first-date last-date tmp)
    (save-excursion
      (set-buffer calendar-buffer)
      (setq m displayed-month)
      (setq y displayed-year))
    (increment-calendar-month m y -1)
    (setq first-date
          (calendar-absolute-from-gregorian (list m 1 y)))
    (increment-calendar-month m y 2)
    (setq last-date
          (calendar-absolute-from-gregorian
           (list m (calendar-last-day-of-month m y) y)))
    (calendar-for-loop date from first-date to last-date do
		       (if (file-exists-p
			    (hnf-filename-from-date
			     (setq tmp
				   (calendar-gregorian-from-absolute date))))
			   (mark-visible-calendar-date tmp)))))

;;; for browse-url
(defun hnf-browse-recent-diary ()
  "browse-url $B$r;H$C$F:G?7$NF|5-$rI=<($7$^$9!#(B"
  (interactive)
  (browse-url hnf-diary-url))

(defun hnf-browse-diary ()
  "browse-url $B$r;H$C$F8=:_JT=8Cf$NF|5-$rI=<($7$^$9!#(B"
  (interactive)
  (let ((date-list (hnf-buffer-hnf-p)))
    (if (null date-list)
	(error "hnf$B$G$O$"$j$^$;$s(B")
      (browse-url (concat hnf-diary-url "?"
			  (mapconcat (function identity) date-list ""))))))

(defvar hnf-temp-file-list '())

(defun hnf-preview-diary ()
  "browse-url $B$r;H$C$F8=:_JT=8Cf$NF|5-$rI=<($7$^$9!#(B"
  (interactive)
  (let ((date-list (hnf-buffer-hnf-p)))
    (if (null date-list)
        (error "hnf$B$G$O$"$j$^$;$s(B")
      (let ((temp-file (concat temporary-file-directory
			       "hnf" (apply (function concat) date-list)
			       ".html"))
            (default-directory (file-name-directory hnf-hns-program)))
        (with-temp-file temp-file
          (call-process hnf-hns-program nil t t
			(apply (function concat) date-list)))
        (browse-url-of-file temp-file)
	(add-to-list 'hnf-temp-file-list temp-file)))))

(defun hnf-delete-temp-file-list ()
  "hnf-temp-file-list $B$N%U%!%$%k$r:o=|$7$^$9!#(B"
  (while hnf-temp-file-list
    (if (file-exists-p (car hnf-temp-file-list))
	(delete-file (car hnf-temp-file-list)))
    (setq hnf-temp-file-list (cdr hnf-temp-file-list))))
(add-hook 'kill-emacs-hook 'hnf-delete-temp-file-list)

;;; $B;(B?$J4X?t(B
(defun hnf-buffer-hnf-p (&optional buffer)
  "buffer $B$,(B hnf $B$+$I$&$+$rH=Dj$7$^$9!#(Bbuffer $B$r>JN,$7$?>l9g$O(B`current-buffer'
$B$KBP$7$FH=Dj$7$^$9!#(Bhnf $B$N>l9g$O(B (year mon day) $B$N%j%9%H$rJV$7$^$9!#(B"
  (let ((fname (buffer-file-name buffer)))
    (if (and fname
	     (string-match 
	      "d\\([0-9][0-9][0-9][0-9]\\)\\([0-1][0-9]\\)\\([0-3][0-9]\\)\\.hnf"
	      (setq fname (file-name-nondirectory fname))))
	(list (match-string 1 fname)
	      (match-string 2 fname)
	      (match-string 3 fname)))))

(defun hnf-header-p ()
  "$B%X%C%@It$+$I$&$+$rD4$Y$^$9!#(B"
  (save-excursion
    (not (search-backward "\n\n" nil t))))

(defun hnf-count-spaces ()
  "$B%+!<%=%k0LCV$^$G$N6uGr?t$r5a$a$^$9!#(B"
  (save-excursion
    (let ((p (point))
	  (cnt 0))
      (beginning-of-line)
      (while (re-search-forward "[ \t]+" p t)
	(setq cnt (+ cnt 1)))
      cnt)))

(defun hnf-cur-word (&optional erase)
  "$B%+!<%=%k0LCV$NC18l$rCj=P$7$^$9!#(B"
  (save-excursion
    (let ((p (point))
	  string)
      (skip-chars-backward "^ \t\n")
      (setq string (buffer-substring-no-properties p (point)))
      (if erase (delete-region p (point)))
      string)))

(defun hnf-command-word ()
  "$B$=$N9T$N%3%^%s%I(B($B$H;W$o$l$k$b$N(B)$B$r<hF@$7$^$9!#(B"
  (save-excursion
    (beginning-of-line)
    (let ((p (point)))
      (skip-chars-forward "^ \t\n")
      (buffer-substring-no-properties p (point)))))

(defun hnf-close-command ()
  "$BJD$8$k%3%^%s%I$rA^F~$7$^$9!#(B"
  (let ((regexp (concat "^/?\\("
			(mapconcat 'identity hnf-close-command-list "\\|")
			"\\)$"))
	stack command)
    (save-excursion
      (while (progn
	       (or (re-search-backward regexp nil t)
		   (error "command unmatch"))
	       (setq command (match-string 1))
	       (if (eq (char-after) ?/)
		   (push command stack)
		 (if (string= command (car stack))
		     (pop stack))))))
    (if stack (error "%s unmatched" command))
    (if command (insert command))))

(defun hnf-current-time (&optional days-ago)
  "`current-time' $B$rJV$7$^$9!#$?$@$7!"(B`hnf-hour-not-today' $B$,@_Dj$5$l$F$$$l$P(B
$B$=$N;~4VJ,$@$10JA0$N;~4V$rJV$7$^$9!#(B"
  (let ((time (current-time)))
    (if (not (numberp days-ago)) (setq days-ago 0))
    (hnf-time-float (- (hnf-float-time time)
		       (* days-ago 24 60 60)
		       (* (or hnf-hour-not-today 0) 60 60)))))

(defun hnf-format-time-string (time-format &optional time)
  "`format-time-string' $B$HF1Ey$G$9!#$?$@$7!"(B%H $B$O(B `hnf-hour-not-today' $B$NJ,$@$1(B
$BL$Mh$N;~4V$H$J$j$^$9!#(B"
  (let ((case-fold-search nil)
	sub-format hour)
    (while (string-match "%[a-zA-Z]" time-format)
      (setq sub-format (match-string 0 time-format))
      (setq time-format
	    (replace-match
	     (if (string= sub-format "%H")
		 (format "%d"
			 (+ (or hnf-hour-not-today 0)
			    (string-to-number (format-time-string "%H" time))))
	       (format-time-string sub-format time))
	     nil nil time-format)))
    time-format))

(defun hnf-float-time (&optional tm)
  (let ((time (or tm (current-time))))
    (+ (* (float (ash 1 16)) (nth 0 time)) (float (nth 1 time)))))
  
(defun hnf-time-float (num)
  (let* ((most-time (floor num 65536))
	 (least-time (floor (- num (* 65536.0 most-time)))))
    (list most-time least-time 0)))

(defun hnf-outline-level ()
  "Return the depth to which a statement is nested in the outline.
Point must be at the beginning of a header line.
See `hnf-outline-level-table'."
  (save-excursion
    (when (looking-at outline-regexp)
      (cdr (assq (intern (buffer-substring (match-beginning 0) (match-end 0)))
                 hnf-outline-level-table)))))

(eval-after-load "speedbar"
  '(speedbar-add-supported-extension ".hnf"))

;;; $BJd40$N>e0L4X?t(B
(defun hnf-complete (alist)
  "$B%+!<%=%k0LCV$K$FJd40$r9T$$$^$9!#(Balist $B$K$F8uJd$r;XDj$7$^$9!#(B"
  (hnf-complete-string (hnf-cur-word t) alist))

(defun hnf-complete-image ()
  "$B2hA|%U%!%$%kL>$NJd40$r9T$$$^$9!#(B"
  (let* ((cur (hnf-cur-word t))
	 (dname (file-name-directory cur))
	 (fname (file-name-nondirectory cur))
	 (files (file-name-all-completions fname
					   (concat hnf-html-dir "/" dname))))
    (if dname (insert dname))
    (hnf-complete-string fname (mapcar (function list) files))))

(defun hnf-complete-link ()
  "LINK $B$NJd40$r9T$$$^$9!#(B"
  (let* ((cur (hnf-cur-word t))
	 (dname (file-name-directory cur))
	 (fname (file-name-nondirectory cur))
	 (files (file-name-all-completions
		 fname
		 (if (char-equal (string-to-char cur) ?/)
		     (concat hnf-document-root dname)
		   (concat hnf-html-dir "/" dname)))))
    (if dname (insert dname))
    (hnf-complete-string fname (mapcar (function list) files))))

(defun hnf-complete-command (command-word)
  "$B%3%^%s%I$NJd40$r9T$$$^$9!#(B"
  (let ((sts (hnf-complete hnf-commands)))
    (if (or (eq sts 'match)
	    (and (eq sts 'match1) (setq command-word (hnf-command-word))))
	(if (eq (cadr (assoc command-word hnf-commands)) 0)
	    (and hnf-complete-command-insert-newline-function
		 (funcall hnf-complete-command-insert-newline-function))
	  (and hnf-complete-command-insert-space-flag (insert " "))))
    sts))

;;; $BJd40$N2<0L4X?t(B
(defun hnf-complete-string (string alist)
  (let* ((completion-ignore-case t)
	 (completions (all-completions string alist))
	 (cur (current-buffer))
	 comp)
    (cond 
     ((= (length completions) 1) ; $BJd408uJd$,(B1$B8D$@$1(B
      (if (string= (car completions) string)
	  (progn
	    (insert string)
	    (hnf-delete-completion-window)
	    'match)
	(insert (car completions))
	'match1))
     ((and (setq comp (try-completion string alist))
	   (not (string= comp string))) ; $BESCf$^$GJd40(B
      (insert comp)
      'complete)
     (t
      (insert string)
      (if (not comp) 
	  (progn ; $B8uJd$,$J$$(B
	    (hnf-delete-completion-window)
	    'no-match)
	; $B8uJd$rI=<((B
	(buffer-disable-undo (get-buffer-create hnf-completion-buffer-name))
	(with-output-to-temp-buffer
	    hnf-completion-buffer-name
	  (display-completion-list (sort completions 'string<)))
	'complete-list)))))
  
(defun hnf-delete-completion-window ()
  (and
   (get-buffer hnf-completion-buffer-name)
   (let ((w (get-buffer-window hnf-completion-buffer-name)))
     (and w
	  (delete-window w))
     (kill-buffer hnf-completion-buffer-name))))

(run-hooks 'hnf-mode-load-hook)

(provide 'hnf-mode)
;;; hnf-mode.el ends here
