/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: HydrogenApp.cpp,v 1.69 2004/07/19 09:20:11 comix Exp $
 *
 */

#include "HydrogenApp.h"
#include "lib/Hydrogen.h"

#include "qdockwindow.h"
#include "qsplitter.h"
#include "qhbox.h"

#include "PlayerControl.h"
#include "PatternEditorPanel.h"
#include "SongEditor.h"
#include "MainForm.h"
#include "Mixer.h"
#include "PreferencesDialog.h"
#include "AudioEngineInfoForm.h"
#include "HelpBrowser.h"
#include "DrumkitManager.h"
#include "LadspaFXProperties.h"
#include "InstrumentPropertiesDialog.h"

HydrogenApp* HydrogenApp::m_pInstance = NULL;

HydrogenApp::HydrogenApp( MainForm *pMainForm, Song *pFirstSong )
 : Object("HydrogenApp")
 , m_pStatusBar( NULL )
 , m_pMainForm( pMainForm )
 , m_pMixer( NULL )
 , m_pPatternEditorPanel( NULL )
 , m_pAudioEngineInfoForm( NULL )
 , m_pSongEditorPanel( NULL )
 , m_pHelpBrowser( NULL )
 , m_pPlayerControl( NULL )
{
	m_pInstance = this;
	
	// Load the LADSPA plugin list
	m_pluginList = LadspaFX::getPluginList();
	m_pFXRootGroup = LadspaFX::getLadspaFXGroup();

	// Create the audio engine :)
	( Hydrogen::getInstance(this) );	// create the engine and set this class as engine listener
	
	(Hydrogen::getInstance())->setSong( pFirstSong );
	( PreferencesMng::getInstance() )->setLastSongFilename( pFirstSong->getFilename() );

	// set initial title
	QString qsSongName( (pFirstSong->getName()).c_str() );
	m_pMainForm->setCaption( ( "Hydrogen " + QString(VERSION) + QString( " - " ) + qsSongName ) );
	m_pMainForm->setDockMenuEnabled(false);

	PreferencesMng *pPref = PreferencesMng::getInstance();
	
	switch ( pPref->getInterfaceMode() ) {
		case PreferencesMng::MDI:
			setupMDIInterface();
			break;
			
		case PreferencesMng::TOP_LEVEL:
			setupTopLevelInterface();
			break;
			
		case PreferencesMng::SINGLE_PANED:
			setupSinglePanedInterface();
			break;
			
		default:
			errorLog( "Bad interface mode: " + toString( pPref->getInterfaceMode() ) );
	}
	
	// restore audio engine form properties
	m_pAudioEngineInfoForm = new AudioEngineInfoForm( 0 );
	WindowProperties audioEngineInfoProp = pPref->getAudioEngineInfoProperties();
	m_pAudioEngineInfoForm->move( audioEngineInfoProp.x, audioEngineInfoProp.y );
	m_pAudioEngineInfoForm->setShown( audioEngineInfoProp.visible );


	m_pStatusBar = new QStatusBar( m_pMainForm );
	m_pStatusBar->setSizeGripEnabled( false );
}



HydrogenApp::~HydrogenApp() {

	delete m_pInstrumentProperties;
	delete m_pPatternEditorPanel;
	delete m_pSongEditorPanel;
	delete m_pHelpBrowser;
	delete m_pAudioEngineInfoForm;

	if (m_pMixer) {
		//m_pMixer->updateStart(false);
		delete m_pMixer;
	}

	delete m_pDrumkitManager;

	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		delete m_pLadspaFXProperties[nFX];
	}

	for (uint i = 0; i < m_pluginList.size(); i++) {
		delete m_pluginList[ i ];
	}

	delete m_pFXRootGroup;


	Hydrogen *engine = Hydrogen::getInstance();
	if (engine) {
		Song *song = engine->getSong();

		delete engine;
		engine = NULL;

		if (song) {
			delete song;
			song = NULL;
		}
	}
}



/// Return an HydrogenApp m_pInstance
HydrogenApp* HydrogenApp::getInstance() {
	if (m_pInstance == NULL) {
		cerr << "Error! HydrogenApp::getInstance (m_pInstance = NULL)" << endl;
	}
	return m_pInstance;
}



void HydrogenApp::setupTopLevelInterface() 
{
	infoLog( "[setupTopLevelInterface]" );
	PreferencesMng *pPref = PreferencesMng::getInstance();

	// TOOLBAR
	QToolBar * fileTools = new QToolBar( m_pMainForm, "file operations" );
	fileTools->setMovingEnabled( false );
	fileTools->setLabel( "Hydrogen control panel" );
	m_pPlayerControl = new PlayerControl( fileTools );
	
	// restore pattern editor properties
	m_pPatternEditorPanel = new PatternEditorPanel( 0 );
	WindowProperties patternEditorProp = pPref->getPatternEditorProperties();
	m_pPatternEditorPanel->resize( patternEditorProp.width, patternEditorProp.height );
	m_pPatternEditorPanel->move( patternEditorProp.x, patternEditorProp.y );
	m_pPatternEditorPanel->setShown( patternEditorProp.visible );

	// restore m_pMixer properties
	m_pMixer = new Mixer(0);
	WindowProperties mixerProp = pPref->getMixerProperties();
	m_pMixer->resize( mixerProp.width, mixerProp.height );
	m_pMixer->move( mixerProp.x, mixerProp.y );
	m_pMixer->setShown( mixerProp.visible );

	// restore song editor properties
	m_pSongEditorPanel = new SongEditorPanel( m_pMainForm );
	m_pMainForm->setCentralWidget( m_pSongEditorPanel );

	// restore drumkit manager properties
	m_pDrumkitManager = new DrumkitManager( 0 );
	WindowProperties drumkitMngProp = pPref->getDrumkitManagerProperties();
	m_pDrumkitManager->move( drumkitMngProp.x, drumkitMngProp.y );
	m_pDrumkitManager->setShown( drumkitMngProp.visible );

	m_pHelpBrowser = new HelpBrowser( 0 );

	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		m_pLadspaFXProperties[nFX] = new LadspaFXProperties( NULL, nFX );
		m_pLadspaFXProperties[nFX]->hide();
		WindowProperties prop = pPref->getLadspaProperties(nFX);
		m_pLadspaFXProperties[nFX]->move( prop.x, prop.y );
		m_pLadspaFXProperties[nFX]->setShown( prop.visible );
	}
	
	m_pInstrumentProperties = new InstrumentPropertiesDialog( 0 );
	WindowProperties instrumentEditorProperties = pPref->getInstrumentEditorProperties();
	m_pInstrumentProperties->move( instrumentEditorProperties.x, instrumentEditorProperties.y );
	m_pInstrumentProperties->setShown( instrumentEditorProperties.visible );
}



void HydrogenApp::setupMDIInterface() 
{
	PreferencesMng *pPref = PreferencesMng::getInstance();

	// TOOLBAR
	QToolBar * fileTools = new QToolBar( m_pMainForm, "file operations" );
	fileTools->setMovingEnabled( false );
	fileTools->setLabel( "Hydrogen control panel" );
	m_pPlayerControl = new PlayerControl( fileTools );

	// create the workspace
	m_pMainForm->workspace = new QWorkspace( m_pMainForm );
	m_pMainForm->workspace->setScrollBarsEnabled( true );
	string mdiBackground_path = string( IMG_PATH ) + string( "/img/mdiBackground.png" );
	m_pMainForm->workspace->setBackgroundPixmap( QPixmap(mdiBackground_path.c_str()) );

	// restore m_pMainForm properties
	WindowProperties mainFormProp = pPref->getMainFormProperties();
//	m_pMainForm->setMinimumSize( QSize( mainFormProp.width, mainFormProp.height ) );
	m_pMainForm->resize( mainFormProp.width, mainFormProp.height );
	m_pMainForm->move( mainFormProp.x, mainFormProp.y );
	m_pMainForm->setCentralWidget( m_pMainForm->workspace );

	// restore PatternEditorPanel properties
	m_pPatternEditorPanel = new PatternEditorPanel(m_pMainForm->workspace);
	WindowProperties patternEditorProp = pPref->getPatternEditorProperties();
	m_pPatternEditorPanel->resize( patternEditorProp.width, patternEditorProp.height );
	m_pPatternEditorPanel->move( patternEditorProp.x, patternEditorProp.y );
	m_pPatternEditorPanel->setShown( patternEditorProp.visible );

	// m_pMixer test
//	QDockWindow *pMixerDock = new QDockWindow( m_pMainForm, "MixerDockWindow");
//	pMixerDock->setResizeEnabled(true);
//	pMixerDock->setMovingEnabled(false);

	// restore m_pMixer properties
	m_pMixer = new Mixer( m_pMainForm->workspace );
//	m_pMixer = new Mixer( pMixerDock );
	WindowProperties mixerProp = pPref->getMixerProperties();
	m_pMixer->resize( mixerProp.width, mixerProp.height );
	m_pMixer->move( mixerProp.x, mixerProp.y );
	m_pMixer->setShown( mixerProp.visible );
//	pMixerDock->setWidget(m_pMixer);
//	m_pMainForm->moveDockWindow( pMixerDock, Qt::DockBottom );

	// restore song editor properties
	m_pSongEditorPanel = new SongEditorPanel( m_pMainForm->workspace );
	WindowProperties songEditorProp = pPref->getSongEditorProperties();
	m_pSongEditorPanel->resize( songEditorProp.width, songEditorProp.height );
	m_pSongEditorPanel->move( songEditorProp.x, songEditorProp.y );
	m_pSongEditorPanel->setShown( songEditorProp.visible );

	// restore drumkit manager properties
	m_pDrumkitManager = new DrumkitManager( m_pMainForm->workspace );
	WindowProperties drumkitMngProp = pPref->getDrumkitManagerProperties();
	m_pDrumkitManager->move( drumkitMngProp.x, drumkitMngProp.y );
	//m_pDrumkitManager->setShown( drumkitMngProp.visible );
	m_pDrumkitManager->hide();
	
	m_pHelpBrowser = new HelpBrowser( 0 );

	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		WindowProperties prop = pPref->getLadspaProperties(nFX);
		
		m_pLadspaFXProperties[nFX] = new LadspaFXProperties( m_pMainForm->workspace, nFX );
		m_pLadspaFXProperties[nFX]->move( prop.x, prop.y );
		m_pLadspaFXProperties[nFX]->setShown( prop.visible );
	}

	m_pInstrumentProperties = new InstrumentPropertiesDialog( m_pMainForm->workspace );
	WindowProperties instrumentEditorProperties = pPref->getInstrumentEditorProperties();
	m_pInstrumentProperties->move( instrumentEditorProperties.x, instrumentEditorProperties.y );
	m_pInstrumentProperties->setShown( instrumentEditorProperties.visible );
}



void HydrogenApp::setupSinglePanedInterface() 
{
	PreferencesMng *pPref = PreferencesMng::getInstance();

	// MAINFORM
	WindowProperties mainFormProp = pPref->getMainFormProperties();
//	m_pMainForm->setMinimumSize( QSize( mainFormProp.width, mainFormProp.height ) );
	m_pMainForm->resize( mainFormProp.width, mainFormProp.height );
	m_pMainForm->move( mainFormProp.x, mainFormProp.y );

	// TOOLBAR
	QToolBar * fileTools = new QToolBar( m_pMainForm, "file operations" );
	fileTools->setMovingEnabled( false );
	fileTools->setLabel( "Hydrogen control panel" );
	m_pPlayerControl = new PlayerControl( fileTools );


	QSplitter *pSplitter = new QSplitter( m_pMainForm );
	pSplitter->setOrientation( QSplitter::Vertical );
	pSplitter->setOpaqueResize( true );
	
	m_pMainForm->setCentralWidget( pSplitter );
	
	m_pSongEditorPanel = new SongEditorPanel( pSplitter );
	WindowProperties songEditorProp = pPref->getSongEditorProperties();
	m_pSongEditorPanel->resize( songEditorProp.width, songEditorProp.height );
	
	// this HBox will contain the Instrument Properties editor and the Pattern editor
	QHBox *pEditorHBox = new QHBox(pSplitter);
	
	// set the background color for the HBox
	UIStyle *pStyle = PreferencesMng::getInstance()->getDefaultUIStyle();
	QColor backgroundColor( pStyle->m_patternEditorPanel_backgroundColor.getRed(), pStyle->m_patternEditorPanel_backgroundColor.getGreen(), pStyle->m_patternEditorPanel_backgroundColor.getBlue() );
	pEditorHBox->setEraseColor( backgroundColor );

	
	// INSTRUMENT PROPERTIES
	m_pInstrumentProperties = new InstrumentPropertiesDialog( pEditorHBox );
	
	m_pPatternEditorPanel = new PatternEditorPanel( pEditorHBox );
	WindowProperties patternEditorProp = pPref->getPatternEditorProperties();
	m_pPatternEditorPanel->resize( patternEditorProp.width, patternEditorProp.height );

	// MIXER
	m_pMixer = new Mixer(0);
	WindowProperties mixerProp = pPref->getMixerProperties();
	m_pMixer->resize( mixerProp.width, mixerProp.height );
	m_pMixer->move( mixerProp.x, mixerProp.y );
	m_pMixer->setShown( mixerProp.visible );

	
	// DRUMKIT MANAGER
	m_pDrumkitManager = new DrumkitManager( 0 );
	WindowProperties drumkitMngProp = pPref->getDrumkitManagerProperties();
	m_pDrumkitManager->move( drumkitMngProp.x, drumkitMngProp.y );
	m_pDrumkitManager->setShown( drumkitMngProp.visible );

	// HELP BROWSER
	m_pHelpBrowser = new HelpBrowser( 0 );

	// LADSPA FX
	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		m_pLadspaFXProperties[nFX] = new LadspaFXProperties( NULL, nFX );
		m_pLadspaFXProperties[nFX]->hide();
		WindowProperties prop = pPref->getLadspaProperties(nFX);
		m_pLadspaFXProperties[nFX]->move( prop.x, prop.y );
		m_pLadspaFXProperties[nFX]->setShown( prop.visible );
	}
	
}



void HydrogenApp::setSong(Song* song) {
	//m_pMixer->updateStart(false);

	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		m_pLadspaFXProperties[nFX]->hide();
	}

	Song* oldSong = (Hydrogen::getInstance())->getSong();
	if (oldSong != NULL) {
		(Hydrogen::getInstance())->removeSong();
		delete oldSong;
		oldSong = NULL;
	}

	(Hydrogen::getInstance())->setSong(song);
	( PreferencesMng::getInstance() )->setLastSongFilename( song->getFilename() );

/*	if (m_pMixer->isVisible()) {
		m_pMixer->updateStart(true);
	}*/
	m_pSongEditorPanel->updateAll();

	QString songName( (song->getName()).c_str() );
	m_pMainForm->setCaption( ( "Hydrogen " + QString(VERSION) + QString( " - " ) + songName ) );

	m_pMainForm->updateRecentUsedSongList();

	QMessageBox::information( m_pMainForm, "Hydrogen", m_pMainForm->m_pQApp->translate("HydrogenApp", "Song Info:     ") + QString("\n\n") + QString( song->getNotes().c_str() ) + QString("\n") );
}



Song *HydrogenApp::getSong() {
	return (Hydrogen::getInstance())->getSong();
}



void HydrogenApp::showMixer(bool show) {
	if (show) {
		m_pMixer->show();
	}
	else {
		m_pMixer->hide();
	}
}



void HydrogenApp::showPreferencesDialog() {
	PreferencesDialog preferencesDialog(m_pMainForm);
	preferencesDialog.exec();
}




void HydrogenApp::setStatusBarMessage(QString msg, int msec) {
	if (msec != 0) {
		m_pStatusBar->message(msg, msec);
	}
	else {
		m_pStatusBar->message(msg, msec);
	}
}



/// Used in the m_pMixer panel, shows the open file dialog and replace an existent instrument
/*
void HydrogenApp::loadNewInstrument(int instrNumber) {
	errorLog( "[loadNewInstrument] not implemented yet" );

	/// \todo save the last used directory in preferences file
	static QString lastUsedDir = "";

	Hydrogen *engine = Hydrogen::getInstance();

	QFileDialog *fd = new QFileDialog(m_pMainForm, "File Dialog", TRUE);
	fd->setMode(QFileDialog::ExistingFile);
	fd->setFilter( m_pMainForm->m_pQApp->translate("HydrogenApp", "Audio files (*.wav *.WAV *.au *.AU *.aiff *.AIFF)") );
	fd->setCaption( m_pMainForm->m_pQApp->translate("HydrogenApp", "Hydrogen - Load instrument") );
	fd->setDir( lastUsedDir );

	FilePreview *pPreview = new FilePreview();
	fd->setContentsPreviewEnabled( TRUE );
	fd->setContentsPreview( pPreview, pPreview );
	fd->setPreviewMode( QFileDialog::Contents );

	QString filename = "";
	if (fd->exec() == QDialog::Accepted) {
		filename = fd->selectedFile();
	}

	if (filename != "") {
		lastUsedDir = fd->dirPath();
		Sample *newSample = Sample::load( filename.latin1() );

		Instrument *instr = NULL;

		engine->lockEngine("HydrogenApp::loadNewInstrument");
		Song *song = engine->getSong();
		InstrumentList *instrList = song->getInstrumentList();
		instr = instrList->get( instrNumber );

		// delete old sample
		Sample *oldSample = instr->getSample();
		delete oldSample;

		// insert new sample from newInstrument
		instr->setSample( newSample );
		instr->setDrumkitName( "" );	// external sample, no drumkit info

		engine->unlockEngine();
	}
}
*/


void HydrogenApp::showAudioEngineInfoForm() {
	m_pAudioEngineInfoForm->hide();
	m_pAudioEngineInfoForm->show();
}



/// This method is called from another thread
void HydrogenApp::engineError( uint nErrorCode ) {
	QString msg;
	QApplication* pApp = m_pMainForm->m_pQApp;

	switch (nErrorCode) {
		case Hydrogen::NO_OSS_SUPPORT:
			msg = pApp->translate( "HydrogenApp", "No OSS driver support" );
			break;

		case Hydrogen::NO_JACK_SUPPORT:
			msg = pApp->translate( "HydrogenApp", "No JACK driver support" );
			break;

		case Hydrogen::UNKNOWN_DRIVER:
			msg = pApp->translate( "HydrogenApp", "Unknown audio driver" );
			break;

		case Hydrogen::ERROR_STARTING_DRIVER:
			msg = pApp->translate( "HydrogenApp", "Error starting audio driver" );
			break;

		case Hydrogen::JACK_SERVER_SHUTDOWN:
			msg = pApp->translate( "HydrogenApp", "Jack driver: server shutdown" );
			break;

		case Hydrogen::JACK_CANNOT_ACTIVATE_CLIENT:
			msg = pApp->translate( "HydrogenApp", "Jack driver: cannot activate client" );
			break;

		case Hydrogen::JACK_CANNOT_CONNECT_OUTPUT_PORT:
			msg = pApp->translate( "HydrogenApp", "Jack driver: cannot connect output port" );
			break;

		case Hydrogen::JACK_ERROR_IN_PORT_REGISTER:
			msg = pApp->translate( "HydrogenApp", "Jack driver: error in port register" );
			break;

		default:
			msg = pApp->translate( "HydrogenApp", "Unknown error" );
	}

	H2TextEvent *ev = new H2TextEvent( "ErrorMsg", msg );
	QApplication::postEvent( m_pMainForm, ev );

}




/*
void HydrogenApp::openPatternEditorPanel(Pattern *pPattern)
{
	infoLog( "[openPatternEditorPanel]" );

	/// \todo devo controllare se ce ne sono gia' aperti con lo stesso pattern..
	PatternEditorPanel *pPatternEditorPanel = NULL;

	PreferencesMng *pref = PreferencesMng::getInstance();
	if ( pref->getInterfaceMode() == "Child frame" ) {	// use the new MDI interface
		pPatternEditorPanel = new PatternEditorPanel(m_pMainForm->workspace, pPattern);
	}
	else {
		pPatternEditorPanel = new PatternEditorPanel( 0, pPattern);
	}
	m_patternEditorPanelList.push_back( pPatternEditorPanel);
	pPatternEditorPanel->show();
}


void HydrogenApp::closePatternEditorPanel(Pattern *pPattern)
{
	infoLog( "[closePatternEditorPanel]" );
	for (uint i = 0; i < m_patternEditorPanelList.size(); i++) {
		PatternEditorPanel *pPanel = m_patternEditorPanelList[i];
		if ( (pPanel->getPattern() )  == pPattern ) {
			m_patternEditorPanelList.erase( m_patternEditorPanelList.begin() + i );
			pPanel->hide();
			delete pPanel;
			return;
		}
	}
	errorLog( "[closePatternEditorPanel] no patternEditorPanel found" );
}
*/
