/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: MainForm.cpp,v 1.90 2004/07/06 10:39:56 comix Exp $
 *
 */


#include "MainForm.h"
#include "AboutDialog.h"
#include "PlayerControl.h"
#include "ExportSongDialog.h"
#include "Mixer.h"
#include "HelpBrowser.h"
#include "PatternEditorPanel.h"
#include "SongEditor.h"
#include "DrumkitManager.h"
#include "AudioEngineInfoForm.h"
#include "LadspaFXProperties.h"
#include "InstrumentPropertiesDialog.h"

#include "../lib/Hydrogen.h"
#include "../lib/smf/SMF.h"

#include "qaccel.h"
#include <sys/time.h>


MainForm::MainForm(QApplication *app, string songFilename ) : QMainWindow( 0, 0, 0 ), Object( "MainForm   " )
{
	setIcon( QPixmap( QString(IMG_PATH) + QString( "/img/icon32.png") ) );

	m_pQApp = app;

	workspace = NULL;

	m_pQApp->processEvents( 1000 );

	// Load default song
	Song *song = NULL;
	if (songFilename != "") {
		song = Song::load( songFilename );
		if (song == NULL) {
			QMessageBox::warning( this, "Hydrogen", trUtf8("Error loading song.") );
			song = Song::getEmptySong();
			song->setFilename( "" );
		}
	}
	else {
		PreferencesMng *pref = PreferencesMng::getInstance();
		bool restoreLastSong = pref->isRestoreLastSongEnabled();
		string filename = pref->getLastSongFilename();
		if ( restoreLastSong && (filename != "" )) {
			song = Song::load( filename );
			if (song == NULL) {
				QMessageBox::warning( this, "Hydrogen", trUtf8("Error restoring last song.") );
				song = Song::getEmptySong();
				song->setFilename( "" );
			}
		}
		else {
			song = Song::getEmptySong();
			song->setFilename( "" );
		}
	}

	h2app = new HydrogenApp( this, song );

	createMenuBar();

	h2app->setStatusBarMessage( trUtf8("Hydrogen Ready."), 10000 );

	initKeyInstMap();

	// we need to do all this to support the keyboard playing
	// for all the window modes
	h2app->getMixer()->installEventFilter (this);
	h2app->getPatternEditorPanel()->installEventFilter (this);
	h2app->getSongEditorPanel()->installEventFilter (this);
	h2app->getPlayerControl()->installEventFilter(this);
	h2app->getInstrumentProperties()->installEventFilter(this);

	QAccel *a = new QAccel( this );
        a->connectItem( a->insertItem(Key_Space), this, SLOT( onPlayStopAccelEvent() ) );
        a->connectItem( a->insertItem(Key_Backspace), this, SLOT( onRestartAccelEvent() ) );
        a->connectItem( a->insertItem(Key_Plus), this, SLOT( onBPMPlusAccelEvent() ) );
        a->connectItem( a->insertItem(Key_Minus), this, SLOT( onBPMMinusAccelEvent() ) );

	a->connectItem( a->insertItem(Key_S + CTRL), this, SLOT( onSaveAccelEvent() ) );
	a->connectItem( a->insertItem(Key_O + CTRL), this, SLOT( onOpenAccelEvent() ) );

	a->connectItem( a->insertItem(Key_Backslash), this, SLOT( onTapTempoAccelEvent() ) );

	installEventFilter( this );
}



MainForm::~MainForm() {
	if ( ((Hydrogen::getInstance())->getState() == PLAYING) ) {
		(Hydrogen::getInstance())->stop();
	}

	hide();

	if (h2app != NULL) {
		delete h2app;
		h2app = NULL;
	}
}



///
/// Create the menubar
///
void MainForm::createMenuBar() {
	// menubar
	m_pMenubar = new QMenuBar( this, "m_pMenubar" );
	
	// FILE menu
	m_pFilePopupMenu = new QPopupMenu( this );			// FILE MENU
	m_pMenubar->insertItem( trUtf8("&File"), m_pFilePopupMenu );

	menuItem_file_new = new QAction( this, "menuItem_file_new" );
	menuItem_file_new->setText( trUtf8( "New" ) );
	menuItem_file_new->addTo( m_pFilePopupMenu );		// FILE->NEW
	connect( menuItem_file_new, SIGNAL( activated() ), this, SLOT( action_file_new() ) );
	
	m_pFilePopupMenu->insertSeparator();				// -----
	
	menuItem_file_open = new QAction( this, "menuItem_file_open" );
	menuItem_file_open->setText( trUtf8( "Open" ) );
	menuItem_file_open->addTo( m_pFilePopupMenu );		// FILE->OPEN
	connect( menuItem_file_open, SIGNAL( activated() ), this, SLOT( action_file_open() ) );
	
	menuItem_file_openDemo = new QAction( this, "menuItem_file_openDemo" );
	menuItem_file_openDemo->setText( trUtf8( "Open Demo" ) );
	menuItem_file_openDemo->addTo( m_pFilePopupMenu );	// FILE->OPEN_DEMO
	connect( menuItem_file_openDemo, SIGNAL( activated() ), this, SLOT( action_file_openDemo() ) );
	
	m_pRecentFilesPopupMenu = new QPopupMenu( this );
	m_pFilePopupMenu->insertItem( trUtf8( "Open recent" ), m_pRecentFilesPopupMenu );	// FILE->OPEN_RECENT
	
	m_pFilePopupMenu->insertSeparator();				// -----
	
	menuItem_file_save = new QAction( this, "menuItem_file_save" );
	menuItem_file_save->setText( trUtf8( "Save" ) );
	menuItem_file_save->addTo( m_pFilePopupMenu );		// FILE->SAVE_AS
	connect( menuItem_file_save, SIGNAL( activated() ), this, SLOT( action_file_save() ) );
	
	menuItem_file_save_as = new QAction( this, "menuItem_file_save_as" );
	menuItem_file_save_as->setText( trUtf8( "Save as..." ) );
	menuItem_file_save_as->addTo( m_pFilePopupMenu );		// FILE->SAVE_AS
	connect( menuItem_file_save_as, SIGNAL( activated() ), this, SLOT( action_file_save_as() ) );
	
	menuItem_file_export = new QAction( this, "menuItem_file_export" );
	menuItem_file_export->setText( trUtf8( "Export" ) );
	menuItem_file_export->addTo( m_pFilePopupMenu );		// FILE->EXPORT
	connect( menuItem_file_export, SIGNAL( activated() ), this, SLOT( action_file_export() ) );
	
#ifdef DEBUG_MESSAGES
	menuItem_file_export_midi = new QAction( this, "menuItem_file_export_midi" );
	menuItem_file_export_midi->setText( trUtf8( "Export MIDI file" ) );
	menuItem_file_export_midi->addTo( m_pFilePopupMenu );		// FILE->EXPORT_MIDI
	connect( menuItem_file_export_midi, SIGNAL( activated() ), this, SLOT( action_file_export_midi() ) );
#endif
	m_pFilePopupMenu->insertSeparator();				// -----
	
	menuItem_file_preferences = new QAction( this, "menuItem_file_preferences" );
	menuItem_file_preferences->setText( trUtf8( "Preferences" ) );
	menuItem_file_preferences->addTo( m_pFilePopupMenu );	// FILE->PREFERENCES
	connect( menuItem_file_preferences, SIGNAL( activated() ), this, SLOT( action_file_preferences() ) );
	
	m_pFilePopupMenu->insertSeparator();				// -----
	
	menuItem_file_quit = new QAction( this, "menuItem_file_quit" );
	menuItem_file_quit->setText( trUtf8( "Quit" ) );
	menuItem_file_quit->addTo( m_pFilePopupMenu);		// FILE->QUIT
	connect( menuItem_file_quit, SIGNAL( activated() ), this, SLOT( action_file_exit() ) );

	updateRecentUsedSongList();
	//~ FILE menu


	// WINDOW menu
	m_pWindowPopupMenu = new QPopupMenu( this );
	m_pMenubar->insertItem( trUtf8( "&Window" ), m_pWindowPopupMenu );
	
	menuItem_window_showMixer = new QAction( this, "menuItem_window_showMixer" );
	menuItem_window_showMixer->setText( trUtf8( "Show mixer" ) );
	menuItem_window_showMixer->addTo( m_pWindowPopupMenu );
	connect( menuItem_window_showMixer, SIGNAL( activated() ), this, SLOT( action_window_showMixer() ) );
	
	
	menuItem_window_showSongEditor = new QAction( this, "menuItem_window_showSongEditor" );
	menuItem_window_showSongEditor->setText( trUtf8( "Show song editor" ) );
	connect( menuItem_window_showSongEditor, SIGNAL( activated() ), this, SLOT( action_window_showSongEditor() ) );
	if ( ( PreferencesMng::getInstance() )->getInterfaceMode() != PreferencesMng::TOP_LEVEL ) {	// show only with child frame interface
		menuItem_window_showSongEditor->addTo( m_pWindowPopupMenu );
	}
	
	menuItem_window_showDrumkitManager = new QAction( this, "menuItem_window_showDrumkitManager" );
	menuItem_window_showDrumkitManager->setText( trUtf8( "Show drumkit manager" ) );
	menuItem_window_showDrumkitManager->addTo( m_pWindowPopupMenu );
	connect( menuItem_window_showDrumkitManager, SIGNAL( activated() ), this, SLOT( action_window_showDrumkitManager() ) );
	
	if ( ( PreferencesMng::getInstance() )->getInterfaceMode() != PreferencesMng::SINGLE_PANED ) {
		menuItem_window_showInstrumentEditor = new QAction( this, "menuItem_showInstrumentEditor" );
		menuItem_window_showInstrumentEditor->setText( trUtf8( "Show instrument editor" ) );
		menuItem_window_showInstrumentEditor->addTo( m_pWindowPopupMenu );
		connect( menuItem_window_showInstrumentEditor, SIGNAL( activated() ), this, SLOT( action_window_showInstrumentEditor() ) );
	}	
	m_pWindowPopupMenu->insertSeparator();
	
	menuItem_window_showAudioEngineInfo = new QAction( this, "menuItem_window_showAudioEngineInfo" );
	menuItem_window_showAudioEngineInfo->setText( trUtf8( "Show audio engine info" ) );
	menuItem_window_showAudioEngineInfo->addTo( m_pWindowPopupMenu );
	connect( menuItem_window_showAudioEngineInfo, SIGNAL( activated() ), this, SLOT( action_window_showAudioEngineInfo() ) );

	//~ WINDOW menu
	

#ifdef DEBUG_MESSAGES
	// DEBUG menu
	m_pDebugPopupMenu = new QPopupMenu( this );
	m_pMenubar->insertItem( trUtf8("&Debug"), m_pDebugPopupMenu );
	
	menuItem_debug_printObjects = new QAction( this, "menuItem_debug_printObjects" );
	menuItem_debug_printObjects->setText( trUtf8( "Print Objects" ) );
	menuItem_debug_printObjects->addTo( m_pDebugPopupMenu );
	connect( menuItem_debug_printObjects, SIGNAL( activated() ), this, SLOT( action_debug_printObjects() ) );
	//~ DEBUG menu
#endif


	// HELP menu
	m_pHelpPopupMenu = new QPopupMenu( this );
	m_pMenubar->insertItem( trUtf8( "?" ), m_pHelpPopupMenu );
	
	menuItem_help_manual = new QAction( this, "menuItem_help_manual" );
	menuItem_help_manual->setText( trUtf8( "User manual" ) );
	menuItem_help_manual->addTo( m_pHelpPopupMenu );
	connect( menuItem_help_manual, SIGNAL( activated() ), this, SLOT( action_help_manual() ) );

	m_pHelpPopupMenu->insertSeparator();
	
	menuItem_help_about = new QAction( this, "menuItem_help_about" );
	menuItem_help_about->setText( trUtf8( "About" ) );
	menuItem_help_about->addTo( m_pHelpPopupMenu );
	connect( menuItem_help_about, SIGNAL( activated() ), this, SLOT( action_help_about() ) );
	//~ HELP menu
	


}



void MainForm::action_file_exit() {
	if ( (h2app->getSong())->isModified() ) {
		switch(
				QMessageBox::information( this, "Hydrogen",
						trUtf8("\nThe song has unsaved changes\n"
						"Do you want to save the changes before exiting?\n"),
						trUtf8("&Save"), trUtf8("&Discard"), trUtf8("Cancel"),
						0,      // Enter == button 0
						2 ) ) { // Escape == button 2
			case 0:
				// Save clicked or Alt+S pressed or Enter pressed.
				if (h2app->getSong()->getFilename() != "") {
					action_file_save();
				} else {
					action_file_save_as();
				}
				// save
				break;
			case 1:
				// Discard clicked or Alt+D pressed
				// don't save but exit
				break;
			case 2:
				// Cancel clicked or Alt+C pressed or Escape pressed
				// don't exit
				return;
				break;
		}
	}

	closeAll();
}



void MainForm::action_file_new() {
	if ( ((Hydrogen::getInstance())->getState() == PLAYING) ) {
		(Hydrogen::getInstance())->stop();
	}

	if ( (h2app->getSong())->isModified() ) {
		switch(
				QMessageBox::information( this, "Hydrogen",
						trUtf8("\nThe document contains unsaved changes\n"
						"Do you want to save the changes before exiting?\n"),
						trUtf8("&Save"), trUtf8("&Discard"), trUtf8("Cancel"),
						0,      // Enter == button 0
						2 ) ) { // Escape == button 2
			case 0: // Save clicked or Alt+S pressed or Enter pressed.
				if (h2app->getSong()->getFilename() != "") {
					action_file_save();
				} else {
					// never been saved
					action_file_save_as();
				}
				// save
				break;
			case 1: // Discard clicked or Alt+D pressed
				// don't save but exit
				break;
			case 2: // Cancel clicked or Alt+C pressed or Escape pressed
				// don't exit
				return;
				break;
		}
	}

	Song * song = Song::getEmptySong();
	song->setFilename( "" );
	h2app->setSong(song);

}



void MainForm::action_file_save_as() 
{
	if ( ((Hydrogen::getInstance())->getState() == PLAYING) ) {
		(Hydrogen::getInstance())->stop();
	}

	QFileDialog *fd = new QFileDialog(this, "File Dialog", TRUE);
	fd->setMode(QFileDialog::AnyFile);
	fd->setFilter( trUtf8("Hydrogen Song (*.h2song)") );

	fd->setCaption( trUtf8( "Save song" ) );
	fd->setIcon( QPixmap( QString(IMG_PATH) + QString( "/img/icon32.png") ) );

	Song *song = h2app->getSong();
	QString defaultFilename;
	QString lastFilename = (song->getFilename()).c_str();

	if (lastFilename == "") {
		defaultFilename = (Hydrogen::getInstance()->getSong()->getName()).c_str();
		defaultFilename += ".h2song";
	} 
	else {
		defaultFilename = lastFilename;
	}

	fd->setSelection(defaultFilename);

	QString filename = "";
	if (fd->exec() == QDialog::Accepted) {
		filename = fd->selectedFile();
	}

	if (filename != "") {


		QString sNewFilename = filename;
		if ( sNewFilename.endsWith(".h2song") == false ) {
			filename += ".h2song";
		}

		song->setFilename(filename.latin1());
		action_file_save();
	}
}



void MainForm::action_file_save()
{
	if ( ((Hydrogen::getInstance())->getState() == PLAYING) ) {
		(Hydrogen::getInstance())->stop();
	}

	Song *song = h2app->getSong();
	QString filename = (song->getFilename()).c_str();

	if (filename == "") {
		// just in case!
		return action_file_save_as();
	}

	LocalFileMng mng;
	song->save( filename.latin1() );
	( PreferencesMng::getInstance() )->setLastSongFilename( song->getFilename() );

	// add the new loaded song in the "last used song" vector
	PreferencesMng *pPref = PreferencesMng::getInstance();
	vector<string> recentFiles = pPref->getRecentFiles();
	recentFiles.insert( recentFiles.begin(), filename.latin1() );
	pPref->setRecentFiles( recentFiles );

	updateRecentUsedSongList();
}




void MainForm::action_help_about() {
	QWidget *parent = this;
	if (workspace) {
		parent = workspace;
	}

	// show modal dialog
	AboutDialog *dialog = new AboutDialog( NULL );
	dialog->exec();
}




void MainForm::action_help_manual() {
	QWidget *parent = this;
	if (workspace) {
		parent = workspace;
	}

	h2app->getHelpBrowser()->hide();
	h2app->getHelpBrowser()->show();
}




void MainForm::action_file_open() {
	if ( ((Hydrogen::getInstance())->getState() == PLAYING) ) {
		(Hydrogen::getInstance())->stop();
	}

	if ( (h2app->getSong())->isModified() ) {
		switch(
			QMessageBox::information( this, "Hydrogen",
					trUtf8("\nThe document contains unsaved changes\n"
					"Do you want to save the changes before exiting?\n"),
					trUtf8("&Save"), trUtf8("&Discard"), trUtf8("Cancel"),
					0,      // Enter == button 0
					2 ) ) { // Escape == button 2
			case 0: // Save clicked or Alt+S pressed or Enter pressed.
				if (h2app->getSong()->getFilename() != "") {
					action_file_save();
				}
				else {
					action_file_save_as();
				}
				// save
				break;
			case 1: // Discard clicked or Alt+D pressed
				// don't save but exit
				break;
			case 2: // Cancel clicked or Alt+C pressed or Escape pressed
				// don't exit
				return;
				break;
		}
	}

	static QString lastUsedDir = "";

	QFileDialog *fd = new QFileDialog(this, "File Dialog", TRUE);
	fd->setMode(QFileDialog::ExistingFile);
	fd->setFilter( trUtf8("Hydrogen Song (*.h2song)") );
	fd->setDir( lastUsedDir );

	fd->setCaption( trUtf8( "Open song" ) );
	fd->setIcon( QPixmap( QString(IMG_PATH) + QString( "/img/icon32.png") ) );

	/// \todo impostare il preview
	/*
	fd->setContentsPreviewEnabled( TRUE );
	fd->setContentsPreview( "uno", "due" );
	fd->setPreviewMode( QFileDialog::Contents );
	*/

	QString filename = "";
	if (fd->exec() == QDialog::Accepted) {
		filename = fd->selectedFile();
		lastUsedDir = fd->dirPath();
	}


	if (filename != "") {
		openSongFile( filename.latin1() );
	}
}




/// \todo parametrizzare il metodo action_file_open ed eliminare il seguente...
void MainForm::action_file_openDemo() {
	if ( ((Hydrogen::getInstance())->getState() == PLAYING) ) {
		(Hydrogen::getInstance())->stop();
	}

	if ( (h2app->getSong())->isModified() ) {
		switch(
			QMessageBox::information( this, "Hydrogen",
					trUtf8("\nThe document contains unsaved changes\n"
					"Do you want to save the changes before exiting?\n"),
					trUtf8("&Save"), trUtf8("&Discard"), trUtf8("Cancel"),
					0,      // Enter == button 0
					2 ) ) { // Escape == button 2
			case 0: // Save clicked or Alt+S pressed or Enter pressed.
				if (h2app->getSong()->getFilename() != "") {
					action_file_save_as();
				}
				else {
					action_file_save_as();
				}
				// save
				break;
			case 1: // Discard clicked or Alt+D pressed
				// don't save but exit
				break;
			case 2: // Cancel clicked or Alt+C pressed or Escape pressed
				// don't exit
				return;
				break;
		}
	}


	QFileDialog *fd = new QFileDialog(this, "File Dialog", TRUE);
	fd->setMode(QFileDialog::ExistingFile);
	fd->setFilter( trUtf8("Hydrogen Song (*.h2song)") );

	fd->setCaption( trUtf8( "Open song" ) );
	fd->setIcon( QPixmap( QString(IMG_PATH) + QString( "/img/icon32.png") ) );

	/// \todo impostare il preview
	/*
	fd->setContentsPreviewEnabled( TRUE );
	fd->setContentsPreview( "uno", "due" );
	fd->setPreviewMode( QFileDialog::Contents );
	*/
	string demoPath = (PreferencesMng::getInstance())->getDemoPath();
	fd->setSelection( QString( demoPath.c_str() ) );


	QString filename = "";
	if (fd->exec() == QDialog::Accepted) {
		filename = fd->selectedFile();
	}


	if (filename != "") {
		openSongFile( filename.latin1() );
		( h2app->getSong() )->setFilename( "" );
	}
}



void MainForm::action_file_preferences() {
	if ( ((Hydrogen::getInstance())->getState() == PLAYING) ) {
		(Hydrogen::getInstance())->stop();
	}

	h2app->showPreferencesDialog();
}




void MainForm::action_window_showMixer() {
	
	h2app->showMixer( true );
}




void MainForm::action_window_showAudioEngineInfo() {
	h2app->showAudioEngineInfoForm();
}



///
/// Shows the song editor
///
void MainForm::action_window_showSongEditor() {
	h2app->getSongEditorPanel()->hide();
	h2app->getSongEditorPanel()->show();
}




///
/// Window close event
///
void MainForm::closeEvent( QCloseEvent* ev ) {
	if ( (h2app->getSong())->isModified() ) {
		switch(
			QMessageBox::information( this, "Hydrogen",
					trUtf8("\nThe song has unsaved changes\n"
					"Do you want to save the changes before exiting?\n"),
					trUtf8("&Save"), trUtf8("&Discard"), trUtf8("Cancel"),
					0,      // Enter == button 0
					2 ) ) { // Escape == button 2
			case 0:
				// Save clicked or Alt+S pressed or Enter pressed.
				if (h2app->getSong()->getFilename() != "") {
					action_file_save();
				}
				else {
					action_file_save_as();
				}
				// save
				break;
			case 1:
				// Discard clicked or Alt+D pressed
				// don't save but exit
				break;
			case 2:
				// Cancel clicked or Alt+C pressed or Escape pressed
				// don't exit
				return;
				break;
		}
	}


	if ( ((Hydrogen::getInstance())->getState() == PLAYING) ) {
		(Hydrogen::getInstance())->stop();
	}


	closeAll();
//	this->app->quit();
	ev->accept();
}



void MainForm::action_file_export() {
	ExportSongDialog *dialog = new ExportSongDialog(this);
	dialog->exec();
	delete dialog;
}



void MainForm::action_window_showDrumkitManager() {
	h2app->getDrumkitManager()->hide();
	h2app->getDrumkitManager()->show();
}




void MainForm::customEvent( QCustomEvent *ev ) {
	if ( ev->type() != H2_TEXT_EVENT ) {	// Must be a H2TextEvent
		return;
	}
	QString message = ((H2TextEvent*)ev)->getText();

	if (message == QString( "ErrorMsg" ) ) {
		QString msg = ((H2TextEvent*)ev)->getErrorMsg();
		QMessageBox::information( this, "Hydrogen", msg );
	}

}




void MainForm::closeAll() {

	// save window properties in the preferences files
	PreferencesMng *pref = PreferencesMng::getInstance();

	// mainform
	WindowProperties mainFormProp;
	mainFormProp.x = x();
	mainFormProp.y = y();
	mainFormProp.height = height();
	mainFormProp.width = width();
	pref->setMainFormProperties( mainFormProp );

	// Instrument properties
	WindowProperties instrumentEditorProp;
	if ( h2app->getInstrumentProperties()->parentWidget() != 0 ) {	// use the new MDI interface
		instrumentEditorProp.x = h2app->getInstrumentProperties()->parentWidget()->x();
		instrumentEditorProp.y = h2app->getInstrumentProperties()->parentWidget()->y();
	}
	else {
		instrumentEditorProp.x = h2app->getInstrumentProperties()->x();
		instrumentEditorProp.y = h2app->getInstrumentProperties()->y();
	}
	instrumentEditorProp.visible = h2app->getInstrumentProperties()->isShown();
	pref->setInstrumentEditorProperties( instrumentEditorProp );
	
	// Save mixer properties
	WindowProperties mixerProp;
	if ( h2app->getMixer()->parentWidget() != 0 ) {	// use the new MDI interface
		mixerProp.x = h2app->getMixer()->parentWidget()->x();
		mixerProp.y = h2app->getMixer()->parentWidget()->y();
//		mixerProp.width = h2app->getMixer()->parentWidget()->width();
//		mixerProp.height = h2app->getMixer()->parentWidget()->height();
		mixerProp.width = h2app->getMixer()->width();
		mixerProp.height = h2app->getMixer()->height();
	}
	else {
		mixerProp.x = h2app->getMixer()->x();
		mixerProp.y = h2app->getMixer()->y();
		mixerProp.width = h2app->getMixer()->width();
		mixerProp.height = h2app->getMixer()->height();
	}
	mixerProp.visible = h2app->getMixer()->isShown();
	pref->setMixerProperties( mixerProp );

	// save pattern editor properties
	WindowProperties patternEditorProp;
	if ( h2app->getPatternEditorPanel()->parentWidget() != 0 ) {	// use the new MDI interface
		patternEditorProp.x = h2app->getPatternEditorPanel()->parentWidget()->x();
		patternEditorProp.y = h2app->getPatternEditorPanel()->parentWidget()->y();
		patternEditorProp.width = h2app->getPatternEditorPanel()->width();
		patternEditorProp.height = h2app->getPatternEditorPanel()->height();
	}
	else {
		patternEditorProp.x = h2app->getPatternEditorPanel()->x();
		patternEditorProp.y = h2app->getPatternEditorPanel()->y();
		patternEditorProp.width = h2app->getPatternEditorPanel()->width();
		patternEditorProp.height = h2app->getPatternEditorPanel()->height();
	}
	patternEditorProp.visible = h2app->getPatternEditorPanel()->isShown();
	pref->setPatternEditorProperties( patternEditorProp );

	// save song editor properties
	WindowProperties songEditorProp;
	if ( h2app->getSongEditorPanel()->parentWidget() != 0 ) {	// use the new MDI interface
		songEditorProp.x = h2app->getSongEditorPanel()->parentWidget()->x();
		songEditorProp.y = h2app->getSongEditorPanel()->parentWidget()->y();
	}
	else {
		songEditorProp.x = h2app->getSongEditorPanel()->x();
		songEditorProp.y = h2app->getSongEditorPanel()->y();
	}
	songEditorProp.width = h2app->getSongEditorPanel()->width();
	songEditorProp.height = h2app->getSongEditorPanel()->height();

//	cout << "song editor size=" << h2app->getSongEditorPanel()->width() << "x" << h2app->getSongEditorPanel()->height() << endl;
	QSize size = h2app->getSongEditorPanel()->frameSize();
//	cout << "song editor size2=" << size.width() << "x" << size.height() << endl;

	songEditorProp.visible = h2app->getSongEditorPanel()->isShown();
	pref->setSongEditorProperties( songEditorProp );

	
	// save drumkit manager properties
	WindowProperties drumkitMngProp;
	if ( h2app->getDrumkitManager()->parentWidget() != 0 ) {	// use the new MDI interface
		drumkitMngProp.x = h2app->getDrumkitManager()->parentWidget()->x();
		drumkitMngProp.y = h2app->getDrumkitManager()->parentWidget()->y();
		drumkitMngProp.width = h2app->getDrumkitManager()->parentWidget()->width();
		drumkitMngProp.height = h2app->getDrumkitManager()->parentWidget()->height();
	}
	else {
		drumkitMngProp.x = h2app->getDrumkitManager()->x();
		drumkitMngProp.y = h2app->getDrumkitManager()->y();
		drumkitMngProp.width = h2app->getDrumkitManager()->width();
		drumkitMngProp.height = h2app->getDrumkitManager()->height();
	}
	drumkitMngProp.visible = h2app->getDrumkitManager()->isShown();
	pref->setDrumkitManagerProperties( drumkitMngProp );

	
	// save audio engine info properties
	WindowProperties audioEngineInfoProp;
	if ( h2app->getAudioEngineInfoForm()->parentWidget() != 0 ) {	// use the new MDI interface
		audioEngineInfoProp.x = h2app->getAudioEngineInfoForm()->parentWidget()->x();
		audioEngineInfoProp.y = h2app->getAudioEngineInfoForm()->parentWidget()->y();
	}
	else {
		audioEngineInfoProp.x = h2app->getAudioEngineInfoForm()->x();
		audioEngineInfoProp.y = h2app->getAudioEngineInfoForm()->y();
	}
	audioEngineInfoProp.visible = h2app->getAudioEngineInfoForm()->isShown();
	pref->setAudioEngineInfoProperties( audioEngineInfoProp );


	// save LADSPA FX window properties
	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		WindowProperties prop;
		if (h2app->getAudioEngineInfoForm()->parentWidget() != 0 ) {	// use MDI interface
			prop.x = h2app->getLadspaFXProperties(nFX)->parentWidget()->x();
			prop.y = h2app->getLadspaFXProperties(nFX)->parentWidget()->y();
		}
		else {
			prop.x = h2app->getLadspaFXProperties(nFX)->x();
			prop.y = h2app->getLadspaFXProperties(nFX)->y();
		}
		prop.visible= h2app->getLadspaFXProperties(nFX)->isShown();
		pref->setLadspaProperties(nFX, prop);
	}

	m_pQApp->quit();
}



// keybindings..

void MainForm::onPlayStopAccelEvent()
{
	int nState = Hydrogen::getInstance()->getState();
	switch (nState) {
		case READY:
			Hydrogen::getInstance()->start();
			break;

		case PLAYING:
			Hydrogen::getInstance()->stop();
			break;

		default:
			cerr << "[MainForm::onPlayStopAccelEvent()] Unhandled case." << endl;
	}
}



void MainForm::onRestartAccelEvent()
{
	Hydrogen* pEngine = ( Hydrogen::getInstance() );
	pEngine->setPatternPos( 0 );
}



void MainForm::onBPMPlusAccelEvent()
{
	Hydrogen* pEngine = ( Hydrogen::getInstance() );
	pEngine->lockEngine( "MainForm::onBPMPlusAccelEvent" );

	Song* pSong = pEngine->getSong();
	if (pSong->getBpm() < 300) {
		pEngine->setBPM( pSong->getBpm() + 1 );
	}
	pEngine->unlockEngine();
}



void MainForm::onBPMMinusAccelEvent()
{
	Hydrogen* pEngine = ( Hydrogen::getInstance() );
	pEngine->lockEngine( "MainForm::onBPMMinusAccelEvent" );

	Song* pSong = pEngine->getSong();
	if (pSong->getBpm() > 40 ) {
		pEngine->setBPM( pSong->getBpm() - 1 );
	}
	pEngine->unlockEngine();
}



void MainForm::onSaveAsAccelEvent()
{
	action_file_save_as();
}



void MainForm::onSaveAccelEvent()
{
	action_file_save();
}



void MainForm::onOpenAccelEvent()
{
	action_file_open();
}



void MainForm::onTapTempoAccelEvent()
{
	infoLog( "tap tempo" );
	static timeval oldTimeVal;

	struct timeval now;
	gettimeofday(&now, NULL);

	float fInterval = (now.tv_sec - oldTimeVal.tv_sec) * 1000.0 + (now.tv_usec - oldTimeVal.tv_usec) / 1000.0;

	oldTimeVal = now;

	if ( fInterval < 1000.0 ) {
		( Hydrogen::getInstance() )->setTapTempo( fInterval );
	}
}



void MainForm::updateRecentUsedSongList()
{
//	infoLog( "updateRecentUsedSongList" );
	m_pRecentFilesPopupMenu->clear();

	PreferencesMng *pPref = PreferencesMng::getInstance();
	vector<string> recentUsedSongs = pPref->getRecentFiles();

	string sFilename = "";

	if ( recentUsedSongs.size() > 0 ) {
		sFilename = recentUsedSongs[ 0 ];
		if ( sFilename != "" ) {
			m_pRecentFileAction0 = new QAction( this, "recent_file0" );
			m_pRecentFileAction0->setText( QString( sFilename.c_str() ) );
			m_pRecentFileAction0->addTo( m_pRecentFilesPopupMenu );
			connect( m_pRecentFileAction0, SIGNAL( activated() ), this, SLOT( action_file_open_recent0() ) );
		}
	}

	if ( recentUsedSongs.size() > 1 ) {
		sFilename = recentUsedSongs[ 1 ];
		if ( sFilename != "" ) {
			m_pRecentFileAction1 = new QAction( this, "recent_file1" );
			m_pRecentFileAction1->setText( QString( sFilename.c_str() ) );
			m_pRecentFileAction1->addTo( m_pRecentFilesPopupMenu );
			connect( m_pRecentFileAction1, SIGNAL( activated() ), this, SLOT( action_file_open_recent1() ) );
		}
	}

	if ( recentUsedSongs.size() > 2 ) {
		sFilename = recentUsedSongs[ 2 ];
		if ( sFilename != "" ) {
			m_pRecentFileAction2 = new QAction( this, "recent_file2" );
			m_pRecentFileAction2->setText( QString( sFilename.c_str() ) );
			m_pRecentFileAction2->addTo( m_pRecentFilesPopupMenu );
			connect( m_pRecentFileAction2, SIGNAL( activated() ), this, SLOT( action_file_open_recent2() ) );
		}
	}

	if ( recentUsedSongs.size() > 3 ) {
		sFilename = recentUsedSongs[ 3 ];
		if ( sFilename != "" ) {
			m_pRecentFileAction3 = new QAction( this, "recent_file3" );
			m_pRecentFileAction3->setText( QString( sFilename.c_str() ) );
			m_pRecentFileAction3->addTo( m_pRecentFilesPopupMenu );
			connect( m_pRecentFileAction3, SIGNAL( activated() ), this, SLOT( action_file_open_recent3() ) );
		}
	}

	if ( recentUsedSongs.size() > 4 ) {
		sFilename = recentUsedSongs[ 4 ];
		if ( sFilename != "" ) {
			m_pRecentFileAction4 = new QAction( this, "recent_file4" );
			m_pRecentFileAction4->setText( QString( sFilename.c_str() ) );
			m_pRecentFileAction4->addTo( m_pRecentFilesPopupMenu );
			connect( m_pRecentFileAction4, SIGNAL( activated() ), this, SLOT( action_file_open_recent4() ) );
		}
	}
}



void MainForm::action_file_open_recent0()
{
	openSongFile( m_pRecentFileAction0->text().latin1() );
}



void MainForm::action_file_open_recent1()
{
	openSongFile( m_pRecentFileAction1->text().latin1() );
}



void MainForm::action_file_open_recent2()
{
	openSongFile( m_pRecentFileAction2->text().latin1() );
}



void MainForm::action_file_open_recent3()
{
	openSongFile( m_pRecentFileAction3->text().latin1() );
}



void MainForm::action_file_open_recent4()
{
	openSongFile( m_pRecentFileAction4->text().latin1() );
}



void MainForm::openSongFile( string sFilename )
{
	LocalFileMng mng;
	Song *pSong = mng.loadSong( sFilename );
	if ( pSong == NULL ) {
		QMessageBox::information( this, "Hydrogen", trUtf8("Error loading song.") );
		return;
	}

	// add the new loaded song in the "last used song" vector
	PreferencesMng *pPref = PreferencesMng::getInstance();
	vector<string> recentFiles = pPref->getRecentFiles();
	recentFiles.insert( recentFiles.begin(), sFilename );
	pPref->setRecentFiles( recentFiles );

	h2app->setSong( pSong );

	updateRecentUsedSongList();
}



void MainForm::initKeyInstMap()
{
	int instr = 0;
	keycodeInstrumentMap[Qt::Key_Z] = instr++;
	keycodeInstrumentMap[Qt::Key_S] = instr++;
	keycodeInstrumentMap[Qt::Key_X] = instr++;
	keycodeInstrumentMap[Qt::Key_D] = instr++;
	keycodeInstrumentMap[Qt::Key_C] = instr++;
	keycodeInstrumentMap[Qt::Key_V] = instr++;
	keycodeInstrumentMap[Qt::Key_G] = instr++;
	keycodeInstrumentMap[Qt::Key_B] = instr++;
	keycodeInstrumentMap[Qt::Key_H] = instr++;
	keycodeInstrumentMap[Qt::Key_N] = instr++;
	keycodeInstrumentMap[Qt::Key_J] = instr++;
	keycodeInstrumentMap[Qt::Key_M] = instr++;

	keycodeInstrumentMap[Qt::Key_Q] = instr++;
	keycodeInstrumentMap[Qt::Key_2] = instr++;
	keycodeInstrumentMap[Qt::Key_W] = instr++;
	keycodeInstrumentMap[Qt::Key_3] = instr++;
	keycodeInstrumentMap[Qt::Key_E] = instr++;
	keycodeInstrumentMap[Qt::Key_R] = instr++;
	keycodeInstrumentMap[Qt::Key_5] = instr++;
	keycodeInstrumentMap[Qt::Key_T] = instr++;
	keycodeInstrumentMap[Qt::Key_6] = instr++;
	keycodeInstrumentMap[Qt::Key_Y] = instr++;
	keycodeInstrumentMap[Qt::Key_7] = instr++;
	keycodeInstrumentMap[Qt::Key_U] = instr++;


	/*
	// QWERTY etc.... rows of the keyboard
	keycodeInstrumentMap[Qt::Key_Q] = instr++;
	keycodeInstrumentMap[Qt::Key_W] = instr++;
	keycodeInstrumentMap[Qt::Key_E] = instr++;
	keycodeInstrumentMap[Qt::Key_R] = instr++;
	keycodeInstrumentMap[Qt::Key_T] = instr++;
	keycodeInstrumentMap[Qt::Key_Y] = instr++;
	keycodeInstrumentMap[Qt::Key_U] = instr++;
	keycodeInstrumentMap[Qt::Key_I] = instr++;
	keycodeInstrumentMap[Qt::Key_O] = instr++;
	keycodeInstrumentMap[Qt::Key_P] = instr++;
	keycodeInstrumentMap[Qt::Key_BracketLeft] = instr++;
	keycodeInstrumentMap[Qt::Key_BracketRight] = instr++;
	keycodeInstrumentMap[Qt::Key_A] = instr++;
	keycodeInstrumentMap[Qt::Key_S] = instr++;
	keycodeInstrumentMap[Qt::Key_D] = instr++;
	keycodeInstrumentMap[Qt::Key_F] = instr++;
	keycodeInstrumentMap[Qt::Key_G] = instr++;
	keycodeInstrumentMap[Qt::Key_H] = instr++;
	keycodeInstrumentMap[Qt::Key_J] = instr++;
	keycodeInstrumentMap[Qt::Key_K] = instr++;
	keycodeInstrumentMap[Qt::Key_L] = instr++;
	keycodeInstrumentMap[Qt::Key_Semicolon] = instr++;
	keycodeInstrumentMap[Qt::Key_Apostrophe] = instr++;
	keycodeInstrumentMap[Qt::Key_Z] = instr++;
	keycodeInstrumentMap[Qt::Key_X] = instr++;
	keycodeInstrumentMap[Qt::Key_C] = instr++;
	keycodeInstrumentMap[Qt::Key_V] = instr++;
	keycodeInstrumentMap[Qt::Key_B] = instr++;
	keycodeInstrumentMap[Qt::Key_N] = instr++;
	keycodeInstrumentMap[Qt::Key_M] = instr++;
	keycodeInstrumentMap[Qt::Key_Comma] = instr++;
	keycodeInstrumentMap[Qt::Key_Period] = instr++;
*/
}



bool MainForm::eventFilter( QObject *o, QEvent *e )
{
	if ( e->type() == QEvent::KeyPress) {
		// special processing for key press
//		PreferencesMng *pref = PreferencesMng::getInstance();

		QKeyEvent *k = (QKeyEvent *)e;

		// qDebug( "Got key press for instrument '%c'", k->ascii() );

		map<int,int>::iterator found = keycodeInstrumentMap.find (k->key());

		if (found != keycodeInstrumentMap.end()) {
			// insert note at the current column in time
			// if event recording enabled
			int row = (*found).second;
			Hydrogen* engine = Hydrogen::getInstance();

			float velocity = 0.8;
			float pan_L = 1.0;
			float pan_R = 1.0;

			engine->addRealtimeNote (row, velocity, pan_L, pan_R);

			return TRUE; // eat event
		}
		else {
			return FALSE; // let it go
		}
        }
	else {
		// standard event processing
		return FALSE;
        }
}



void MainForm::action_window_showInstrumentEditor()
{
	HydrogenApp::getInstance()->getInstrumentProperties()->hide();
	HydrogenApp::getInstance()->getInstrumentProperties()->show();
}



/// print the object map
void MainForm::action_debug_printObjects()
{
	infoLog( "[action_debug_printObjects]" );
	Object::printObjectMap();
}




void MainForm::action_file_export_midi()
{
	warningLog( "[action_file_export_midi] not implemented yet" );

	if ( ((Hydrogen::getInstance())->getState() == PLAYING) ) {
		(Hydrogen::getInstance())->stop();
	}

	QFileDialog *fd = new QFileDialog(this, "File Dialog", TRUE);
	fd->setMode(QFileDialog::AnyFile);
	fd->setFilter( trUtf8("Midi file (*.mid)") );

	fd->setCaption( trUtf8( "Export MIDI file" ) );
	fd->setIcon( QPixmap( QString(IMG_PATH) + QString( "/img/icon32.png") ) );

	QString sFilename = "";
	if ( fd->exec() == QDialog::Accepted ) {
		sFilename = fd->selectedFile();
	}

	if ( sFilename != "" ) {
		if ( sFilename.endsWith(".mid") == false ) {
			sFilename += ".mid";
		}
		
		Song *pSong = Hydrogen::getInstance()->getSong();
		
		// create the Standard Midi File object
		SMFWriter *pSmfWriter = new SMFWriter();
		pSmfWriter->save( sFilename.latin1(), pSong );
	
		delete pSmfWriter;
	}
}
