/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: PlayerControl.cpp,v 1.59 2004/07/19 09:20:11 comix Exp $
 *
 */


#include "PlayerControl.h"
#include "Button.h"
#include "HydrogenApp.h"
#include "CpuLoadWidget.h"
#include "MidiActivityWidget.h"

#include "lib/Hydrogen.h"
#include "lib/drivers/JackDriver.h"

PlayerControl::PlayerControl(QWidget *parent) : QWidget(parent), Object("PlayerControl")
{
	setMinimumSize( width, height );
	setMaximumSize( width, height );
	resize( width, height );

/*	// Background image
	string background_path = string( IMG_PATH ) + string( "/img/playerControlPanel/background.png" );
	bool ok = m_background.load( background_path.c_str() );
	if( ok == false ){
		errorLog( "Error loading pixmap" );
	}
	setBackgroundPixmap( m_background );
*/	
	// Background solid color
	UIStyle *pStyle = PreferencesMng::getInstance()->getDefaultUIStyle();
	QColor backgroundColor( pStyle->m_patternEditorPanel_backgroundColor.getRed(), pStyle->m_patternEditorPanel_backgroundColor.getGreen(), pStyle->m_patternEditorPanel_backgroundColor.getBlue() );
	this->setEraseColor( backgroundColor );

	
	
	// Play button
	string play_on_path = string( IMG_PATH ) + string( "/img/playerControlPanel/btn_play_on.png" );
	string play_off_path = string( IMG_PATH ) + string( "/img/playerControlPanel/btn_play_off.png" );
	string play_over_path = string( IMG_PATH ) + string( "/img/playerControlPanel/btn_play_over.png" );
	m_pPlayBtn = new ToggleButton(this, QSize(20, 20), play_on_path, play_off_path, play_over_path);
	m_pPlayBtn->move(5, 5);
	m_pPlayBtn->setPressed(false);
	QToolTip::add( m_pPlayBtn, trUtf8("Play") );
	connect(m_pPlayBtn, SIGNAL(clicked(Button*)), this, SLOT(playBtnClicked(Button*)));

	// Stop button
	string stop_on_path = string( IMG_PATH ) + string( "/img/playerControlPanel/btn_stop_on.png" );
	string stop_off_path = string( IMG_PATH ) + string( "/img/playerControlPanel/btn_stop_off.png" );
	string stop_over_path = string( IMG_PATH ) + string( "/img/playerControlPanel/btn_stop_over.png" );
	m_pStopBtn = new ToggleButton(this, QSize(20, 20), stop_on_path, stop_off_path, stop_over_path);
	m_pStopBtn->move(30, 5);
	m_pStopBtn->setPressed(true);
	QToolTip::add( m_pStopBtn, trUtf8("Stop") );
	connect(m_pStopBtn, SIGNAL(clicked(Button*)), this, SLOT(stopBtnClicked(Button*)));


	// Song mode button
	string songMode_on_path = string( IMG_PATH ) + string( "/img/playerControlPanel/btn_song_on.png" );
	string songMode_off_path = string( IMG_PATH ) + string( "/img/playerControlPanel/btn_song_off.png" );
	string songMode_over_path = string( IMG_PATH) + string( "/img/playerControlPanel/btn_song_over.png" );
	m_pSongModeBtn = new ToggleButton(this, QSize(20, 20), songMode_on_path, songMode_off_path, songMode_over_path);
	m_pSongModeBtn->move(95, 5);
	m_pSongModeBtn->setPressed(false);
	QToolTip::add( m_pSongModeBtn, trUtf8("Song Mode") );
	connect(m_pSongModeBtn, SIGNAL(clicked(Button*)), this, SLOT(songModeBtnClicked(Button*)));

	// Live mode button
	string liveMode_on_path = string( IMG_PATH ) + string( "/img/playerControlPanel/btn_live_on.png" );
	string liveMode_off_path = string( IMG_PATH ) + string( "/img/playerControlPanel/btn_live_off.png" );
	string liveMode_over_path = string( IMG_PATH ) + string( "/img/playerControlPanel/btn_live_over.png" );
	m_pLiveModeBtn = new ToggleButton(this, QSize(20, 20), liveMode_on_path, liveMode_off_path, liveMode_over_path);
	m_pLiveModeBtn->move(70, 5);
	m_pLiveModeBtn->setPressed(true);
	QToolTip::add( m_pLiveModeBtn, trUtf8("Pattern Mode") );
	connect(m_pLiveModeBtn, SIGNAL(clicked(Button*)), this, SLOT(liveModeBtnClicked(Button*)));


	// BPM
	m_pBpmSpinBox = new QSpinBox( this, "m_pBpmSpinBox" );
	m_pBpmSpinBox->setMaxValue( 300 );
	m_pBpmSpinBox->setMinValue( 40 );
	m_pBpmSpinBox->setLineStep( 1 );
	m_pBpmSpinBox->setValue( 120 );
	m_pBpmSpinBox->move( 135, 5 );
	m_pBpmSpinBox->resize( 45, 20 );
	m_pBpmSpinBox->setFocusPolicy(QWidget::ClickFocus);
	m_pBpmSpinBox->setEnabled( true );
	QToolTip::add( m_pBpmSpinBox, trUtf8("BPM") );
	connect(m_pBpmSpinBox, SIGNAL(valueChanged(int)), this, SLOT(bpmChanged()));


	m_pEngine = Hydrogen::getInstance();

	// CPU load widget
	m_pCpuLoadWidget = new CpuLoadWidget( this );
	m_pCpuLoadWidget->move( 200, 5 );

	// Midi Activity widget
	m_pMidiActivityWidget = new MidiActivityWidget( this );
	m_pMidiActivityWidget->move( 320, 5 );

	// Jack transport mode button
	string jackTransportMode_on_path = string( IMG_PATH ) + string( "/img/playerControlPanel/jackTransportBtn_on.png" );
	string jackTransportMode_off_path = string( IMG_PATH ) + string( "/img/playerControlPanel/jackTransportBtn_off.png" );
	string jackTransportMode_over_path = string( IMG_PATH ) + string( "/img/playerControlPanel/jackTransportBtn_over.png" );
	m_pJackTransportBtn = new ToggleButton(this, QSize(75, 20), jackTransportMode_on_path, jackTransportMode_off_path, jackTransportMode_over_path);
	m_pJackTransportBtn->move(380, 5);
	m_pJackTransportBtn->setPressed(true);
	QToolTip::add( m_pJackTransportBtn, trUtf8("Jack-transport on/off") );
	connect(m_pJackTransportBtn, SIGNAL(clicked(Button*)), this, SLOT(jackTransportBtnClicked(Button*)));


	QTimer *timer = new QTimer(this);
	connect(timer, SIGNAL(timeout()), this, SLOT(updatePlayerControl()));

	timer->start(200);	// update player control at 5 fps
}




PlayerControl::~PlayerControl() {
}





/// Start audio engine
void PlayerControl::playBtnClicked(Button* ref) {
	if (ref->isPressed()) {
		m_pStopBtn->setPressed(false);
		// Start audio m_pEngine
		m_pEngine->start();
		m_pStopBtn->setPressed(false);
		(HydrogenApp::getInstance())->setStatusBarMessage(trUtf8("Playing."), 5000);
	}
	else {
		m_pPlayBtn->setPressed(true);
		m_pStopBtn->setPressed(false);
	}
}




/// Stop audio engine
void PlayerControl::stopBtnClicked(Button* ref) {
	if (ref->isPressed()) {
		m_pPlayBtn->setPressed(false);
		m_pEngine->stop();
		m_pBpmSpinBox->setEnabled(true);
		(HydrogenApp::getInstance())->setStatusBarMessage(trUtf8("Stopped."), 5000);
	}
	else {
		m_pStopBtn->setPressed(true);
		m_pPlayBtn->setPressed(false);
		m_pBpmSpinBox->setEnabled(true);
	}
}




/// Set Song mode
void PlayerControl::songModeBtnClicked(Button* ref) {
	m_pEngine->stop();
	m_pEngine->setPatternPos( 0 );	// from start
	m_pEngine->getSong()->setMode( Song::SONG_MODE );
	m_pSongModeBtn->setPressed(true);
	m_pLiveModeBtn->setPressed(false);
	(HydrogenApp::getInstance())->setStatusBarMessage(trUtf8("Song mode selected."), 5000);
}




///Set Live mode
void PlayerControl::liveModeBtnClicked(Button* ref) {
	m_pEngine->stop();
	m_pEngine->getSong()->setMode( Song::PATTERN_MODE );
	m_pEngine->setNextPattern( m_pEngine->getSelectedPatternNumber() );	// imposto il pattern correntemente selezionato come il prossimo da suonare
	m_pSongModeBtn->setPressed(false);
	m_pLiveModeBtn->setPressed(true);
	(HydrogenApp::getInstance())->setStatusBarMessage(trUtf8("Pattern mode selected."), 5000);
}




void PlayerControl::updatePlayerControl() {
	int state = m_pEngine->getState();
	if (state == PLAYING ) {
		m_pPlayBtn->setPressed(true);
		m_pStopBtn->setPressed(false);
	}
	else {
		m_pPlayBtn->setPressed(false);
		m_pStopBtn->setPressed(true);
	}

	Song *song = m_pEngine->getSong();

	/// \todo IL nuovo componente dovra' gestire i float!!
	float bpm = song->getBpm();
	m_pBpmSpinBox->setValue( (int) bpm);

	if ( song->getMode() == Song::PATTERN_MODE ) {
		m_pLiveModeBtn->setPressed( true );
		m_pSongModeBtn->setPressed( false );
	}
	else {
		m_pLiveModeBtn->setPressed( false );
		m_pSongModeBtn->setPressed( true );
	}

	PreferencesMng *pPref = PreferencesMng::getInstance();

	if (pPref->getAudioDriver() == "Jack") {
		m_pJackTransportBtn->show();
		int nMode = pPref->getJackTransportMode();
		switch (nMode) {
		case PreferencesMng::NO_JACK_TRANSPORT:
			m_pJackTransportBtn->setPressed(false);
			break;

		case PreferencesMng::USE_JACK_TRANSPORT:
			m_pJackTransportBtn->setPressed(true);
			break;
		}
	}
	else {
		m_pJackTransportBtn->hide();
	}



}




void PlayerControl::bpmChanged() {
	int newBpmValue = m_pBpmSpinBox->value();
	m_pEngine->lockEngine("PlayerControl::bpmChanged");
//	m_pEngine->getSong()->setBpm(newBpmValue);

	m_pEngine->setBPM( newBpmValue );

	m_pEngine->unlockEngine();
}



void PlayerControl::jackTransportBtnClicked(Button* ref)
{
	PreferencesMng *pPref = PreferencesMng::getInstance();

	if (m_pJackTransportBtn->isPressed()) {
		m_pEngine->lockEngine( "PlayerControl::jackTransportBtnClicked" );
		pPref->setJackTransportMode(PreferencesMng::USE_JACK_TRANSPORT);
		m_pEngine->unlockEngine();
		(HydrogenApp::getInstance())->setStatusBarMessage(trUtf8("Jack-transport mode = On"), 5000);
	}
	else {
		m_pEngine->lockEngine( "PlayerControl::jackTransportBtnClicked" );
		pPref->setJackTransportMode(PreferencesMng::NO_JACK_TRANSPORT);
		m_pEngine->unlockEngine();
		(HydrogenApp::getInstance())->setStatusBarMessage(trUtf8("Jack-transport mode = Off"), 5000);
	}

}


