/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: PreferencesDialog.cpp,v 1.55 2004/06/29 10:56:47 comix Exp $
 *
 */


#include "PreferencesDialog.h"
#include "HydrogenApp.h"
#include "MainForm.h"

#include "qmessagebox.h"
#include "qstylefactory.h"

#include "lib/Hydrogen.h"
#include "lib/PreferencesMng.h"
#include "lib/drivers/AlsaMidiDriver.h"

/**
 * Constructor
 */
PreferencesDialog::PreferencesDialog(QWidget* parent) : PreferencesDialog_UI(parent, 0, true), Object( "PreferencesDialog" )
{
//	cout << "PreferencesDIalog INIT" << endl;
	setCaption( trUtf8( "Preferences" ) );
	setIcon( QPixmap( QString(IMG_PATH) + QString( "/img/icon32.png") ) );

	setMinimumSize( width(), height() );
	setMaximumSize( width(), height() );


	PreferencesMng *preferencesMng = PreferencesMng::getInstance();
	preferencesMng->loadPreferences();

	//m_pLadspaPathText->setText( preferencesMng->getLadspaPath().c_str() );


	// BUffer size
	bufferSizeTxt->setValue( preferencesMng->getBufferSize() );

	// AUdio Driver
	string audioDriver = preferencesMng->getAudioDriver();
	if (audioDriver == "Oss") {
		driverComboBox->setCurrentItem(0);
	}
	else if (audioDriver == "Jack") {
		driverComboBox->setCurrentItem(1);
	}
	else {
		errorLog( "Unknown audio driver from preferences [" + audioDriver + "]" );
	}

	// Sample rate
	int sampleRate = preferencesMng->getSampleRate();
	if (sampleRate == 22050) {
		sampleRateComboBox->setCurrentItem(0);
	}
	else if (sampleRate == 44100) {
		sampleRateComboBox->setCurrentItem(1);
	}
	else if (sampleRate == 48000) {
		sampleRateComboBox->setCurrentItem(2);
	}

	// oss device
	ossDeviceTxt->setText( QString( preferencesMng->getOssDevice().c_str() ) );


	// Use metronome
	if (preferencesMng->isMetronomeEnabled()) {
		metronomeEnabledCheckBox->setChecked(true);
	}
	else {
		metronomeEnabledCheckBox->setChecked(false);
	}

	// metronome volume
	uint metronomeVol = (uint)(preferencesMng->getMetronomeVolume() * 100.0);
	metronomeVolumeSpinBox->setValue(metronomeVol);


/*
	// jack transport
	int nMode = preferencesMng->getJackTransportMode();
	if (nMode == PreferencesMng::NO_JACK_TRANSPORT ) {
		transportCheckBox->setChecked( false );
	}
	else if ( nMode == PreferencesMng::USE_JACK_TRANSPORT ) {
		transportCheckBox->setChecked( true );
	}
	*/

	trackOutsCheckBox->setChecked(preferencesMng->isJackTrackOuts());
	connectDefaultsCheckBox->setChecked(preferencesMng->isJackConnectDefaults());

	updateDriverInfo();

	// max voices
	maxVoicesTxt->setValue( preferencesMng->getMaxNotes() );



	// Appearance tab
	QString applicationFamily = (preferencesMng->getApplicationFontFamily()).c_str();
	int applicationPointSize = preferencesMng->getApplicationFontPointSize();

	QFont applicationFont( applicationFamily, applicationPointSize );
	applicationFontLbl->setFont( applicationFont );
	applicationFontLbl->setText( applicationFamily + QString("  %1").arg( applicationPointSize ) );

	QString mixerFamily = (preferencesMng->getMixerFontFamily()).c_str();
	int mixerPointSize = preferencesMng->getMixerFontPointSize();
	QFont mixerFont( mixerFamily, mixerPointSize );
	mixerFontLbl->setFont( mixerFont );
	mixerFontLbl->setText( mixerFamily + QString("  %1").arg( mixerPointSize ) );


	float falloffSpeed = preferencesMng->getMixerFalloffSpeed();
	if (falloffSpeed == FALLOFF_SLOW) {
		mixerFalloffComboBox->setCurrentItem(0);
	}
	else if (falloffSpeed == FALLOFF_NORMAL) {
		mixerFalloffComboBox->setCurrentItem(1);
	}
	else if (falloffSpeed == FALLOFF_FAST) {
		mixerFalloffComboBox->setCurrentItem(2);
	}
	else {
		errorLog( "PreferencesDialog: wrong mixerFalloff value = " + toString(falloffSpeed) );
	}

	switch ( preferencesMng->getInterfaceMode() ) {
		case PreferencesMng::MDI:
			interfaceModeComboBox->setCurrentItem( 0 );
			break;
		
		case PreferencesMng::TOP_LEVEL:
			interfaceModeComboBox->setCurrentItem( 1 );
			break;	
			
		case PreferencesMng::SINGLE_PANED:
			interfaceModeComboBox->setCurrentItem( 2 );
			break;
		
		default:
			errorLog( "[INIT] Wrong interfaceMode: " + toString( preferencesMng->getInterfaceMode() ) );	
	}
	
	// Style
	QStringList list = QStyleFactory::keys();
	uint i = 0;
	for ( QStringList::Iterator it = list.begin(); it != list.end(); it++) {
		m_styleComboBox->insertItem( *it );
		string sStyle = (*it).latin1();
		if (sStyle == preferencesMng->getQTStyle() ) {
			m_styleComboBox->setCurrentItem( i );
		}
		i++;
	}


	// midi tab
	#ifndef USE_ALSA_SEQ
		midiPortChannelComboBox->setEnabled( false );
		midiPortComboBox->setEnabled( false );
	#endif

	#ifdef USE_ALSA_SEQ
		// list alsa output ports
		vector<MidiPortInfo> midiOutList = ( ( Hydrogen::getInstance() )->getMidiDriver() )->listAlsaOutputPorts();
		midiPortComboBox->insertItem( "None" );
		for (uint i = 0; i < midiOutList.size(); i++) {
			MidiPortInfo info = midiOutList[ i ];
			char tmp[200];
			sprintf( tmp, "%s [%d:%d]",
					info.name.c_str(),
					info.client,
					info.port
			);

			midiPortComboBox->insertItem( QString( tmp ) );

			if ( ( info.client == preferencesMng->getMidiDest_client() ) && ( info.port == preferencesMng->getMidiDest_port() ) ) {
				midiPortComboBox->setCurrentItem( i + 1 );
			}
		}


	#endif

	int midiPortChannel = preferencesMng->getMidiPortChannel();
	if ( midiPortChannel == -1 ) {
		midiPortChannelComboBox->setCurrentItem( 0 );
	}
	else {
		midiPortChannelComboBox->setCurrentItem( midiPortChannel + 1 );
	}



	// General tab
	restoreLastUsedSongCheckbox->setChecked( preferencesMng->isRestoreLastSongEnabled() );

	needDriverRestart = false;
}




/**
 * Destructor
 */
PreferencesDialog::~PreferencesDialog() {
//	cout << "PreferencesDIalog Destroy" << endl;
}




/**
 * Cancel Button clicked
 */
void PreferencesDialog::cancelBtnClicked() {
	PreferencesMng *preferencesMng = PreferencesMng::getInstance();
	preferencesMng->loadPreferences();	// reload old preferences
	reject();
}



/**
 * Ok button clicked
 */
void PreferencesDialog::okBtnClicked() {
	PreferencesMng *preferencesMng = PreferencesMng::getInstance();

	//preferencesMng->setLadspaPath( m_pLadspaPathText->text().latin1() );

	// Audio buffer size
	preferencesMng->setBufferSize( bufferSizeTxt->value() );

	// Selected audio driver
	int nDriver = driverComboBox->currentItem();
	if ( nDriver == 0 ) {	// OSS
		preferencesMng->setAudioDriver( "Oss" );
	}
	else if ( nDriver == 1 ) {	// JACK
		preferencesMng->setAudioDriver( "Jack" );
	}
	else {
		string selectedDriver = (driverComboBox->currentText()).latin1();
		errorLog( "Unknown driver = " + selectedDriver );
	}

	// sample rate
	uint sampleRate = 44100;
	string sampleRateTxt = (sampleRateComboBox->currentText()).latin1();
	if (sampleRateTxt == "22050") {
		sampleRate = 22050;
	}
	else if (sampleRateTxt == "44100") {
		sampleRate = 44100;
	}
	else if (sampleRateTxt == "48000") {
		sampleRate = 48000;
	}
	preferencesMng->setSampleRate(sampleRate);

	// oss device
	string ossDevice = ossDeviceTxt->text().latin1();
	if (ossDevice != preferencesMng->getOssDevice() ) {
		needDriverRestart = true;
	}
	preferencesMng->setOssDevice(ossDevice);

	// metronome
	bool isMetronomeEnabled = metronomeEnabledCheckBox->isChecked();
	preferencesMng->setUseMetronome(isMetronomeEnabled);

	float metronomeVolume = (metronomeVolumeSpinBox->value()) / 100.0;
	preferencesMng->setMetronomeVolume(metronomeVolume);


	// maxVoices
	preferencesMng->setMaxNotes(maxVoicesTxt->value() );


	// Mixer falloff
	QString falloffStr = mixerFalloffComboBox->currentText().latin1();
	if ( falloffStr== trUtf8("Slow") ) {
		preferencesMng->setMixerFalloffSpeed(FALLOFF_SLOW);
	}
	else if ( falloffStr == trUtf8("Normal") ) {
		preferencesMng->setMixerFalloffSpeed(FALLOFF_NORMAL);
	}
	else if ( falloffStr == trUtf8("Fast") ) {
		preferencesMng->setMixerFalloffSpeed(FALLOFF_FAST);
	}
	else {
		errorLog( "[okBtnClicked()] Unknown mixerFallOffSpeed: " + string(falloffStr.latin1()) );
	}

	// interface mode
	switch( interfaceModeComboBox->currentItem() ) {
		case PreferencesMng::TOP_LEVEL:
			infoLog( "[okBtnClicked] TOP LEVEL interface selected" );
			preferencesMng->setInterfaceMode( PreferencesMng::TOP_LEVEL );	
			break;
			
		case PreferencesMng::MDI:
			infoLog( "[okBtnClicked] MDI interface selected" );
			preferencesMng->setInterfaceMode( PreferencesMng::MDI );	
			break;
		
		case PreferencesMng::SINGLE_PANED:
			infoLog( "[okBtnClicked] SINGLE PANED interface selected" );
			preferencesMng->setInterfaceMode( PreferencesMng::SINGLE_PANED );	
			break;
			
		default:
			errorLog( "[okBtnClicked] Unknown interface mode: " + toString( interfaceModeComboBox->currentItem() ) + ", " + interfaceModeComboBox->currentText().latin1() );
	}
	




	// Midi channel
	if ( midiPortComboBox->currentItem() == 0 ) {
//		preferencesMng->setMidiPortChannel( -1 );
		preferencesMng->setMidiDest_name( "-" );
		preferencesMng->setMidiDest_client( -1 );
		preferencesMng->setMidiDest_port( -1 );
		needDriverRestart = true;
	}
	else {
		string newMidiDest = midiPortComboBox->currentText().latin1();

		char midiDest_client_old[200];
		sprintf( midiDest_client_old, "%s [%d:%d]",
				preferencesMng->getMidiDest_name().c_str(),
				preferencesMng->getMidiDest_client(),
				preferencesMng->getMidiDest_port()
		);

		#ifdef USE_ALSA_SEQ
			if ( string(midiDest_client_old) != newMidiDest ) {
				// find the selected midi port in the list
				vector<MidiPortInfo> midiOutList = ( ( Hydrogen::getInstance() )->getMidiDriver() )->listAlsaOutputPorts();
				for (uint i = 0; i < midiOutList.size(); i++) {
					MidiPortInfo info = midiOutList[ i ];
					char tmp[200];
					sprintf( tmp, "%s [%d:%d]", info.name.c_str(), info.client, info.port );

					if ( string(tmp) == newMidiDest ) {
						preferencesMng->setMidiDest_name( info.name );
						preferencesMng->setMidiDest_client( info.client );
						preferencesMng->setMidiDest_port( info.port );
					}
				}
				needDriverRestart = true;
			}
		#endif

	}
	int midiPortChannel = midiPortChannelComboBox->currentItem() - 1;
	if ( preferencesMng->getMidiPortChannel() != midiPortChannel ) {
		needDriverRestart = true;
	}
	preferencesMng->setMidiPortChannel( midiPortChannel );


/*	// jack transport
	if ( transportCheckBox->isChecked() ) {
		preferencesMng->setJackTransportMode( PreferencesMng::USE_JACK_TRANSPORT );
	}
	else {
		preferencesMng->setJackTransportMode( PreferencesMng::NO_JACK_TRANSPORT );
	}
*/
	preferencesMng->setJackTrackOuts( trackOutsCheckBox->isChecked() );
	preferencesMng->setJackConnectDefaults( connectDefaultsCheckBox->isChecked() );

	// General tab
	preferencesMng->setRestoreLastSongEnabled( restoreLastUsedSongCheckbox->isChecked() );


	preferencesMng->savePreferences();

	if (needDriverRestart) {
		(Hydrogen::getInstance())->restartDrivers();
	}
	accept();
}





/**
 *
 */
void PreferencesDialog::driverChanged() {
	string selectedDriver = (driverComboBox->currentText()).latin1();
	updateDriverInfo();
	needDriverRestart = true;
}






/**
 *
 */
void PreferencesDialog::updateDriverInfo() {
	QString info = "";

	bool oss_support = false;
	#ifdef OSS_SUPPORT
		oss_support = true;
	#endif

	bool jack_support = false;
	#ifdef JACK_SUPPORT
		jack_support = true;
	#endif

	int nDriver = driverComboBox->currentItem();
	if ( nDriver == 0 ) {	// OSS
		info += trUtf8("<b>Open Sound System</b><br>Simple audio driver [/dev/dsp]");
		if (oss_support == false) {
			info += trUtf8("<br><b><font color=\"red\">Not compiled</font></b>");
		}
		bufferSizeTxt->setEnabled( true );
		sampleRateComboBox->setEnabled( true );

		//oss driver widgets
		bufferSizeTxt->show();
		sampleRateComboBox->show();
		buffer_sizeLbl->show();
		sampleRateLbl->show();
		ossDeviceTxt->show();
		ossDeviceLbl->show();

		// jack driver widgets
//		transportCheckBox->hide();
		trackOutsCheckBox->hide();
		connectDefaultsCheckBox->hide();
	}
	else if ( nDriver == 1 ) {	// JACK
		info += trUtf8("<b>Jack Audio Connection Kit Driver</b><br>Low latency audio driver");
		if (jack_support == false) {
			info += trUtf8("<br><b><font color=\"red\">Not compiled</font></b>");
		}
		bufferSizeTxt->setEnabled( false );
		sampleRateComboBox->setEnabled( false );

		//oss driver widgets
		bufferSizeTxt->hide();
		sampleRateComboBox->hide();
		buffer_sizeLbl->hide();
		sampleRateLbl->hide();
		ossDeviceTxt->hide();
		ossDeviceLbl->hide();

		// jack driver widgets
//		transportCheckBox->show();
		trackOutsCheckBox->show();
		connectDefaultsCheckBox->show();
	}
	else {
		string selectedDriver = (driverComboBox->currentText()).latin1();
		errorLog( "Unknown driver = " + selectedDriver );
	}

	driverInfoLbl->setText(info);
}




/**
 *
 */
void PreferencesDialog::useMetronomeCheckboxChanged() {
	metronomeVolumeSpinBox->setEnabled(metronomeEnabledCheckBox->isChecked());
}






/**
 *
 */
void PreferencesDialog::selectApplicationFont() {
	PreferencesMng *preferencesMng = PreferencesMng::getInstance();

	QString family = (preferencesMng->getApplicationFontFamily()).c_str();
	int pointSize = preferencesMng->getApplicationFontPointSize();


	bool ok;
	QFont font = QFontDialog::getFont( &ok, QFont( family, pointSize ), this );
	if ( ok ) {
		// font is set to the font the user selected
		family = font.family();
		pointSize = font.pointSize();
		string familyStr = family.latin1();
		preferencesMng->setApplicationFontFamily(familyStr);
		preferencesMng->setApplicationFontPointSize(pointSize);


	} else {
		// the user cancelled the dialog; font is set to the initial
		// value, in this case Times, 12.
	}

	QFont newFont(family, pointSize);
	applicationFontLbl->setFont(newFont);
	applicationFontLbl->setText(family + QString("  %1").arg(pointSize));
}




/**
 *
 */
void PreferencesDialog::bufferSizeChanged() {
	needDriverRestart = true;
}




/**
 *
 */
void PreferencesDialog::sampleRateChanged() {
	needDriverRestart = true;
}



/**
 *
 */
void PreferencesDialog::restartDriverBtnClicked() {
	// update audio driver preferences

	PreferencesMng *preferencesMng = PreferencesMng::getInstance();

	// Audio buffer size
	preferencesMng->setBufferSize( bufferSizeTxt->value() );

	// Selected audio driver
	string selectedDriver = (driverComboBox->currentText()).latin1();
	preferencesMng->setAudioDriver(selectedDriver);

	// sample rate
	uint sampleRate = 44100;
	string sampleRateTxt = (sampleRateComboBox->currentText()).latin1();
	if (sampleRateTxt == "22050") {
		sampleRate = 22050;
	}
	else if (sampleRateTxt == "44100") {
		sampleRate = 44100;
	}
	else if (sampleRateTxt == "48000") {
		sampleRate = 48000;
	}
	preferencesMng->setSampleRate(sampleRate);

	(Hydrogen::getInstance())->restartDrivers();
	needDriverRestart = false;
}



/**
 *
 */
void PreferencesDialog::selectMixerFont() {
	PreferencesMng *preferencesMng = PreferencesMng::getInstance();

	QString family = (preferencesMng->getMixerFontFamily()).c_str();
	int pointSize = preferencesMng->getMixerFontPointSize();

	bool ok;
	QFont font = QFontDialog::getFont( &ok, QFont( family, pointSize ), this );
	if ( ok ) {
		// font is set to the font the user selected
		family = font.family();
		pointSize = font.pointSize();
		string familyStr = family.latin1();
		preferencesMng->setMixerFontFamily(familyStr);
		preferencesMng->setMixerFontPointSize(pointSize);
	}
	QFont newFont(family, pointSize);
	mixerFontLbl->setFont(newFont);
	mixerFontLbl->setText(family + QString("  %1").arg(pointSize));
}





void PreferencesDialog::midiPortChannelChanged()
{
	needDriverRestart = true;
}


void PreferencesDialog::guiStyleChanged()
{
//	infoLog( "[guiStyleChanged]" );
	QApplication *pQApp = (HydrogenApp::getInstance())->getMainForm()->m_pQApp;
	QString sStyle = m_styleComboBox->currentText();
	pQApp->setStyle( sStyle );

	PreferencesMng *pPref = PreferencesMng::getInstance();
	pPref->setQTStyle( sStyle.latin1() );
}


